/*!\file: control_core.cpp
 * \brief: core of the control solution 
 */ 
#include <config.h>
#include "../toolkits/toolkits.h"
#include "../objects/objects.h"
#include "../shared/shared.h"
#include "../io/io.h"
#include "../EnumDefinitions/EnumDefinitions.h"
#include "./solutions.h"
#include "../modules/modules.h"
#include "../include/include.h"
#include "../solvers/solvers.h"

#if defined (_HAVE_TAO_) && (_PETSC_MAJOR_ == 3 && _PETSC_MINOR_ == 2)
#include <tao.h>

/*Local prototype*/
int FormFunctionGradient(TaoSolver tao,Vec,double*,Vec,void*);
typedef struct {
	FemModel* femmodel;
} AppCtx;

void controltao_core(FemModel* femmodel){

	/*TAO*/
	int       ierr,numberofvertices;
	AppCtx    user;
	TaoSolver tao;
	Vec       X  = NULL;
	Vec       XL = NULL;
	Vec       XU = NULL;

	/*Initialize TAO*/
	int argc; char **args=NULL;
	PetscGetArgs(&argc,&args);
	ierr = TaoInitialize(&argc,&args,(char*)0,"");
	if(ierr) _error_("Could not initialize Tao");

	/*Initialize TAO*/
	TaoCreate(PETSC_COMM_WORLD,&tao);
	PetscOptionsSetValue("-tao_monitor","");
	TaoSetFromOptions(tao);
	TaoSetType(tao,"tao_blmvm");

	/*Prepare all TAO parameters*/
	TaoSetMaximumFunctionEvaluations(tao,50);
	TaoSetMaximumIterations(tao,10);
	TaoSetTolerances(tao,10e-28,0.0000,0.0000,0.0000,10e-28);

	numberofvertices = femmodel->vertices->NumberOfVertices();
	XL=NewVec(numberofvertices); VecSet(XL,1.);
	XU=NewVec(numberofvertices); VecSet(XU,200.);
	TaoSetVariableBounds(tao,XL,XU); VecFree(&XL); VecFree(&XU);

	GetVectorFromInputsx(&X,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,FrictionCoefficientEnum,VertexEnum);
	TaoSetInitialVector(tao,X);

	user.femmodel=femmodel;
	TaoSetObjectiveAndGradientRoutine(tao,FormFunctionGradient,(void*)&user); 

	/*Solver optimization problem*/
	TaoSolve(tao);
	TaoView(tao,PETSC_VIEWER_STDOUT_WORLD);
	TaoGetSolutionVector(tao,&X);
	InputUpdateFromVectorx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,X,FrictionCoefficientEnum,VertexEnum);
	InputToResultx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,FrictionCoefficientEnum);

	/*Clean up and return*/
	VecFree(&X);
	TaoDestroy(&tao);
	TaoFinalize();
}
int FormFunctionGradient(TaoSolver tao, Vec X, double *fcn,Vec G,void *userCtx){

	/*Retreive arguments*/
	AppCtx   *user     = (AppCtx*)userCtx;
	FemModel *femmodel = user->femmodel;
	Vec       gradient = NULL;

	int costfunction=SurfaceAbsVelMisfitEnum;
	femmodel->parameters->SetParam(&costfunction,1,1,StepResponsesEnum);
	//InputUpdateFromVectorx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,X,MaterialsRheologyBbarEnum,VertexEnum);
	InputUpdateFromVectorx(femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,X,FrictionCoefficientEnum,VertexEnum);
	adjointdiagnostic_core(user->femmodel);
	//Gradjx(&gradient,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,MaterialsRheologyBbarEnum);
	Gradjx(&gradient,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters,FrictionCoefficientEnum);
	VecScale(gradient,-1.);
	VecCopy(gradient,G);
	VecFree(&gradient);
	CostFunctionx(fcn,femmodel->elements,femmodel->nodes,femmodel->vertices,femmodel->loads,femmodel->materials,femmodel->parameters);
	return 0;
}

#else
void controltao_core(FemModel* femmodel){
	_error_("TAO not installed or PETSc version not supported");
}
#endif //_HAVE_TAO_ 
