%PFE class definition
%
%   Usage:
%      cluster=greenplanet();
%      cluster=greenplanet('np',3);
%      cluster=greenplanet('np',3,'login','username');

classdef greenplanet
    properties (SetAccess=public)  
		 % {{{1
		 name='greenplanet'
		 login='';
		 numnodes=20;
		 cpuspernode=8; 
		 port=8000;
		 queue='rignot';
		 codepath='';
		 executionpath='';
		 interactive=0;
	 end
	 properties (SetAccess=private) 
		 np=20*8;
		 % }}}
	 end
	 methods
		 function cluster=greenplanet(varargin) % {{{1

			 %initialize cluster using default settings if provided
			 if (exist('greenplanet_settings')==2), greenplanet_settings; end

			 %use provided options to change fields
			 options=pairoptions(varargin{:});
			 for i=1:size(options.list,1),
				 fieldname=options.list{i,1};
				 fieldvalue=options.list{i,2};
				 if ismember(fieldname,properties('greenplanet')),
					 cluster.(fieldname)=fieldvalue;
				 else
					 disp(['''' fieldname ''' is not a property of cluster greenplanet']);
				 end
			 end
		 end
		 %}}}
		 function disp(cluster) % {{{1
			 %  display the object
			 disp(sprintf('class ''%s'' object ''%s'' = ',class(cluster),inputname(1)));
			 disp(sprintf('    name: %s',cluster.name));
			 disp(sprintf('    login: %s',cluster.login));
			 disp(sprintf('    port: %i',cluster.port));
			 disp(sprintf('    numnodes: %i',cluster.numnodes));
			 disp(sprintf('    cpuspernode: %i',cluster.cpuspernode));
			 disp(sprintf('    np: %i',cluster.cpuspernode*cluster.numnodes));
			 disp(sprintf('    queue: %s',cluster.queue));
			 disp(sprintf('    codepath: %s',cluster.codepath));
			 disp(sprintf('    executionpath: %s',cluster.executionpath));
			 disp(sprintf('    interactive: %i',cluster.interactive));
		 end
		 %}}}
		 function checkconsistency(cluster,md,solution,analyses) % {{{1

			 available_queues={'rignot','default'};
			 queue_requirements_time=[Inf Inf];
			 queue_requirements_np=[80 80];

			 QueueRequirements(available_queues,queue_requirements_time,queue_requirements_np,cluster.queue,cluster.np,1)

			 %Miscelaneous
			 if isempty(cluster.login), checkmessage('login empty'); end
			 if isempty(cluster.codepath), checkmessage('codepath empty'); end
			 if isempty(cluster.executionpath), checkmessage('executionpath empty'); end

		 end
		 %}}}
		 function BuildQueueScript(cluster,md) % {{{1

			 %retrieve parameters 
			 modelname=md.miscellaneous.name; 
			 solution=md.private.solution;
			 isvalgrind=md.debug.valgrind;

			 %compute number of processors
			 cluster.np=cluster.numnodes*cluster.cpuspernode;

			 %open file for writing: 
			 fid=fopen([modelname '.queue'],'w');

			 fprintf(fid,'#PBS -S /bin/bash\n');
			 fprintf(fid,'#PBS -N %s\n',modelname);
			 fprintf(fid,'#PBS -q %s \n',cluster.queue);
			 fprintf(fid,'#PBS -l nodes=%i:ppn=%i\n',cluster.numnodes,cluster.cpuspernode);
			 fprintf(fid,'#PBS -m bea\n');
			 fprintf(fid,'#PBS -M mmorligh@uci.edu\n');
			 fprintf(fid,'#PBS -o %s.outlog \n',modelname);
			 fprintf(fid,'#PBS -e %s.errlog \n\n',modelname);

			 fprintf(fid,'cd %s/%s\n\n',cluster.executionpath,md.private.runtimename);
			 fprintf(fid,'/sopt/mpi/mvapich-1.1/intel/bin/mpiexec -mpich-p4-no-shmem -np %i %s/issm.exe %s %s %s\n',cluster.np,cluster.codepath,EnumToString(solution),cluster.executionpath,modelname);

			 if ~md.settings.io_gather,
				 %concatenate the output files:
				 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
			 end

			 %close file
			 fclose(fid);

			 %in interactive mode, create a run file, and errlog and outlog file
			 if cluster.interactive,
				 fid=fopen([modelname '.run'],'w');
				 fprintf(fid,'mpiexec -np %i %s/issm.exe %s %s %s\n',cluster.np,cluster.codepath,EnumToString(solution),cluster.executionpath,modelname);

				 if ~md.settings.io_gather,
					 %concatenate the output files:
					 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
				 end
				 fclose(fid);
				 fid=fopen([modelname '.errlog'],'w');
				 fclose(fid);
				 fid=fopen([modelname '.outlog'],'w');
				 fclose(fid);
			 end
		 end %}}}
		 function LaunchQueueJob(cluster,md,options)% {{{1
			 
			 %lauch command, to be executed via ssh
			 if ~cluster.interactive, 
				launchcommand=['cd ' cluster.executionpath ' && rm -rf ./' md.private.runtimename ' && mkdir ' md.private.runtimename ...
			                ' && cd ' md.private.runtimename ' && mv ../' md.private.runtimename '.tar.gz ./ && tar -zxf ' md.private.runtimename '.tar.gz  && qsub ' md.miscellaneous.name '.queue '];
			else
				launchcommand=['cd ' cluster.executionpath '/Interactive' num2str(cluster.interactive) ' && tar -zxf ' md.private.runtimename '.tar.gz'];
			end

			if ~strcmpi(options.batch,'yes'),
				
				%compress the files into one zip.
				compressstring=['tar -zcf ' md.private.runtimename '.tar.gz ' md.miscellaneous.name '.bin ' md.miscellaneous.name '.queue '  md.miscellaneous.name '.petsc '];
				if md.qmu.isdakota,
					compressstring=[compressstring md.miscellaneous.name '.qmu.in '];
				end
				if cluster.interactive,
					compressstring=[compressstring md.miscellaneous.name '.run ' md.miscellaneous.name '.errlog ' md.miscellaneous.name '.outlog '];
				end
				system(compressstring);
				
				disp('uploading input file and queueing script');
				if cluster.interactive,
					directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive)];
				else 
					directory=cluster.executionpath;
				end
				
				issmscpout(cluster.name,directory,cluster.login,cluster.port,{[md.private.runtimename '.tar.gz']});
				
				disp('launching solution sequence on remote cluster');
				issmssh(cluster.name,cluster.login,cluster.port,launchcommand);

			else
				disp('batch mode requested: not launching job interactively');
				disp('launch solution sequence on remote cluster by hand');
			end
		 end
		 %}}}
		 function Download(cluster,md)% {{{1

			%some check
			if isempty(md.private.runtimename),
				if ~cluster.interactive,
					error('greenplanet Download error message: supply runtime name for results to be loaded!');
				end
			end

			%Figure out the  directory where all the files are in: 
			if ~cluster.interactive,
				directory=[cluster.executionpath '/' md.private.runtimename '/'];
			else
				directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive) '/'];
			end

			%What packages are we picking up from remote cluster
			if ~cluster.interactive,
				packages={[md.miscellaneous.name '.outlog'],[md.miscellaneous.name '.errlog']};
			else
				packages={};
			end
			if md.qmu.isdakota,
				packages{end+1}=[md.miscellaneous.name '.qmu.err'];
				packages{end+1}=[md.miscellaneous.name '.qmu.out'];
				if isfield(md.qmu.params,'tabular_graphics_data'),
					if md.qmu.params.tabular_graphics_data==true,
						packages{end+1}='dakota_tabular.dat'; 
					end
				end
			else
				packages{end+1}=[md.miscellaneous.name '.outbin'];
			end

			%copy files from cluster to present directory
			issmscpin(cluster.name, cluster.login, cluster.port, directory, packages);

		end %}}}
	end
end
