// Implementation of UPC related translation
// All things are inside the namespace by default. 
#include "rose.h"
#include "upc_translation.h" 
#include <algorithm>
#include <functional>      // For greater<int>( )
#include <boost/functional/hash.hpp>
#include <map>
#include <fstream>
#include <iostream>
#include <string>
using namespace std;
using namespace SageInterface;
using namespace SageBuilder;
namespace upcTranslation{
  // Transfer content of upcc-sizes generated by Berkeley UPC compiler driver
  // useful for cross compilation also
   StructCustomizedSizes upc_sizes;
  //! Add headers and preprocessor declarations
  void addHeadersAndPreprocessorDecls(SgGlobal* global_scope)
  {
    ROSE_ASSERT(global_scope);
    ROSE_ASSERT(upc_sizes.sz_shared_ptr!=0);// check if sizes are initialized

      //Add #define xx, not an ideal anchor point for it now
     buildCpreprocessorDefineDeclaration(global_scope, "#define UPCR_WANT_MAJOR 3",
                  PreprocessingInfo::before);
     buildCpreprocessorDefineDeclaration(global_scope, "#define UPCR_WANT_MINOR 6",
                 PreprocessingInfo::before);
     stringstream ss1,ss2;
     ss1<<upc_sizes.sz_shared_ptr;
     ss2<<upc_sizes.sz_pshared_ptr;
     buildCpreprocessorDefineDeclaration(global_scope, "#define UPCR_SHARED_SIZE_ "+ss1.str(),
                  PreprocessingInfo::before);
     buildCpreprocessorDefineDeclaration(global_scope, "#define UPCR_PSHARED_SIZE_ "+ss2.str(),
                  PreprocessingInfo::before);

  // Add UPCR system headers
     insertHeader("upcr.h",PreprocessingInfo::after,false,global_scope); 
   //  insertHeader("whirl2c.h",false,global_scope); 
     insertHeader("upcr_proxy.h",PreprocessingInfo::after,false,global_scope); 
     //insertHeader("upcr_geninclude/stdio.h",false,global_scope); 
  }

  //! Check if a variable declaration declares a thread local datum, including 
  //unshared global and static local variables
  // global extern variables are counted as TLD even though no translation is needed sometimes
  // At least in UPCR_INIT_xxx(), Adresses of other TLD count them 
  bool isThreadLocalDataDecl(SgVariableDeclaration* vardecl)
  {
    ROSE_ASSERT(vardecl!=NULL);
    SgVariableSymbol* varsymbol = getFirstVarSym(vardecl);
    SgType* t = varsymbol->get_type();
    bool isGlobal = isSgGlobal(vardecl->get_scope());
    return (!isUpcSharedType(t)) && (isGlobal || isStatic(vardecl));
  }

    //! Check if a variable reference is referring to a thread local data
  bool isThreadLocalDataRef(SgVarRefExp* exp)
  {
    ROSE_ASSERT(exp);
    SgVariableDeclaration* decl = isSgVariableDeclaration(
      exp->get_symbol()->get_declaration()->get_declaration());
    ROSE_ASSERT(decl);
    return isThreadLocalDataDecl(decl);
  } // isThreadLocalDataRef()


    //! Check if an initializer involves addresses of other TLD data, used for generating 
  // void UPCRI_INIT_filename_xxx()
  bool hasAddressOfOtherTLD (SgInitializer* initor)
  {
    ROSE_ASSERT(initor!=NULL);
    bool result = false;
    Rose_STL_Container<SgNode*> nodelist = NodeQuery::querySubTree(initor, V_SgAddressOfOp);
    for (Rose_STL_Container<SgNode*>::iterator i = nodelist.begin(); i!=nodelist.end();i++)
    {
      SgAddressOfOp * operation = isSgAddressOfOp(*i);
      // the operand could be an expression of other TLD, so search it again
      Rose_STL_Container<SgNode*> nodelist2 = NodeQuery::querySubTree(operation, V_SgVarRefExp);
      for (Rose_STL_Container<SgNode*>::iterator j = nodelist2.begin(); j!=nodelist2.end();j++)
      {
        SgVarRefExp * varref = isSgVarRefExp(*j);
        result = isThreadLocalDataRef(varref);
      } // variable ref list
    } // end for

    return result;
  } // hasAddressOfOtherTLD()


 //-----------------------------------------------------------------
  //! Insert macros and helper function prototypes: upcrt_gcd() etc.
  // extern int upcrt_gcd (int a, int b);
  /* 
   extern int _upcrt_forall_start(int start_thread, int step, int lo UPCRI_PT_ARG);
   #define upcrt_forall_start(start_thread, step, lo)  \
       _upcrt_forall_start(start_thread, step, lo UPCRI_PT_PASS)
   int32_t UPCR_TLD_DEFINE_TENTATIVE(upcrt_forall_control, 4, 4);
   #define upcr_forall_control upcrt_forall_control

     #ifndef UPCR_EXIT_FUNCTION
    #define UPCR_EXIT_FUNCTION() ((void)0)
    #endif

    #if UPCR_RUNTIME_SPEC_MAJOR > 3 || (UPCR_RUNTIME_SPEC_MAJOR == 3 && UPCR_RUNTIME_SPEC_MINOR >= 8)
    #define UPCRT_STARTUP_SHALLOC(sptr, blockbytes, numblocks, mult_by_threads, elemsz, typestr) \
      { &(sptr), (blockbytes), (numblocks), (mult_by_threads), (elemsz), #sptr, (typestr) }
    #else
    #define UPCRT_STARTUP_SHALLOC(sptr, blockbytes, numblocks, mult_by_threads, elemsz, typestr) \
      { &(sptr), (blockbytes), (numblocks), (mult_by_threads) }
    #endif
    #define UPCRT_STARTUP_PSHALLOC UPCRT_STARTUP_SHALLOC
 
    TODO
    extern int _upcrt_forall_start (int start_thread, int step,
    int lo UPCRI_PT_ARG);
    #define upcrt_forall_start(start_thread, step, lo)  \
       _upcrt_forall_start(start_thread, step, lo UPCRI_PT_PASS)
     int32_t UPCR_TLD_DEFINE_TENTATIVE (upcrt_forall_control, 4, 4);
    #define upcr_forall_control upcrt_forall_control

   */
  void addHelperFunctionPrototypes(SgFile* cur_file)
  {
    SgSourceFile* src_file = isSgSourceFile(cur_file);
    if (src_file==NULL) return;
    SgGlobal* global_scope = src_file->get_globalScope();
    pushScopeStack(global_scope);
    SgInitializedName* arg1 = buildInitializedName(SgName("a"),buildIntType());
    SgInitializedName* arg2 = buildInitializedName(SgName("b"),buildIntType());
    SgFunctionParameterList * paraList = buildFunctionParameterList();
    appendArg(paraList,arg1);  
    appendArg(paraList,arg2);  
    SgFunctionDeclaration * func1 = buildNondefiningFunctionDeclaration \
       ("upcrt_gcd",buildIntType(),paraList);
    SgStatement* old_first_stmt = getFirstStatement(global_scope);
    prependStatement (func1);
    func1->get_declarationModifier().get_storageModifier().setExtern();
    popScopeStack();
    // We treat the macros as arbitrary string for simplicity
    string macros="";
    macros+="extern int _upcrt_forall_start(int start_thread, int step, int lo UPCRI_PT_ARG);\n#define upcrt_forall_start(start_thread, step, lo)  \\\n\t_upcrt_forall_start(start_thread, step, lo UPCRI_PT_PASS)\nint32_t UPCR_TLD_DEFINE_TENTATIVE(upcrt_forall_control, 4, 4);\n#define upcr_forall_control upcrt_forall_control\n";
    macros+="#ifndef UPCR_EXIT_FUNCTION\n#define UPCR_EXIT_FUNCTION() ((void)0)\n#endif\n";
    macros+="#if UPCR_RUNTIME_SPEC_MAJOR > 3 || (UPCR_RUNTIME_SPEC_MAJOR == 3 && UPCR_RUNTIME_SPEC_MINOR >= 8)\n#define UPCRT_STARTUP_SHALLOC(sptr, blockbytes, numblocks, mult_by_threads, elemsz, typestr) \\\n\t{ &(sptr), (blockbytes), (numblocks), (mult_by_threads), (elemsz), #sptr, (typestr) }\n#else\n#define UPCRT_STARTUP_SHALLOC(sptr, blockbytes, numblocks, mult_by_threads, elemsz, typestr) \\\n\t{ &(sptr), (blockbytes), (numblocks), (mult_by_threads) }\n#endif\n#define UPCRT_STARTUP_PSHALLOC UPCRT_STARTUP_SHALLOC\n"; 
    // attach them after the statement, ensure it is after #include ... sequence
    attachArbitraryText(func1,macros,PreprocessingInfo::after);
    
    if (old_first_stmt) // prepend to global scope, but after all #include 
    { 
#if 0      
      cout<<"Find a preexist statement in the global scope, with VariantT:"
     <<old_first_stmt->sage_class_name() 
     << " unparseToString: "
     <<old_first_stmt->unparseToString()<<endl;
#endif
       moveUpPreprocessingInfo(func1,old_first_stmt);
     }  
     else
     {
       // cout<<"Debug:Cannot find a preexist statement in the global scope!"<<endl;
     }

  } // addHelperFunctionPrototypes

 //----------------------------------------------------
  //! Change output file's suffix to .c when necessary 
  //! Consider from .upc to .c for now
   void setOutputFileName(SgFile* cur_file)
   {
     ROSE_ASSERT(cur_file);
      string orig_name = cur_file->get_file_info()->get_filenameString(); 
      //cout<<"orig file name is :"<<orig_name<<endl;
      string file_suffix = StringUtility::fileNameSuffix(orig_name);
      if (CommandlineProcessing::isUPCFileNameSuffix(file_suffix))
      {  
        orig_name = StringUtility::stripPathFromFileName(orig_name);
        string naked_name = StringUtility::stripFileSuffixFromFileName(orig_name);
        cur_file->set_unparse_output_filename("rose_"+naked_name+".c");
        string output_name =  cur_file->get_unparse_output_filename();
        //cout<<"output file name is :"<<output_name<<endl;
       }
   } // setOutputFileName()


 //----------------------------------------------------
  //! Translation driver
  void translationDriver::visit(SgNode* node)
  {
    switch (node->variantT())
    {
      case V_SgGlobal:
      {
        // Should be moved to the point right before the postorder traversal for translation
        // since top-down translation is used in transStaticallyAllocatedData()
        // to prepare statically allocated global variables.
        //
        // Otherwise, there will be undesired conflicts between the postorder transVarRefExp()
        // and the top down transStaticallyAllocatedData()
        //transStaticallyAllocatedData(node);
        break;
      }
      case V_SgUpcMythread:
      {
        transUpcMythread(node);
        break;
      }
       case V_SgUpcThreads:
      {
        transUpcThreads(node);
        break;
      }
       case V_SgUpcBarrierStatement:
      {
        transUpcBarrier(node);
        break;
      }
       case V_SgVarRefExp:
       {
         transVarRefExp(node);
         break;
       }
       case V_SgAssignOp:
       {
         transAssignOp(node);
         break;
       }
       case V_SgFunctionDefinition:
       {
         addUpcrBeginAndExitFunctions(isSgFunctionDefinition(node));
         break;
       }
       case V_SgProject:
       {
         //cout<<"Debug: sgproject is found!!!!~~~~~~~~~~~"<<endl;
         break;
       }
     default:
      {
        // do nothing here    
      }
    }// switch
         
  }//translationDriver::visit()

 //----------------------------------------------------
  //! Replace it with a function call to upcr_mythread()
  // example case: printf (.., MYTHREAD) with AST 
  // SgFunctionCallExp (SgFunctionRefExp, SgExprListExp)
  // SgExprListExp(..., SgUpcMythread)
  void transUpcMythread(SgNode * node)
  {
    SgUpcMythread * sg_mythread = isSgUpcMythread(node);
    ROSE_ASSERT(sg_mythread);
    //cout<<"Found a SgUpcMythread node!"<<endl;
    
    //build unsigned int upcr_mythread();
    SgScopeStatement* cur_scope = SageInterface::getScope(node); 
    ROSE_ASSERT(cur_scope);
    pushScopeStack(cur_scope);
    SgType* return_type = buildUnsignedIntType();
    SgFunctionCallExp* call_exp = buildFunctionCallExp("upcr_mythread",
                           return_type, buildExprListExp());

   // replace variable ref to call exp
    replaceExpression(sg_mythread,call_exp); 

    popScopeStack(); 
  } //transUpcMythread

 //----------------------------------------------------
  //! Replace it with a function call to upcr_threads()
  void transUpcThreads(SgNode * node)
  {
    SgUpcThreads * sg_node = isSgUpcThreads(node);
    ROSE_ASSERT(sg_node);
   // cout<<"Found a SgUpcMythread node!"<<endl;
    
    //build unsigned int upcr_threads();
    SgScopeStatement* cur_scope = SageInterface::getScope(node); 
    ROSE_ASSERT(cur_scope);
    pushScopeStack(cur_scope);
    SgType* return_type = buildUnsignedIntType();
    SgFunctionCallExp* call_exp = buildFunctionCallExp("upcr_threads",
                           return_type, buildExprListExp());
   // replace variable ref to call exp
    replaceExpression(sg_node,call_exp); 
    popScopeStack(); 
  } //transUpcMythread

 //----------------------------------------------------
  //! Replace upc_barrier exp_opt with upcr_barrier(barrierval, int flag);
  //! flag ==0 means barrierval matters, otherwise it is anonymous barrier.
  void transUpcBarrier(SgNode * node)
  {
    SgUpcBarrierStatement * sg_node = isSgUpcBarrierStatement(node);
    ROSE_ASSERT(sg_node);
    //cout<<"Found a SgUpcBarrierStatment node!"<<endl;
    SgScopeStatement* cur_scope = SageInterface::getScope(node); 
    ROSE_ASSERT(cur_scope);

    pushScopeStack(cur_scope);
    SgExpression* exp = sg_node->get_barrier_expression();
    SgExpression* arg1 = NULL;
    SgExpression* arg2 = NULL;

    if (exp)
    {  
      arg1 = copyExpression(exp);  // should not share expressions in AST
      arg2 = buildIntVal(0);
    }
      else // anonymous barrier
    {
      arg1 = buildIntVal(UPCR_BARRIERVAL_ANONYMOUS);
      arg2 = buildIntVal(UPCR_BARRIERFLAG_ANONYMOUS);
    }
    SgExprListExp * arg_list = buildExprListExp();
    appendExpression(arg_list, arg1);
    appendExpression(arg_list, arg2);
    SgType* return_type = buildVoidType();
    SgExprStatement* call_stmt = buildFunctionCallStmt("upcr_barrier",
                           return_type, arg_list);
    replaceStatement(sg_node,call_stmt);   
    popScopeStack();
  } // transUpcBarrier

 //----------------------------------------------------
  //! Add UPCR_BEGIN_FUNCTION () and  UPCR_EXIT_FUNCTION() into function definitions
   void addUpcrBeginAndExitFunctions(SgFunctionDefinition* funcdef)
   { 
     ROSE_ASSERT(funcdef!=NULL);

     // prepend UPCR_BEGIN_FUNCTION () 
     SgBasicBlock * body = funcdef->get_body();
     SgExprStatement* call_stmt = buildFunctionCallStmt 
            ("UPCR_BEGIN_FUNCTION",buildVoidType(), buildExprListExp(),body); 
     prependStatement(call_stmt, body);       
     
     // instrument UPCR_EXIT_FUNCTION() 
     //if (isMain(funcdef->get_declaration()))
      { 
       SgExprStatement* call_stmt2 =  buildFunctionCallStmt
            ("UPCR_EXIT_FUNCTION",buildVoidType(),buildExprListExp(),body);
       int result = instrumentEndOfFunction(funcdef->get_declaration(), call_stmt2); 
       ROSE_ASSERT(result >0);
     }
   } // addUpcrBeginAndExitFunctions

 //----------------------------------------------------
   //! Rename main() to extern  user_main()
   void renameMainToUserMain(SgFunctionDeclaration* sg_func)
   {
     ROSE_ASSERT(isMain(sg_func)); 

     // grab symbol before any modifications. 
     SgGlobal* global_scope= isSgGlobal(sg_func->get_scope());
     ROSE_ASSERT(global_scope);
     SgFunctionSymbol * symbol = global_scope->lookup_function_symbol
            (SgName("main"),sg_func->get_type());
     ROSE_ASSERT(symbol == sg_func->get_symbol_from_symbol_table());
     global_scope->remove_symbol(symbol);
     delete (symbol); // avoid dangling symbol!!

     // rename it
     SgName new_name = SgName("user_main"); 
     sg_func->set_name(new_name);
     sg_func->get_declarationModifier().get_storageModifier().setExtern();

     //handle function symbol:remove the original one, insert a new one
     symbol = new SgFunctionSymbol(sg_func);
     global_scope->insert_symbol(new_name,symbol);
   } // renameMainToUserMain()

 //----------------------------------------------------
 //! A startup function for runtime library, call user_main() internally
   void generateNewMainFunction(SgFile* sg_file)
   {
     SgSourceFile* src_file = isSgSourceFile(sg_file);
     ROSE_ASSERT(src_file!=NULL);// should not apply to binary files
     SgGlobal* global_scope = src_file->get_globalScope();
     // some global variables
     // const char * UPCRL_main_name = "user_main";
     SgInitializer * init =  buildAssignInitializer(buildStringVal("user_main"));
     SgType* var_type = buildConstType(buildCharType());

     SgVariableDeclaration * var_decl = buildVariableDeclaration
        ("UPCRL_main_name",buildPointerType(var_type),init,global_scope);
        
     appendStatement(var_decl,global_scope);   
     
     // int main( int arc, char** argc)
     SgInitializedName* arg1 = buildInitializedName("argc", buildIntType()); 
     SgType* type2= buildPointerType(buildPointerType(buildCharType()));
     SgInitializedName* arg2 = buildInitializedName("argv", type2);
     SgFunctionParameterList * paraList = buildFunctionParameterList();
     appendArg(paraList, arg1);  
     appendArg(paraList, arg2);

     SgFunctionDeclaration * func = buildDefiningFunctionDeclaration
             ("main",buildIntType(),paraList,global_scope);
     appendStatement(func,global_scope); 

     // fill main body:
     SgBasicBlock* body = func->get_definition()->get_body();

      //bupc_init_reentrant(&argc, &argv, &user_main);
      SgExpression * bupc_arg1 = buildAddressOfOp(buildVarRefExp("argc",body));
      SgExpression * bupc_arg2 = buildAddressOfOp(buildVarRefExp("argv",body));
      SgExpression * bupc_arg3 = buildAddressOfOp(buildFunctionRefExp("user_main",body));
      SgExprListExp* arg_list = buildExprListExp();
      appendExpression(arg_list,bupc_arg1);
      appendExpression(arg_list,bupc_arg2);
      appendExpression(arg_list,bupc_arg3);

      SgExprStatement* stmt1 = buildFunctionCallStmt
          ("bupc_init_reentrant",buildVoidType(),arg_list,body);
      appendStatement(stmt1, body);

      //return 0;
     SgReturnStmt * stmt2 = buildReturnStmt(buildIntVal(0));
     appendStatement(stmt2, body);
   } //generateNewMainFunction()

  #if 0
  //! Internal function to help sorting SgNodes by their memory addresses
  static bool SgNode_lesser (SgNode* decl_1, SgNode* decl_2)
  {
    return (long) decl_1 <  (long) decl_2;
  }
  #endif

//----------------------------------------
 //! Translate shared variables: replace them with generic shared pointers, generate allocation and initialization functions.
 /*! Handling of statically-allocated shared data (SSD), 
  *  global shared data + function scope static shared data
  *  find shared ones of different types scalar/pointer & arrays
  *     x. private to shared 
  * They are divided into two categories: regular and phase-less shared variables
  * a. phase-less: blocksize ==1 or blocksize == 0 or blocksize == unspecified
  * b. regular : the rest
  */
  int transStaticallyAllocatedData(SgNode* global)
  {
    SgGlobal * global_scope = isSgGlobal(global); 
    ROSE_ASSERT(global_scope);

    Rose_STL_Container<SgVariableDeclaration*> decl_list;
    decl_list = querySubTree<SgVariableDeclaration>(global, V_SgVariableDeclaration); 

    // One list for shared data: SSD
    // One list for all unshared data: statically allocated unshared data,
    Rose_STL_Container<SgVariableDeclaration*> ssd_list, tld_list,global_ssd_list, local_ssd_list;
    
    // collect global shared declarations 
    Rose_STL_Container<SgVariableDeclaration*>::iterator iter; 
    for (iter = decl_list.begin();iter != decl_list.end(); iter++)
    { 
      SgVariableDeclaration * var_decl = isSgVariableDeclaration(*iter);
      ROSE_ASSERT(var_decl != NULL );

      bool isGlobal = isSgGlobal(var_decl->get_scope());  // local or global?
      //ROSE AST has only one variable in a declaration. 
      SgType* t = getFirstVarType(isSgVariableDeclaration(var_decl));

      if (isUpcSharedType(t))
      {  
        ssd_list.push_back(var_decl);
        if (isGlobal) 
          global_ssd_list.push_back(var_decl);
        else
          local_ssd_list.push_back(var_decl);
      }  
      else
      { 
        if (isGlobal)
          tld_list.push_back(var_decl);
        else // only static local or private-to-shared local variables matter
          if (isStatic(var_decl)||isUpcPrivateToSharedType(t)) 
            tld_list.push_back(var_decl);
      }
    } //for decl_list

   //--------------allocation function-----------------------
   // global and local shared
     
   // invoke for each file/SgGlobal (global_scope)?
    SgFunctionDeclaration* func_alloc = generateUpcriAllocFunc(
      isSgFile(global_scope->get_parent()),global_ssd_list, local_ssd_list,global_scope);
    appendStatement(func_alloc,global_scope);
      // insert its prototype at the beginning
#if 0     // not necessary
    SgFunctionDeclaration* func_alloc2 = buildNondefiningFunctionDeclaration(func_alloc,global_scope);
    SgStatement* old_first_stmt = getFirstStatement(global_scope);
    prependStatement(func_alloc2,global_scope);
    if (old_first_stmt) // prepend to global scope, but after all #include 
         moveUpPreprocessingInfo(func_alloc2,old_first_stmt,PreprocessingInfo::before);
#endif
 
   //---------------initialization function-------------------
   //Generate TLD data initializations before generating their declarations using
   //  transThreadLocalData(), which will remove the original decls 
   //  needed by the initialization process
    SgFunctionDeclaration* func_init = generateUpcriInitFunc(
      isSgFile(global_scope->get_parent()),ssd_list, tld_list,global_scope);
    appendStatement(func_init,global_scope);
      // insert its prototype at the beginning
#if 0     // not necessary
    SgFunctionDeclaration* func_init2 = buildNondefiningFunctionDeclaration(func_init,global_scope);
    old_first_stmt = getFirstStatement(global_scope);
    prependStatement(func_init2,global_scope);
    if (old_first_stmt) // prepend to global scope, but after all #include 
       moveUpPreprocessingInfo(func_init2,old_first_stmt,PreprocessingInfo::before);
#endif
    // handle SSD variables
    transStaticallyAllocatedSharedData(ssd_list);

   // handle TLD variable declarations
    transThreadLocalData (tld_list,global_scope); 

 
   //Finally, fix symbols referenced since we generate var ref before their declarations
    fixVariableReferences(global_scope);
    
    return decl_list.size();
  } // transStaticallyAllocatedData()

  /*!
   * This function implements the translation of Statically-allocated shared Data (SSD), 
   * defined in The Berkeley UPC Runtime Specification V 3.9. It returns the number of 
   * variables handled. 
   */
  int transStaticallyAllocatedSharedData(Rose_STL_Container<SgVariableDeclaration*> ssd_list)
  {
    if (ssd_list.size()==0) return 0;

    SgGlobal * global_scope = getGlobalScope(*(ssd_list.begin())); 
    ROSE_ASSERT(global_scope);
   // Generate upcr_pshared_ptr_t var or upcr_shared_ptr_t var
   // and insert them into the global scope
     // prepare types 
    SgType* regular_type = lookupNamedTypeInParentScopes("upcr_shared_ptr_t",global_scope);
    SgType* phaseLess_type = lookupNamedTypeInParentScopes("upcr_pshared_ptr_t",global_scope);

    if (regular_type==NULL) 
      regular_type = buildOpaqueType("upcr_shared_ptr_t",global_scope);
    if( phaseLess_type == NULL ) 
      phaseLess_type = buildOpaqueType("upcr_pshared_ptr_t",global_scope);

      // handle them one by one
    Rose_STL_Container<SgVariableDeclaration*>::iterator iter2; 
   for (iter2 = ssd_list.begin(); iter2 != ssd_list.end(); iter2++) 
   {
     SgVariableDeclaration* orig_decl = *iter2;
     SgVariableDeclaration* new_decl = NULL;

     bool isGlobal = isSgGlobal(orig_decl->get_scope());  // local or global?
     bool hasInitializer = 
        (getFirstVarSym(orig_decl)->get_declaration()->get_initptr ()==NULL)?false:true;

     SgName name;
     if (isGlobal)
       name = getFirstVarSym(orig_decl)->get_name();
     else
       name = generateUniqueGlobalName(orig_decl);

     if (isUpcPhaseLessSharedType(getFirstVarType(orig_decl)))
     {
       if (hasInitializer)
         new_decl = buildVariableDeclaration(name, phaseLess_type, \
         buildAssignInitializer(buildOpaqueVarRefExp("UPCR_INITIALIZED_PSHARED",global_scope)), \
         global_scope );
       else
       {  // Only global ones need handling for now 
         //if (isGlobal) // not if a local one has been referenced later on
           new_decl = buildVariableDeclaration(name, phaseLess_type,NULL, global_scope );
       }  
     }
     else  // regular shared types  
     {
       if (hasInitializer)
         new_decl = buildVariableDeclaration(name, regular_type, \
        buildAssignInitializer(buildOpaqueVarRefExp("UPCR_INITIALIZED_SHARED",global_scope)), \
        global_scope );
       else
         // if (isGlobal)
           new_decl = buildVariableDeclaration(name, regular_type,NULL, global_scope );
     }
     if (new_decl)
     {
       if (isGlobal) // // insert before the original place for global ones
       {
         insertStatementBefore(orig_decl, new_decl);
         moveUpPreprocessingInfo(new_decl,orig_decl);
       }
       else
        //local SSD's global version  needs to be inserted as the last one in the 
        //decl stmt sequence to avoid undeclared global variables in the initializer         
       {
          SgStatement* anchor_stmt = findFirstDefiningFunctionDecl(global_scope);  
          if (anchor_stmt)
              insertStatementBefore(anchor_stmt, new_decl);
          else
             appendStatement(new_decl,global_scope);
       }
       removeStatement(orig_decl);
     } // if (new_decl)

   } // for ssd_list
    return ssd_list.size();
  }
   /*!
   * This function implements the translation of Statically-allocated unshared Data, 
   * namely Thread-Local Data (TLD)
   * defined in The Berkeley UPC Runtime Specification V 3.9. 
   *
   * We extend the TLD list to contain non-static local pointer-to-shared variables
   * since they demand the similar translation, although they are not TLD variables.
   */
  int transThreadLocalData(Rose_STL_Container<SgVariableDeclaration*> tld_list,SgGlobal* global_scope)
  {
    ROSE_ASSERT(global_scope);
    ofstream tld_file; 
    string orig_file_name= global_scope->get_file_info()->get_filenameString();   
    string tld_file_name =StringUtility::stripFileSuffixFromFileName(
                            StringUtility::stripPathFromFileName(orig_file_name));
    tld_file_name+="_obj.tld";
    //tld_file_name="global.tld";

   // cerr<<"Debug: tld file name is :"<<tld_file_name<<endl;
    tld_file.open(tld_file_name.c_str(),ios::app);

    tld_file<<"UPCR_TLD_DEFINE(upcrt_forall_control, 4, 4)"<<"\n";
    Rose_STL_Container<SgVariableDeclaration*>::iterator iter; 
   for (iter = tld_list.begin(); iter != tld_list.end(); iter++) 
   {
     SgVariableDeclaration* decl = *iter;
     SgVariableDeclaration* vardecl = isSgVariableDeclaration(decl);
     SgVariableSymbol* varsymbol = getFirstVarSym(vardecl);
     string varname = varsymbol->get_name().str();
     SgType* t = varsymbol->get_type();
     SgInitializer * initor = varsymbol->get_declaration()->get_initptr();
     string srcstr; // we create a source string here since they have non-compilable macros.
     bool isGlobal = isSgGlobal(vardecl->get_scope());
     bool isp2s = isUpcPrivateToSharedType(t);
     bool isphaseless = isUpcPhaseLessSharedType(t);
     bool isArrayType = isSgArrayType(t);
     bool isFunctionPointer = false;
     bool hasNullValue = false;
     if (isSgPointerType(t))
       if (isSgFunctionType(isSgPointerType(t)->get_base_type()))
         isFunctionPointer = true;
 
    // prepare types 
    SgType* regular_type = lookupNamedTypeInParentScopes("upcr_shared_ptr_t",global_scope);
    SgType* phaseLess_type = lookupNamedTypeInParentScopes("upcr_pshared_ptr_t",global_scope);
    if (regular_type==NULL) 
      regular_type = buildOpaqueType("upcr_shared_ptr_t",global_scope);
    if( phaseLess_type == NULL ) 
      phaseLess_type = buildOpaqueType("upcr_pshared_ptr_t",global_scope);

    if (isGlobal&&isExtern(vardecl)&&(!isp2s)) 
     //No action needed for extern global variables which are not private pointers to shared
     //e.g. extern int quux;
        continue;
    else if ((isGlobal&&isExtern(vardecl) 
                ||!isGlobal&&!isStatic(vardecl)) 
             &&isp2s) 
     // Handle special cases for private to shared pointers
     //  a)  global: must have extern modifier 
     //  b)  local: must not be static , 
      transSpecialPrivate2Shared(vardecl);
    else if (isGlobal||(isStatic(vardecl))) 
     // static local variables without initializers also have global version! 
     // The rest TLD variables, include private to shared pointers
     // Global variables which are not shared
     // a) global private to shared pointer variables
     //   upcr_shared_ptr_t UPCR_TLD_DEFINE_TENTATIVE (p2s_p1, 8, 4);
     //   upcr_pshared_ptr_t UPCR_TLD_DEFINE (p2s_p2, 8, 4) = UPCR_INITIALIZED_PSHARED;
     // b)regular global variables 
     //  int UPCR_TLD_DEFINE_TENTATIVE(counter, 4, 4);
     //  int UPCR_TLD_DEFINE(counter2, 4, 4) = 100;
     // c) array and pointer to function types need one more type definition
     //     typedef double _type_myarray[10];
     //     _type_myarray UPCR_TLD_DEFINE_TENTATIVE(myarray, 80, 4);
     // The recommended translation using macro is put as source comment 
     // to 'fool' Berkeley UPC compiler's linker script to get list of TLD_DEFINE items
     // We generate a legal declaration to facilitate analysis of AST
    //
     // Similar handling for local TLD 
     //   a) Must be static variables with initializers
     { 
       SgTypedefDeclaration* typedef_decl = NULL;
       string typedef_name, tldstr;
       stringstream ss,ss2;
       SgTypedefType * typedef_type =NULL;

       //Extra typedef declaration for array types
       //typedef double _type__N9_fooArray2_N10_1665815976_[2];
       // _type__N9_fooArray2_N10_1665815976_ UPCR_TLD_DEFINE(_N9_fooArray2_N10_1665815976_, 16, 4) = {3.1,1.3,};
         if (isArrayType||isFunctionPointer)
       { 
         string prefix = "_type_";
         if (isFunctionPointer) 
           prefix = "_funptr_";
         if (isGlobal)
           typedef_name = prefix+varname; 
         else  
           typedef_name = prefix +generateUniqueGlobalName(vardecl);  // local only
         typedef_decl = buildTypedefDeclaration(typedef_name,t);
         typedef_type = typedef_decl->get_type();
       }
      
       if (isp2s)
       {  
         if (isphaseless)
           srcstr = "upcr_pshared_ptr_t";
         else
           srcstr = "upcr_shared_ptr_t"; 
       }
       else if (isArrayType||isFunctionPointer)// scalar or array types are handled differently
         srcstr = typedef_name;
       else
         srcstr = t->unparseToString();

       srcstr += " UPCR_TLD_DEFINE";  
       if (initor==NULL)
         srcstr += "_TENTATIVE (";
        else
         srcstr += " (";
       // file_obj.tld always has  UPCR_TLD_DEFINE(..) 
       tldstr = "UPCR_TLD_DEFINE(";  
        if (isGlobal) 
        {
          srcstr += varname;  
          tldstr += varname;
        }
        else
        {
          srcstr +=generateUniqueGlobalName(vardecl);  
          tldstr += generateUniqueGlobalName(vardecl); 
        }
        srcstr += ", ";
        tldstr += ", ";
        ss<<customsizeof(varsymbol->get_type());
        srcstr += ss.str();
        tldstr += ss.str();
        srcstr += ", ";
        tldstr += ", ";
        ss2<<customalignof(varsymbol->get_type());
        srcstr += ss2.str() + " )";
        tldstr += ss2.str() + " )";
        if (initor)
        {  
          if (isp2s) // pointer to shared cases
          {
            SgAssignInitializer * assign_initor = isSgAssignInitializer(initor);
            if (assign_initor)
            {
              hasNullValue = isNullValued(assign_initor->get_operand());
              if (hasNullValue)
                srcstr+=" = UPCR_NULL_SHARED";
            }
            else
            {
              if (isphaseless)
                srcstr+=" = UPCR_INITIALIZED_PSHARED";
              else 
                srcstr+=" = UPCR_INITIALIZED_SHARED";
            }  
          } 
          else // regular cases
            srcstr+= " = "+initor->unparseToString();
        }
        srcstr += ";";  
        // write file_obj.tld 
         tld_file<<tldstr.c_str()<<"\n";
        // Generate the real variable declaration in our translation
        // new_type name = initializer;
        // distinguish between null value and others
        SgType* new_type=NULL;//regular_type;
       if (isUpcPhaseLessSharedType(t))
          new_type = phaseLess_type;
        else 
          new_type = regular_type;

        SgInitializer* initializer = NULL;
        if (initor ==NULL) 
          initializer = NULL;
        else if (isp2s)
        {
          if (hasNullValue)
          {
            if (isUpcPhaseLessSharedType(t))
            {
               initializer = buildAssignInitializer(
                 buildOpaqueVarRefExp("UPCR_NULL_PSHARED",global_scope));
            }
             else
             {
               initializer = buildAssignInitializer(
                 buildOpaqueVarRefExp("UPCR_NULL_SHARED",global_scope));
             }
          }
          else
          {
            if (isUpcPhaseLessSharedType(t))
            {
              initializer = buildAssignInitializer(
                  buildOpaqueVarRefExp("UPCR_INITIALIZED_PSHARED",global_scope));
            }
            else
            {
              initializer= buildAssignInitializer(
                  buildOpaqueVarRefExp("UPCR_INITIALIZED_SHARED",global_scope));
             }
           }

        } // end if (isp2s)
        else  if (!isp2s) // adjust for regular global variables
        {
          new_type = t;
          initializer = deepCopy<SgInitializer>(initor);
        }

        if (isArrayType||isFunctionPointer) // adjust for array and function pointer types
          new_type = typedef_type;
        if (!isGlobal) // global variable for local static variable needs mangled name 
          varname = generateUniqueGlobalName(vardecl);

        SgVariableDeclaration* new_decl = buildVariableDeclaration(varname, new_type,initializer, global_scope);
        attachComment(new_decl,srcstr,PreprocessingInfo::after);

        // different insert point for local and global TLD variables
        //SgStatement* anchor_stmt = getFirstStatement(global_scope);  
        if (isGlobal) // insert before the original place for global ones
        {
          insertStatementBefore(decl, new_decl);
          moveUpPreprocessingInfo(new_decl,decl,PreprocessingInfo::before);
        }
        else
        //local TLD need to be inserted as the last one in the decl stmt sequence 
        //to avoid undeclared global variables being used in the initializer
        {
           SgStatement* anchor_stmt = findFirstDefiningFunctionDecl(global_scope);  
           if (anchor_stmt)
              insertStatementBefore(anchor_stmt, new_decl);
            else
              appendStatement(new_decl,global_scope);
        }

        //keep the base defining type declaration like: union xxx {} a;
        // TODO refactor it to a function
        SgDeclarationStatement* base_decl = decl->get_baseTypeDefiningDeclaration(); 
        if (base_decl)
         { 
          ROSE_ASSERT(base_decl);
          SgClassDeclaration* classDecl = isSgClassDeclaration (base_decl); 
          if (classDecl)
          {
            SgClassDeclaration* newdecl2 = deepCopy<SgClassDeclaration>(classDecl);     
            ROSE_ASSERT(newdecl2);
            setOneSourcePositionForTransformation(newdecl2);
            newdecl2->set_definingDeclaration(classDecl->get_definingDeclaration());
            newdecl2->set_firstNondefiningDeclaration(
                     classDecl->get_firstNondefiningDeclaration());
            // insert the type declaration newdecl2  
            // before the use of the type in the new declaration new_decl;
            insertStatementBefore(new_decl, newdecl2);
            moveUpPreprocessingInfo(newdecl2,new_decl,PreprocessingInfo::before);
          }
          else {
           //We only handle struct, union for now, may need enumerate later
            cout<<"Unhandled base type declaration type:"<<base_decl->sage_class_name()<<endl;
            ROSE_ASSERT(false);
           }
         } // end if for keeping base defining type declaration

        removeStatement(decl);
        if (isArrayType||isFunctionPointer) // insert typedef declaration for array or function pointer types
        {
          insertStatementBefore(new_decl,typedef_decl);
          moveUpPreprocessingInfo(typedef_decl,new_decl,PreprocessingInfo::before);
        }
     } // end if  for eligible variables for handling 
     // global 
   } // for
   // return the actual statements processed
   tld_file.close();
   return tld_list.size();
  } // transThreadLocalData()

   /*! Per file UPCRI allocation function for shared variables
    *  Remember to insert the returned function declaration to the right position
    *  of the scope after calling this function. 
    *
    * e.g.: 
  void
  UPCRI_ALLOC_flename_3447388659 ()
  {
    UPCR_BEGIN_FUNCTION ();
    upcr_startup_shalloc_t info[] = {
      UPCRT_STARTUP_SHALLOC (array, 40, 20, 1, 8, "A100H_R5_d"),
    };
    upcr_startup_pshalloc_t pinfo[] = {
      UPCRT_STARTUP_PSHALLOC (_N9_scounter2_N10_2477035566_, 4, 1, 0, 4,
                              "R1_i"),
    };

    UPCR_SET_SRCPOS ("_filename_3447388659_ALLOC", 0);
    upcr_startup_shalloc (info,
                          sizeof (info) / sizeof (upcr_startup_shalloc_t));
    upcr_startup_pshalloc (pinfo,
                           sizeof (pinfo) / sizeof (upcr_startup_pshalloc_t));
  }
   */
  SgFunctionDeclaration * generateUpcriAllocFunc(SgFile* file, 
                    std::vector<SgVariableDeclaration*>global_list, 
                    std::vector<SgVariableDeclaration*>local_list,
                    SgScopeStatement* scope)
  {
    string fullfilename, filename; 
    boost::hash<std::string> string_hash;
    fullfilename = file->get_file_info()->get_filename();
    size_t file_num =  string_hash(fullfilename);
    // cout<<"file name is: "<<file->get_file_info()->get_filename()<<endl;
    // get file name without path and suffix
    filename = StringUtility::stripFileSuffixFromFileName(
               StringUtility::stripPathFromFileName(fullfilename)); 
    stringstream ss;
    ss<<file_num;

    string func_name = "UPCRI_ALLOC_"+filename+"_"+ss.str();
    
    SgFunctionDeclaration* result = buildDefiningFunctionDeclaration (func_name, 
          buildVoidType(), buildFunctionParameterList(),scope);

    SgBasicBlock * body = result->get_definition()->get_body();
 
    // handled in addUpcrBeginAndExitFunctions() now
    //SgExprStatement* begin_stmt = buildFunctionCallStmt 
     //         ("UPCR_BEGIN_FUNCTION",buildVoidType(), buildExprListExp(),body); 
    //prependStatement(begin_stmt, body);       
    // ------------ prepare regular and phaseless shared variable lists-----
    vector<SgVariableDeclaration*> regular_list, phaseless_list;
    vector<SgVariableDeclaration*>::iterator iter1, iter2;

        // collect static local shared variables 
    for (iter1 = local_list.begin(); iter1 != local_list.end(); iter1 ++)
    {
      SgVariableDeclaration* decl = * iter1;
      //bool hasInitializer = (getFirstInitializedName(decl)->get_initptr ()==NULL)?false:true;
      // Ideally, local shared variables without initializers should not be allocated
      // if they are not used later. We simply it by allocating them all. 
      //if (hasInitializer)  
      {  
        if  (isUpcPhaseLessSharedType(getFirstVarType(decl)))
        {
          phaseless_list.push_back(decl);
          //cout<<"found local phaseless shared with initializer...."<<endl;
        }
        else
        {
          regular_list.push_back(decl);
          //cout<<"found local regular shared with initializer...."<<endl;
        }  
      }  
    } // end for local list
        // collect global shared variables
    for (iter2 = global_list.begin(); iter2 != global_list.end(); iter2 ++)
    {
      SgVariableDeclaration* decl = * iter2;
      if  (isUpcPhaseLessSharedType(getFirstVarType(decl)))
      {  
       phaseless_list.push_back(decl);
       // cout<<"found global phaseless shared var...."<<endl;
      } 
      else
      {  
        regular_list.push_back(decl);
       // cout<<"found global regular shared var...."<<endl;
      }
    } // end for local list

    // ------------------regular shared--------------------------------
    // upcr_startup_shalloc_t info[] = {a,b,c,..};
    // SgAggregateInitializer -->SgExprListExp -> SgAssignInitializer

    SgAggregateInitializer * initializer = generateUpcriAllocFuncInitializer(regular_list,false, body); 
    bool hasRegElements=true, hasPhaselessElements=true; 
    if (initializer->get_initializers()->get_expressions ().size()==0)
      hasRegElements= false;
    if (hasRegElements)  
    {
      SgVariableDeclaration* info_decl = buildVariableDeclaration
      ("info", buildArrayType(buildOpaqueType("upcr_startup_shalloc_t",body)),initializer,body);  
      appendStatement(info_decl, body); 
    }
    // --------------------phaseless shared-----------------------------
    //upcr_startup_pshalloc_t pinfo[] = {...};
    
    SgAggregateInitializer * initializer2 = generateUpcriAllocFuncInitializer(phaseless_list,true, body); 
    if (initializer2->get_initializers()->get_expressions ().size()==0)
      hasPhaselessElements= false;
    if (hasPhaselessElements)  
    {
     
      SgVariableDeclaration* pinfo_decl = buildVariableDeclaration
        ("pinfo", buildArrayType(buildOpaqueType("upcr_startup_pshalloc_t",body)),
          initializer2,body);  
      appendStatement(pinfo_decl, body); 
    } 
    // -----------------------Other statements--------------------------
    // UPCR_SET_SRCPOS ("_filename_3447388659_ALLOC", 0);
    SgExprListExp* arg_list = buildExprListExp();
    appendExpression(arg_list, buildStringVal("_"+filename+"_"+ss.str()+"_ALLOC")); 
    appendExpression(arg_list, buildIntVal(0)); 
    SgExprStatement* src_stmt = buildFunctionCallStmt
         ("UPCR_SET_SRCPOS", buildVoidType(), arg_list, body);
    appendStatement(src_stmt, body);       

    //upcr_startup_shalloc (info, sizeof (info) / sizeof (upcr_startup_shalloc_t));
    if (hasRegElements)
    {
      SgExprListExp* shalloc_args = buildExprListExp();
      appendExpression(shalloc_args, buildVarRefExp("info",body));
      SgDivideOp * ss_arg2 = buildDivideOp(buildSizeOfOp(buildVarRefExp("info",body)),
                  buildSizeOfOp(buildOpaqueType("upcr_startup_shalloc_t",body)));
      appendExpression(shalloc_args, ss_arg2);
      SgExprStatement* ss_stmt = buildFunctionCallStmt
           ("upcr_startup_shalloc", buildVoidType(), shalloc_args, body);
      appendStatement(ss_stmt, body);   
    }

    if (hasPhaselessElements)
    {
      //upcr_startup_pshalloc (pinfo, sizeof (pinfo) / sizeof (upcr_startup_pshalloc_t));
      SgExprListExp* pshalloc_args = buildExprListExp();
      appendExpression(pshalloc_args, buildVarRefExp("pinfo",body));
      SgDivideOp * sp_arg2 = buildDivideOp(buildSizeOfOp(buildVarRefExp("pinfo",body)),
                  buildSizeOfOp(buildOpaqueType("upcr_startup_pshalloc_t",body)));
      appendExpression(pshalloc_args, sp_arg2);
      SgExprStatement* sp_stmt = buildFunctionCallStmt
           ("upcr_startup_pshalloc", buildVoidType(), pshalloc_args, body);
      appendStatement(sp_stmt, body);   
    }

    return result;
  } // generateUpcriAllocFunc()

  //!Generate the UPCRI_INIT_filename_xxx() function for per-file initialization of
  //shared and unshared (TLD) data
  /* Per-file initialization for  shared and unshared(TLD) data
  *
   */
  SgFunctionDeclaration* generateUpcriInitFunc(SgFile* file, 
                    std::vector<SgVariableDeclaration*>shared_list, 
                    std::vector<SgVariableDeclaration*>tld_list, 
                    SgScopeStatement* scope)
  {
    string fullfilename, filename; 
    boost::hash<std::string> string_hash;
    SgSourceFile* srcfile = isSgSourceFile(file);
    ROSE_ASSERT(srcfile!=NULL);
    fullfilename = srcfile->get_file_info()->get_filename();
    SgGlobal* global_scope = srcfile->get_globalScope();
    size_t file_num =  string_hash(fullfilename);
    filename = StringUtility::stripFileSuffixFromFileName(
               StringUtility::stripPathFromFileName(fullfilename)); 
    stringstream ss;
    ss<<file_num;
    //   void UPCRI_INIT_filename_xxx()  {}
    string func_name = "UPCRI_INIT_"+filename+"_"+ss.str();
    SgFunctionDeclaration* result = buildDefiningFunctionDeclaration (func_name, 
          buildVoidType(), buildFunctionParameterList(),scope);

    SgBasicBlock * body = result->get_definition()->get_body();
       //handled in addUpcrBeginAndExitFunctions() now
       //UPCR_BEGIN_FUNCTION ();
    //SgExprStatement* begin_stmt = buildFunctionCallStmt 
      //        ("UPCR_BEGIN_FUNCTION",buildVoidType(), buildExprListExp(),body); 
   // prependStatement(begin_stmt, body);       

       // UPCR_SET_SRCPOS ("_filename_3447388659_INIT", 0);
    SgExprListExp* arg_list = buildExprListExp();
    appendExpression(arg_list, buildStringVal("_"+filename+"_"+ss.str()+"_INIT")); 
    appendExpression(arg_list, buildIntVal(0)); 
    SgExprStatement* src_stmt = buildFunctionCallStmt
         ("UPCR_SET_SRCPOS", buildVoidType(), arg_list, body);
    appendStatement(src_stmt, body);      
    //generate initialization statements for shared declarations
    generateUpcriInitStmtForShared(shared_list,body, global_scope);

    // generate initialization statements for TLD declarations
     generateUpcriInitStmtForTLD (tld_list,body,global_scope); 
     return result;
  } // generateUpcriInitFunc()

  //!Generate the initialization statements for shared variables and insert them into  UPCRI_INIT_filename_xxx() 
  /*!
   * two categories for shared variables:
   *   1) scalar: only thread 0 does the initialization
   *     int32_t _N11_lsscounter1_N8_63317268__val = 0;
   *     if (upcr_mythread () == 0)
   *     {
   *       upcr_put_pshared (_N11_lsscounter2_N10_3305040721_, 0,
   *       &_N11_lsscounter2_N10_3305040721__val, 4);
   *     }
   *   2) arrays: special macro: upcr_startup_initarray();
 
   */
  void generateUpcriInitStmtForShared(std::vector<SgVariableDeclaration*>shared_list,
                       SgBasicBlock* func_body,SgScopeStatement* global_scope)
  {
     ROSE_ASSERT(global_scope);
     Rose_STL_Container<SgVariableDeclaration*>::iterator iter; 
     for (iter = shared_list.begin(); iter != shared_list.end(); iter++) 
     {
       SgVariableDeclaration* vardecl = *iter;
       SgVariableSymbol* varsymbol = getFirstVarSym(vardecl);
       string varname = varsymbol->get_name().str();
       SgType* t = varsymbol->get_type();
       SgInitializer * initor = varsymbol->get_declaration()->get_initptr();
       bool isGlobal = isSgGlobal(vardecl->get_scope());
       bool isArrayType = isSgArrayType(t);
       bool isphaseless = isUpcPhaseLessSharedType(t);
      if (initor) 
       {  
         if (isArrayType)
         {//TODO
           
           ROSE_ASSERT(false);
         } 
         else // scalar case
         {
           SgModifierType* modifier_type=NULL;
           hasUpcSharedType(t,&modifier_type);
           ROSE_ASSERT(modifier_type); 
           SgType* base_type = modifier_type->get_base_type();
           string var_val_name;
           string global_var_name;
           if (isGlobal)
             global_var_name = varname;
           else  
             global_var_name= generateUniqueGlobalName(vardecl);
           var_val_name =  global_var_name+"_val";
           //int32_t _N11_lsscounter1_N8_63317268__val = 0;
           SgVariableDeclaration* decl = buildVariableDeclaration(var_val_name,
                     base_type, deepCopy<SgInitializer>(initor), func_body);  
           appendStatement(decl,func_body);

            // if (upcr_mythread() ==0 ) {...}
            SgFunctionCallExp *func_call= buildFunctionCallExp("upcr_mythread",buildIntType(),
                       buildExprListExp(),global_scope); 
            SgExprStatement* condition = buildExprStatement(
              buildEqualityOp(func_call,buildIntVal(0))); 
            SgBasicBlock* true_body = buildBasicBlock();
            SgIfStmt* ifstmt = buildIfStmt(condition,true_body,NULL);
            appendStatement(ifstmt,func_body);
            //void upcr_put_shared (upcr_shared_ptr_t dest,  ptrdiff_t destoffset, 
            //              const void *src, size_t nbytes);
            SgExprListExp* arg_list = buildExprListExp();
            appendExpression(arg_list,buildVarRefExp(global_var_name,global_scope));
            appendExpression(arg_list,buildIntVal(0));
            appendExpression(arg_list,buildAddressOfOp(buildVarRefExp(decl)));
            appendExpression(arg_list,buildIntVal(sizeof(base_type)));
            string func_name;
            if (isphaseless)
              func_name = "upcr_put_pshared";
            else  
              func_name = "upcr_put_shared";
            SgExprStatement* func_call_stmt = buildFunctionCallStmt(func_name,
                  buildVoidType(),arg_list,global_scope);
            appendStatement(func_call_stmt,true_body);
         }
       } // end if has initializer
     }// end for loop

  } //generateUpcriInitStmtForShared()

  /* TLD: two cases
   *   1) initialization involving addresses of other TLD data: 
   *      int* pcounter= &counter; // is translated into -->
   *      *((int* *) UPCR_TLD_ADDR(pcounter)) = (int* )UPCR_TLD_ADDR(counter);
   *   2) declaration for local pointer to shared data: 
   *     shared int* p2s_p2 = & shareddata; // is translated into-->
   *     *((upcr_pshared_ptr_t *) UPCR_TLD_ADDR(p2s_p2)) = shareddata;
   */
   void generateUpcriInitStmtForTLD(std::vector<SgVariableDeclaration*>tld_list,
                       SgBasicBlock* body,SgScopeStatement* global_scope)
   {
     ROSE_ASSERT(global_scope);
     SgType* shared_type;
     // prepare types 
     SgType* regular_type = lookupNamedTypeInParentScopes("upcr_shared_ptr_t",global_scope);
    SgType* phaseLess_type = lookupNamedTypeInParentScopes("upcr_pshared_ptr_t",global_scope);
    if (regular_type==NULL) 
      regular_type = buildOpaqueType("upcr_shared_ptr_t",global_scope);
    if( phaseLess_type == NULL ) 
      phaseLess_type = buildOpaqueType("upcr_pshared_ptr_t",global_scope);

        Rose_STL_Container<SgVariableDeclaration*>::iterator iter; 
       for (iter = tld_list.begin(); iter != tld_list.end(); iter++) 
     {
       SgVariableDeclaration* vardecl = *iter;
       SgVariableSymbol* varsymbol = getFirstVarSym(vardecl);
       string varname = varsymbol->get_name().str();
       SgType* t = varsymbol->get_type();
       SgInitializer * initor = varsymbol->get_declaration()->get_initptr();
       string srcstr; // we create a source string here since they have non-compilable macros.
       bool isGlobal = isSgGlobal(vardecl->get_scope());
       bool isp2s = isUpcPrivateToSharedType(t);
       bool isphaseless = isUpcPhaseLessSharedType(t);
       bool hasNullValue = false;
       if (!isGlobal) 
         varname = generateUniqueGlobalName(vardecl);
       if (isphaseless) 
         shared_type = phaseLess_type;
       else
         shared_type = regular_type;
       // a) involves addresses of other TLD: AddessOf 
       // *((int* *) UPCR_TLD_ADDR(pcounter)) = &(*(int* )UPCR_TLD_ADDR(counter));
       if (initor) 
       {  
         if (isp2s) // pointer to shared cases
         {
            SgAssignInitializer * assign_initor = isSgAssignInitializer(initor);
            if (assign_initor)
                hasNullValue = isNullValued(assign_initor->get_operand());
         }
         if (hasAddressOfOtherTLD(initor)) 
          {
           //cout<<"Found address of other TLD:"<<initor->unparseToString()<<endl;
           SgType* return_type = buildPointerType(buildVoidType());
           SgFunctionCallExp * leftFunc= buildFunctionCallExp("UPCR_TLD_ADDR",return_type,
              buildExprListExp(buildVarRefExp(varname,vardecl->get_scope())), global_scope);
           SgType* cast_type = buildPointerType(t); //type **
           SgExpression* lefthand = buildPointerDerefExp(buildCastExp(leftFunc,cast_type));
           // There may be complex cases like = &tld1 +&tld2*&tld3
           SgAssignInitializer *initializer = isSgAssignInitializer(initor);
           ROSE_ASSERT(initializer);
           SgExpression* righthand=copyExpression(initializer->get_operand());
           // We defer the variable substitution for the right hand until 
           // the postorder processing by transVarRefExp() 
           SgExprStatement* assign_stmt = buildAssignStatement(lefthand,righthand);
           appendStatement(assign_stmt, body);      
          }
         //b) pointer to shared data: global and static local 
         //   shared int * p2s_p2 = &gsj ; // --translated into--->
         // *((upcr_pshared_ptr_t *) UPCR_TLD_ADDR(p2s_p2)) = *(&gsj);
         // we use *(... &gsj..) since the initializer could be expressions of &gsj
          else if ((isp2s) && (isGlobal||isStatic(vardecl))) 
          // we extend tld list to have some unstatic local pointer-2-shared variables, 
          // should exclude them here
          {
            if (!isp2s||!hasNullValue)// ignore p2s with 0 value !(isp2s&&hasNullValue)
            {
             SgType* return_type = buildPointerType(buildVoidType());
             // local variables need to be replaced to be their global versions
             //if (!isGlobal) 
             //  varname = SgName(generateUniqueGlobalName(vardecl));
             SgFunctionCallExp * leftFunc= buildFunctionCallExp("UPCR_TLD_ADDR",return_type,
                buildExprListExp(buildVarRefExp(varname,vardecl->get_scope())), global_scope);
             SgType* cast_type = buildPointerType(shared_type); 
             SgExpression* lefthand = buildCastExp(leftFunc,cast_type);

           SgAssignInitializer *initializer = isSgAssignInitializer(initor);
           ROSE_ASSERT(initializer);
           SgExpression* righthand=copyExpression(initializer->get_operand());
           SgExprStatement* assign_stmt = buildAssignStatement(
                               buildPointerDerefExp(lefthand),buildPointerDerefExp(righthand));
           appendStatement(assign_stmt, body);      
          }
        } 
     }//end if (initor)
    } // end for TLD_list 
   }

  //! TODO read upcc-sizes file from UPC driver or configuration script
  // to set up ABI and values
  void initUpcTranslation()
  {
    upc_sizes.str_abi="x86"; // default to x86 (i386, ILP32)
    upc_sizes.sz_shared_ptr= 8;
    upc_sizes.sz_alignof_shared_ptr= 4;
    upc_sizes.sz_pshared_ptr= 8;
    upc_sizes.sz_alignof_pshared_ptr= 4;
    upc_sizes.sz_mem_handle= 4;
    upc_sizes.sz_alignof_mem_handle= 4;
    upc_sizes.sz_reg_handle= 4;
    upc_sizes.sz_alignof_reg_handle=4;
    upc_sizes.sz_void_ptr= 4;
    upc_sizes.sz_alignof_void_ptr= 4;
    upc_sizes.sz_ptrdiff_t= 4;
    upc_sizes.sz_alignof_ptrdiff_t=4;

    // implied by ABI, redundant here actually
    upc_sizes.sz_char=1;
    upc_sizes.sz_alignof_char=1;
    upc_sizes.sz_int=4;
    upc_sizes.sz_alignof_int=4;
    upc_sizes.sz_short=2;
    upc_sizes.sz_alignof_short=2;
    upc_sizes.sz_long=4;
    upc_sizes.sz_alignof_long=4;
    upc_sizes.sz_longlong=8;
    upc_sizes.sz_alignof_longlong=4;
    upc_sizes.sz_float=4;
    upc_sizes.sz_alignof_float=4;
    upc_sizes.sz_double=8;
    upc_sizes.sz_alignof_double=4;
    upc_sizes.sz_longdouble=12;
    upc_sizes.sz_alignof_longdouble=4;
 
    upc_sizes.sz_size_t=4;
    upc_sizes.sz_alignof_size_t=4;
    upc_sizes.sz_alignof_dbl_1st=0;
    upc_sizes.sz_alignof_int64_1st=0;
    upc_sizes.sz_alignof_sharedptr_1st =0;
    upc_sizes.sz_alignof_psharedptr_1st =0;
    upc_sizes.sz_alignof_dbl_innerstruct=0;
    upc_sizes.sz_alignof_int64_innerstruct=0;
    upc_sizes.sz_alignof_sharedptr_innerstruct =0;
    upc_sizes.sz_alignof_psharedptr_innerstruct= 0;
    upc_sizes.sz_maxblocksz= 4194304;
 /* NOT used yet, put it into another structure later  
    upc_sizes.UPCRConfig="VERSION=2.6.0,PLATFORMENV=shared,SHMEM=pthreads,SHAREDPTRREP=packed,TRANS=berkeleyupc,debug,nogasp,notv";
    upc_sizes.GASNetConfig="RELEASE=1.10.0,SPEC=1.8,CONDUIT=SMP(SMP-1.7/REFERENCE-1.10),THREADMODEL=PAR,SEGMENT=FAST,PTR=32bit,align,debug,trace,stats,srclines,timers_native,membars_native,atomics_native,atomic32_native,atomic64_native";
    upc_sizes.runtime_spec="3.9";
    upc_sizes.upc_header_dir="/upcr_preinclude/";
    upc_sizes.use_type_interface=false;
    upc_sizes.system_header_dirs=":";
  */  
  } 

  //! Get size of a type, using customized UPC fundamental sizes
  size_t customsizeof(SgType* t)
  {
    ROSE_ASSERT(upc_sizes.sz_int!=0); // make sure upc_sizes has been initialized
    CustomizedPrimitiveTypeLayoutGenerator gen1_upc(NULL,&upc_sizes);
    NonpackedTypeLayoutGenerator gen_upc(&gen1_upc);
    return gen_upc.layoutType(t).size;
  }

  //! Get alignment of a type, using customized UPC fundamental sizes
  size_t customalignof(SgType* t)
  {
    ROSE_ASSERT(upc_sizes.sz_alignof_int!=0); // make sure upc_sizes has been initialized
    CustomizedPrimitiveTypeLayoutGenerator gen1_upc(NULL,&upc_sizes);
    NonpackedTypeLayoutGenerator gen_upc(&gen1_upc);
    return gen_upc.layoutType(t).alignment;
  }


  //! Get size of an element type
  size_t getElementSize(SgType* t)
  {
    size_t result = customsizeof(t); 
    if (isSgArrayType(t))
    {
      result = customsizeof(getArrayElementType(isSgArrayType(t)));
    }
    return result;  
  }
 
  //! Get number of blocks of a UPC shared type
  size_t getNumberOfBlocks(SgType* t)
  {
    size_t size = customsizeof(t);
    size_t block_bytes = getBlockBytes(t);
    return (size % block_bytes ==0 )? (size/block_bytes):(size/block_bytes +1);

  }

  //! Get the size of a block for a UPC type
  size_t getBlockBytes(SgType* t)
  {
    size_t block_bytes;
    size_t size = customsizeof(t);
    size_t element_size = getElementSize(t);
    ROSE_ASSERT(isUpcSharedType(t));
    block_bytes = SageInterface::getUpcSharedBlockSize(t);
    if (!isSgArrayType(t) || block_bytes == 0)
      { block_bytes = size; }
    else
      {block_bytes = min (block_bytes*element_size, size);}
    return block_bytes;  
  }

  //! Is a type is a UPC shared array with THREADS related dimension?
  bool hasUpcThreadsDimension(SgType* t)
  {
    if(!isUpcSharedType(t)) return false;
    if (isSgArrayType(t))
        if (isUpcArrayWithThreads(isSgArrayType(t)))
          return true;
    return false;      
  }

  SgAggregateInitializer*
  generateUpcriAllocFuncInitializer(std::vector<SgVariableDeclaration*>decl_list, 
           bool phaseless, SgScopeStatement* scope)
  {
     vector<SgVariableDeclaration*>::iterator iter1, iter2;
        // upcr_startup_shalloc_t info[] = {a,b,c,..};
        // upcr_startup_pshalloc_t info[] = {a,b,c,..};
    // SgAggregateInitializer -->SgExprListExp -> SgAssignInitializer
    SgExprListExp * exp_list = buildExprListExp(); 
    for (iter1 = decl_list.begin();iter1 != decl_list.end(); iter1++)
    {
      SgVariableDeclaration * decl = *iter1;
      SgInitializedName * var = getFirstInitializedName(decl);
      SgType* t = var->get_type();
      SgName name = var->get_name();
      // get the global version of static local shared variables with initializers
      if (isStatic(decl)&&isUpcSharedType(t)) 
        if (decl->get_scope()!=getGlobalScope(decl))
         // if (var->get_initializer()!=NULL) 
         // we need this actually if the variable is used later
            name = SgName(generateUniqueGlobalName(decl));

      SgExprListExp* para_list = buildExprListExp();
      // UPCRT_STARTUP_SHALLOC (array, 40, 20, 1, 8, "A100H_R5_d"),
  // shared pointer, block bytes, number of blocks, multi_by_threads, elementsize, type string
      //shaky here: the var decl will be removed later actually, 
      //rename them should be a better solution. 
      SgVarRefExp* arg1 = buildVarRefExp(name, scope);

      appendExpression(para_list,arg1);
      
      SgIntVal* arg2 = buildIntVal(getBlockBytes(t));
      appendExpression(para_list,arg2);

      SgIntVal* arg3 = buildIntVal(getNumberOfBlocks(t));
      appendExpression(para_list,arg3);

      int multiByThreads=0;
      if (hasUpcThreadsDimension(t)) multiByThreads = 1;
      SgIntVal* arg4 = buildIntVal(multiByThreads);
      appendExpression(para_list,arg4);

      SgIntVal* arg5 = buildIntVal(getElementSize(t));
      appendExpression(para_list,arg5);

      string mangled_type_name = mangleType(t);
      appendExpression(para_list, buildStringVal(mangled_type_name));
      std::string startup_name="UPCRT_STARTUP_SHALLOC";
      if (phaseless) startup_name="UPCRT_STARTUP_PSHALLOC";
      // assume return int to match the opaque int type of the array
      SgFunctionCallExp* callExp = buildFunctionCallExp
           (startup_name,buildIntType(), para_list,scope);
      // append function call expression into aggregate initializer
      SgAssignInitializer* initor2 =  buildAssignInitializer(callExp);
       // must not have () around to enable correct macro expansion!!  
      //initor2->set_need_paren(false);
      appendExpression(exp_list,initor2);

    }

    SgAggregateInitializer * initializer = buildAggregateInitializer(exp_list); 
    return initializer;

  } //generateUpcriAllocFuncInitializer()

  //!Algorithm is: _OrigName_Hash(Mangled_name(OrigName))_
  std::string generateUniqueGlobalName(SgVariableDeclaration* decl)
  {
    boost::hash<std::string> string_hash;
    std::string result;
    SgName orig_name =  getFirstInitializedName(decl)->get_name();
    SgName mangled_name = getFirstInitializedName(decl)->get_mangled_name();
    size_t hash_num = string_hash(orig_name.getString());
    stringstream ss;
     ss<<hash_num;
    result ="_"+orig_name.getString()+"_"+ss.str()+"_";
    return result;
  }

  //! Check if a SgVarRefExp is a parameter of a function call expression named UPCR_TLD_ADDR
  // used to avoid duplicated variable substitution for shared variable accesses
  bool isParameterOfFunctionCall(const std::string &funcName, SgVarRefExp* varRef)
  {
    bool result = false;
    ROSE_ASSERT(varRef!=NULL);
    SgExprListExp* exp_list = isSgExprListExp(varRef->get_parent());
    if (exp_list)
    {  
      SgFunctionCallExp* callExp = isSgFunctionCallExp(exp_list->get_parent());
      if (callExp)
       {
         string funcName0= isSgFunctionRefExp(callExp->get_function())->
                 get_symbol()->get_name().getString();
         if (funcName0.compare(funcName)==0) result = true; 
       }
    }
    return result;
  } 


  /*! 
   * Translate a variable reference expression, mostly replace it with another expression
   * Variable substitution: 
   * for thread-local data (TLD): treat as pointer deference  
   *   global regular variable
   *   *((int *) UPCR_TLD_ADDR (counter))
   *   TODO array references
   *   Exclude initializer in global declaration
   *   int *UPCR_TLD_DEFINE (pcounter, 4, 4) = &counter;
   *                                     not = &( *((int *)(UPCR_TLD_ADDR(counter))));
   * Skip the var ref already inside UPCR_TLD_ADDR() to avoid duplicated substitution!!
   */
  void transVarRefExp(SgNode* node) 
  {
    SgVarRefExp* exp = isSgVarRefExp(node);
    SgVarRefExp* exp2 = NULL;
    SgVariableDeclaration* vardecl=NULL;
    ROSE_ASSERT(exp);
    SgGlobal* global_scope = getGlobalScope(node);
    ROSE_ASSERT(global_scope);
    

    // skip the one already translated into runtime library calls or macro calls
    if (isParameterOfFunctionCall("UPCR_TLD_ADDR",exp)||
        isParameterOfFunctionCall("UPCRT_STARTUP_PSHALLOC",exp)|| 
        isParameterOfFunctionCall("upcr_put_shared",exp)|| 
        isParameterOfFunctionCall("UPCR_PUT_SHARED_VAL",exp)|| 
        isParameterOfFunctionCall("upcr_put_pshared",exp)|| 
        isParameterOfFunctionCall("UPCR_PUT_PSHARED_VAL",exp)|| 
        isParameterOfFunctionCall("UPCR_GET_PSHARED",exp)|| 
        isParameterOfFunctionCall("UPCR_GET_SHARED",exp)|| 
        isParameterOfFunctionCall("UPCRT_STARTUP_SHALLOC",exp)) 
      return;
    
    SgExpression* new_exp=NULL;
    
    // var_scope is the declaration scope of the referenced variable
    // varexp_scope is the enclosing scope of the current node. 
    vardecl = isSgVariableDeclaration(exp->get_symbol()->get_declaration()->get_declaration());
    ROSE_ASSERT(vardecl!=NULL);
    SgScopeStatement * var_scope = exp->get_symbol()->get_declaration()->get_scope();
    SgScopeStatement* varexp_scope = getScope(node);
    //bool isGlobal = isSgGlobal(vardecl);
    SgVariableSymbol* varsymbol = getFirstVarSym(vardecl);
    string varname = varsymbol->get_name().str();    

    // unshared, regular scalar or local static scalar (mangled global name) 
    // The expression types remain after removing the original shared declarations??
    SgType* t = exp->get_type();
#if 0    
    // should not do this replacement here , moved to transAssignOp()
    // postorder traversal will crash when replacing a parent node (assignop) of the varRef
    // with the function call exp
    //---------------------- shared variables----------------------
    if (isUpcSharedType(t)&&(varexp_scope!=global_scope))
    {
       bool isphaseless = isUpcPhaseLessSharedType(t);
       bool isArrayType = isSgArrayType(t);
       bool isWritten = false; // if the varRefExp is the left hand of an assignment stmt
       SgExpression * rhs = NULL;
      // write references
      SgAssignOp *assignop = isSgAssignOp(exp->get_parent());
      if (assignop)
      {
        //.........
      } // end if (assignop)
    } else
#endif    
    //------------------------TLD cases ----------------------------
    //if (isScalarType(t)&&!isUpcSharedType(t))
    if ((!isUpcSharedType(t))&&(varexp_scope!=global_scope))
      if ((var_scope == global_scope) || 
          (isStatic(exp->get_symbol()->get_declaration()->get_declaration())))
      {
        //cout<<"Found unshared static/global variables!!"<<endl;
        SgType* return_type = buildPointerType(buildVoidType());
        // must use deepcopy here, otherwise its parent pointer will be changed to exprList 
        // The replacement later on won't operate on the right parent!!
        if (var_scope==global_scope) 
          exp2 = isSgVarRefExp(copyExpression(exp));
        else // local TLD, use reference to its global version
        { 
          exp2= buildVarRefExp(generateUniqueGlobalName(vardecl),global_scope);
        }
        ROSE_ASSERT(exp2!=NULL);

        SgFunctionCallExp*  call_exp = buildFunctionCallExp("UPCR_TLD_ADDR",return_type,
              buildExprListExp(exp2), global_scope);
        new_exp = buildPointerDerefExp(buildCastExp(call_exp,buildPointerType(t)));
        //cout<<"new exp is:"<< new_exp->unparseToString() <<endl;
        //replace expression , keep old one used as a function parameter now 
        replaceExpression(exp, new_exp);
        //replaceExpression(exp, new_exp,true);
      }  // end if
  } //transVarRefExp()

  //!Translate an assignment operations
  /*!
   * Cases are:
   * a)  lhs is a shared variable being written:
   *    lsscounter=99; //--> is translated into
   *    UPCR_PUT_PSHARED_VAL(_N10_lsscounter_N9_781042953_, 0, 99, 4);
   */
  void transAssignOp(SgNode* node)
  {
    SgAssignOp *assignop = isSgAssignOp(node);
    ROSE_ASSERT(assignop);
    SgGlobal* global_scope = getGlobalScope(node);
    ROSE_ASSERT(global_scope);

    SgVarRefExp* exp = isSgVarRefExp(assignop->get_lhs_operand());
    SgExpression* rhs = assignop->get_rhs_operand();
    if (rhs==NULL)
    {
      cerr<<"Error:empty rhs of an assignment is not allowed"<<endl;
      cerr<<"assign op is:"<<assignop->unparseToString()<<endl;
      ROSE_ASSERT(rhs);
    }
    if (exp) // if left hand is a variable reference expression
    {
      // var_scope is the declaration scope of the referenced variable
      // varexp_scope is the enclosing scope of the current node. 
      SgVariableDeclaration* vardecl = isSgVariableDeclaration(exp->get_symbol()->get_declaration()->get_declaration());
      ROSE_ASSERT(vardecl!=NULL);
      //SgScopeStatement * var_scope = exp->get_symbol()->get_declaration()->get_scope();
      SgScopeStatement* varexp_scope = getScope(node);
      bool isGlobal = isSgGlobal(vardecl);
      SgVariableSymbol* varsymbol = getFirstVarSym(vardecl);
      string varname = varsymbol->get_name().str();    

      SgType* t = exp->get_type();
      if (isUpcSharedType(t)&&(varexp_scope!=global_scope))
      {
        bool isphaseless = isUpcPhaseLessSharedType(t);
        bool isArrayType = isSgArrayType(t);
        string global_name;
        if (isGlobal) // scope of original shared declaration
          global_name = varname;
        else
          global_name = generateUniqueGlobalName(vardecl);
        // base type for shared scalar variables  
        SgModifierType* modifier_type=NULL;
        hasUpcSharedType(t,&modifier_type);
        ROSE_ASSERT(modifier_type); 
        SgType* base_type = modifier_type->get_base_type();

        if (isArrayType)
        {
          cerr<<"Error: unhandled array types when translating a variable reference to a shared array"<<endl;
          ROSE_ASSERT(false);
        }
        else
        {
          SgExprListExp* arg_list = buildExprListExp();
          appendExpression(arg_list,buildVarRefExp(global_name,global_scope));
          appendExpression(arg_list,buildIntVal(0));
          appendExpression(arg_list,deepCopy<SgExpression>(rhs));
          appendExpression(arg_list,buildIntVal(sizeof(base_type)));
          string func_name;
          if (isphaseless)
             func_name = "UPCR_PUT_PSHARED_VAL";
          else  
             func_name = "UPCR_PUT_SHARED_VAL";
          SgFunctionCallExp* func_call_exp = buildFunctionCallExp(func_name,
                    buildVoidType(),arg_list,global_scope);
          replaceExpression(assignop,func_call_exp);
        }//scalar case
       } // end if shared variable ref inside a local scope
    } // if lhs is a varRefExp
  } //transAssignOp()
 
   //! Linking stage processing: insert per-file alloc/init functions etc.
  void transProject(SgNode* node)
  {
    SgProject * project = isSgProject(node);
    SgFilePtrList filelist = project->get_fileList();
    //printf("Number of files: %zu \n", filelist.size());
    //No action for single file project
    if (filelist.size()<=1)
      return; 
    vector<SgFile*>  src_files;
    SgFile* target_file=NULL;
    // collect source(anything else) and target (bupc_magic.c) files
    for (SgFilePtrList::iterator iter = filelist.begin();iter!= filelist.end();iter++)
    {
      SgFile* file = isSgFile(*iter);
      ROSE_ASSERT(file);
      string filename =  file->get_file_info()->get_filename();
      string file_no_path = StringUtility::stripPathFromFileName(filename);
      if (file_no_path == "bupcr_magic.c")
        target_file = file;
      else
        src_files.push_back(file);
    }//end for

    if (target_file)
    {
      ROSE_ASSERT(src_files.size()>0);
      SgSourceFile* target_src_file = isSgSourceFile(target_file);
      ROSE_ASSERT(target_src_file!=NULL); //should only have source files
      SgGlobal* global = target_src_file->get_globalScope();

      SgFunctionDeclaration* target_allocs =
        findDeclarationStatement<SgFunctionDeclaration>(global,"perfile_allocs",global, true);
      ROSE_ASSERT(target_allocs);
      SgBasicBlock* allocs_body = target_allocs->get_definition()->get_body();
      ROSE_ASSERT(allocs_body);
      SgFunctionDeclaration* target_inits =
        findDeclarationStatement<SgFunctionDeclaration>(global,"perfile_inits",global, true);
      ROSE_ASSERT(target_inits);
      SgBasicBlock* inits_body = target_inits->get_definition()->get_body();
      ROSE_ASSERT(inits_body);
      // add function declaration first
      // then insert function calls into perfile_allocs() and perfile_inits();
      for (vector<SgFile*>::iterator iter=src_files.begin();
         iter != src_files.end();iter++)
      {
        SgFile* src_file = *iter;
       SgFunctionDeclaration* alloc_func = findUpcriAllocInit(src_file,true);
       if (alloc_func)
       {  
         SgFunctionDeclaration* func = buildNondefiningFunctionDeclaration(alloc_func
           ,global);
        prependStatement(func,global); 
        appendStatement(func, allocs_body);
       } 

       SgFunctionDeclaration* init_func = findUpcriAllocInit(src_file,false);
       if (init_func)
       {  
         SgFunctionDeclaration* func = buildNondefiningFunctionDeclaration(init_func
           ,global);
        prependStatement(func,global); 
        appendStatement(func, inits_body);
       } 
      } // end for
    } 
    else
    {
      cout<<"Cannot find the target magic file to add perfile processing functions."<<endl;
      ROSE_ASSERT(false);
    }
  } //transProject()

  //! Check if a node is a defining UPCRI_ALLOC_file_xxx() or UPCRI_INIT_file_xxx()
  bool isUpcriAllocInit(SgNode* n, bool checkAlloc)
  {
    string matchStr="UPCRI_ALLOC_";
    if (!checkAlloc)
      matchStr="UPCRI_INIT_";

    bool result = false;
    if (isSgFunctionDeclaration(n) &&
     isSgGlobal(isSgStatement(n)->get_scope())&&
     (isSgFunctionDeclaration(n)->get_name().getString().find(matchStr,0) == 0)) 
   result = true;

   return result;
  }

  //! Find defining UPCRI_ALLCO_file_xxx() or UPCRI_INIT_file_xxx() from a file
  SgFunctionDeclaration* findUpcriAllocInit(SgNode* n, bool checkAlloc)
  {
    if (!n) return 0;   

    if (isUpcriAllocInit(n,checkAlloc)) 
    {
      return isSgFunctionDeclaration(n);
    }
    vector<SgNode*> children = n->get_traversalSuccessorContainer();
    for (vector<SgNode*>::const_iterator i = children.begin(); 
    i != children.end(); ++i) 
    {
      SgFunctionDeclaration* target= findUpcriAllocInit(*i,checkAlloc);
      if (target) 
        return target;
    }
    return 0;
  }

  //! Translate special cases for private-to-shared pointers
  /*!
   * Special cases for private to shared pointers
   *
   * a) in global scope: must have 'extern' modifier
   *  extern shared int * ep2s_p1; // translated into
   *  extern upcr_pshared_ptr_t ep2s_p1;
   *
   * b) in local scope: must not have 'static' modifier
   *   
   *  shared int *lp2s_p;
   *  shared int *lp2s_p1=0;
   *  shared int *lp2s_p2=&global_counter2;
   *  lp2s_p1 = &global_counter2;
   *  // translated into-->
   *    // declaration without initializers
   *   upcr_pshared_ptr_t lp2s_p;
   *   upcr_pshared_ptr_t lp2s_p1;
   *   upcr_pshared_ptr_t lp2s_p2;
   *
   *   // special handling for NULL-valued initializer
   *   UPCR_SETNULL_PSHARED ((_UINT32) & lp2s_p1);
   *   lp2s_p2 = global_counter2;
   *   lp2s_p1 = global_counter2;
   */
   void transSpecialPrivate2Shared(SgVariableDeclaration* vardecl)
   {
     ROSE_ASSERT(vardecl!=NULL);
     SgGlobal* global_scope = getGlobalScope(vardecl); 
     ROSE_ASSERT(global_scope!=NULL);

     SgVariableSymbol* varsymbol = getFirstVarSym(vardecl);
     string varname = varsymbol->get_name().str();
     SgType* t = varsymbol->get_type();
     SgInitializer * initor = varsymbol->get_declaration()->get_initptr();

     bool isGlobal = isSgGlobal(vardecl->get_scope());
     bool isp2s = isUpcPrivateToSharedType(t);
     //bool isphaseless = isUpcPhaseLessSharedType(t);
     // must be private-to-shared pointers
     ROSE_ASSERT(isp2s); 
     // must be either global with extern or non-static local
     ROSE_ASSERT((isGlobal && isExtern(vardecl)) ||
                  (!isGlobal&&!isStatic(vardecl)));
        // prepare types 
    SgType* regular_type = lookupNamedTypeInParentScopes("upcr_shared_ptr_t",global_scope);
    SgType* phaseLess_type = lookupNamedTypeInParentScopes("upcr_pshared_ptr_t",global_scope);
    if (regular_type==NULL) 
      regular_type = buildOpaqueType("upcr_shared_ptr_t",global_scope);
    if( phaseLess_type == NULL ) 
      phaseLess_type = buildOpaqueType("upcr_pshared_ptr_t",global_scope);
     
    SgType* new_type=regular_type;
    SgScopeStatement* new_scope = global_scope;
    if (!isGlobal) new_scope = vardecl->get_scope();
    if (isUpcPhaseLessSharedType(t)) new_type = phaseLess_type;
    SgVariableDeclaration* new_decl = buildVariableDeclaration(
           varsymbol->get_name(),new_type, NULL, new_scope);
    if (isExtern(vardecl)) setExtern(new_decl);
    insertStatementBefore(vardecl, new_decl);
    moveUpPreprocessingInfo(new_decl,vardecl);
    // for local pointer to shared data with an initializer
    // add an extra assignment statement. 
    // e.g:
    //  shared int * lp2s_p2 = &gsj // is translated into 
    //  upcr_pshared_ptr_t lp2s_p2;
    //   lp2s_p2 = gsj;
    //
    // // Special handling for null-valued initializer
    //  shared int *lp2s_p1=0; 
    //  UPCR_SETNULL_PSHARED ((_UINT32) & lp2s_p1);
    if ((!isGlobal) &&(initor))  
    {
      SgExpression* lhs = buildVarRefExp(new_decl);
      // Only handle SgAssignInitializer for now, 
      // TODO SgAggregateInitializer and others
      SgAssignInitializer * assign_initor = isSgAssignInitializer(initor); 
      if (!assign_initor)
      {
         cerr<<"Error: unhandled SgInitializer type in transThreadLocalData()"<<endl;
         ROSE_ASSERT(assign_initor);
      }
       // distinguish between null value and others
      bool hasNullValue = isNullValued(assign_initor->get_operand()); 
     if (hasNullValue) 
      {
        SgType* return_type = buildVoidType();
        SgVarRefExp* var1 = buildVarRefExp(new_decl);
        //grab the base type for private-to-shared type
        SgModifierType* modifier_type=NULL;
        hasUpcSharedType(t,&modifier_type);
        ROSE_ASSERT(modifier_type); 
        SgType* base_type = modifier_type->get_base_type();

        SgExpression* arg1= buildCastExp(buildAddressOfOp(var1),base_type);
        SgExprListExp* arg_list = buildExprListExp();
        appendExpression(arg_list,arg1);
        SgExprStatement* callStmt1 = buildFunctionCallStmt("UPCR_SETNULL_PSHARED",
                 return_type, arg_list);
        insertStatementAfter(vardecl,callStmt1);
      }
      else
      {
       // cout<<"Debug: 1624: not null value:"<<assign_initor->unparseToString()<<endl;
        SgExpression* rhs =buildPointerDerefExp(
               copyExpression(assign_initor->get_operand()));
        SgExprStatement* assign_stmt = buildAssignStatement(lhs, rhs);
        insertStatementAfter(vardecl,assign_stmt);
      } // end if (null value)
    } // end if local non-static p2s variables
    removeStatement(vardecl);
     // TODO replace variable reference from global -> *global ??
     // lp2s_p1 = &global_counter2; // translated into -->
     // lp2s_p1 = global_counter2;
      
   } //transSpecialPrivate2Shared()

  //shared int* pp =0; 
  //initializer is represented as SgAssignInitializer->SgCastExp->SgIntVal
   bool isNullValued(SgExpression* exp)
   {
     ROSE_ASSERT(exp);
      bool hasNullValue = false; 
      SgCastExp* cast_exp = isSgCastExp(exp);
      if (cast_exp)
      {
        SgIntVal* intVal= isSgIntVal(cast_exp->get_operand());
        if (intVal)
        {
         // cout<<"Debug: 1606: int val is:"<<intVal->get_value()<<endl;
         if (intVal->get_value()==0)
          hasNullValue = true;
        }
      }
      return hasNullValue;
   }

//----------------------------------------
} // end of namespace

