/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

#include <cstdlib>
#include <iostream>
#include <fstream>
#include <vector>
#include <string>
using namespace std;


int main(int argc, char** argv)
{
  ifstream fin(argv[1]);
  if (!fin) {
    cerr << "\nError: failure opening " << argv[1] << endl;
    exit(-1);
  }
  size_t i, j, num_vars, num_fns, num_deriv_vars;
  string vars_text, fns_text, dvv_text;

  // Get the parameter vector and ignore the labels
  fin >> num_vars >> vars_text;
  vector<double> x(num_vars);
  for (i=0; i<num_vars; i++) {
    fin >> x[i];
    fin.ignore(256, '\n');
  }

  // Get the ASV vector and ignore the labels
  fin >> num_fns >> fns_text;
  vector<int> ASV(num_fns);
  for (i=0; i<num_fns; i++) {
    fin >> ASV[i];
    fin.ignore(256, '\n');
  }

  // Get the DVV vector and ignore the labels
  fin >> num_deriv_vars >> dvv_text;
  vector<int> DVV(num_deriv_vars);
  for (i=0; i<num_deriv_vars; i++) {
    fin >> DVV[i];
    fin.ignore(256, '\n');
  }

  if (num_vars != 9 || num_fns != 1) {
    cerr << "Error: wrong number of inputs/outputs in steel_column_perf."<<endl;
    exit(-1);
  }

  // Compute the results and output them directly to argv[2] (the NO_FILTER
  // option is used).  Response tags are optional; output them for ease
  // of results readability.
  ofstream fout(argv[2]);
  if (!fout) {
    cerr << "\nError: failure creating " << argv[2] << endl;
    exit(-1);
  }
  fout.precision(15); // 16 total digits
  fout.setf(ios::scientific);
  fout.setf(ios::right);

  // In the steel column description in Kuschel & Rackwitz, Cost is _not_
  // defined as a random variable.  That is Cost is not a fn(B, D, H), but
  // is rather defined as a fn(b, d, h).  Since dCost/dX|_{X=mean} is not the
  // same as dCost/dmean for non-normal X (jacobian_dX_dS is not 1), dCost/dX
  // may not be used and an optional interface must be defined for Cost.

  // set effective length s based on assumed boundary conditions
  // actual length of the column is 7500 mm
  double s = 7500;

  // F0 = x[0] = init. deflection (normal    unc. var.)
  // P1 = x[1] = dead weight load (normal    unc. var.)
  // B  = x[2] = flange breadth   (lognormal unc. var., mean is design var.)
  // D  = x[3] = flange thickness (lognormal unc. var., mean is design var.)
  // H  = x[4] = profile height   (lognormal unc. var., mean is design var.)
  // Fs = x[5] = yield stress     (lognormal unc. var.)
  // P2 = x[6] = variable load    (gumbel    unc. var.)
  // P3 = x[7] = variable load    (gumbel    unc. var.)
  // E  = x[8] = elastic modulus  (weibull   unc. var.)

  double F0 = x[0], B = x[2], D = x[3], H = x[4], Fs = x[5], E = x[8],
         P = x[1]+x[6]+x[7], Pi = 3.1415926535897932385, Pi2 = Pi*Pi,
         Pi4 = Pi2*Pi2, Pi6 = Pi2*Pi4, B2 = B*B, D2 = D*D,
         H2 = H*H, H3 = H*H2, H5 = H2*H3, E2 = E*E, E3 = E*E2, s2 = s*s,
         X = Pi2*E*B*D*H2 - 2.*s2*P, X2 = X*X, X3 = X*X2;
  /*
  // Alternate ordering with all Normal and Lognormal variables
  // (for PCE verification)
  double F0 = x[0], B = x[5], D = x[6], H = x[7], Fs = x[8], E = x[4],
         P = x[1]+x[2]+x[3], Pi = 3.1415926535897932385, Pi2 = Pi*Pi,
         Pi4 = Pi2*Pi2, Pi6 = Pi2*Pi4, B2 = B*B, D2 = D*D,
         H2 = H*H, H3 = H*H2, H5 = H2*H3, E2 = E*E, E3 = E*E2, s2 = s*s,
         X = Pi2*E*B*D*H2 - 2.*s2*P, X2 = X*X, X3 = X*X2;
  */

  // **** g (limit state):
  if (ASV[0] & 1)
    fout << "                     "
	 << Fs - P*(1./2./B/D + Pi2*F0*E*H/X) << " g\n";

  // **** dg/dx (w.r.t. active/uncertain variables):
  if (ASV[0] & 2) {
    fout << "[ ";
    for (i=0; i<num_deriv_vars; i++) {
      int var_index = DVV[i] - 1;
      switch (var_index) {
      case 0: // df/dF0
	fout << -E*H*P*Pi2/X << ' ';
	break;
      case 1: case 6: case 7: // df/dP1, df/dP2, df/dP3
	fout << -1./2./B/D - B*D*E2*F0*H3*Pi4/X2 << ' ';
	break;
      case 2: // df/dB
	fout << P*(1./2./B2/D + D*E2*F0*H3*Pi4/X2) << ' ';
	break;
      case 3: // df/dD
	fout << P*(1./2./B/D2 + B*E2*F0*H3*Pi4/X2) << ' ';
	break;
      case 4: // df/dH
	fout << E*F0*P*Pi2*(X + 4.*P*s2)/X2 << ' ';
	break;
      case 5: // df/dFs
	fout << "1. ";
	break;
      case 8: // df/dE
	fout << 2.*F0*H*P*P*Pi2*s2/X2;
	break;
      }
    }
    fout << "]\n";
  }

  // **** d^2g/dx^2: (SORM)
  if (ASV[0] & 4) {
    fout << "[[ ";
    for (i=0; i<num_deriv_vars; i++) {
      int var_index_i = DVV[i] - 1;
      for (j=0; j<num_deriv_vars; j++) {
	int var_index_j = DVV[j] - 1;
	if (var_index_i == 5 || var_index_j == 5)          // d^2g/dFsdx
	  fout << "0. ";
	else if ( (var_index_i == 1 && var_index_j == 1) ||
                  (var_index_i == 1 && var_index_j == 6) ||
                  (var_index_i == 1 && var_index_j == 7) ||
		  (var_index_i == 6 && var_index_j == 1) ||
		  (var_index_i == 6 && var_index_j == 6) ||
		  (var_index_i == 6 && var_index_j == 7) ||
		  (var_index_i == 7 && var_index_j == 1) ||
		  (var_index_i == 7 && var_index_j == 6) ||
		  (var_index_i == 7 && var_index_j == 7) ) // d^2g/dPdP
	  fout << -4.*B*D*E2*F0*H3*Pi4*s2/X3 << ' ';
	else if ( (var_index_i == 1 && var_index_j == 2) ||
 		  (var_index_i == 6 && var_index_j == 2) ||
		  (var_index_i == 7 && var_index_j == 2) ||
		  (var_index_i == 2 && var_index_j == 1) ||
		  (var_index_i == 2 && var_index_j == 6) ||
		  (var_index_i == 2 && var_index_j == 7) ) // d^2g/dPdB
	  fout << 1./2./B2/D + D*E2*F0*H3*Pi4/X2*(2.*B*D*E*H2*Pi2/X - 1.) <<' ';
	else if ( (var_index_i == 1 && var_index_j == 3) ||
 		  (var_index_i == 6 && var_index_j == 3) ||
		  (var_index_i == 7 && var_index_j == 3) ||
		  (var_index_i == 3 && var_index_j == 1) ||
		  (var_index_i == 3 && var_index_j == 6) ||
		  (var_index_i == 3 && var_index_j == 7) ) // d^2g/dPdD
	  fout << 1./2./B/D2 + B*E2*F0*H3*Pi4/X2*(2.*B*D*E*H2*Pi2/X - 1.) <<' ';
	else if ( (var_index_i == 1 && var_index_j == 4) ||
 		  (var_index_i == 6 && var_index_j == 4) ||
		  (var_index_i == 7 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 1) ||
		  (var_index_i == 4 && var_index_j == 6) ||
		  (var_index_i == 4 && var_index_j == 7) ) // d^2g/dPdH
	  fout << B*D*E2*F0*H2*Pi4*(X+8.*P*s2)/X3 << ' ';
	else if ( (var_index_i == 1 && var_index_j == 0) ||
 		  (var_index_i == 6 && var_index_j == 0) ||
		  (var_index_i == 7 && var_index_j == 0) ||
		  (var_index_i == 0 && var_index_j == 1) ||
		  (var_index_i == 0 && var_index_j == 6) ||
		  (var_index_i == 0 && var_index_j == 7) ) // d^2g/dPdF0
	  fout << -B*D*E2*H3*Pi4/X2 << ' ';
	else if ( (var_index_i == 1 && var_index_j == 8) ||
 		  (var_index_i == 6 && var_index_j == 8) ||
		  (var_index_i == 7 && var_index_j == 8) ||
		  (var_index_i == 8 && var_index_j == 1) ||
		  (var_index_i == 8 && var_index_j == 6) ||
		  (var_index_i == 8 && var_index_j == 7) ) // d^2g/dPdE
	  fout << 4.*B*D*E*F0*H3*P*Pi4*s2/X3 << ' ';
	else if (var_index_i == 2 && var_index_j == 2)     // d^2g/dB^2
	  fout << -P*(1./B2/B/D + 2.*D2*E3*F0*H5*Pi6/X3) << ' ';
	else if ( (var_index_i == 2 && var_index_j == 3) ||
		  (var_index_i == 3 && var_index_j == 2) ) // d^2g/dBdD
	  fout << -P*(1./2./B2/D2 + E2*F0*H3*Pi4/X2*(2.*B*D*E*H2*Pi2/X - 1.))
               << ' ';
	else if ( (var_index_i == 2 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 2) ) // d^2g/dBdH
	  fout << -D*E2*F0*H2*P*Pi4*(X + 8.*P*s2)/X3 << ' ';
	else if ( (var_index_i == 0 && var_index_j == 2) ||
		  (var_index_i == 2 && var_index_j == 0) ) // d^2g/dBdF0
	  fout << D*E2*H3*P*Pi4/X2 << ' ';
	else if ( (var_index_i == 2 && var_index_j == 8) ||
		  (var_index_i == 8 && var_index_j == 2) ) // d^2g/dBdE
	  fout << -4.*D*E*F0*H3*P*P*Pi4*s2/X3 << ' ';
	else if (var_index_i == 3 && var_index_j == 3)     // d^2g/dD^2
	  fout << -P*(1./B/D2/D + 2.*B2*E3*F0*H5*Pi6/X3) << ' ';
	else if ( (var_index_i == 3 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 3) ) // d^2g/dDdH
	  fout << -B*E2*F0*H2*P*Pi4*(X + 8.*P*s2)/X3 << ' ';
	else if ( (var_index_i == 0 && var_index_j == 3) ||
		  (var_index_i == 3 && var_index_j == 0) ) // d^2g/dDdF0
	  fout << B*E2*H3*P*Pi4/X2 << ' ';
	else if ( (var_index_i == 3 && var_index_j == 8) ||
		  (var_index_i == 8 && var_index_j == 3) ) // d^2g/dDdE
	  fout << -4.*B*E*F0*H3*P*P*Pi4*s2/X3 << ' ';
	else if (var_index_i == 4 && var_index_j == 4)     // d^2g/dH^2
	  fout << -2.*B*D*E2*F0*H*P*Pi4*(X + 8.*P*s2)/X3 << ' ';
	else if ( (var_index_i == 0 && var_index_j == 4) ||
		  (var_index_i == 4 && var_index_j == 0) ) // d^2g/dHdF0
	  fout << E*P*Pi2*(X + 4.*P*s2)/X2 << ' ';
	else if ( (var_index_i == 4 && var_index_j == 8) ||
		  (var_index_i == 8 && var_index_j == 4) ) // d^2g/dHdE
	  fout << -2.*F0*P*P*Pi2*s2*(3.*X + 8.*P*s2)/X3 << ' ';
	else if (var_index_i == 0 && var_index_j == 0)     // d^2g/dF0^2
	  fout << "0. ";
	else if ( (var_index_i == 0 && var_index_j == 8) ||
		  (var_index_i == 8 && var_index_j == 0) ) // d^2g/dF0dE
	  fout << 2.*H*P*P*Pi2*s2/X2 << ' ';
	else if (var_index_i == 8 && var_index_j == 8)     // d^2g/dE^2
	  fout << -4.*B*D*F0*H3*P*P*Pi4*s2/X3 << ' ';
	else { // unsupported derivative
	  cerr << "Error: unsupported Hessian cross term in steel_column."
	       << endl;
	  exit(-1);
	}
      }
    }
    fout << "]]\n";
  }

  fout.flush();
  fout.close();
  return 0;
}
