#!/bin/csh -f
#
# This script traverses the platform test directories, builds an html page
# with links to results and sends mail for the last nights build.
#
#####################################################

if ( $#argv < 3 ) then
  echo "Usage: $0 DAKOTA_ROOT web_server pass_number"
  exit
endif

echo "Running $0 on `uname -n` at `date`"

# set up the environment variables (global)
setenv DAKOTA_ROOT $1
setenv TEST_SQA $DAKOTA_ROOT/Dakota/test/sqa
source $TEST_SQA/environment.csh
setenv TEST_DIR_FQ $DAKOTA_ROOT/$TEST_DIR

set dakota_dir      = "$TEST_DIR_FQ/Dakota"

# define base directory, html directory and html_address
# assumes ~/public_html 
set web_serv = $2
set pass_number = $3

# destination for html reports, local and web server
set html_dir = $HOME/public_html/nightly_dakota
set html_loc = https://$web_serv/$USER/nightly_dakota

# if html directory does not exist, create
if ( -e $html_dir ) then
  echo "HTML directory is $html_dir"
else
  mkdir -p $html_dir
  echo "Creating HTML directory $html_dir"
endif

# change into html directory
cd $html_dir

# set date and htmlize command
set date = `date +"%Y%m%d"`
if ( -e /usr/netpub/bin/emacs ) then
  set emacs = /usr/netpub/bin/emacs
else if ( -e /usr/bin/emacs ) then
    set emacs = /usr/bin/emacs
else  #Don't know where emacs is...
    set emacs = emacs
endif
  
set htmlize = "$TEST_SQA/htmlize.el"
set lhtmlize = "$TEST_SQA/log-htmlize.el"
set log_htmlize = "$emacs -batch -l $htmlize -l $lhtmlize"

# initialize build error flag
set dakota_err = 0

# email pass 1 is run per platform in run_base_test.csh to parse files
# that might be purged during subsequent builds
# pass 0 does all except copy create_votd.log, for use in 
# run_base_test.csh to copy interim dakota_report files
#
# it creates dakota_report.out in the platform-specific directory
echo "${0}: pass_number = $pass_number"
if ( $pass_number <= 1 ) then

  # platform-dependent input filenames
  set platform_report = $TEST_DIR_FQ/dakota_report.out
  set buildhost       = `cat $TEST_DIR_FQ/buildhost` 
  set dakota_dir      = "$TEST_DIR_FQ/Dakota"

  # platform-dependent output filenames
  set coutput_report = "config_dakota""_${SNLSYSTEM}_$TEST_PLATFORM""_$date.out"
  set cerror_report  = "config_dakota""_${SNLSYSTEM}_$TEST_PLATFORM""_$date.err"

  # platform header
  echo "*** PLATFORM: $TEST_PLATFORM ($buildhost) ***" >! $platform_report
  # presently disabled due to use of svn exported tarballs
  #echo "* svn info:" >> $platform_report
  #svn info $dakota_dir | grep URL >> $platform_report
  #svn info $dakota_dir | grep Revision >> $platform_report

  set fail_copy = 0     # want to use this for checking $? after network scp, but some platform problems

  # check that configure succeeded
  if ( -e $dakota_dir/config_dakota.err || ! -e $dakota_dir/config.status ) then
    if ( -e $dakota_dir/config_dakota.out ) then
      echo "DAKOTA failed to configure correctly." >> $platform_report
      cp -p $dakota_dir/config_dakota.out $coutput_report
      scp -Bp $coutput_report "$USER@$web_serv"":$html_dir/$coutput_report"
#      set fail_copy = $?
      if ( $fail_copy != 0 ) then
        echo "WARNING:  failed to copy $coutput_report to $web_serv "
      endif
      echo "" >> $platform_report
      echo "$html_loc/$coutput_report" >> $platform_report
      if ( -e $dakota_dir/config_dakota.err ) then
        cp -p $dakota_dir/config_dakota.err $cerror_report
        scp -Bp $cerror_report "$USER@$web_serv"":$html_dir/$cerror_report"
#        set fail_copy = $?
        if ( $fail_copy != 0 ) then
          echo "WARNING:  failed to copy $cerror_report to $web_serv "
        endif
        echo "$html_loc/$cerror_report" >> $platform_report
      endif
    else
      echo "" >> $platform_report
      echo "DAKOTA failed to configure: no output." >> $platform_report
      echo "Check network, download/checkout, or script/crontab." >> $platform_report
    endif
    set dakota_err = 1
  else
    # output status of DAKOTA PACKAGE BUILDS
    echo "" >> $platform_report
    echo "DAKOTA PACKAGE BUILDS:" >> $platform_report
    # iterate over the externals in the DAKOTA top-level directory
    foreach p ( `cat $dakota_dir/test/sqa/package-list-externals` )
      # since p is an absolute path, get the package after the last /
      set pp = `echo $p | awk -F\/ '{print $NF}'`

      # package and platform dependent input filenames (build logs)
      set moutput = "make_$pp.out"
      set moutput_report = "make_$pp""_${SNLSYSTEM}_$TEST_PLATFORM""_$date.html"
      set merror  = "make_$pp.err"
      set merror_report  = "make_$pp""_${SNLSYSTEM}_$TEST_PLATFORM""_$date.err"

      if ( ! -e $dakota_dir/$p/$moutput ) then
        echo "   $pp has no build information" >> $platform_report
      else 
        set mdate = `ls -l $dakota_dir/$p/$moutput | tr -s ' ' | cut -d' ' -f 6-7`        
	if ( -e $dakota_dir/$p/$merror ) then
	  echo "   $pp build failed on $mdate" >> $platform_report
	  $log_htmlize $dakota_dir/$p/$moutput > $moutput_report
	  scp -Bp $moutput_report "$USER@$web_serv"":$html_dir/$moutput_report"
#          set fail_copy = $?
          if ( $fail_copy != 0 ) then
            echo "WARNING:  failed to copy $moutput_report to $web_serv "
          endif
	  cp -p $dakota_dir/$p/$merror $merror_report
	  scp -Bp $merror_report "$USER@$web_serv"":$html_dir/$merror_report"
#          set fail_copy = $?
          if ( $fail_copy != 0 ) then
            echo "WARNING:  failed to copy $merror_report to $web_serv "
          endif
        else
	  echo "   $pp built successfully on $mdate" >> $platform_report
        endif
      endif
    end

    # output resulting DAKOTA configuration (with package failure workarounds)
    echo "" >> $platform_report
    echo "DAKOTA configured as follows" >> $platform_report
    head -7 $dakota_dir/config.log | tail -1 >> $platform_report
    echo "" >> $platform_report

    # platform dependent input filenames (build logs)
    set moutput = "make_dakota.out"
    set moutput_report = "make_dakota""_${SNLSYSTEM}_$TEST_PLATFORM""_$date.html"
    set merror  = "make_dakota.err"
    set merror_report  = "make_dakota""_${SNLSYSTEM}_$TEST_PLATFORM""_$date.err"
    set dinput  = "$TEST_DIR_FQ/int_full_dakota_diffs.out"
    set pinput  = "$TEST_DIR_FQ/int_full_dakota_pdiffs.out"
    set doutput = "dakota_${SNLSYSTEM}_$TEST_PLATFORM""_diffs_$date.out"
    set poutput = "dakota_${SNLSYSTEM}_$TEST_PLATFORM""_pdiffs_$date.out"    

    # check that make executed without errors
    if ( ! -e $dakota_dir/make_dakota.out ) then
      echo "DAKOTA has no build information" >> $platform_report
    else
      set mdate = `ls -l $dakota_dir/make_dakota.out | tr -s ' ' | cut -d' ' -f 6-7`
      if ( -e $dakota_dir/make_dakota.err || ! -e $dakota_dir/src/dakota ) then
        echo "DAKOTA failed to build on $mdate" >> $platform_report
        set dakota_err = 1
        $log_htmlize $dakota_dir/make_dakota.out > $moutput_report
        scp -Bp $moutput_report "$USER@$web_serv"":$html_dir/$moutput_report"
#        set fail_copy = $?
        if ( $fail_copy != 0 ) then 
          echo "WARNING:  failed to copy $moutput_report to $web_serv "
        endif 
        cp -p $dakota_dir/make_dakota.err $merror_report
        scp -Bp $merror_report "$USER@$web_serv"":$html_dir/$merror_report"
#        set fail_copy = $?
        if ( $fail_copy != 0 ) then  
          echo "WARNING:  failed to copy $merror_report to $web_serv "
        endif 
        echo "" >> $platform_report
        echo "$html_loc/$moutput_report" >> $platform_report
        echo "$html_loc/$merror_report" >> $platform_report
      else
        echo "DAKOTA built successfully on $mdate" >> $platform_report
        # Serial test reports
        echo "" >> $platform_report
	if ( ! -e $dinput ) then
	  echo "Serial test results unavailable" >> $platform_report
        else
          set pass = `grep PASS $dinput | wc -l`
          set fail = `grep FAIL $dinput | wc -l`
          set diff = `grep DIFF $dinput | wc -l`
          cp -p $dinput $doutput
          scp -Bp $doutput "$USER@$web_serv"":$html_dir/$doutput"
#          set fail_copy = $?
          if ( $fail_copy != 0 ) then  
            echo "WARNING:  failed to copy $doutput to $web_serv "
          endif 
          echo "Serial test results for $TEST_PLATFORM" >> $platform_report
          echo "PASS: $pass  FAIL: $fail  DIFF: $diff" >> $platform_report
          echo "$html_loc/$doutput" >> $platform_report
	endif
        # Parallel test reports
        if ( $TEST_PARALLEL == 'yes' ) then
 	  if ( ! -e $pinput ) then
	    echo "Parallel test results unavailable" >> $platform_report
	  else
            echo "" >> $platform_report
            set pass = `grep PASS $pinput | wc -l`
            set fail = `grep FAIL $pinput | wc -l`
            set diff = `grep DIFF $pinput | wc -l`
            cp -p $pinput $poutput
            scp -Bp $poutput "$USER@$web_serv"":$html_dir/$poutput"
#            set fail_copy = $?
            if ( $fail_copy != 0 ) then
              echo "WARNING:  failed to copy $poutput to $web_serv " 
            endif
            echo "Parallel test results for $TEST_PLATFORM" >> $platform_report
            echo "PASS: $pass  FAIL: $fail  DIFF: $diff" >> $platform_report
            echo "$html_loc/$poutput" >> $platform_report
          endif
        endif 
      endif # is make error
    endif # is make output
  endif

  # temporary hack to be replaced with multi-filesystem mail plan
  if ( `uname -n` == 'purple' ) then
    cp -p /projects/dakota/$TEST_DIR/dakota_report.out /usr/sharelan/dakota/$TEST_DIR
    if ( $pass_number == 1) then
      cp -p /projects/dakota/$TEST_DIR/create_votd.log /usr/sharelan/dakota/$TEST_DIR
    endif
  endif

  # KH NOTES:
  #   - Destination directory for all multi-filesystem dakota_report.out
  #     is on SCICO:/usr/sharelan/dakota/nightly/reports/
  #   - Because we cannot create directories on another machine, the destination
  #     report directory on SCICO can't be 
  #       SCICO:/usr/sharelan/dakota/nightly/reports/$TEST_DIR
  #     If that directory didn't exist, the scp will (most likely) just create
  #     a file called $TEST_DIR; but if a directory did exist, the file 
  #     would get copied into the $TEST_DIR as dakota_report.out
  #     This would be very confusing.
  #     So, the $platform_report file will get copied into the directory
  #       SCICO:/usr/sharelan/dakota/nightly/reports/
  #     and will be named $TEST_DIR.report
  set nightly_report_dir = /usr/sharelan/dakota/nightly/reports
  set nightly_report_dest = "$nightly_report_dir/$TEST_DIR.report"
  if ( $SNLSYSTEM == 'scico' ) then
    cp -p $platform_report $nightly_report_dest
  else
    scp -Bp $platform_report sahp7377:$nightly_report_dest
#    set fail_copy = $?
    if ( $fail_copy != 0 ) then
      echo "WARNING:  failed to copy $platform_report to $nightly_report_dest "
    endif
  endif
  # if create_votd.log exists, copy it over also
  if ( -e $TEST_DIR_FQ/create_votd.log ) then
   if ( $pass_number == 1) then
    if ( $SNLSYSTEM == 'scico' ) then
      cp -p $TEST_DIR_FQ/create_votd.log $nightly_report_dir/$TEST_DIR.create_votd.report
    else
      scp -Bp $TEST_DIR_FQ/create_votd.log sahp7377:$nightly_report_dir/$TEST_DIR.create_votd.report
#      set fail_copy = $?
      if ( $fail_copy != 0 ) then
        echo "WARNING:  failed to copy $TEST_DIR_FQ/create_votd.log to nightly_report_dir"
      endif
    endif
   endif
  endif
  #temporary purple copy...not sure why scp above doesn't work?  Something 
  # about .ssh/config permissions
  if ( `uname -n` == 'purple' ) then
    cp -p /projects/dakota/$TEST_DIR/dakota_report.out $nightly_report_dest
   if ( $pass_number == 1) then
    cp -p /projects/dakota/$TEST_DIR/create_votd.log $nightly_report_dir/$TEST_DIR.create_votd.report
   endif
  endif


endif  #-- email pass 1 --#


# email pass 2 is run once per day on a single platform as a separate cron job
if ( $pass_number == 2 ) then

#New pass 2 - create dakota_report.log from platform reports in common 
#             directory
  cd $html_dir
  # Build DAKOTA report in Web browsable form (overwrite if present)
  echo "*********************************************" >! dakota_report.log
  echo "*                                           *" >> dakota_report.log
  echo "* BUILD/TEST LOG FROM NIGHTLY DAKOTA BUILDS *" >> dakota_report.log
  echo "*                                           *" >> dakota_report.log
  echo "*********************************************" >> dakota_report.log

# iterate through each platform and report stats
# (all platform reports are assumed in a common directory on scico)
  set nightly_report_dir = /usr/sharelan/dakota/nightly/reports/

  foreach platform ( `cat $dakota_dir/test/sqa/nightly-test-email-list` )
    echo "" >> dakota_report.log
    echo "*********************************************" >> dakota_report.log
    set platform_report    = "$platform.report"
    set platform_report_fq = "$nightly_report_dir/$platform_report"
    echo "*** $platform ***" >> dakota_report.log
    echo "*********************************************" >> dakota_report.log
    if ( ! -e $platform_report_fq ) then
      echo "No build report available" >> dakota_report.log
      set dakota_err = 1
    else
      set rpt_date = `ls -l --time-style +"%Y%m%d"  $nightly_report_dir/$platform_report  | tr -s ' ' | cut -d' ' -f 6`
      if ( $rpt_date != $date ) then
        echo "Stale $platform_report  detected dated $rpt_date (likely cron failure)" >> dakota_report.log
        set dakota_err = 1
      else
      # Check if DAKOTA failed to configure or build..
        set config_ok = `grep -c "DAKOTA failed to config" $platform_report_fq`
        set build_ok  = `grep -c "DAKOTA failed to build" $platform_report_fq`
        if ($config_ok != 0 || $build_ok != 0) then
          set dakota_err = 1
        endif
        cat $platform_report_fq >> dakota_report.log
#       add votd report
        set platform_votd    = "$platform.create_votd.report"
        set platform_votd_fq = "$nightly_report_dir/$platform_votd"
        echo "" >> dakota_report.log 
        if ( ! -e $platform_votd_fq ) then 
          echo "No VOTD report available for $platform" >> dakota_report.log 
        else 
          set rpt_date = `ls -l  --time-style +"%Y%m%d" $platform_votd_fq | tr -s ' ' | cut -d' ' -f 6`  
          if ( $rpt_date != $date ) then  
            echo "Stale $platform_votd  detected dated $rpt_date (likely cron or build failure)" >> dakota_report.log 
          else  
            if ( `grep -c "VOTD creation and push succeeded" $platform_votd_fq` > 0 ) then
              echo "VOTD creation and push succeeded" >> dakota_report.log
              set push_stable = `grep -c "to stable" $platform_votd_fq`
              if ( $push_stable > 0 ) then
                echo "Stable version updated" >> dakota_report.log
              else
                echo "Stable version not updated" >> dakota_report.log
              endif
            else if ( `grep -c "Failed VOTD push of tar" $platform_votd_fq` >  0 ) then
              echo "VOTD tar push failed" >> dakota_report.log
            else if ( `grep -c "Failed VOTD push of src" $platform_votd_fq` >  0 ) then
              echo "VOTD src push failed" >> dakota_report.log 
            else if ( `grep -c "build failed" $platform_votd_fq` > 0) then
              echo "Build failed; no VOTD/Stable push" >> dakota_report.log
            else if ( `grep -c "Distribution push currently enabled for 'scico' and 'csri' only" $platform_votd_fq` > 0) then
              echo "VOTD/Stable push not enabled for this platform" >> dakota_report.log
            else       
              echo "Internal ERROR: send_base_mail logic not finding other case for platform $platform ?" 
            endif
          endif
        endif
      endif
    endif
  end # foreach
#end new pass 2 report creation

  #####################
  # SEND EMAIL REPORT #
  #####################
  # (alter to_string for testing)
#send email with report in common directory
  set to_string = "dakota-developers@development.sandia.gov"
  if ( $dakota_err == 0 && `grep -c "DAKOTA failed" dakota_report.log` == 0 ) then
    mailx -s "DAKOTA BUILD SUCCESS" $to_string < dakota_report.log
  else
    mailx -s "DAKOTA BUILD FAILURE" $to_string < dakota_report.log
  endif


endif  #-- email pass 2 --#
