/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

// Class:        ProblemDescDB
//- Description: Implementation code for the ProblemDescDB class.
//-              It provides storage for problem description database entries
//-              and defines the keyword handlers that yacc calls to populate
//-              the database based on the parsed input.
//- Owner:       Mike Eldred
//- Checked by:

#include "system_defs.h"
#include "ProblemDescDB.H"
#include "ParallelLibrary.H"
#include "NIDRProblemDescDB.H"
#include "CommandLineHandler.H"
#include "DakotaIterator.H"
#include "DakotaInterface.H"
#include <string>

static const char rcsId[]="@(#) $Id: ProblemDescDB.C 5851 2009-05-01 21:23:51Z wjbohnh $";

namespace Dakota {

extern ParallelLibrary dummy_lib; // defined in global_defs.C


/** This constructor is the one which must build the base class data for all
    derived classes.  get_db() instantiates a derived class letter and the
    derived constructor selects this base class constructor in its
    initialization list (to avoid the recursion of the base class constructor
    calling get_db() again).  Since the letter IS the representation, its
    representation pointer is set to NULL (an uninitialized pointer causes
    problems in ~ProblemDescDB). */
ProblemDescDB::ProblemDescDB(BaseConstructor, ParallelLibrary& parallel_lib):
  parallelLib(parallel_lib), strategyCntr(0), methodDBLocked(true),
  modelDBLocked(true), variablesDBLocked(true), interfaceDBLocked(true),
  responsesDBLocked(true), dbRep(NULL), referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "ProblemDescDB::ProblemDescDB(BaseConstructor) called to build base "
       << "class data for letter object." << endl;
#endif
}


/** The default constructor: dbRep is NULL in this case.  This makes
    it necessary to check for NULL in the copy constructor, assignment
    operator, and destructor. */
ProblemDescDB::ProblemDescDB(): parallelLib(dummy_lib), dbRep(NULL),
  referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "ProblemDescDB::ProblemDescDB() called to build empty db object."
       << endl;
#endif
}


/** This is the primary envelope constructor which uses problem_db to
    build a fully populated db object.  It only needs to
    extract enough data to properly execute get_db(problem_db),
    since the constructor overloaded with BaseConstructor builds the
    actual base class data inherited by the derived classes. */
ProblemDescDB::
ProblemDescDB(ParallelLibrary& parallel_lib,
	      CommandLineHandler& cmd_line_handler):
  parallelLib(parallel_lib),
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "ProblemDescDB::ProblemDescDB(ParallelLibrary&, CommandLineHandler&) "
       << "called to instantiate envelope." << endl;
#endif

  // Set the rep pointer to the appropriate db type
  dbRep = get_db(parallel_lib, cmd_line_handler);
  if (!dbRep) // bad settings or insufficient memory
    abort_handler(-1);
}


/** Initializes dbRep to the appropriate derived type.  The standard
    derived class constructors are invoked.  */
ProblemDescDB* ProblemDescDB::
get_db(ParallelLibrary& parallel_lib, CommandLineHandler& cmd_line_handler)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_db(ParallelLibrary&, "
       << "CommandLineHandler&)." << endl;
#endif

  //if (idr_flag)
  const char *cs;
  char *s;
  if (!( cs = cmd_line_handler.retrieve("parser") ) &&
       ( cs = getenv("DAKOTA_PARSER") )             &&
       (  s = (char*)malloc(strlen(cs)+1) ) ) {
    strcpy(s,cs);
    cs = s;
  }
#if 0
  if (cs && !strcmp(cs,"idr"))
    return new IDRProblemDescDB(parallel_lib);      // deprecated
  //else if (xml_flag)
  //  return new XMLProblemDescDB(parallel_lib);
  else
#endif
    return new NIDRProblemDescDB(parallel_lib, cs); // default

  // TO DO: when removing old IDR and the CommandLineHandler is no longer needed
  // for the derived class selection, move cs options to manage_inputs() so that
  // the ProblemDescDB ctor/get_db functions may be collapsed.
}


/** This is the library mode envelope constructor which does not have
    access to command line options. */
ProblemDescDB::ProblemDescDB(ParallelLibrary& parallel_lib):
  parallelLib(parallel_lib),
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "ProblemDescDB::ProblemDescDB(ParallelLibrary&) called to "
       << "instantiate envelope." << endl;
#endif

  // Set the rep pointer to the appropriate db type
  dbRep = get_db(parallel_lib);
  if (!dbRep) // bad settings or insufficient memory
    abort_handler(-1);
}


/** Initializes dbRep to an NIDR instance for library mode.  The standard
    derived class constructors are invoked.  */
ProblemDescDB* ProblemDescDB::get_db(ParallelLibrary& parallel_lib)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_db(ParallelLibrary&)." << endl;
#endif

  const char *cs = NULL; // no special options passed to NIDR
  return new NIDRProblemDescDB(parallel_lib, cs);
}


/** Copy constructor manages sharing of dbRep and incrementing of
    referenceCount. */
ProblemDescDB::ProblemDescDB(const ProblemDescDB& db):
  parallelLib(db.parallel_library())
{
  // Increment new (no old to decrement)
  dbRep = db.dbRep;
  if (dbRep) // Check for an assignment of NULL
    dbRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "ProblemDescDB::ProblemDescDB(ProblemDescDB&)" << endl;
  if (dbRep)
    Cout << "dbRep referenceCount = " << dbRep->referenceCount << endl;
#endif
}


/** Assignment operator decrements referenceCount for old dbRep, assigns
    new dbRep, and increments referenceCount for new dbRep. */
ProblemDescDB ProblemDescDB::operator=(const ProblemDescDB& db)
{
  if (dbRep != db.dbRep) { // normal case: old != new
    // Decrement old
    if (dbRep) // Check for NULL
      if ( --dbRep->referenceCount == 0 )
	delete dbRep;
    // Assign and increment new
    dbRep = db.dbRep;
    if (dbRep) // Check for NULL
      dbRep->referenceCount++;
  }
  // else if assigning same rep, then do nothing since referenceCount
  // should already be correct

#ifdef REFCOUNT_DEBUG
  Cout << "ProblemDescDB::operator=(ProblemDescDB&)" << endl;
  if (dbRep)
    Cout << "dbRep referenceCount = " << dbRep->referenceCount << endl;
#endif

  return *this; // calls copy constructor since returned by value
}


/** Destructor decrements referenceCount and only deletes dbRep
    when referenceCount reaches zero. */
ProblemDescDB::~ProblemDescDB()
{
  // Check for NULL pointer
  if (dbRep) {
    --dbRep->referenceCount;
#ifdef REFCOUNT_DEBUG
    Cout << "dbRep referenceCount decremented to " << dbRep->referenceCount
	 << endl;
#endif
    if (dbRep->referenceCount == 0) {
#ifdef REFCOUNT_DEBUG
      Cout << "deleting dbRep" << endl;
#endif
      delete dbRep;
    }
  }
}


/** Manage command line inputs using the CommandLineHandler class and
    parse the input file. */
void ProblemDescDB::
manage_inputs(CommandLineHandler& cmd_line_handler)
{
  if (dbRep)
    dbRep->manage_inputs(cmd_line_handler);
  else {
    // Only the master parses the input file.
    const char* dakota_input_file = (parallelLib.world_rank())
      ? NULL : cmd_line_handler.retrieve("input");
    manage_inputs(dakota_input_file); // non-NULL only for worldRank == 0
  }
}


/** Parse the input file, broadcast it to all processors, and
    post-process the data on all processors. */
void ProblemDescDB::
manage_inputs(const char* dakota_input_file, void (*callback)(void*),
	      void *callback_data)
{
  if (dbRep)
    dbRep->manage_inputs(dakota_input_file, callback, callback_data);
  else {
    // parse the input file, execute the callback (if present), and
    // check the keyword counts
    parse_inputs(dakota_input_file, callback, callback_data);

    // bcast a minimal MPI buffer containing the input specification
    // data prior to post-processing
    broadcast();

    // After broadcast, perform post-processing on all processors to
    // size default variables/responses specification vectors (avoid
    // sending large vectors over an MPI buffer).
    post_process();
  }
}


/** Parse the input file, execute the callback function (if present), and
    perform basic checks on keyword counts. */
void ProblemDescDB::
parse_inputs(const char* dakota_input_file, void (*callback)(void*),
	      void *callback_data)
{
  if (dbRep)
    dbRep->parse_inputs(dakota_input_file, callback, callback_data);
  else {
    // Only the master parses the input file.
    if (parallelLib.world_rank() == 0) {
      // Parse the input file using one of the derived parser-specific classes
      derived_manage_inputs(dakota_input_file);

      // Allow user input by callback function.
      // Note: the DB is locked and the list iterators are not defined.  Thus,
      // the user function must do something to put the DB in a usable set/get
      // state (e.g., resolve_top_method() or set_db_list_nodes()).
      if (callback)
	(*callback)(callback_data);

      // Check to make sure at least one of each of the keywords was found
      // in the problem specification file
      check_input();
    }
  }
}


void ProblemDescDB::broadcast()
{
  if (dbRep)
    dbRep->broadcast();
  else { // base class definition

    // DAKOTA's old design for reading the input file was for the master to get
    // the input filename from cmd_line_handler (after MPI_Init) and broadcast
    // the character buffer to all other processors (having every processor
    // query the cmd_line_handler was failing because of the effect of MPI_Init
    // on argc and argv).  Then every processor yyparsed.  This worked fine but
    // was not scalable for MP machines with a limited number of I/O devices.

    // The new design is for the master to yyparse and send all the parsed data
    // in a single buffer to the slaves.
    if (parallelLib.world_size() > 1) {
      if (parallelLib.world_rank() == 0) {
	// check and complete variable arrays, except labels, before transmitting
	NIDRProblemDescDB::finish_variables(&dataVariablesList);
	send_db_buffer();
	return;
	}
      else
	receive_db_buffer();
    }
  // for non-parallel or receive case, post_process now
  NIDRProblemDescDB::finish_variables(&dataVariablesList);
  }
}


void ProblemDescDB::derived_manage_inputs(const char* dakota_input_file)
{
  if (dbRep)
    dbRep->derived_manage_inputs(dakota_input_file);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual derived_manage_"
	 << "inputs function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
}


void ProblemDescDB::check_input()
{
  if (dbRep)
    dbRep->check_input();
  else {
    // NOTE: when using library mode in a parallel application, check_input()
    // should either be called only on worldRank 0, or it should follow a
    // matched send_db_buffer()/receive_db_buffer() pair.

    int num_errors = 0;
    //if (!strategyCntr) { // Allow strategy omission (default = single_method)
    //  Cerr << "No strategy specification found in input file.\n";
    //  num_errors++;
    //}
    if (strategyCntr > 1) {
      Cerr << "Multiple strategy specifications not allowed in input file.\n";
      num_errors++;
    }
    if (dataMethodList.empty()) {
      Cerr << "No method specification found in input file.\n";
      num_errors++;
    }
    if (dataModelList.empty()) { // Allow model omission
      DataModel data_model; // use defaults: modelType == "single"
      dataModelList.insert(data_model);
    }
    if (dataVariablesList.empty()) {
      Cerr << "No variables specification found in input file.\n";
      num_errors++;
    }
    if (dataInterfaceList.empty()) {
      Cerr << "No interface specification found in input file.\n";
      num_errors++;
    }
    if (dataResponsesList.empty()) {
      Cerr << "No responses specification found in input file.\n";
      num_errors++;
    }

    if (num_errors) {
      Cerr << num_errors << " input specification errors detected." << endl;
      abort_handler(-1);
    }
  }
}


void ProblemDescDB::post_process()
{
  if (dbRep)
    dbRep->post_process();
  else {
    // NOTE: when using library mode in a parallel application, post_process()
    // should be called on all processors following broadcast() of a minimal
    // problem specification.

    // finish processing dataVariableList
    NIDRProblemDescDB::make_variable_labels(&dataVariablesList);

    // check entries in dataResponsesList one at a time
    List<DataResponses>::iterator It  = dataResponsesList.begin();
    List<DataResponses>::iterator Ite = dataResponsesList.end();
    for(; It != Ite; It++)
      NIDRProblemDescDB::post_process_response_data(&*It);
  }
}


void ProblemDescDB::set_db_list_nodes(const String& method_tag)
{
  if (dbRep)
    dbRep->set_db_list_nodes(method_tag);
  else {
    set_db_method_node(method_tag);
    set_db_model_nodes(dataMethodIter->dataMethodRep->modelPointer);
  }
}


void ProblemDescDB::set_db_list_nodes(const size_t& method_index)
{
  if (dbRep)
    dbRep->set_db_list_nodes(method_index);
  else {
    // Set the correct Index values for all Data class lists.
    set_db_method_node(method_index);
    set_db_model_nodes(dataMethodIter->dataMethodRep->modelPointer);
  }
}


void ProblemDescDB::resolve_top_method()
{
  if (dbRep)
    dbRep->resolve_top_method();
  else { // deduce which method spec sits on top
    DataStrategyRep *SRep = strategySpec.dataStratRep;
    if (SRep->strategyType != "single_method") {
      // this fn should only be called for single_method/default strategies
      // since all other strategies have required, possibly multi-component,
      // method pointer specifications
      Cerr << "\nError: ProblemDescDB::resolve_top_method() only supported "
	   << "for single_method/default strategies.\n";
      abort_handler(-1);
    }
    size_t num_method_spec = dataMethodList.entries();
    if (num_method_spec == 1)
      dataMethodIter = dataMethodList.begin();
    else if (!SRep->methodPointer.empty())
      dataMethodIter = dataMethodList.find(data_method_id_compare,
					   &SRep->methodPointer);
    else { // identify which id_method does not appear in a method_pointer
      // Collect list of all method id's (including empty ids)
      StringList method_ids;
      for (List<DataMethod>::iterator it=dataMethodList.begin();
	   it!=dataMethodList.end(); it++)
	method_ids.insert(it->dataMethodRep->idMethod);
      // Eliminate sub-method pointers from method specs
      for (List<DataMethod>::iterator it=dataMethodList.begin();
	   it!=dataMethodList.end(); it++)
	if (!it->dataMethodRep->subMethodPointer.empty())
	  method_ids.remove(it->dataMethodRep->subMethodPointer);
      // Eliminate method_pointers from model specs
      for (List<DataModel>::iterator it=dataModelList.begin();
	   it!=dataModelList.end(); it++)
	if (!it->dataModelRep->subMethodPointer.empty())
	  method_ids.remove(it->dataModelRep->subMethodPointer);
      // by process of elimination, select the top method
      if (method_ids.empty() || method_ids.entries() > 1) {
	Cerr << "\nError: ProblemDescDB::resolve_top_method() failed to "
	     << "determine active method specification.\n       Please resolve "
	     << "method pointer ambiguities." << endl;
	abort_handler(-1);
      }
      else {
	const String& method_id = *method_ids.begin();
	dataMethodIter = dataMethodList.find(data_method_id_compare,&method_id);
      }
    }
    methodDBLocked = false; // unlock
    // set all subordinate list nodes for this method
    set_db_model_nodes(dataMethodIter->dataMethodRep->modelPointer);
  }
}


void ProblemDescDB::set_db_method_node(const String& method_tag)
{
  if (dbRep)
    dbRep->set_db_method_node(method_tag);
  else {
    // set the correct Index values for all Data class lists.
    if (method_tag.empty()) { // no pointer specification
      if (dataMethodList.entries() == 1) // no ambiguity if only one spec
	dataMethodIter = dataMethodList.begin();
      else { // try to match to a method without an id
	dataMethodIter
	  = dataMethodList.find(data_method_id_compare, &method_tag);
	if (dataMethodIter == dataMethodList.end()) {
	  if (parallelLib.world_rank() == 0)
	    Cerr << "\nWarning: empty method id string not found.\n         "
		 << "Last method specification parsed will be used.\n";
	  dataMethodIter--; // last entry in list
	}
	else if (parallelLib.world_rank() == 0 &&
		 dataMethodList.count(data_method_id_compare, &method_tag) > 1)
	  Cerr << "\nWarning: empty method id string is ambiguous.\n         "
	       << "First matching method specification will be used.\n";
      }
    }
    else {
      dataMethodIter = dataMethodList.find(data_method_id_compare, &method_tag);
      if (dataMethodIter == dataMethodList.end()) {
	Cerr << "\nError: " << method_tag
	     << " is not a valid method identifier string." << endl;
	abort_handler(-1);
      }
      else if (parallelLib.world_rank() == 0 &&
	       dataMethodList.count(data_method_id_compare, &method_tag) > 1)
	Cerr << "\nWarning: method id string " << method_tag << " is ambiguous."
	     << "\n         First matching method specification will be used."
	     << '\n';
    }
    methodDBLocked = false; // unlock
  }
}


void ProblemDescDB::set_db_method_node(const size_t& method_index)
{
  if (dbRep)
    dbRep->set_db_method_node(method_index);
  else {
    if (method_index >= dataMethodList.entries()) {
      Cerr << "\nError: method_index sent to set_db_list_nodes is out of range."
	   << endl;
      abort_handler(-1);
    }
    dataMethodIter = dataMethodList.begin();
    advance(dataMethodIter, method_index);
    methodDBLocked = false; // unlock
  }
}


void ProblemDescDB::set_db_model_nodes(const size_t& model_index)
{
  if (dbRep)
    dbRep->set_db_model_nodes(model_index);
  else {
    // Set the correct Index values for the Data class lists.
    if (model_index >= dataModelList.entries()) {
      Cerr << "\nError: model_index sent to set_db_list_nodes is out of range."
	   << endl;
      abort_handler(-1);
    }
    dataModelIter = dataModelList.begin();
    advance(dataModelIter, model_index);
    modelDBLocked = false; // unlock

    DataModelRep *MoRep = dataModelIter->dataModelRep;
    set_db_variables_node(MoRep->variablesPointer);
    set_db_interface_node(MoRep->interfacePointer);
    set_db_responses_node(MoRep->responsesPointer);
  }
}


void ProblemDescDB::set_db_model_nodes(const String& model_tag)
{
  if (dbRep)
    dbRep->set_db_model_nodes(model_tag);
  else {
    // set dataModelIter from model_tag
    if (model_tag.empty()) { // no pointer specification
      if (dataModelList.empty()) {
	DataModel data_model; // for library mode
	dataModelList.insert(data_model);
	}
      if (dataModelList.entries() == 1) // no ambiguity if only one spec
	dataModelIter = dataModelList.begin();
      else { // try to match to a model without an id
	dataModelIter = dataModelList.find(data_model_id_compare, &model_tag);
	if (dataModelIter == dataModelList.end()) {
	  if (parallelLib.world_rank() == 0)
	    Cerr << "\nWarning: empty model id string not found.\n         "
		 << "Last model specification parsed will be used.\n";
	  dataModelIter--; // last entry in list
	}
	else if (parallelLib.world_rank() == 0 &&
		 dataModelList.count(data_model_id_compare, &model_tag) > 1)
	  Cerr << "\nWarning: empty model id string is ambiguous.\n         "
	       << "First matching model specification will be used.\n";
      }
    }
    else {
      dataModelIter = dataModelList.find(data_model_id_compare, &model_tag);
      if (dataModelIter == dataModelList.end()) {
	Cerr << "\nError: " << model_tag
	     << " is not a valid model identifier string." << endl;
	abort_handler(-1);
      }
      else if (parallelLib.world_rank() == 0 &&
	       dataModelList.count(data_model_id_compare, &model_tag) > 1)
	Cerr << "\nWarning: model id string " << model_tag << " is ambiguous."
	     << "\n         First matching model specification will be used.\n";
    }
    modelDBLocked = false; // unlock

    DataModelRep *MoRep = dataModelIter->dataModelRep;
    set_db_variables_node(MoRep->variablesPointer);
    set_db_interface_node(MoRep->interfacePointer);
    set_db_responses_node(MoRep->responsesPointer);
  }
}


void ProblemDescDB::set_db_variables_node(const String& variables_tag)
{
  if (dbRep)
    dbRep->set_db_variables_node(variables_tag);
  else {
    // set dataVariablesIter from variables_tag
    if (variables_tag.empty()) { // no pointer specification
      if (dataVariablesList.entries() == 1) // no ambiguity if only one spec
	dataVariablesIter = dataVariablesList.begin();
      else { // try to match to a variables without an id
	dataVariablesIter
	  = dataVariablesList.find(data_variables_id_compare, &variables_tag);
	if (dataVariablesIter == dataVariablesList.end()) {
	  if (parallelLib.world_rank() == 0)
	    Cerr << "\nWarning: empty variables id string not found.\n         "
		 << "Last variables specification parsed will be used.\n";
	  dataVariablesIter--; // last entry in list
	}
	else if (parallelLib.world_rank() == 0 &&
		 dataVariablesList.count(data_variables_id_compare,
					 &variables_tag) > 1)
	  Cerr << "\nWarning: empty variables id string is ambiguous."
	       << "\n         First matching variables specification will be "
	       << "used.\n";
      }
    }
    else {
      dataVariablesIter
	= dataVariablesList.find(data_variables_id_compare, &variables_tag);
      if (dataVariablesIter == dataVariablesList.end()) {
	Cerr << "\nError: " << variables_tag
	     << " is not a valid variables identifier string." << endl;
	abort_handler(-1);
      }
      else if (parallelLib.world_rank() == 0 &&
	       dataVariablesList.count(data_variables_id_compare,
				       &variables_tag) > 1)
	Cerr << "\nWarning: variables id string " << variables_tag
	     << " is ambiguous.\n         First matching variables "
	     << "specification will be used.\n";
    }
    variablesDBLocked = false; // unlock
  }
}


void ProblemDescDB::set_db_interface_node(const String& interface_tag)
{
  if (dbRep)
    dbRep->set_db_interface_node(interface_tag);
  else {
    DataModelRep *MoRep = dataModelIter->dataModelRep;
    // set dataInterfaceIter from interface_tag
    if (interface_tag.empty()) { // no pointer specification
      if (dataInterfaceList.entries() == 1) // no ambiguity if only one spec
	dataInterfaceIter = dataInterfaceList.begin();
      else { // try to match to a interface without an id
	dataInterfaceIter
	  = dataInterfaceList.find(data_interface_id_compare, &interface_tag);
	// echo warning if a default interface list entry will be used and more
	// than 1 interface specification is present.  Currently this can only
	// happen for single models, since surrogate model specifications do not
	// contain interface pointers and the omission of an optional interface
	// pointer in nested models indicates the omission of an optional
	// interface (rather than the presence of an unidentified interface).
	if (dataInterfaceIter == dataInterfaceList.end()) {
	  if (parallelLib.world_rank() == 0 &&
	      MoRep->modelType == "single")
	    Cerr << "\nWarning: empty interface id string not found.\n         "
		 << "Last interface specification parsed will be used.\n";
	  dataInterfaceIter--; // last entry in list
	}
	else if (parallelLib.world_rank() == 0        &&
		 MoRep->modelType == "single" &&
		 dataInterfaceList.count(data_interface_id_compare,
					 &interface_tag) > 1)
	  Cerr << "\nWarning: empty interface id string is ambiguous."
	       << "\n         First matching interface specification will be "
	       << "used.\n";
      }
    }
    else {
      dataInterfaceIter
	= dataInterfaceList.find(data_interface_id_compare, &interface_tag);
      if (dataInterfaceIter == dataInterfaceList.end()) {
	Cerr << "\nError: " << interface_tag
	     << " is not a valid interface identifier string." << endl;
	abort_handler(-1);
      }
      else if (parallelLib.world_rank() == 0 &&
	       dataInterfaceList.count(data_interface_id_compare,
				       &interface_tag) > 1)
	Cerr << "\nWarning: interface id string " << interface_tag
	     << " is ambiguous.\n         First matching interface "
	     << "specification will be used.\n";
    }
    interfaceDBLocked = false; // unlock
  }
}


void ProblemDescDB::set_db_responses_node(const String& responses_tag)
{
  if (dbRep)
    dbRep->set_db_responses_node(responses_tag);
  else {
    // set dataResponsesIter from responses_tag
    if (responses_tag.empty()) { // no pointer specification
      if (dataResponsesList.entries() == 1) // no ambiguity if only one spec
	dataResponsesIter = dataResponsesList.begin();
      else { // try to match to a responses without an id
	dataResponsesIter
	  = dataResponsesList.find(data_responses_id_compare, &responses_tag);
	if (dataResponsesIter == dataResponsesList.end()) {
	  if (parallelLib.world_rank() == 0)
	    Cerr << "\nWarning: empty responses id string not found.\n         "
		 << "Last responses specification parsed will be used.\n";
	  dataResponsesIter--; // last entry in list
	}
	else if (parallelLib.world_rank() == 0 &&
		 dataResponsesList.count(data_responses_id_compare,
					 &responses_tag) > 1)
	  Cerr << "\nWarning: empty responses id string is ambiguous."
	       << "\n         First matching responses specification will be "
	       << "used.\n";
      }
    }
    else {
      dataResponsesIter
	= dataResponsesList.find(data_responses_id_compare, &responses_tag);
      if (dataResponsesIter == dataResponsesList.end()) {
	Cerr << "\nError: " << responses_tag
	     << " is not a valid responses identifier string." << endl;
	abort_handler(-1);
      }
      else if (parallelLib.world_rank() == 0 &&
	       dataResponsesList.count(data_responses_id_compare,
				       &responses_tag) > 1)
	Cerr << "\nWarning: responses id string " << responses_tag
	     << " is ambiguous.\n         First matching responses "
	     << "specification will be used.\n";
    }
    responsesDBLocked = false; // unlock
  }
}


void ProblemDescDB::send_db_buffer()
{
  MPIPackBuffer send_buffer;
  send_buffer << strategySpec      << dataMethodList    << dataModelList
	      << dataVariablesList << dataInterfaceList << dataResponsesList;

#ifdef MPI_DEBUG
  Cout << "DB buffer to send on world rank " << parallelLib.world_rank()
       << ":\n" << strategySpec << dataMethodList << dataVariablesList
       << dataInterfaceList << dataResponsesList << endl;
#endif // MPI_DEBUG

  // Broadcast length of buffer so that slaves can allocate MPIUnpackBuffer
  int buffer_len = send_buffer.size();
  parallelLib.bcast_w(buffer_len);

  // Broadcast actual buffer
  parallelLib.bcast_w(send_buffer);
}


void ProblemDescDB::receive_db_buffer()
{
  // receive length of incoming buffer and allocate space for MPIUnpackBuffer
  int buffer_len;
  parallelLib.bcast_w(buffer_len);

  // receive incoming buffer
  MPIUnpackBuffer recv_buffer(buffer_len);
  parallelLib.bcast_w(recv_buffer);
  recv_buffer >> strategySpec      >> dataMethodList    >> dataModelList
	      >> dataVariablesList >> dataInterfaceList >> dataResponsesList;

#ifdef MPI_DEBUG
  Cout << "DB buffer received on world rank " << parallelLib.world_rank()
       << ":\n" << strategySpec << dataMethodList << dataVariablesList
       << dataInterfaceList << dataResponsesList << endl;
#endif // MPI_DEBUG
}


const Iterator& ProblemDescDB::get_iterator(Model& model)
{
  // ProblemDescDB::get_<object> functions operate at the envelope level
  // so that any passing of *this provides the envelope object.
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_iterator() called for letter object."
	 << endl;
    abort_handler(-1);
  }

  // As get_response() must use a Variables instance, get_iterator() must
  // use a Model instance.  However, since the Iterator either points to a
  // particular Model or uses the last one parsed, model ambiguity is not
  // possible in reused iterator specifications.

  // In general, have to worry about loss of encapsulation and use of context
  // _above_ this specification.  However, any dependence on the strategy
  // specification is OK since there is only one.  All other specifications
  // are identified via model_pointer.

  // The DB list nodes are set prior to calling get_iterator():
  // >    method_ptr spec -> id_method must be defined
  // > no method_ptr spec -> id_method is ignored, method spec is last parsed
  // Reuse logic works in both cases -> only a single unreferenced iterator
  // may exist, which corresponds to the last method spec and is reused for
  // all untagged instantiations.
  const String& id_method = dbRep->dataMethodIter->dataMethodRep->idMethod;
  IterLIter i_it = dbRep->iteratorList.find(method_id_compare, &id_method);
  if (i_it == dbRep->iteratorList.end()) {
    Iterator new_iterator(model);
    dbRep->iteratorList.insert(new_iterator);
    i_it = --dbRep->iteratorList.end();
  }
  return *i_it;
}


const Model& ProblemDescDB::get_model()
{
  // ProblemDescDB::get_<object> functions operate at the envelope level
  // so that any passing of *this provides the envelope object.
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_model() called for letter object."<<endl;
    abort_handler(-1);
  }

  // A model specification identifies its variables, interface, and responses.
  // Have to worry about loss of encapsulation and use of context _above_ this
  // specification, i.e., any dependence on an iterator specification
  // (dependence on the strategy specification is OK since there is only one).
  // > method.output
  // > Constraints: method.linear_*, variables view

  // The DB list nodes are set prior to calling get_model():
  // >    model_ptr spec -> id_model must be defined
  // > no model_ptr spec -> id_model is ignored, model spec is last parsed
  const String& id_model = dbRep->dataModelIter->dataModelRep->idModel;
  ModelLIter m_it = dbRep->modelList.find(model_id_compare, &id_model);
  if (m_it == dbRep->modelList.end()) {
    Model new_model(*this);
    dbRep->modelList.insert(new_model);
    m_it = --dbRep->modelList.end();
  }
  return *m_it;
}


const Variables& ProblemDescDB::get_variables()
{
  // ProblemDescDB::get_<object> functions operate at the envelope level
  // so that any passing of *this provides the envelope object.
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_variables() called for letter object."
	 << endl;
    abort_handler(-1);
  }

  // Have to worry about loss of encapsulation and use of context _above_ this
  // specification, i.e., any dependence on iterator/model/interface/responses
  // specifications (dependence on the strategy specification is OK since there
  // is only one).
  // > variables view is method/strategy dependent

  // The DB list nodes are set prior to calling get_variables():
  // >    variables_ptr spec -> id_variables must be defined
  // > no variables_ptr spec -> id_variables ignored, vars spec = last parsed
  //const String& id_variables = dbRep->dataVariablesIter->idVariables;

  // Turn off variables reuse for now, since it is problematic with surrogates:
  // a top level variables set followed by a subModel eval which sets subModel
  // vars (where the subModel vars object is reused) results in a top level
  // eval with the wrong vars (e.g., surrogate auto-build in
  // dakota_textbook_lhs_approx.in).
  //
  // In general, variables object reuse should be fine for objects with peer
  // relationships, but are questionable for use among nested/layered levels.
  // Need a way to detect peer vs. nested/layered relationships.
  VarsLIter v_it;
  // = dbRep->variablesList.find(variables_id_compare, &id_variables);
  //if ( v_it == dbRep->variablesList.end() ||
  //     v_it->view() != v_it->get_view(*this) ) {
    Variables new_variables(*this);
    dbRep->variablesList.insert(new_variables);
    v_it = --dbRep->variablesList.end();
  //}
  return *v_it;
}


const Interface& ProblemDescDB::get_interface()
{
  // ProblemDescDB::get_<object> functions operate at the envelope level
  // so that any passing of *this provides the envelope object.
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_interface() called for letter object."
	 << endl;
    abort_handler(-1);
  }

  // Have to worry about loss of encapsulation and use of context _above_ this
  // specification, i.e., any dependence on iterator/model/variables/responses
  // specifications (dependence on the strategy specification is OK since there
  // is only one):
  // > Interface, AnalysisCode: method.output
  // > ApplicationInterface: responses.gradient_type, responses.hessian_type,
  //     responses.gradients.mixed.id_analytic
  // > DakotaInterface: responses.labels

  // ApproximationInterfaces and related classes are OK, since they are
  // instantiated with assign_rep() for each unique DataFitSurrModel instance:
  // > ApproximationInterface: model.surrogate.function_ids
  // > Approximation: method.output, model.surrogate.type,
  //     model.surrogate.gradient_usage
  // > SurfpackApproximation: model.surrogate.polynomial_order,
  //     model.surrogate.kriging_correlations
  // > TaylorApproximation: model.surrogate.actual_model_pointer,
  //     responses.hessian_type
  // > OrthogPolyApproximation: method.nond.expansion_{terms,order}

  // The DB list nodes are set prior to calling get_interface():
  // >    interface_ptr spec -> id_interface must be defined
  // > no interface_ptr spec -> id_interf ignored, interf spec = last parsed
  const String& id_interface
    = dbRep->dataInterfaceIter->dataIfaceRep->idInterface;
  InterfLIter i_it
    = dbRep->interfaceList.find(interface_id_compare, &id_interface);
  if (i_it == dbRep->interfaceList.end()) {
    Interface new_interface(*this);
    dbRep->interfaceList.insert(new_interface);
    i_it = --dbRep->interfaceList.end();
  }
  return *i_it;
}


const Response& ProblemDescDB::get_response(const Variables& vars)
{
  // ProblemDescDB::get_<object> functions operate at the envelope level
  // so that any passing of *this provides the envelope object.
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_response() called for letter object."
	 << endl;
    abort_handler(-1);
  }

  // Have to worry about loss of encapsulation and use of context _above_ this
  // specification, i.e., any dependence on iterator/model/variables/interface
  // specifications (dependence on the strategy specification is OK since there
  // is only one).
  // > mismatch in vars attributes (cv(),continuous_variable_ids()) should be OK
  //   since derivative arrays are dynamically resized based on current active
  //   set content

  // The DB list nodes are set prior to calling get_response():
  // >    responses_ptr spec -> id_responses must be defined
  // > no responses_ptr spec -> id_responses ignored, resp spec = last parsed
  //const String& id_responses
  //  = dbRep->dataResponsesIter->dataRespRep->idResponses;

  // Turn off response reuse for now, even though it has not yet been
  // problematic.  In general, response object reuse should be fine for objects
  // with peer relationships, but are questionable for use among nested/layered
  // levels.  Need a way to detect peer vs. nested/layered relationships.
  RespLIter r_it;
  // = dbRep->responseList.find(responses_id_compare,&id_responses);
  //if (r_it == dbRep->responseList.end()) { // ||
    //r_it->active_set_derivative_vector() != vars.continuous_variable_ids()) {
    Response new_response(vars, *this);
    dbRep->responseList.insert(new_response);
    r_it = --dbRep->responseList.end();
  //}
  return *r_it;
}


const RealDenseVector& ProblemDescDB::get_rdv(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_rdv() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  if (entry_name.begins("strategy.")) {
    if (entry_name == "strategy.concurrent.parameter_sets")
      return dbRep->strategySpec.dataStratRep->concurrentParameterSets;
    else
      err_flag = true;
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.linear_inequality_constraints")
      return MeRep->linearIneqConstraintCoeffs;
    else if (entry_name == "method.linear_inequality_lower_bounds")
      return MeRep->linearIneqLowerBnds;
    else if (entry_name == "method.linear_inequality_upper_bounds")
      return MeRep->linearIneqUpperBnds;
    else if (entry_name == "method.linear_equality_constraints")
      return MeRep->linearEqConstraintCoeffs;
    else if (entry_name == "method.linear_equality_targets")
      return MeRep->linearEqTargets;
    else if (entry_name == "method.parameter_study.final_point")
      return MeRep->finalPoint;
    else if (entry_name == "method.parameter_study.step_vector")
      return MeRep->stepVector;
    else if (entry_name == "method.parameter_study.list_of_points")
      return MeRep->listOfPoints;
    else
      err_flag = true;
  }
  // WJB: ask for dmgay review!
  else if (entry_name.begins("model.")) {
    if (dbRep->modelDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataModelRep *MoRep = dbRep->dataModelIter->dataModelRep;
    if (entry_name == "model.nested.primary_response_mapping")
      return MoRep->primaryRespCoeffs;
    else if (entry_name == "model.nested.secondary_response_mapping")
      return MoRep->secondaryRespCoeffs;
    else
      err_flag = true;
  }
  else if (entry_name.begins("variables.")) {
    DataVariablesRep* VRep = dbRep->dataVariablesIter->dataVarsRep;
    if (entry_name == "variables.continuous_design.initial_point")
      return VRep->continuousDesignVars;
    else if (entry_name == "variables.continuous_design.lower_bounds")
      return VRep->continuousDesignLowerBnds;
    else if (entry_name == "variables.continuous_design.upper_bounds")
      return VRep->continuousDesignUpperBnds;
    else if (entry_name == "variables.uncertain.initial_point")
      return VRep->uncertainVars;
    else if (entry_name == "variables.uncertain.lower_bounds")
      return VRep->uncertainLowerBnds;
    else if (entry_name == "variables.uncertain.upper_bounds")
      return VRep->uncertainUpperBnds;
    else if (entry_name == "variables.continuous_state.initial_state")
      return VRep->continuousStateVars;
    else if (entry_name == "variables.continuous_state.lower_bounds")
      return VRep->continuousStateLowerBnds;
    else if (entry_name == "variables.continuous_state.upper_bounds")
      return VRep->continuousStateUpperBnds;
    else if (entry_name == "variables.normal_uncertain.means")
      return VRep->normalUncMeans;
    else if (entry_name == "variables.normal_uncertain.std_deviations")
      return VRep->normalUncStdDevs;
    else if (entry_name == "variables.normal_uncertain.lower_bounds")
      return VRep->normalUncLowerBnds;
    else if (entry_name == "variables.normal_uncertain.upper_bounds")
      return VRep->normalUncUpperBnds;
    else if (entry_name == "variables.lognormal_uncertain.means")
      return VRep->lognormalUncMeans;
    else if (entry_name == "variables.lognormal_uncertain.std_deviations")
      return VRep->lognormalUncStdDevs;
    else if (entry_name == "variables.lognormal_uncertain.error_factors")
      return VRep->lognormalUncErrFacts;
    else if (entry_name == "variables.lognormal_uncertain.lower_bounds")
      return VRep->lognormalUncLowerBnds;
    else if (entry_name == "variables.lognormal_uncertain.upper_bounds")
      return VRep->lognormalUncUpperBnds;
    else if (entry_name == "variables.uniform_uncertain.lower_bounds")
      return VRep->uniformUncLowerBnds;
    else if (entry_name == "variables.uniform_uncertain.upper_bounds")
      return VRep->uniformUncUpperBnds;
    else if (entry_name == "variables.loguniform_uncertain.lower_bounds")
      return VRep->loguniformUncLowerBnds;
    else if (entry_name == "variables.loguniform_uncertain.upper_bounds")
      return VRep->loguniformUncUpperBnds;
    else if (entry_name == "variables.triangular_uncertain.modes")
      return VRep->triangularUncModes;
    else if (entry_name == "variables.triangular_uncertain.lower_bounds")
      return VRep->triangularUncLowerBnds;
    else if (entry_name == "variables.triangular_uncertain.upper_bounds")
      return VRep->triangularUncUpperBnds;
    else if (entry_name == "variables.exponential_uncertain.betas")
      return VRep->exponentialUncBetas;
    else if (entry_name == "variables.beta_uncertain.alphas")
      return VRep->betaUncAlphas;
    else if (entry_name == "variables.beta_uncertain.betas")
      return VRep->betaUncBetas;
    else if (entry_name == "variables.beta_uncertain.lower_bounds")
      return VRep->betaUncLowerBnds;
    else if (entry_name == "variables.beta_uncertain.upper_bounds")
      return VRep->betaUncUpperBnds;
    else if (entry_name == "variables.gamma_uncertain.alphas")
      return VRep->gammaUncAlphas;
    else if (entry_name == "variables.gamma_uncertain.betas")
      return VRep->gammaUncBetas;
    else if (entry_name == "variables.gumbel_uncertain.alphas")
      return VRep->gumbelUncAlphas;
    else if (entry_name == "variables.gumbel_uncertain.betas")
      return VRep->gumbelUncBetas;
    else if (entry_name == "variables.frechet_uncertain.alphas")
      return VRep->frechetUncAlphas;
    else if (entry_name == "variables.frechet_uncertain.betas")
      return VRep->frechetUncBetas;
    else if (entry_name == "variables.weibull_uncertain.alphas")
      return VRep->weibullUncAlphas;
    else if (entry_name == "variables.weibull_uncertain.betas")
      return VRep->weibullUncBetas;
  }
  else if (entry_name.begins("interface.")) {
    if (dbRep->interfaceDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    else if (entry_name == "interface.failure_capture.recovery_fn_vals")
      return dbRep->dataInterfaceIter->dataIfaceRep->recoveryFnVals;
    else
      err_flag = true;
  }
  else if (entry_name.begins("responses.")) {
    if (dbRep->responsesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataResponsesRep *RRep = dbRep->dataResponsesIter->dataRespRep;
    if (entry_name == "responses.primary_response_fn_weights")
      return RRep->primaryRespFnWeights;
    else if (entry_name == "responses.nonlinear_inequality_lower_bounds")
      return RRep->nonlinearIneqLowerBnds;
    else if (entry_name == "responses.nonlinear_inequality_upper_bounds")
      return RRep->nonlinearIneqUpperBnds;
    else if (entry_name == "responses.nonlinear_equality_targets")
      return RRep->nonlinearEqTargets;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_rdv: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const RealDenseVector&>(-1);
}

const RealVector& ProblemDescDB::get_drv(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_drv() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.linear_inequality_scales")
      return MeRep->linearIneqScales;
    else if (entry_name == "method.linear_equality_scales")
      return MeRep->linearEqScales;
    else if (entry_name == "method.jega.niche_vector")
      return MeRep->nicheVector;
    else if (entry_name == "method.jega.distance_vector")
      return MeRep->distanceVector;
    else
      err_flag = true;
  }
  else if (entry_name.begins("model.")) {
    if (dbRep->modelDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataModelRep *MoRep = dbRep->dataModelIter->dataModelRep;
    if (entry_name == "model.surrogate.kriging_correlations")
      return MoRep->krigingCorrelations;
    else if (entry_name == "model.surrogate.kriging_conmin_seed")
      return MoRep->krigingConminSeed;
    else if (entry_name == "model.surrogate.kriging_max_correlations")
      return MoRep->krigingMaxCorrelations;
    else if (entry_name == "model.surrogate.kriging_min_correlations")
      return MoRep->krigingMinCorrelations;
// WJB: primary/secondary RespCoeffs used to be here! (del comment after review)
    else
      err_flag = true;
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataVariablesRep* VRep = dbRep->dataVariablesIter->dataVarsRep;
    if (entry_name == "variables.continuous_design.scales")
      return VRep->continuousDesignScales;
    else
      err_flag = true;
  }
  else if (entry_name.begins("responses.")) {
    if (dbRep->responsesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataResponsesRep *RRep = dbRep->dataResponsesIter->dataRespRep;
    if (entry_name == "responses.primary_response_fn_scales")
      return RRep->primaryRespFnScales;
    else if (entry_name == "responses.nonlinear_inequality_scales")
      return RRep->nonlinearIneqScales;
    else if (entry_name == "responses.nonlinear_equality_scales")
      return RRep->nonlinearEqScales;
    else if (entry_name == "responses.fd_gradient_step_size")
      return RRep->fdGradStepSize;
    else if (entry_name == "responses.fd_hessian_step_size")
      return RRep->fdHessStepSize;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_drv: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const RealVector&>(-1);
}


const IntDenseVector& ProblemDescDB::get_idv(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_idv() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataVariablesRep* VRep = dbRep->dataVariablesIter->dataVarsRep;
    if (entry_name == "variables.discrete_design.initial_point")
      return VRep->discreteDesignVars;
    else if (entry_name == "variables.discrete_design.lower_bounds")
      return VRep->discreteDesignLowerBnds;
    else if (entry_name == "variables.discrete_design.upper_bounds")
      return VRep->discreteDesignUpperBnds;
    else if (entry_name == "variables.discrete_state.initial_state")
      return VRep->discreteStateVars;
    else if (entry_name == "variables.discrete_state.lower_bounds")
      return VRep->discreteStateLowerBnds;
    else if (entry_name == "variables.discrete_state.upper_bounds")
      return VRep->discreteStateUpperBnds;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_idv: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const IntDenseVector&>(-1);
}


const IntVector& ProblemDescDB::get_div(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_div() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.fsu_quasi_mc.sequenceStart")
      return MeRep->sequenceStart;
    else if (entry_name == "method.fsu_quasi_mc.sequenceLeap")
      return MeRep->sequenceLeap;
    else if (entry_name == "method.fsu_quasi_mc.primeBase")
      return MeRep->primeBase;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_div: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const IntVector&>(-1);
}


const UShortArray& ProblemDescDB::get_dusa(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_dusa() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.nond.expansion_order")
      return MeRep->expansionOrder;
    else if (entry_name == "method.nond.quadrature_order")
      return MeRep->quadratureOrder;
    else if (entry_name == "method.nond.sparse_grid_level")
      return MeRep->sparseGridLevel;
    else if (entry_name == "method.partitions")
      return MeRep->varPartitions;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_dusa: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const UShortArray&>(-1);
}


const RealSymDenseMatrix& ProblemDescDB::
get_rsdm(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_rsdm() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    else if (entry_name == "variables.uncertain.correlation_matrix")
      return dbRep->dataVariablesIter->dataVarsRep->uncertainCorrelations;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_rsdm: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const RealSymDenseMatrix&>(-1);
}


const RealDenseVectorArray& ProblemDescDB::
get_rdva(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_rdva() called with NULL representation."
	 <<endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataVariablesRep* VRep = dbRep->dataVariablesIter->dataVarsRep;
    if (entry_name == "variables.histogram_uncertain.bin_pairs")
      return VRep->histogramUncBinPairs;
    else if (entry_name == "variables.histogram_uncertain.point_pairs")
      return VRep->histogramUncPointPairs;
    else if (entry_name == "variables.interval_uncertain.basic_probs")
      return VRep->intervalUncBasicProbs;
    else if (entry_name == "variables.interval_uncertain.bounds")
      return VRep->intervalUncBounds;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_rdva: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const RealDenseVectorArray&>(-1);
}


const RealVectorArray& ProblemDescDB::get_drva(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_drva() called with NULL representation."
	 <<endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.nond.response_levels")
      return MeRep->responseLevels;
    else if (entry_name == "method.nond.probability_levels")
      return MeRep->probabilityLevels;
    else if (entry_name == "method.nond.reliability_levels")
      return MeRep->reliabilityLevels;
    else if (entry_name == "method.nond.gen_reliability_levels")
      return MeRep->genReliabilityLevels;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_drva: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const RealVectorArray&>(-1);
}


const IntList& ProblemDescDB::get_dil(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_dil() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("responses.")) {
    if (dbRep->responsesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataResponsesRep *RRep = dbRep->dataResponsesIter->dataRespRep;
    if (entry_name == "responses.gradients.mixed.id_analytic")
      return RRep->idAnalyticGrads;
    else if (entry_name == "responses.gradients.mixed.id_numerical")
      return RRep->idNumericalGrads;
    else if (entry_name == "responses.hessians.mixed.id_analytic")
      return RRep->idAnalyticHessians;
    else if (entry_name == "responses.hessians.mixed.id_numerical")
      return RRep->idNumericalHessians;
    else if (entry_name == "responses.hessians.mixed.id_quasi")
      return RRep->idQuasiHessians;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_dil: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const IntList&>(-1);
}


const IntSet& ProblemDescDB::get_dis(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_dis() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("model.")) {
    if (dbRep->modelDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    else if (entry_name == "model.surrogate.function_indices")
      return dbRep->dataModelIter->dataModelRep->surrogateFnIndices;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_dis: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const IntSet&>(-1);
}


const StringArray& ProblemDescDB::get_dsa(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_dsa() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("strategy.")) {
    if (entry_name == "strategy.hybrid.method_list")
      return dbRep->strategySpec.dataStratRep->hybridMethodList;
    else
      err_flag = true;
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.linear_inequality_scale_types")
      return MeRep->linearIneqScaleTypes;
    else if (entry_name == "method.linear_equality_scale_types")
      return MeRep->linearEqScaleTypes;
    else if (entry_name == "method.coliny.misc_options")
      return MeRep->miscOptions;
    else
      err_flag = true;
  }
  else if (entry_name.begins("model.")) {
    if (dbRep->modelDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataModelRep *MoRep = dbRep->dataModelIter->dataModelRep;
    if (entry_name == "model.nested.primary_variable_mapping")
      return MoRep->primaryVarMaps;
    else if (entry_name == "model.nested.secondary_variable_mapping")
      return MoRep->secondaryVarMaps;
    else if (entry_name == "model.diagnostics")
      return MoRep->diagMetrics;
    else
      err_flag = true;
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataVariablesRep* VRep = dbRep->dataVariablesIter->dataVarsRep;
    if (entry_name == "variables.continuous_design.labels")
      return VRep->continuousDesignLabels;
    else if (entry_name == "variables.continuous_design.scale_types")
      return VRep->continuousDesignScaleTypes;
    else if (entry_name == "variables.discrete_design.labels")
      return VRep->discreteDesignLabels;
    else if (entry_name == "variables.uncertain.labels")
      return VRep->uncertainLabels;
    else if (entry_name == "variables.continuous_state.labels")
      return VRep->continuousStateLabels;
    else if (entry_name == "variables.discrete_state.labels")
      return VRep->discreteStateLabels;
    else
      err_flag = true;
  }
  else if (entry_name.begins("interface.")) {
    if (dbRep->interfaceDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    else if (entry_name == "interface.application.analysis_drivers")
      return dbRep->dataInterfaceIter->dataIfaceRep->analysisDrivers;
    else
      err_flag = true;
  }
  else if (entry_name.begins("responses.")) {
    if (dbRep->responsesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataResponsesRep *RRep = dbRep->dataResponsesIter->dataRespRep;
    if (entry_name == "responses.labels")
      return RRep->responseLabels;
    else if (entry_name == "responses.primary_response_fn_scale_types")
      return RRep->primaryRespFnScaleTypes;
    else if (entry_name == "responses.nonlinear_inequality_scale_types")
      return RRep->nonlinearIneqScaleTypes;
    else if (entry_name == "responses.nonlinear_equality_scale_types")
      return RRep->nonlinearEqScaleTypes;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_dsa: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const StringArray&>(-1);
}


const String2DArray& ProblemDescDB::get_ds2a(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_d2sa() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("interface.")) {
    if (dbRep->interfaceDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    else if (entry_name == "interface.application.analysis_components")
      return dbRep->dataInterfaceIter->dataIfaceRep->analysisComponents;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_ds2a: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const String2DArray&>(-1);
}


const String& ProblemDescDB::get_string(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_string() called with NULL "
	 << "representation." << endl;
    abort_handler(-1);
  }
  if (entry_name.begins("strategy.")) {
    DataStrategyRep *SRep = dbRep->strategySpec.dataStratRep;
    if (entry_name == "strategy.type")
      return SRep->strategyType;
    else if (entry_name == "strategy.tabular_graphics_file")
      return SRep->tabularDataFile;
    else if (entry_name == "strategy.iterator_scheduling")
      return SRep->iteratorScheduling;
    else if (entry_name == "strategy.method_pointer")
      return SRep->methodPointer;
    else if (entry_name == "strategy.hybrid.type")
      return SRep->hybridType;
    else if (entry_name == "strategy.hybrid.global_method_pointer")
      return SRep->hybridGlobalMethodPointer;
    else if (entry_name == "strategy.hybrid.local_method_pointer")
      return SRep->hybridLocalMethodPointer;
    else
      err_flag = true;
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.id")
      return MeRep->idMethod;
    else if (entry_name == "method.algorithm")
      return MeRep->methodName;
    else if (entry_name == "method.model_pointer")
      return MeRep->modelPointer;
    else if (entry_name == "method.sub_method_name")
      return MeRep->subMethodName;
    else if (entry_name == "method.sub_method_pointer")
      return MeRep->subMethodPointer;
    else if (entry_name == "method.asynch_pattern_search.synchronization")
      return MeRep->evalSynchronize;
    else if (entry_name == "method.asynch_pattern_search.merit_function")
      return MeRep->meritFunction;
     else if (entry_name == "method.coliny.synchronization")
      return MeRep->evalSynchronization;
    else if (entry_name == "method.coliny.exploratory_moves")
      return MeRep->exploratoryMoves;
    else if (entry_name == "method.coliny.pattern_basis")
      return MeRep->patternBasis;
    else if (entry_name == "method.coliny.division")
      return MeRep->boxDivision;
    else if (entry_name == "method.dl_solver.dlDetails")
      return MeRep->dlDetails;
    else if (entry_name == "method.dot.minmax_type")
      return MeRep->minMaxType;
    else if (entry_name == "method.optpp.search_method")
      return MeRep->searchMethod;
    else if (entry_name == "method.optpp.merit_function")
      return MeRep->meritFn;
    else if (entry_name == "method.optpp.central_path")
      return MeRep->centralPath;
    else if (entry_name == "method.replacement_type")
      return MeRep->replacementType;
    else if (entry_name == "method.crossover_type")
      return MeRep->crossoverType;
    else if (entry_name == "method.mutation_type")
      return MeRep->mutationType;
    else if (entry_name == "method.sample_type")
      return MeRep->sampleType;
    else if (entry_name == "method.trial_type")
      return MeRep->trialType;
    else if (entry_name == "method.nond.distribution")
      return MeRep->distributionType;
    else if (entry_name == "method.nond.response_level_mapping_type")
      return MeRep->responseLevelMappingType;
    else if (entry_name == "method.nond.expansion_sample_type")
      return MeRep->expansionSampleType;
    else if (entry_name == "method.nond.collocation_sample_reuse")
      return MeRep->collocSampleReuse;
    else if (entry_name == "method.nond.expansion_import_file")
      return MeRep->expansionImportFile;
    else if (entry_name == "method.nond.reliability_search_type")
      return MeRep->reliabilitySearchType;
    else if (entry_name == "method.nond.reliability_search_algorithm")
      return MeRep->reliabilitySearchAlgorithm;
    else if (entry_name == "method.nond.reliability_integration")
      return MeRep->reliabilityIntegration;
    else if (entry_name == "method.nond.reliability_integration_refinement")
      return MeRep->reliabilityIntegrationRefine;
    else if (entry_name == "method.dace_method")
      return MeRep->daceMethod;
    else if (entry_name == "method.initialization_type")
      return MeRep->initializationType;
    else if (entry_name == "method.flat_file")
      return MeRep->flatFile;
    else if (entry_name == "method.log_file")
      return MeRep->logFile;
    else if (entry_name == "method.fitness_type")
      return MeRep->fitnessType;
    else if (entry_name == "method.jega.convergence_type")
      return MeRep->convergenceType;
    else if (entry_name == "method.jega.niching_type")
      return MeRep->nichingType;
    else if (entry_name == "method.jega.postprocessor_type")
      return MeRep->postProcessorType;
    else
      err_flag = true;
  }
  else if (entry_name.begins("model.")) {
    if (dbRep->modelDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataModelRep *MoRep = dbRep->dataModelIter->dataModelRep;
    if (entry_name == "model.id")
      return MoRep->idModel;
    else if (entry_name == "model.type")
      return MoRep->modelType;
    else if (entry_name == "model.nested.sub_method_pointer")
      return MoRep->subMethodPointer;
    else if (entry_name == "model.optional_interface_responses_pointer")
      return MoRep->optionalInterfRespPointer;
    else if (entry_name == "model.interface_pointer")
      return MoRep->interfacePointer;
    else if (entry_name == "model.surrogate.type")
      return MoRep->surrogateType;
    else if (entry_name == "model.surrogate.actual_model_pointer")
      return MoRep->truthModelPointer;
    else if (entry_name == "model.surrogate.low_fidelity_model_pointer")
      return MoRep->lowFidelityModelPointer;
    else if (entry_name == "model.surrogate.high_fidelity_model_pointer")
      return MoRep->truthModelPointer;
    else if (entry_name == "model.surrogate.dace_method_pointer")
      return MoRep->subMethodPointer;
    else if (entry_name == "model.surrogate.sample_reuse")
      return MoRep->approxSampleReuse;
    else if (entry_name == "model.surrogate.sample_reuse_file")
      return MoRep->approxSampleReuseFile;
    else if (entry_name == "model.surrogate.correction_type")
      return MoRep->approxCorrectionType;
    else if (entry_name == "model.surrogate.mars_interpolation")
      return MoRep->marsInterpolation;
    else
      err_flag = true;
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    else if (entry_name == "variables.id")
      return dbRep->dataVariablesIter->dataVarsRep->idVariables;
    else
      err_flag = true;
  }
  else if (entry_name.begins("interface.")) {
    if (dbRep->interfaceDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataInterfaceRep* IRep = dbRep->dataInterfaceIter->dataIfaceRep;
    if (entry_name == "interface.id")
      return IRep->idInterface;
    else if (entry_name == "interface.type")
      return IRep->interfaceType;
    else if (entry_name == "interface.algebraic_mappings")
      return IRep->algebraicMappings;
    else if (entry_name == "interface.application.input_filter")
      return IRep->inputFilter;
    else if (entry_name == "interface.application.output_filter")
      return IRep->outputFilter;
    else if (entry_name == "interface.application.parameters_file")
      return IRep->parametersFile;
    else if (entry_name == "interface.application.results_file")
      return IRep->resultsFile;
    else if (entry_name == "interface.synchronization")
      return IRep->interfaceSynchronization;
    else if (entry_name == "interface.evaluation_scheduling")
      return IRep->evalScheduling;
    else if (entry_name == "interface.analysis_scheduling")
      return IRep->analysisScheduling;
    else if (entry_name == "interface.failure_capture.action")
      return IRep->failAction;
    else
      err_flag = true;
  }
  else if (entry_name.begins("responses.")) {
    if (dbRep->responsesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataResponsesRep *RRep = dbRep->dataResponsesIter->dataRespRep;
    if (entry_name == "responses.id")
      return RRep->idResponses;
    else if (entry_name == "responses.least_squares_data_file")
      return RRep->leastSqDataFile;
    else if (entry_name == "responses.gradient_type")
      return RRep->gradientType;
    else if (entry_name == "responses.hessian_type")
      return RRep->hessianType;
    else if (entry_name == "responses.quasi_hessian_type")
      return RRep->quasiHessianType;
    else if (entry_name == "responses.method_source")
      return RRep->methodSource;
    else if (entry_name == "responses.interval_type")
      return RRep->intervalType;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_string: " << entry_name
         << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const String&>(-1);
}


const Real& ProblemDescDB::get_real(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_real() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  if (entry_name.begins("strategy.")) {
    //if (entry_name == "strategy.hybrid.progress_threshold")
    //  return dbRep->strategySpec.dataStratRep->hybridProgThresh;
    if (entry_name == "strategy.hybrid.local_search_probability")
      return dbRep->strategySpec.dataStratRep->hybridLSProb;
    else
      err_flag = true;
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.sbl.trust_region.initial_size")
      return MeRep->surrBasedLocalTRInitSize;
    else if (entry_name == "method.sbl.trust_region.minimum_size")
      return MeRep->surrBasedLocalTRMinSize;
    else if (entry_name == "method.sbl.trust_region.contract_threshold")
      return MeRep->surrBasedLocalTRContractTrigger;
    else if (entry_name == "method.sbl.trust_region.expand_threshold")
      return MeRep->surrBasedLocalTRExpandTrigger;
    else if (entry_name == "method.sbl.trust_region.contraction_factor")
      return MeRep->surrBasedLocalTRContract;
    else if (entry_name == "method.sbl.trust_region.expansion_factor")
      return MeRep->surrBasedLocalTRExpand;
    else if (entry_name == "method.asynch_pattern_search.initial_delta")
      return MeRep->initStepLength;
    else if (entry_name == "method.asynch_pattern_search.contraction_factor")
      return MeRep->contractStepLength;
    else if (entry_name == "method.asynch_pattern_search.threshold_delta")
      return MeRep->threshStepLength;
    else if (entry_name == "method.asynch_pattern_search.solution_target")
      return MeRep->solnTarget;
    else if (entry_name == "method.asynch_pattern_search.constraint_penalty")
      return MeRep->constrPenalty;
    else if (entry_name == "method.asynch_pattern_search.smoothing_factor")
      return MeRep->smoothFactor;
    else if (entry_name == "method.solution_accuracy")
      return MeRep->solnAccuracy;
     else if (entry_name == "method.coliny.initial_delta")
      return MeRep->initDelta;
    else if (entry_name == "method.coliny.threshold_delta")
      return MeRep->threshDelta;
    else if (entry_name == "method.coliny.contraction_factor")
      return MeRep->contractFactor;
    else if (entry_name == "method.constraint_penalty")
      return MeRep->constraintPenalty;
    else if (entry_name == "method.coliny.global_balance_parameter")
      return MeRep->globalBalanceParam;
    else if (entry_name == "method.coliny.local_balance_parameter")
      return MeRep->localBalanceParam;
    else if (entry_name == "method.coliny.max_boxsize_limit")
      return MeRep->maxBoxSize;
    else if (entry_name == "method.min_boxsize_limit")
      return MeRep->minBoxSize;
    else if (entry_name == "method.volume_boxsize_limit")
      return MeRep->volBoxSize;
    else if (entry_name == "method.parameter_study.step_length")
      return MeRep->stepLength;
    else if (entry_name == "method.parameter_study.percent_delta")
      return MeRep->percentDelta;
    else if (entry_name == "method.optpp.gradient_tolerance")
      return MeRep->gradientTolerance;
    else if (entry_name == "method.optpp.max_step")
      return MeRep->maxStep;
    else if (entry_name == "method.optpp.steplength_to_boundary")
      return MeRep->stepLenToBoundary;
    else if (entry_name == "method.optpp.centering_parameter")
      return MeRep->centeringParam;
    else if (entry_name == "method.crossover_rate")
      return MeRep->crossoverRate;
    else if (entry_name == "method.mutation_rate")
      return MeRep->mutationRate;
    else if (entry_name == "method.mutation_scale")
      return MeRep->mutationScale;
    else if (entry_name == "method.npsol.linesearch_tolerance")
      return MeRep->lineSearchTolerance;
    else if (entry_name == "method.function_precision")
      return MeRep->functionPrecision;
    else if (entry_name == "method.constraint_tolerance")
      return MeRep->constraintTolerance;
    else if (entry_name == "method.jega.percent_change")
      return MeRep->convergenceTolerance;
    else if (entry_name == "method.convergence_tolerance")
      return MeRep->convergenceTolerance;
    else if (entry_name == "method.jega.shrinkage_percentage")
      return MeRep->shrinkagePercent;
    else if (entry_name == "method.jega.fitness_limit")
      return MeRep->fitnessLimit;
    else if (entry_name == "method.nl2sol.absolute_conv_tol")
      return MeRep->absConvTol;
    else if (entry_name == "method.nl2sol.x_conv_tol")
      return MeRep->xConvTol;
    else if (entry_name == "method.nl2sol.singular_conv_tol")
      return MeRep->singConvTol;
    else if (entry_name == "method.nl2sol.singular_radius")
      return MeRep->singRadius;
    else if (entry_name == "method.nl2sol.false_conv_tol")
      return MeRep->falseConvTol;
    else if (entry_name == "method.nl2sol.initial_trust_radius")
      return MeRep->initTRRadius;
    else if (entry_name == "method.nond.collocation_ratio")
      return MeRep->collocationRatio;
    else
      err_flag = true;
  }
  else if (entry_name.begins("model.")) {
    if (dbRep->modelDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    if (entry_name == "model.surrogate.neural_network_range")
      return dbRep->dataModelIter->dataModelRep->annRange;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_real: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<const Real&>(-1);
}


int ProblemDescDB::get_int(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_int() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  if (entry_name.begins("strategy.")) {
    DataStrategyRep *SRep = dbRep->strategySpec.dataStratRep;
    if (entry_name == "strategy.iterator_servers")
      return SRep->iteratorServers;
    else if (entry_name == "strategy.concurrent.random_jobs")
      return SRep->concurrentRandomJobs;
    else if (entry_name == "strategy.concurrent.random_seed")
      return SRep->concurrentSeed;
    else
      err_flag = true;
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.sbl.soft_convergence_limit")
      return MeRep->surrBasedLocalSoftConvLimit;
    //else if (entry_name == "method.bandb.num_samples_at_root")
    //  return MeRep->branchBndNumSamplesRoot;
    //else if (entry_name == "method.bandb.num_samples_at_node")
    //  return MeRep->branchBndNumSamplesNode;
    else if (entry_name == "method.coliny.contract_after_failure")
      return MeRep->contractAfterFail;
    else if (entry_name == "method.coliny.expand_after_success")
      return MeRep->expandAfterSuccess;
    else if (entry_name == "method.coliny.total_pattern_size")
      return MeRep->totalPatternSize;
    else if (entry_name == "method.coliny.new_solutions_generated")
      return MeRep->newSolnsGenerated;
    else if (entry_name == "method.coliny.number_retained")
      return MeRep->numberRetained;
    else if (entry_name == "method.coliny.mutation_range")
      return MeRep->mutationRange;
    else if (entry_name == "method.max_iterations")
      return MeRep->maxIterations;
    else if (entry_name == "method.max_function_evaluations")
      return MeRep->maxFunctionEvaluations;
    else if (entry_name == "method.npsol.verify_level")
      return MeRep->verifyLevel;
    else if (entry_name == "method.optpp.search_scheme_size")
      return MeRep->searchSchemeSize;
    else if (entry_name == "method.population_size")
      return MeRep->populationSize;
    else if (entry_name == "method.nond.expansion_terms")
      return MeRep->expansionTerms;
    else if (entry_name == "method.nond.expansion_samples")
      return MeRep->expansionSamples;
    else if (entry_name == "method.nond.collocation_points")
      return MeRep->collocationPoints;
    else if (entry_name == "method.random_seed")
      return MeRep->randomSeed;
    else if (entry_name == "method.samples")
      return MeRep->numSamples;
    else if (entry_name == "method.previous_samples")
      return MeRep->previousSamples;
    else if (entry_name == "method.symbols")
      return MeRep->numSymbols;
    else if (entry_name == "method.parameter_study.num_steps")
      return MeRep->numSteps;
    else if (entry_name == "method.parameter_study.deltas_per_variable")
      return MeRep->deltasPerVariable;
    else if (entry_name == "method.nl2sol.covariance")
      return MeRep->covarianceType;
    else if (entry_name == "method.fsu_cvt.num_trials")
      return MeRep->numTrials;
    else
      err_flag = true;
  }
  else if (entry_name.begins("interface.")) {
    if (dbRep->interfaceDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataInterfaceRep* IRep = dbRep->dataInterfaceIter->dataIfaceRep;
    if (entry_name == "interface.failure_capture.retry_limit")
      return IRep->retryLimit;
    else if (entry_name == "interface.direct.processors_per_analysis")
      return IRep->procsPerAnalysis;
    else if (entry_name == "interface.asynch_local_evaluation_concurrency")
      return IRep->asynchLocalEvalConcurrency;
    else if (entry_name == "interface.asynch_local_analysis_concurrency")
      return IRep->asynchLocalAnalysisConcurrency;
    else if (entry_name == "interface.evaluation_servers")
      return IRep->evalServers;
    else if (entry_name == "interface.analysis_servers")
      return IRep->analysisServers;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_int: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<int>(-1);
}


short ProblemDescDB::get_short(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_short() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.output")
      return MeRep->methodOutput;
    else if (entry_name == "method.sbl.subproblem_objective")
      return MeRep->surrBasedLocalSubProbObj;
    else if (entry_name == "method.sbl.subproblem_constraints")
      return MeRep->surrBasedLocalSubProbCon;
    else if (entry_name == "method.sbl.merit_function")
      return MeRep->surrBasedLocalMeritFn;
    else if (entry_name == "method.sbl.acceptance_logic")
      return MeRep->surrBasedLocalAcceptLogic;
    else if (entry_name == "method.sbl.constraint_relax")
      return MeRep->surrBasedLocalConstrRelax;
    else if (entry_name == "method.parameter_study.type")
      return MeRep->paramStudyType;
    else
      err_flag = true;
  }
  else if (entry_name.begins("model.")) {
    if (dbRep->modelDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataModelRep *MoRep = dbRep->dataModelIter->dataModelRep;
    if (entry_name == "model.surrogate.polynomial_order")
      return MoRep->polynomialOrder;
    else if (entry_name == "model.surrogate.trend_order")
      return MoRep->trendOrder;
    else if (entry_name == "model.surrogate.correction_order")
      return MoRep->approxCorrectionOrder;
    else if (entry_name == "model.surrogate.kriging_max_trials")
      return MoRep->krigingMaxTrials;
    else if (entry_name == "model.surrogate.rbf_bases")
      return MoRep->rbfBases;
    else if (entry_name == "model.surrogate.rbf_min_partition")
      return MoRep->rbfMinPartition;
    else if (entry_name == "model.surrogate.rbf_max_pts")
      return MoRep->rbfMaxPts;
    else if (entry_name == "model.surrogate.rbf_max_subsets")
      return MoRep->rbfMaxSubsets;
    else if (entry_name == "model.surrogate.mls_weight_function")
      return MoRep->mlsWeightFunction;
    else if (entry_name == "model.surrogate.mls_poly_order")
      return MoRep->mlsPolyOrder;
    else if (entry_name == "model.surrogate.mars_max_bases")
      return MoRep->marsMaxBases;
    else if (entry_name == "model.surrogate.neural_network_random_weight")
      return MoRep->annRandomWeight;
    else if (entry_name == "model.surrogate.neural_network_nodes")
      return MoRep->annNodes;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_short: " << entry_name
	 << endl;
    abort_handler(-1);
  }

  return abort_handler_t<short>(-1);
}


size_t ProblemDescDB::get_sizet(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_sizet() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("strategy.")) {
    if (entry_name == "strategy.hybrid.num_solutions_transferred")
      return dbRep->strategySpec.dataStratRep->hybridNumSolnsTrans;
    else
      err_flag = true;
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.jega.num_cross_points")
      return MeRep->numCrossPoints;
    else if (entry_name == "method.jega.num_parents")
      return MeRep->numParents;
    else if (entry_name == "method.jega.num_offspring")
      return MeRep->numOffspring;
    else if (entry_name == "method.jega.num_generations")
      return MeRep->numGenerations;
    else
      err_flag = true;
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    List<DataVariables>::iterator v_iter = dbRep->dataVariablesIter;
    DataVariablesRep* VRep = v_iter->dataVarsRep;
    // DataVariables helper functions
    if (entry_name == "variables.design")
      return v_iter->design();
    else if (entry_name == "variables.aleatory_uncertain")
      return v_iter->aleatory_uncertain();
    else if (entry_name == "variables.epistemic_uncertain")
      return v_iter->epistemic_uncertain();
    else if (entry_name == "variables.uncertain")
      return v_iter->uncertain();
    else if (entry_name == "variables.state")
      return v_iter->state();
    else if (entry_name == "variables.continuous")
      return v_iter->continuous_variables();
    else if (entry_name == "variables.discrete")
      return v_iter->discrete_variables();
    else if (entry_name == "variables.total")
      return v_iter->total_variables();
    // normal DB lookups
    else if (entry_name == "variables.continuous_design")
      return VRep->numContinuousDesVars;
    else if (entry_name == "variables.discrete_design")
      return VRep->numDiscreteDesVars;
    else if (entry_name == "variables.normal_uncertain")
      return VRep->numNormalUncVars;
    else if (entry_name == "variables.lognormal_uncertain")
      return VRep->numLognormalUncVars;
    else if (entry_name == "variables.uniform_uncertain")
      return VRep->numUniformUncVars;
    else if (entry_name == "variables.loguniform_uncertain")
      return VRep->numLoguniformUncVars;
    else if (entry_name == "variables.triangular_uncertain")
      return VRep->numTriangularUncVars;
    else if (entry_name == "variables.exponential_uncertain")
      return VRep->numExponentialUncVars;
    else if (entry_name == "variables.beta_uncertain")
      return VRep->numBetaUncVars;
    else if (entry_name == "variables.gamma_uncertain")
      return VRep->numGammaUncVars;
    else if (entry_name == "variables.gumbel_uncertain")
      return VRep->numGumbelUncVars;
    else if (entry_name == "variables.frechet_uncertain")
      return VRep->numFrechetUncVars;
    else if (entry_name == "variables.weibull_uncertain")
      return VRep->numWeibullUncVars;
    else if (entry_name == "variables.histogram_uncertain")
      return VRep->numHistogramUncVars;
    else if (entry_name == "variables.interval_uncertain")
      return VRep->numIntervalUncVars;
    else if (entry_name == "variables.continuous_state")
      return VRep->numContinuousStateVars;
    else if (entry_name == "variables.discrete_state")
      return VRep->numDiscreteStateVars;
    else
      err_flag = true;
  }
  else if (entry_name.begins("responses.")) {
    if (dbRep->responsesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataResponsesRep *RRep = dbRep->dataResponsesIter->dataRespRep;
    if (entry_name == "responses.num_objective_functions")
      return RRep->numObjectiveFunctions;
    else if (entry_name == "responses.num_nonlinear_inequality_constraints")
      return RRep->numNonlinearIneqConstraints;
    else if (entry_name == "responses.num_nonlinear_equality_constraints")
      return RRep->numNonlinearEqConstraints;
    else if (entry_name == "responses.num_least_squares_terms")
      return RRep->numLeastSqTerms;
    else if (entry_name == "responses.num_response_functions")
      return RRep->numResponseFunctions;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_sizet: " << entry_name
         << endl;
    abort_handler(-1);
  }

  return abort_handler_t<size_t>(-1);
}


bool ProblemDescDB::get_bool(const String& entry_name) const
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::get_bool() called with NULL representation."
	 << endl;
    abort_handler(-1);
  }
  if (entry_name.begins("strategy.")) {
    DataStrategyRep *SRep = dbRep->strategySpec.dataStratRep;
    if (entry_name == "strategy.graphics")
      return SRep->graphicsFlag;
    else if (entry_name == "strategy.tabular_graphics_data")
      return SRep->tabularDataFlag;
    else
      err_flag = true;
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.scaling")
      return MeRep->methodScaling;
    else if (entry_name == "method.sbl.truth_surrogate_bypass")
      return MeRep->surrBasedLocalLayerBypass;
    else if (entry_name == "method.sbg.replace_points")
      return MeRep->surrBasedGlobalReplacePts;
    else if (entry_name == "method.coliny.constant_penalty")
      return MeRep->constantPenalty;
    else if (entry_name == "method.coliny.randomize")
      return MeRep->randomizeOrderFlag;
    else if (entry_name == "method.coliny.expansion")
      return MeRep->expansionFlag;
    else if (entry_name == "method.coliny.show_misc_options")
      return MeRep->showMiscOptions;
    else if (entry_name == "method.mutation_adaptive")
      return MeRep->mutationAdaptive;
    else if (entry_name == "method.speculative")
      return MeRep->speculativeFlag;
    else if (entry_name == "method.nl2sol.regression_diagnostics")
      return MeRep->regressDiag;
    else if (entry_name == "method.fixed_seed")
      return MeRep->fixedSeedFlag;
    else if (entry_name == "method.fsu_quasi_mc.fixed_sequence")
      return MeRep->fixedSequenceFlag;
    else if (entry_name == "method.nond.all_variables")
      return MeRep->allVarsFlag;
    else if (entry_name == "method.latinize")
      return MeRep->latinizeFlag;
    else if (entry_name == "method.quality_metrics")
      return MeRep->volQualityFlag;
    else if (entry_name == "method.main_effects")
      return MeRep->mainEffectsFlag;
    else if (entry_name == "method.variance_based_decomp")
      return MeRep->varBasedDecompFlag;
    else if (entry_name == "method.print_each_pop")
      return MeRep->printPopFlag;
    else
      err_flag = true;
  }
  else if (entry_name.begins("model.")) {
    if (dbRep->modelDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataModelRep *MoRep = dbRep->dataModelIter->dataModelRep;
    if (entry_name == "model.surrogate.gradient_usage")
      return MoRep->approxGradUsageFlag;
    if (entry_name == "model.surrogate.point_selection")
      return MoRep->pointSelection;
    else
      err_flag = true;
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    else if (entry_name == "variables.histogram_uncertain.skyline")
      return dbRep->dataVariablesIter->dataVarsRep->histogramUncSkylineFlag;
    else
      err_flag = true;
  }
  else if (entry_name.begins("interface.")) {
    if (dbRep->interfaceDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataInterfaceRep* IRep = dbRep->dataInterfaceIter->dataIfaceRep;
    if (entry_name == "interface.application.verbatim")
      return IRep->verbatimFlag;
    else if (entry_name == "interface.application.aprepro")
      return IRep->apreproFlag;
    else if (entry_name == "interface.application.file_tag")
      return IRep->fileTagFlag;
    else if (entry_name == "interface.application.file_save")
      return IRep->fileSaveFlag;
    else if (entry_name == "interface.active_set_vector")
      return IRep->activeSetVectorFlag;
    else if (entry_name == "interface.evaluation_cache")
      return IRep->evalCacheFlag;
    else if (entry_name == "interface.restart_file")
      return IRep->restartFileFlag;
    else
      err_flag = true;
  }
  else if (entry_name.begins("responses.")) {
    if (dbRep->responsesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataResponsesRep *RRep = dbRep->dataResponsesIter->dataRespRep;
    if (entry_name == "responses.ignore_bounds")
      return RRep->ignoreBounds;
    else if (entry_name == "responses.central_hess")
      return RRep->centralHess;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::get_bool: " << entry_name << endl;
    abort_handler(-1);
  }

  return abort_handler_t<bool>(-1);
}


void ProblemDescDB::set(const String& entry_name, const RealDenseVector& rdv)
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::set(RealDenseVector&) called with NULL "
	 << "representation." << endl;
    abort_handler(-1);
  }
  if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.linear_inequality_constraints")
      MeRep->linearIneqConstraintCoeffs = rdv;
    else if (entry_name == "method.linear_inequality_lower_bounds")
      MeRep->linearIneqLowerBnds = rdv;
    else if (entry_name == "method.linear_inequality_upper_bounds")
      MeRep->linearIneqUpperBnds = rdv;
    else if (entry_name == "method.linear_equality_constraints")
      MeRep->linearEqConstraintCoeffs = rdv;
    else if (entry_name == "method.linear_equality_targets")
      MeRep->linearEqTargets = rdv;
    else
      err_flag = true;
  }
  else if (entry_name.begins("model.")) {
    if (dbRep->modelDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataModelRep *MoRep = dbRep->dataModelIter->dataModelRep;
    if (entry_name == "model.nested.primary_response_mapping")
      MoRep->primaryRespCoeffs = rdv;
    else if (entry_name == "model.nested.secondary_response_mapping")
      MoRep->secondaryRespCoeffs = rdv;
    else
      err_flag = true;
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataVariablesRep* VRep = dbRep->dataVariablesIter->dataVarsRep;
    if (entry_name == "variables.continuous_design.initial_point")
      VRep->continuousDesignVars = rdv;
    else if (entry_name == "variables.continuous_design.lower_bounds")
      VRep->continuousDesignLowerBnds = rdv;
    else if (entry_name == "variables.continuous_design.upper_bounds")
      VRep->continuousDesignUpperBnds = rdv;
    else if (entry_name == "variables.uncertain.initial_point")
      VRep->uncertainVars = rdv;
    else if (entry_name == "variables.uncertain.lower_bounds")
      VRep->uncertainLowerBnds = rdv;
    else if (entry_name == "variables.uncertain.upper_bounds")
      VRep->uncertainUpperBnds = rdv;
    else if (entry_name == "variables.continuous_state.initial_state")
      VRep->continuousStateVars = rdv;
    else if (entry_name == "variables.continuous_state.lower_bounds")
      VRep->continuousStateLowerBnds = rdv;
    else if (entry_name == "variables.continuous_state.upper_bounds")
      VRep->continuousStateUpperBnds = rdv;
    else if (entry_name == "variables.normal_uncertain.means")
      VRep->normalUncMeans = rdv;
    else if (entry_name == "variables.normal_uncertain.std_deviations")
      VRep->normalUncStdDevs = rdv;
    else if (entry_name == "variables.normal_uncertain.lower_bounds")
      VRep->normalUncLowerBnds = rdv;
    else if (entry_name == "variables.normal_uncertain.upper_bounds")
      VRep->normalUncUpperBnds = rdv;
    else if (entry_name == "variables.lognormal_uncertain.means")
      VRep->lognormalUncMeans = rdv;
    else if (entry_name == "variables.lognormal_uncertain.std_deviations")
      VRep->lognormalUncStdDevs = rdv;
    else if (entry_name == "variables.lognormal_uncertain.error_factors")
      VRep->lognormalUncErrFacts = rdv;
    else if (entry_name == "variables.lognormal_uncertain.lower_bounds")
      VRep->lognormalUncLowerBnds = rdv;
    else if (entry_name == "variables.lognormal_uncertain.upper_bounds")
      VRep->lognormalUncUpperBnds = rdv;
    else if (entry_name == "variables.uniform_uncertain.lower_bounds")
      VRep->uniformUncLowerBnds = rdv;
    else if (entry_name == "variables.uniform_uncertain.upper_bounds")
      VRep->uniformUncUpperBnds = rdv;
    else if (entry_name == "variables.loguniform_uncertain.lower_bounds")
      VRep->loguniformUncLowerBnds = rdv;
    else if (entry_name == "variables.loguniform_uncertain.upper_bounds")
      VRep->loguniformUncUpperBnds = rdv;
    else if (entry_name == "variables.triangular_uncertain.modes")
      VRep->triangularUncModes = rdv;
    else if (entry_name == "variables.triangular_uncertain.lower_bounds")
      VRep->triangularUncLowerBnds = rdv;
    else if (entry_name == "variables.triangular_uncertain.upper_bounds")
      VRep->triangularUncUpperBnds = rdv;
    else if (entry_name == "variables.exponential_uncertain.betas")
      VRep->exponentialUncBetas = rdv;
    else if (entry_name == "variables.beta_uncertain.alphas")
      VRep->betaUncAlphas = rdv;
    else if (entry_name == "variables.beta_uncertain.betas")
      VRep->betaUncBetas = rdv;
    else if (entry_name == "variables.beta_uncertain.lower_bounds")
      VRep->betaUncLowerBnds = rdv;
    else if (entry_name == "variables.beta_uncertain.upper_bounds")
      VRep->betaUncUpperBnds = rdv;
    else if (entry_name == "variables.gamma_uncertain.alphas")
      VRep->gammaUncAlphas = rdv;
    else if (entry_name == "variables.gamma_uncertain.betas")
      VRep->gammaUncBetas = rdv;
    else if (entry_name == "variables.gumbel_uncertain.alphas")
      VRep->gumbelUncAlphas = rdv;
    else if (entry_name == "variables.gumbel_uncertain.betas")
      VRep->gumbelUncBetas = rdv;
    else if (entry_name == "variables.frechet_uncertain.alphas")
      VRep->frechetUncAlphas = rdv;
    else if (entry_name == "variables.frechet_uncertain.betas")
      VRep->frechetUncBetas = rdv;
    else if (entry_name == "variables.weibull_uncertain.alphas")
      VRep->weibullUncAlphas = rdv;
    else if (entry_name == "variables.weibull_uncertain.betas")
      VRep->weibullUncBetas = rdv;
    else
      err_flag = true;
  }
  else if (entry_name.begins("responses.")) {
    if (dbRep->responsesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataResponsesRep *RRep = dbRep->dataResponsesIter->dataRespRep;
    if (entry_name == "responses.primary_response_fn_weights")
      RRep->primaryRespFnWeights = rdv;
    else if (entry_name == "responses.nonlinear_inequality_lower_bounds")
      RRep->nonlinearIneqLowerBnds = rdv;
    else if (entry_name == "responses.nonlinear_inequality_upper_bounds")
      RRep->nonlinearIneqUpperBnds = rdv;
    else if (entry_name == "responses.nonlinear_equality_targets")
      RRep->nonlinearEqTargets = rdv;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::set(RealDenseVector&): "
	 << entry_name << endl;
    abort_handler(-1);
  }
}


void ProblemDescDB::set(const String& entry_name, const RealVector& drv)
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::set(RealVector&) called with NULL "
	 << "representation." << endl;
    abort_handler(-1);
  }
  if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.linear_inequality_scales")
      MeRep->linearIneqScales = drv;
    else if (entry_name == "method.linear_equality_scales")
      MeRep->linearEqScales = drv;
    else
      err_flag = true;
  }
// WJB: primary/secondary RespCoeffs used to be here! (del comment after review)
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataVariablesRep* VRep = dbRep->dataVariablesIter->dataVarsRep;
    if (entry_name == "variables.continuous_design.scales")
      VRep->continuousDesignScales = drv;
    else
      err_flag = true;
  }
  else if (entry_name.begins("responses.")) {
    if (dbRep->responsesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataResponsesRep *RRep = dbRep->dataResponsesIter->dataRespRep;
    if (entry_name == "responses.primary_response_fn_scales")
      RRep->primaryRespFnScales = drv;
    else if (entry_name == "responses.nonlinear_inequality_scales")
      RRep->nonlinearIneqScales = drv;
    else if (entry_name == "responses.nonlinear_equality_scales")
      RRep->nonlinearEqScales = drv;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::set(RealVector&): " << entry_name
	 << endl;
    abort_handler(-1);
  }
}


void ProblemDescDB::set(const String& entry_name, const IntDenseVector& idv)
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::set(IntVector&) called with NULL "
	 << "representation." << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataVariablesRep* VRep = dbRep->dataVariablesIter->dataVarsRep;
    if (entry_name == "variables.discrete_design.initial_point")
      VRep->discreteDesignVars = idv;
    else if (entry_name == "variables.discrete_design.lower_bounds")
      VRep->discreteDesignLowerBnds = idv;
    else if (entry_name == "variables.discrete_design.upper_bounds")
      VRep->discreteDesignUpperBnds = idv;
    else if (entry_name == "variables.discrete_state.initial_state")
      VRep->discreteStateVars = idv;
    else if (entry_name == "variables.discrete_state.lower_bounds")
      VRep->discreteStateLowerBnds = idv;
    else if (entry_name == "variables.discrete_state.upper_bounds")
      VRep->discreteStateUpperBnds = idv;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::set(IntVector&): " << entry_name
	 << endl;
    abort_handler(-1);
  }
}


void ProblemDescDB::
set(const String& entry_name, const RealSymDenseMatrix& rsdm)
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::set(RealSymDenseMatrix&) called with NULL "
	 << "representation." << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    else if (entry_name == "variables.uncertain.correlation_matrix")
      dbRep->dataVariablesIter->dataVarsRep->uncertainCorrelations = rsdm;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::set(RealSymDenseMatrix&): "
	 << entry_name << endl;
    abort_handler(-1);
  }
}


void ProblemDescDB::
set(const String& entry_name, const RealDenseVectorArray& rdva)
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::set(RealDenseVectorArray&) called with NULL "
	 << "representation." <<endl;
    abort_handler(-1);
  }
  if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataVariablesRep* VRep = dbRep->dataVariablesIter->dataVarsRep;
    if (entry_name == "variables.histogram_uncertain.bin_pairs")
      VRep->histogramUncBinPairs = rdva;
    else if (entry_name == "variables.histogram_uncertain.point_pairs")
      VRep->histogramUncPointPairs = rdva;
    else if (entry_name == "variables.interval_uncertain.basic_probs")
      VRep->intervalUncBasicProbs = rdva;
    else if (entry_name == "variables.interval_uncertain.bounds")
      VRep->intervalUncBounds = rdva;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::set(RealDenseVectorArray&): "
	 << entry_name << endl;
    abort_handler(-1);
  }
}


void ProblemDescDB::set(const String& entry_name, const RealVectorArray& drva)
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::set(RealVectorArray&) called with NULL "
	 << "representation." << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.nond.response_levels")
      MeRep->responseLevels = drva;
    else if (entry_name == "method.nond.probability_levels")
      MeRep->probabilityLevels = drva;
    else if (entry_name == "method.nond.reliability_levels")
      MeRep->reliabilityLevels = drva;
    else if (entry_name == "method.nond.gen_reliability_levels")
      MeRep->genReliabilityLevels = drva;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::set(RealVectorArray&): "
	 << entry_name << endl;
    abort_handler(-1);
  }
}


void ProblemDescDB::set(const String& entry_name, const StringArray& dsa)
{
  bool err_flag = false;
  if (!dbRep) {
    Cerr << "Error: ProblemDescDB::set(StringArray&) called with NULL "
	 << "representation." << endl;
    abort_handler(-1);
  }
  else if (entry_name.begins("method.")) {
    if (dbRep->methodDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataMethodRep *MeRep = dbRep->dataMethodIter->dataMethodRep;
    if (entry_name == "method.linear_inequality_scale_types")
      MeRep->linearIneqScaleTypes = dsa;
    else if (entry_name == "method.linear_equality_scale_types")
      MeRep->linearEqScaleTypes = dsa;
    else
      err_flag = true;
  }
  else if (entry_name.begins("model.")) {
    if (dbRep->modelDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataModelRep *MoRep = dbRep->dataModelIter->dataModelRep;
    if (entry_name == "model.nested.primary_variable_mapping")
      MoRep->primaryVarMaps = dsa;
    else if (entry_name == "model.nested.secondary_variable_mapping")
      MoRep->secondaryVarMaps = dsa;
    else if (entry_name == "model.diagnostics")
      MoRep->diagMetrics = dsa;
    else
      err_flag = true;
  }
  else if (entry_name.begins("variables.")) {
    if (dbRep->variablesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataVariablesRep* VRep = dbRep->dataVariablesIter->dataVarsRep;
    if (entry_name == "variables.continuous_design.labels")
      VRep->continuousDesignLabels = dsa;
    else if (entry_name == "variables.continuous_design.scale_types")
      VRep->continuousDesignScaleTypes = dsa;
    else if (entry_name == "variables.discrete_design.labels")
      VRep->discreteDesignLabels = dsa;
    else if (entry_name == "variables.uncertain.labels")
      VRep->uncertainLabels = dsa;
    else if (entry_name == "variables.continuous_state.labels")
      VRep->continuousStateLabels = dsa;
    else if (entry_name == "variables.discrete_state.labels")
      VRep->discreteStateLabels = dsa;
    else
      err_flag = true;
  }
  else if (entry_name.begins("responses.")) {
    if (dbRep->responsesDBLocked) {
      Cerr << "\nError: database is locked.  You must first unlock the database"
	   << "\n       by setting the list nodes." << endl;
      abort_handler(-1);
    }
    DataResponsesRep *RRep = dbRep->dataResponsesIter->dataRespRep;
    if (entry_name == "responses.labels")
      RRep->responseLabels = dsa;
    else if (entry_name == "responses.primary_response_fn_scale_types")
      RRep->primaryRespFnScaleTypes = dsa;
    else if (entry_name == "responses.nonlinear_inequality_scale_types")
      RRep->nonlinearIneqScaleTypes = dsa;
    else if (entry_name == "responses.nonlinear_equality_scale_types")
      RRep->nonlinearEqScaleTypes = dsa;
    else
      err_flag = true;
  }
  else
    err_flag = true;

  if (err_flag) {
    Cerr << "Bad entry_name in ProblemDescDB::set(StringArray&): " << entry_name
	 << endl;
    abort_handler(-1);
  }
}

} // namespace Dakota
