/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       NonDExpansion
//- Description: Iterator base class for polynomial expansion methods for UQ
//- Owner:       Mike Eldred, Sandia National Laboratories

#ifndef NOND_EXPANSION_H
#define NOND_EXPANSION_H

#include "DakotaNonD.H"


namespace Dakota {


/// Base class for polynomial chaos expansions (PCE) and stochastic
/// collocation (SC)

/** The NonDExpansion class provides a base class for methods that use
    polynomial expansions to approximate the effect of parameter
    uncertainties on response functions of interest. */

class NonDExpansion: public NonD
{
public:

  //
  //- Heading: Constructors and destructor
  //

  NonDExpansion(Model& model); ///< constructor
  ~NonDExpansion();            ///< destructor

  //
  //- Heading: Virtual function redefinitions
  //

  /// print the final statistics
  void print_results(ostream& s);

protected:

  //
  //- Heading: Member function definitions
  //

  /// recast iteratedModel from x-space to u-space to create g_u_model
  void construct_g_u_model(Model& g_u_model);

  /// assign a NonDQuadrature instance within u_space_sampler
  void construct_quadrature(Iterator& u_space_sampler, Model& g_u_model,
			    const UShortArray& quad_order);
  /// assign a NonDSparsegrid instance within u_space_sampler
  void construct_sparse_grid(Iterator& u_space_sampler, Model& g_u_model,
			     const UShortArray& sparse_grid_level);
  /// assign a NonDLHSSampling instance within u_space_sampler
  void construct_lhs(Iterator& u_space_sampler, Model& g_u_model,
		     int orig_seed);

  /// construct the expansionSampler operating on uSpaceModel
  void construct_expansion_sampler();
  /// initialize random variable definitions and final stats arrays
  void initialize_expansion();
  /// form the expansion by calling uSpaceModel.build_approximation()
  void compute_expansion();

  /// calculate analytic and numerical statistics from the expansion
  void compute_statistics();
  /// update finalStatistics
  void update_final_statistics();

  //
  //- Heading: Data
  //

  /// Model representing the approximate response function in u-space, after
  /// u-space recasting and orthogonal polynomial data fit recursions
  Model uSpaceModel;

  /// Iterator used for sampling on the uSpaceModel to generate approximate
  /// probability/reliability/response level statistics.  Currently this is
  /// an LHS sampling instance, but AIS could also be used.
  Iterator expansionSampler;

  /// method for collocation point generation and subsequent
  /// calculation of the expanion coefficients
  short expansionCoeffsApproach;

  /// number of invocations of quantify_uncertainty()
  size_t numUncertainQuant;

  /// number of truth samples performed on g_u_model to form the expansion
  int numSamplesOnModel;
  /// number of approximation samples performed on the polynomial
  /// expansion in order to estimate probabilities
  int numSamplesOnExpansion;

  /// stores the initial variables data in u-space
  RealDenseVector initialPtU;

  /// derivative of the expansion with respect to the x-space variables
  /// evaluated at the means (used as uncertainty importance metrics)
  RealMatrix expGradsMeanX;
};

} // namespace Dakota

#endif
