/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        Analyzer
//- Description:  Base class for NonD, DACE, and ParamStudy branches of the
//-               iterator hierarchy.
//- Owner:        Mike Eldred
//- Version: $Id: DakotaAnalyzer.H 5780 2009-03-26 22:14:19Z mseldre $

#ifndef DAKOTA_ANALYZER_H
#define DAKOTA_ANALYZER_H

#include "DakotaIterator.H"
//#include "DakotaResponse.H"


namespace Dakota {

/// Base class for NonD, DACE, and ParamStudy branches of the iterator
/// hierarchy.

/** The Analyzer class provides common data and functionality for
    various types of systems analysis, including nondeterministic
    analysis, design of experiments, and parameter studies. */

class Analyzer: public Iterator
{
public:

  //
  //- Heading: Virtual member function redefinitions
  //

  const VariablesArray&  all_variables() const;
  const ResponseArray&   all_responses() const;

  /// return multiple final iterator solutions (variables) 
  const VariablesArray& variables_array_results() const; 
  /// return multiple final iterator solutions (response) 
  const ResponseArray&  response_array_results()  const; 


protected:

  //
  //- Heading: Constructors and destructor
  //

  /// default constructor
  Analyzer();
  /// standard constructor
  Analyzer(Model& model);
  /// alternate constructor for instantiations "on the fly" with a Model
  Analyzer(NoDBBaseConstructor, Model& model);
  /// alternate constructor for instantiations "on the fly" without a Model
  Analyzer(NoDBBaseConstructor);
  /// destructor
  ~Analyzer();

  //
  //- Heading: Virtual functions
  //

  /// compares current evaluation to best evaluation and updates best
  virtual void update_best(const Variables& vars, const Response& response,
			   const int eval_num);

  /// sets varyPattern in derived classes that support it
  virtual void vary_pattern(bool pattern_flag);

  /// Returns one block of samples (ndim * num_samples)
  virtual void get_parameter_sets(const Model& model);

  //
  //- Heading: Member functions
  //

  /// perform function evaluations to map parameter sets (allVariables)
  /// into response sets (allResponses)
  void evaluate_parameter_sets(Model& model, bool log_resp_flag,
			       bool log_best_flag);

  void var_based_decomp(const int ndim, const int num_samples);
 
  /// Calculation of volumetric quality measures
  void volumetric_quality(int ndim, int num_samples, double* sample_points);

  /// Printing of VBD results
  void print_vbd(ostream& s, const RealVectorArray& S,
		 const RealVectorArray& T) const;

  //
  //- Heading: Data
  //

  /// array of all variables evaluated
  VariablesArray allVariables;
  /// array of all responses computed
  ResponseArray allResponses;
  /// array of headers to insert into output while evaluating allVariables
  StringArray allHeaders;
  /// flag to indicated if quality metrics were calculated
  bool qualityFlag;
  /// quality measure
  double chiMeas;
  /// quality measure
  double dMeas;
  /// quality measure
  double hMeas;
  /// quality measure
  double tauMeas;

private:

  //
  //- Heading: Data
  //
};


inline Analyzer::Analyzer()
{ }

inline Analyzer::~Analyzer() { }

inline const VariablesArray& Analyzer::all_variables() const
{ return allVariables; }

inline const ResponseArray& Analyzer::all_responses() const
{ return allResponses; }

inline const VariablesArray& Analyzer::variables_array_results() const
{ return allVariables; }

inline const ResponseArray& Analyzer::response_array_results() const
{ return allResponses; }

} // namespace Dakota

#endif
