/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

// $Id: CommandLineHandler.C 5731 2009-03-10 03:40:39Z mseldre $
// S Manoharan. Advanced Computer Research Institute. Lyon. France

#include "CommandLineHandler.H"
#include "DakotaBuildInfo.H"
#ifdef HAVE_MPI
#include <mpi.h>
#endif // HAVE_MPI


namespace Dakota {

GetLongOpt::GetLongOpt(const char optmark)
{
   table = last = 0;
   ustring = "[valid options and arguments]";
   enroll_done = 0;
   optmarker = optmark;
}

GetLongOpt::~GetLongOpt()
{
   Cell *t = table;

   while ( t ) {
      Cell *tmp = t;
      t = t->next;
      delete tmp;
   }
}

// MSE, 10/8/99: "pname" changed to "p" to quiet compiler warning
char * GetLongOpt::basename(char * const p) const
{
   char *s;

   s = strrchr(p, '/');
   if ( s == 0 ) s = p;
   else ++s;

   return s;
}

int GetLongOpt::enroll(const char * const opt, const OptType t,
		       const char * const desc, const char * const val)
{
   if ( enroll_done ) return 0;

   Cell *c = new Cell;
   c->option = opt;
   c->type = t;
   c->description = desc ? desc : "no description available";
   c->value = val;
   c->next = 0;

   if ( last == 0 ) {
      table = last = c;
   }
   else {
      last->next = c;
      last = c;
   }

   return 1;
}

const char * GetLongOpt::retrieve(const char * const opt) const
{
   Cell *t;
   for ( t = table; t != 0; t = t->next ) {
      if ( strcmp(opt, t->option) == 0 )
	 return t->value;
   }
   Cerr << "GetLongOpt::retrieve - unenrolled option "
        << optmarker << opt << '\n';
   return 0;
}

int GetLongOpt::parse(int argc, char * const *argv)
{
   int optind = 1;

   pname = basename(*argv);
   enroll_done = 1;
   if ( argc-- <= 1 ) return optind;

   while ( argc >= 1 ) {
      char *token = *++argv; --argc;

      if ( token[0] != optmarker || !token[1] )	/* allow "-" for stdin */
	 break;	/* end of options */
      if ( token[1] == optmarker ) {
		/* "--" ==> end of options; omit it and break */
		if ( !token[2] ) {
			++optind;
			break;
			}
		/* treat --opname as -opname */
		++token;
		}

      ++optind;
      char *tmptoken = ++token;
      while ( *tmptoken && *tmptoken != '=' )
	 ++tmptoken;
      /* (tmptoken - token) is now equal to the command line option
	 length. */

      Cell *t;
      enum { NoMatch, ExactMatch, PartialMatch } matchStatus = NoMatch;
      Cell *pc = 0;	// pointer to the partially-matched cell
      for ( t = table; t != 0; t = t->next ) {
	 if ( strncmp(t->option, token, (tmptoken - token)) == 0 ) {
	    if ( strlen(t->option) == (tmptoken - token) ) {
	       /* an exact match found */
	       int stat = setcell(t, tmptoken, *(argv+1), pname);
	       if ( stat == -1 ) return -1;
	       else if ( stat == 1 ) {
		  ++argv; --argc; ++optind;
	       }
	       matchStatus = ExactMatch;
	       break;
	    }
	    else {
	       /* partial match found */
	       matchStatus = PartialMatch;
	       pc = t;
	    }
	 } /* end if */
      } /* end for */

      if ( matchStatus == PartialMatch ) {
	 int stat = setcell(pc, tmptoken, *(argv+1), pname);
	 if ( stat == -1 ) return -1;
	 else if ( stat == 1 ) {
	    ++argv; --argc; ++optind;
	 }
      }
      else if ( matchStatus == NoMatch ) {
	 Cerr << pname << ": unrecognized option "
	      << optmarker << strtok(token,"= ") << '\n';
	 return -1;		/* no match */
      }

   } /* end while */

   return optind;
}

int GetLongOpt::parse(char * const str, char * const p)
{
   enroll_done = 1;
   char *token = strtok(str, " \t");
   const char *name = p ? p : "GetLongOpt";

   while ( token ) {
      if ( token[0] != optmarker || token[1] == optmarker ) {
	 Cerr << name << ": nonoptions not allowed\n";
	 return -1;	/* end of options */
      }

      char *ladtoken = 0;	/* lookahead token */
      char *tmptoken = ++token;
      while ( *tmptoken && *tmptoken != '=' )
	 ++tmptoken;
      /* (tmptoken - token) is now equal to the command line option
	 length. */

      Cell *t;
      enum { NoMatch, ExactMatch, PartialMatch } matchStatus = NoMatch;
      Cell *pc =0;	// pointer to the partially-matched cell
      for ( t = table; t != 0; t = t->next ) {
	 if ( strncmp(t->option, token, (tmptoken - token)) == 0 ) {
	    if ( strlen(t->option) == (tmptoken - token) ) {
	       /* an exact match found */
	       ladtoken = strtok(0, " \t");
	       int stat = setcell(t, tmptoken, ladtoken, name);
	       if ( stat == -1 ) return -1;
	       else if ( stat == 1 ) {
		  ladtoken = 0;
	       }
	       matchStatus = ExactMatch;
	       break;
	    }
	    else {
	       /* partial match found */
	       matchStatus = PartialMatch;
	       pc = t;
	    }
	 } /* end if */
      } /* end for */

      if ( matchStatus == PartialMatch ) {
	 ladtoken = strtok(0, " \t");
	 int stat = setcell(pc, tmptoken, ladtoken, name);
	 if ( stat == -1 ) return -1;
	 else if ( stat == 1 ) {
	    ladtoken = 0;
	 }
      }
      else if ( matchStatus == NoMatch ) {
	 Cerr << name << ": unrecognized option "
	      << optmarker << strtok(token,"= ") << '\n';
	 return -1;		/* no match */
      }

      token = ladtoken ? ladtoken : strtok(0, " \t");
   } /* end while */

   return 1;
}

/* ----------------------------------------------------------------
GetLongOpt::setcell returns
   -1	if there was an error
    0	if the nexttoken was not consumed
    1	if the nexttoken was consumed
------------------------------------------------------------------- */

int GetLongOpt::setcell(Cell *c, char *valtoken, char *nexttoken,
			const char *name)
{
   if ( c == 0 ) return -1;

   switch ( c->type ) {
   case GetLongOpt::Valueless :
      if ( *valtoken == '=' ) {
	 Cerr << name << ": unsolicited value for flag "
	      << optmarker << c->option << '\n';
	 return -1;	/* unsolicited value specification */
      }
      c->value = (c->value) ? 0 : (char *) ~0;
      return 0;
   case GetLongOpt::OptionalValue :
      if ( *valtoken == '=' ) {
	 c->value = ++valtoken;
	 return 0;
      }
      else {
	 if ( nexttoken != 0 && nexttoken[0] != optmarker ) {
	    c->value = nexttoken;
	    return 1;
	 }
	 else return 0;
      }
   case GetLongOpt::MandatoryValue :
      if ( *valtoken == '=' ) {
	 c->value = ++valtoken;
	 return 0;
      }
      else {
	 if ( nexttoken != 0 && nexttoken[0] != optmarker ) {
	    c->value = nexttoken;
	    return 1;
	 }
	 else {
	    Cerr << name << ": mandatory value for "
	         << optmarker << c->option << " not specified\n";
	    return -1;	/* mandatory value not specified */
	 }
      }
   default :
      break;
   }
   return -1;
}

void GetLongOpt::usage(ostream &outfile) const
{
   Cell *t;

   outfile << "usage: " << pname << " " << ustring << '\n';
   for ( t = table; t != 0; t = t->next ) {
      outfile << "\t" << optmarker << t->option;
      if ( t->type == GetLongOpt::MandatoryValue )
	 outfile << " <$val>";
      else if ( t->type == GetLongOpt::OptionalValue )
	 outfile << " [$val]";
      outfile << " (" << t->description << ")\n";
   }
   outfile.flush();
}

void GetLongOpt::store(const char *name, const char *value)
{
	Cell *t;
	for(t = table; t; t = t->next)
		if (!strcmp(name, t->option)) {
			t->value = value;
			break;
			}
	}


void CommandLineHandler::initialize_options()
{
  // NOTE: MandatoryValue means that a value must be specified _if_ the command
  // line option is used.  This does not make the command line option itself
  // mandatory.  Required command line inputs are enforced in check_usage().

  usage("[options and <args>]");

  enroll("help",    GetLongOpt::Valueless, "Print this summary", NULL);

  enroll("version", GetLongOpt::Valueless, "Print DAKOTA version number", NULL);

  enroll("check",   GetLongOpt::Valueless, "Perform input checks", NULL);

  enroll("dry_run", GetLongOpt::Valueless, "Perform dry run of study", NULL);

  enroll("input",   GetLongOpt::MandatoryValue,
	 "REQUIRED DAKOTA input file $val", NULL);

  enroll("output",  GetLongOpt::MandatoryValue,
	 "Redirect DAKOTA standard output to file $val", NULL);

  enroll("error",   GetLongOpt::MandatoryValue,
         "Redirect DAKOTA standard error to file $val", NULL);

  enroll("parser",  GetLongOpt::MandatoryValue,
	 "Parsing technology: nidr[strict][:dumpfile]", NULL);

  // The read restart filename string is mandatory since we want the user to be
  // explicit about the file to use when invoking the -read_restart option.  If
  // -read_restart is not invoked, then NULL will be returned by retrieve().
  enroll("read_restart", GetLongOpt::MandatoryValue,
         "Read an existing DAKOTA restart file $val", NULL);

  enroll("stop_restart", GetLongOpt::MandatoryValue,
         "Stop restart file processing at evaluation $val", NULL);

  // As for read restart, the write restart filename string is mandatory since
  // we want the user to be explicit about the file to use when invoking the
  // -write_restart option.  If -write_restart is not invoked, then NULL will
  // be returned by retrieve().
  //enroll("write_restart", GetLongOpt::OptionalValue,
  //       "Write a new DAKOTA restart file $val", "dakota.rst");
  enroll("write_restart", GetLongOpt::MandatoryValue,
         "Write a new DAKOTA restart file $val", NULL);

  //enroll("mpi", GetLongOpt::Valueless,
  //       "Turn on message passing within an executable built with MPI", 0);
}


void CommandLineHandler::check_usage(int argc, char** argv)
{
  const char *cs;
  int optind = parse(argc, argv);
  //Cout << "CommandLineHandler::check_usage: optind = " << optind << endl;

  if (optind < 1) {
//#ifdef USE_MPI
// Must comment out "return -1" for NoMatch case in parse
//    Cerr << "Warning: some unrecognized tokens in CommandLineHandler::"
//         << "check_usage.\n         Assuming tokens are MPI arguments." 
//         << endl;
//#else // if no MPI, then echo usage message and exit
    usage();
    abort_handler(-1);
//#endif // USE_MPI
  }

  if (retrieve("help") != NULL) {
    usage();
    abort_handler(0);
  }

  if (retrieve("version") != NULL) {
    output_version(Cout);
    abort_handler(0);
  }

  if (retrieve("input") == NULL) { // an input file is REQUIRED
    if (optind == argc - 1)
      GetLongOpt::store("input", argv[optind]);
    else {
      usage();
      Cerr << "Missing input file command line argument." << endl;
      abort_handler(-1);
    }
  }

  if (retrieve("read_restart") == NULL && retrieve("stop_restart") != NULL) {
    usage();
    Cerr <<  "\nread_restart is REQUIRED for use with stop_restart." << endl;
    abort_handler(-1);
  }

  cs = retrieve("parser");
  if (cs /*&& strncmp(cs,"idr",3)*/ && strncmp(cs,"nidr",4)) {
    usage();
    Cerr << "\n-parser must specify nidr...." << endl;
    abort_handler(-1);
  }
}


void CommandLineHandler::output_version(ostream& s) const
{
  int rank = 0;
#ifdef HAVE_MPI // mpi available
  int initialized = 0;
  MPI_Initialized(&initialized);
  if (initialized)
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);
#endif // HAVE_MPI

  if (rank == 0) {
    // Major/interim releases:
    //s << "DAKOTA version 4.2 released 11/05/2008." << endl;

    // Developmental/VOTD releases:
    s << "DAKOTA version 4.2+ developmental release." << endl;
    s << "Subversion revision " << DakotaBuildInfo::getInstance().getRev()
      << " built " << DakotaBuildInfo::getInstance().getDate()
      << " " << DakotaBuildInfo::getInstance().getTime() << "." << endl;
  }
}

} // namespace Dakota
