/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        BasisPolynomial
//- Description:  Class implementation of base class for basis polynomials
//-               
//- Owner:        Mike Eldred, Sandia National Laboratories

#include "BasisPolynomial.H"
#include "ProblemDescDB.H"
#include "HermiteOrthogPolynomial.H"
#include "LegendreOrthogPolynomial.H"
#include "JacobiOrthogPolynomial.H"
#include "LaguerreOrthogPolynomial.H"
#include "GenLaguerreOrthogPolynomial.H"
#include "GolubWelschOrthogPolynomial.H"
#include "LagrangeInterpPolynomial.H"


namespace Dakota {

/** This constructor is the one which must build the base class data
    for all derived classes.  get_polynomial() instantiates a derived
    class letter and the derived constructor selects this base class
    constructor in its initialization list (to avoid recursion in the
    base class constructor calling get_polynomial() again).  Since the
    letter IS the representation, its rep pointer is set to NULL (an
    uninitialized pointer causes problems in ~BasisPolynomial). */
BasisPolynomial::BasisPolynomial(BaseConstructor):// basisPolyType(-1),
  wtFactor(1.), ptFactor(1.), polyRep(NULL), referenceCount(1)
{

#ifdef REFCOUNT_DEBUG
  Cout << "BasisPolynomial::BasisPolynomial(BaseConstructor) called "
       << "to build base class for letter." << endl;
#endif
}


/** The default constructor is used in Array<BasisPolynomial>
    instantiations and by the alternate envelope constructor.  polyRep
    is NULL in this case (problem_db is needed to build a meaningful
    instance).  This makes it necessary to check for NULL in the copy
    constructor, assignment operator, and destructor. */
BasisPolynomial::BasisPolynomial(): polyRep(NULL), referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "BasisPolynomial::BasisPolynomial() called to build empty "
       << "basis polynomial object." << endl;
#endif
}


/** Envelope constructor which does not require access to problem_db.
    This constructor executes get_polynomial(type), which invokes the
    default constructor of the derived letter class, which in turn
    invokes the BaseConstructor of the base class. */
BasisPolynomial::BasisPolynomial(short poly_type):
  referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "BasisPolynomial::BasisPolynomial(short) called to "
       << "instantiate envelope." << endl;
#endif

  // Set the rep pointer to the appropriate derived type
  polyRep = get_polynomial(poly_type);
  if ( !polyRep ) // bad type or insufficient memory
    abort_handler(-1);
}


/** Used only by the envelope constructor to initialize polyRep to the 
    appropriate derived type. */
BasisPolynomial* BasisPolynomial::get_polynomial(short poly_type)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_polynomial(short)." << endl;
#endif

  BasisPolynomial* polynomial;
  if (poly_type == HERMITE)       // var_type == "normal"
    polynomial = new HermiteOrthogPolynomial();
  else if (poly_type == LEGENDRE) // var_type == "uniform"
    polynomial = new LegendreOrthogPolynomial();
  else if (poly_type == LAGUERRE) // var_type == "exponential"
    polynomial = new LaguerreOrthogPolynomial();
  else if (poly_type == JACOBI)   // var_type == "beta"
    polynomial = new JacobiOrthogPolynomial();
  else if (poly_type == GENERALIZED_LAGUERRE) // var_type == "gamma"
    polynomial = new GenLaguerreOrthogPolynomial();
  else if (poly_type == GOLUB_WELSCH)
    polynomial = new GolubWelschOrthogPolynomial();
  else if (poly_type == LAGRANGE)
    polynomial = new LagrangeInterpPolynomial();
  else {
    Cerr << "Error: BasisPolynomial type " << poly_type << " not available."
	 << endl;
    polynomial = NULL;
  }
  // Note: basisPolyType is not available at construct time, but is thereafter
  //if (polynomial)
  //  polynomial->basisPolyType = poly_type;
  return polynomial;
}


/** Copy constructor manages sharing of polyRep and incrementing of
    referenceCount. */
BasisPolynomial::BasisPolynomial(const BasisPolynomial& polynomial)
{
  // Increment new (no old to decrement)
  polyRep = polynomial.polyRep;
  if (polyRep) // Check for an assignment of NULL
    polyRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "BasisPolynomial::BasisPolynomial(BasisPolynomial&)" << endl;
  if (polyRep)
    Cout << "polyRep referenceCount = " << polyRep->referenceCount << endl;
#endif
}


/** Assignment operator decrements referenceCount for old polyRep,
    assigns new polyRep, and increments referenceCount for new polyRep. */
BasisPolynomial BasisPolynomial::operator=(const BasisPolynomial& polynomial)
{
  if (polyRep != polynomial.polyRep) { // normal case: old != new
    // Decrement old
    if (polyRep) // Check for NULL
      if ( --polyRep->referenceCount == 0 ) 
	delete polyRep;
    // Assign and increment new
    polyRep = polynomial.polyRep;
    if (polyRep) // Check for NULL
      polyRep->referenceCount++;
  }
  // else if assigning same rep, then do nothing since referenceCount
  // should already be correct

#ifdef REFCOUNT_DEBUG
  Cout << "BasisPolynomial::operator=(BasisPolynomial&)" << endl;
  if (polyRep)
    Cout << "polyRep referenceCount = " << polyRep->referenceCount << endl;
#endif

  return *this; // calls copy constructor since returned by value
}


/** Destructor decrements referenceCount and only deletes polyRep when
    referenceCount reaches zero. */
BasisPolynomial::~BasisPolynomial()
{ 
  // Check for NULL pointer 
  if (polyRep) {
    --polyRep->referenceCount;
#ifdef REFCOUNT_DEBUG
    Cout << "polyRep referenceCount decremented to " << polyRep->referenceCount
	 << endl;
#endif
    if (polyRep->referenceCount == 0) {
#ifdef REFCOUNT_DEBUG
      Cout << "deleting polyRep" << endl;
#endif
      delete polyRep;
    }
  }
}


const Real& BasisPolynomial::get_value(const Real& x, unsigned short n)
{
  if (!polyRep) {
    Cerr << "Error: get_value() not available for this basis polynomial type."
	 << endl;
    abort_handler(-1);
  }
  return polyRep->get_value(x, n);
}


const Real& BasisPolynomial::get_gradient(const Real& x, unsigned short n)
{
  if (!polyRep) {
    Cerr << "Error: get_gradient() not available for this basis polynomial "
	 << "type." << endl;
    abort_handler(-1);
  }
  return polyRep->get_gradient(x, n);
}


const Real& BasisPolynomial::norm_squared(unsigned short n)
{
  if (!polyRep) {
    Cerr << "Error: norm_squared() not available for this basis polynomial "
	 << "type." << endl;
    abort_handler(-1);
  }
  return polyRep->norm_squared(n);
}


const RealVector& BasisPolynomial::gauss_points(unsigned short n)
{
  if (!polyRep) {
    Cerr << "Error: gauss_points() not available for this basis polynomial "
	 << "type." << endl;
    abort_handler(-1);
  }
  return polyRep->gauss_points(n);
}


const RealVector& BasisPolynomial::gauss_weights(unsigned short n)
{
  if (!polyRep) {
    Cerr << "Error: gauss_weights() not available for this basis polynomial "
	 << "type." << endl;
    abort_handler(-1);
  }
  return polyRep->gauss_weights(n);
}


void BasisPolynomial::reset_gauss()
{
  if (polyRep)
    polyRep->reset_gauss();
  else {
    Cerr << "Error: reset_gauss() not available for this basis polynomial type."
	 << endl;
    abort_handler(-1);
  }
}


const Real& BasisPolynomial::point_factor()
{
  if (polyRep)
    return polyRep->point_factor();
  else // default is used whenever ptFactor does not need to be updated
    return ptFactor;
}


const Real& BasisPolynomial::weight_factor()
{
  if (polyRep)
    return polyRep->weight_factor();
  else // default is used whenever wtFactor does not need to be updated
    return wtFactor;
}


void BasisPolynomial::alpha_polynomial(const Real& alpha)
{
  if (polyRep)
    polyRep->alpha_polynomial(alpha);
  else {
    Cerr << "Error: alpha_polynomial() not available for this basis polynomial "
	 << "type." << endl;
    abort_handler(-1);
  }
}


void BasisPolynomial::beta_polynomial(const Real& beta)
{
  if (polyRep)
    polyRep->beta_polynomial(beta);
  else {
    Cerr << "Error: beta_polynomial() not available for this basis polynomial "
	 << "type." << endl;
    abort_handler(-1);
  }
}


void BasisPolynomial::alpha_stat(const Real& alpha)
{
  if (polyRep)
    polyRep->alpha_stat(alpha);
  else {
    Cerr << "Error: alpha_stat() not available for this basis polynomial "
	 << "type." << endl;
    abort_handler(-1);
  }
}


void BasisPolynomial::beta_stat(const Real& beta)
{
  if (polyRep)
    polyRep->beta_stat(beta);
  else {
    Cerr << "Error: beta_stat() not available for this basis polynomial "
	 << "type." << endl;
    abort_handler(-1);
  }
}


void BasisPolynomial::interpolation_points(const RealVector& interpolation_pts)
{
  if (polyRep)
    polyRep->interpolation_points(interpolation_pts);
  else {
    Cerr << "Error: interpolation_points() not available for this basis "
	 << "polynomial type." << endl;
    abort_handler(-1);
  }
}

} // namespace Dakota
