# include "sandia_rules.H"
# include "sparse_grid_mixed_growth.H"

# include <cstdlib>
# include <iomanip>
# include <iostream>
# include <cmath>

int main ( );

void product_mixed_growth_weight_tests ( );
void product_mixed_growth_weight_test ( int dim_num, int order_1d[], 
  int order_nd, int rule[], double alpha[], double beta[],
  void ( *gw_compute_weights[] ) ( int order, double alpha, double beta, double w[] ) );

void sparse_grid_mixed_growth_index_tests ( double tol );
void sparse_grid_mixed_growth_index_test ( int dim_num, int level_max_min, 
  int level_max_max, int rule[], double alpha[], double beta[], 
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  double tol );

void sparse_grid_mixed_growth_point_tests ( double tol );
void sparse_grid_mixed_growth_point_test ( int dim_num, int level_max_min, 
  int level_max_max, int rule[], double alpha[], double beta[], 
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  double tol );

void sparse_grid_mixed_growth_size_tests ( double tol );
void sparse_grid_mixed_growth_size_test ( int dim_num, int level_max_min, 
  int level_max_max, int rule[], double alpha[], double beta[], 
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  double tol );

void sparse_grid_mixed_growth_unique_index_tests ( double tol );
void sparse_grid_mixed_growth_unique_index_test ( int dim_num, int level_max_min, 
  int level_max_max, int rule[], double alpha[], double beta[], 
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  double tol );

void sparse_grid_mixed_growth_weight_tests ( double tol );
void sparse_grid_mixed_growth_weight_test ( int dim_num, int level_max_min, 
  int level_max_max, int rule[], double alpha[], double beta[],
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  void ( *gw_compute_weights[] ) ( int order, double alpha, double beta, double w[] ),
  double tol );

void sparse_grid_mixed_growth_write_tests ( double tol );
void sparse_grid_mixed_growth_write_test ( int dim_num, int level_max, 
  int rule[], double alpha[], double beta[], 
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  void ( *gw_compute_weights[] ) ( int order, double alpha, double beta, double w[] ),
  double tol, std::string file_name );

typedef void ( *GWPointer ) ( int order, double alpha, double beta, double w[] );

//****************************************************************************80

int main ( )

//****************************************************************************80
//
//  Purpose:
//
//    MAIN is the main program for SPARSE_GRID_MIXED_GROWTH_PRB.
//
//  Discussion:
//
//    SPARSE_GRID_MIXED_GROWTH_PRB tests the SPARSE_GRID_MIXED_GROWTH routines.
//
//    The amount of output from these tests is somewhat extensive.  To control
//    this, I've put a logical statement around the call to each test.
//    I typically run just one test, and I choose it by setting the logical
//    constant that controls that call to TRUE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    16 April 2009
//
//  Author:
//
//    John Burkardt
//
{
  double tol;

  webbur::timestamp ( );
  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_PRB\n";
  std::cout << "  C++ version\n";
  std::cout << "\n";
  std::cout << "  Test the routines in the SPARSE_GRID_MIXED_GROWTH library.\n";
//
//  Make sure the individual product rule weights are computed correctly.
//
  if ( false )
  {
    product_mixed_growth_weight_tests ( );
  }
//
//  1) Using a tolerance that is less than 0 means that there will be no
//  consolidation of duplicate points.
//
//  2) Using a small positive tolerance means there will be consolidation of
//  points whose maximum difference is no more than TOL.
//
  if ( false )
  {
    tol = - 1.0;
    sparse_grid_mixed_growth_size_tests ( tol );

    tol = std::sqrt ( webbur::r8_epsilon ( ) );
    sparse_grid_mixed_growth_size_tests ( tol );
  }
//
//  Check that we can determine a UNIQUE_INDEX mapping, so that we can
//  generate all the points, and then select unique representatives, and
//  then match each point to its representative.
//
  if ( false )
  {
    sparse_grid_mixed_growth_unique_index_tests ( tol );
  }
//
//  Compute the INDEX and ORDER arrays which describe, for a given abscissa 
//  in a sparse grid, for each component of that abscissa, the index of the 
//  1D component as a member of a particular 1D quadrature rule of given ORDER.
//
  if ( false )
  {
    sparse_grid_mixed_growth_index_tests ( tol );
  }
//
//  Generate the abscissas for sparse grid rules.
//
  if ( false ) 
  {
    sparse_grid_mixed_growth_point_tests ( tol );
  }
//
//  Generate the weights for sparse grid rules.
//
  if ( true )
  {
    sparse_grid_mixed_growth_weight_tests ( tol );
  }
//
//  Generate sparse grid rules and write them to files.
//
  if ( false )
  {
    sparse_grid_mixed_growth_write_tests ( tol );
  }
//
//  That's all.
//
  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_PRB\n";
  std::cout << "  Normal end of execution.\n";

  std::cout << "\n";
  webbur::timestamp ( );
  
  return 0;
}
//****************************************************************************80

void product_mixed_growth_weight_tests ( )

//****************************************************************************80
//
//  Purpose:
//
//    PRODUCT_MIXED_GROWTH_WEIGHT_TESTS calls PRODUCT_MIXED_GROWTH_WEIGHT_TEST.
//
//  Discussion:
//
//    To test Golub Welsch rules for a spatial dimension DIM, we can
//    set RULE[DIM] = 10, and set the corresponding entry of 
//    GW_COMPUTE_WEIGHTS to the name of a function that we know is already
//    available, such as "webbur::clenshaw_curtis_compute_weights".
//
//    Note that, for ALL the tests, we set every entry of the GW_COMPUTE_WEIGHTS
//    array.  However, a particular entry is only inspected if the corresponding
//    entry of RULE is 10.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    16 April 2009
//
//  Author:
//
//    John Burkardt
//
{
  double *alpha;
  double *beta;
  int dim_num;
  GWPointer *gw_compute_weights;
  int *order_1d;
  int order_nd;
  int *rule;

  std::cout << "\n";
  std::cout << "PRODUCT_MIXED_GROWTH_WEIGHT_TESTS\n";
  std::cout << "  Call PRODUCT_MIXED_GROWTH_WEIGHT_TEST with various arguments.\n";

  dim_num = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  order_1d = new int[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  order_1d[0] = 3;
  order_1d[1] = 5;
  order_nd = webbur::i4vec_product ( dim_num, order_1d );
  rule[0] = 1;
  rule[1] = 1;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::clenshaw_curtis_compute_weights;
  product_mixed_growth_weight_test ( dim_num, order_1d, order_nd, rule, 
    alpha, beta, gw_compute_weights );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_weights;
  delete [] order_1d;
  delete [] rule;

  dim_num = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  order_1d = new int[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  order_1d[0] = 3;
  order_1d[1] = 7;
  order_nd = webbur::i4vec_product ( dim_num, order_1d );
  rule[0] = 1;
  rule[1] = 5;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::hermite_compute_weights;
  product_mixed_growth_weight_test ( dim_num, order_1d, order_nd, rule, 
    alpha, beta, gw_compute_weights );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_weights;
  delete [] order_1d;
  delete [] rule;

  dim_num = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  order_1d = new int[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  order_1d[0] = 3;
  order_1d[1] = 3;
  order_nd = webbur::i4vec_product ( dim_num, order_1d );
  rule[0] = 3;
  rule[1] = 7;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::patterson_lookup_weights;
  gw_compute_weights[1] = webbur::laguerre_compute_weights;
  product_mixed_growth_weight_test ( dim_num, order_1d, order_nd, rule, 
    alpha, beta, gw_compute_weights );
  delete [] alpha;
  delete [] beta;
  delete [] order_1d;
  delete [] rule;

  dim_num = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  order_1d = new int[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 1.5;
  beta[0] = 0.0;
  beta[1] = 0.0;
  order_1d[0] = 5;
  order_1d[1] = 5;
  order_nd = webbur::i4vec_product ( dim_num, order_1d );
  rule[0] = 1;
  rule[1] = 8;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::gen_laguerre_compute_weights;
  product_mixed_growth_weight_test ( dim_num, order_1d, order_nd, rule, 
    alpha, beta, gw_compute_weights );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_weights;
  delete [] order_1d;
  delete [] rule;

  dim_num = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  order_1d = new int[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.5;
  beta[0] = 0.0;
  beta[1] = 1.5;
  order_1d[0] = 5;
  order_1d[1] = 5;
  order_nd = webbur::i4vec_product ( dim_num, order_1d );
  rule[0] = 2;
  rule[1] = 9;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::fejer2_compute_weights;
  gw_compute_weights[1] = webbur::jacobi_compute_weights;
  product_mixed_growth_weight_test ( dim_num, order_1d, order_nd, rule, 
    alpha, beta, gw_compute_weights );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_weights;
  delete [] order_1d;
  delete [] rule;

  dim_num = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  order_1d = new int[dim_num];
  rule = new int[dim_num];
  alpha[0] = 2.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  order_1d[0] = 7;
  order_1d[1] = 7;
  order_nd = webbur::i4vec_product ( dim_num, order_1d );
  rule[0] = 6;
  rule[1] = 4;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::gen_hermite_compute_weights;
  gw_compute_weights[1] = webbur::legendre_compute_weights;
  product_mixed_growth_weight_test ( dim_num, order_1d, order_nd, rule, 
    alpha, beta, gw_compute_weights );
  delete [] alpha;
  delete [] beta;
  delete [] order_1d;
  delete [] rule;

  dim_num = 3;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  order_1d = new int[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  order_1d[0] = 2;
  order_1d[1] = 3;
  order_1d[2] = 3;
  order_nd = webbur::i4vec_product ( dim_num, order_1d );
  rule[0] = 1;
  rule[1] = 4;
  rule[2] = 5;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::legendre_compute_weights;
  gw_compute_weights[2] = webbur::hermite_compute_weights;
  product_mixed_growth_weight_test ( dim_num, order_1d, order_nd, rule, 
    alpha, beta, gw_compute_weights );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_weights;
  delete [] order_1d;
  delete [] rule;
//
//  Repeat, treating  rules #2 and #3 as Golub Welsch rules.
//
  dim_num = 3;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  order_1d = new int[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  order_1d[0] = 2;
  order_1d[1] = 3;
  order_1d[2] = 3;
  order_nd = webbur::i4vec_product ( dim_num, order_1d );
  rule[0] = 1;
  rule[1] = 10;
  rule[2] = 10;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::legendre_compute_weights;
  gw_compute_weights[2] = webbur::hermite_compute_weights;
  product_mixed_growth_weight_test ( dim_num, order_1d, order_nd, rule, 
    alpha, beta, gw_compute_weights );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_weights;
  delete [] order_1d;
  delete [] rule;

  return;
}
//***************************************************************************80

void product_mixed_growth_weight_test ( int dim_num, int order_1d[], 
  int order_nd, int rule[], double alpha[], double beta[],
  void ( *gw_compute_weights[] ) ( int order, double alpha, double beta, double w[] ) )

//***************************************************************************80
//
//  Purpose:
//
//    PRODUCT_MIXED_GROWTH_WEIGHT_TEST: weights of a mixed factor product rule.
//
//  Discussion:
//
//    This routine computes a sparse grid and compares the sum of the weights
//    to the expected exact value.
//
//    The routine cannot produce a result for rules that include one or more
//    component rules of type 10, that is, Golub-Welsch rules.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    30 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int ORDER_1D[DIM_NUM], the order of the 1D rules.
//
//    Input, int ORDER_ND, the order of the product rule.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, double ALPHA[DIM_NUM], BETA[DIM_NUM], parameters used for
//    Generalized Gauss Hermite, Generalized Gauss Laguerre, 
//    and Gauss Jacobi rules.
//
//    Input, void ( *GW_COMPUTE_WEIGHTS[] ) ( int order, double alpha, double beta, double w[] ),
//    an array of pointers to functions which return the 1D quadrature weights 
//    associated with each spatial dimension for which a Golub Welsch rule 
//    is used.
//
{
  double arg1;
  double arg2;
  double arg3;
  double arg4;
  int dim;
  double pi = 3.141592653589793;
  double value1;
  double value2;
  double *weight;
  double weight_sum;
  double weight_sum_error;
  double weight_sum_exact;

  weight_sum_exact = 1.0;

  for ( dim = 0; dim < dim_num; dim++ )
  {
    if ( rule[dim] == 1 )
    {
      weight_sum_exact = weight_sum_exact * 2.0;
    }
    else if ( rule[dim] == 2 )
    {
      weight_sum_exact = weight_sum_exact * 2.0;
    }
    else if ( rule[dim] == 3 )
    {
      weight_sum_exact = weight_sum_exact * 2.0;
    }
    else if ( rule[dim] == 4 )
    {
      weight_sum_exact = weight_sum_exact * 2.0;
    }
    else if ( rule[dim] == 5 )
    {
      weight_sum_exact = weight_sum_exact * std::sqrt ( pi );
    }
    else if ( rule[dim] == 6 )
    {
      weight_sum_exact = weight_sum_exact 
        * webbur::r8_gamma ( 0.5 * ( alpha[dim] + 1.0 ) );
    }
    else if ( rule[dim] == 7 )
    {
      weight_sum_exact = weight_sum_exact * 1.0;
    }
    else if ( rule[dim] == 8 )
    {
      weight_sum_exact = weight_sum_exact * webbur::r8_gamma ( alpha[dim] + 1.0 );
    }
    else if ( rule[dim] == 9 )
    {
      arg1 = - alpha[dim];
      arg2 = 1.0;
      arg3 = beta[dim] + 2.0;
      arg4 = - 1.0;
      value1 = webbur::r8_hyper_2f1 ( arg1, arg2, arg3, arg4 );
      arg1 = - beta[dim];
      arg2 = 1.0;
      arg3 = alpha[dim] + 2.0;
      arg4 = - 1.0;
      value2 = webbur::r8_hyper_2f1 ( arg1, arg2, arg3, arg4 );
      weight_sum_exact = weight_sum_exact * ( 
        value1 / ( beta[dim] + 1.0 ) + value2 / ( alpha[dim] + 1.0 ) );
    }
    else if ( rule[dim] == 10 )
    {
      weight_sum_exact = 0.0;
    }
    else
    {
      std::cerr << "\n";
      std::cerr << "PRODUCT_MIXED_GROWTH_WEIGHT_TEST - Fatal error!\n";
      std::cerr << "  Unexpected value of RULE[" << dim << "] = " 
           << rule[dim] << ".\n";
      std::exit ( 1 );
    }
  }

  std::cout << "\n";
  std::cout << "PRODUCT_MIXED_GROWTH_WEIGHT_TEST:\n";
  std::cout << "  Compute the weights of a mixed factor product grid.\n";
  std::cout << "\n";
  std::cout << "  As a simple test, sum these weights.\n";
  std::cout << "  They should sum to exactly " << weight_sum_exact << "\n";
  std::cout << "\n";
  std::cout << "  Spatial dimension DIM_NUM = " << dim_num << "\n";

  std::cout << "\n";
  std::cout << " Dimension      Rule     Order        Alpha          Beta\n";
  std::cout << "\n";

  for ( dim = 0; dim < dim_num; dim++ )
  {
    if ( rule[dim] == 1 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << std::setw(8) << order_1d[dim] << "\n";
    }
    else if ( rule[dim] == 2 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << std::setw(8) << order_1d[dim] << "\n";
    }
    else if ( rule[dim] == 3 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << std::setw(8) << order_1d[dim] << "\n";
    }
    else if ( rule[dim] == 4 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << std::setw(8) << order_1d[dim] << "\n";
    }
    else if ( rule[dim] == 5 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << std::setw(8) << order_1d[dim] << "\n";
    }
    else if ( rule[dim] == 6 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << std::setw(8) << order_1d[dim] 
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 7 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << std::setw(8) << order_1d[dim] << "\n";
    }
    else if ( rule[dim] == 8 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << std::setw(8) << order_1d[dim] 
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 9 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << std::setw(8) << order_1d[dim] 
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else if ( rule[dim] == 10 )
    {
      std::cout << "  " << std::setw(8) << dim
                << "  " << std::setw(8) << rule[dim]
                << "  " << std::setw(8) << order_1d[dim] 
                << "  " << std::setw(14) << alpha[dim] 
                << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else
    {
      std::cerr << "\n";
      std::cerr << "PRODUCT_MIXED_GROWTH_WEIGHT_TEST - Fatal error!\n";
      std::cerr << "  Cannot perform test for rule = " << rule[dim] << "\n";
      std::exit ( 1 );
    }
  }
//
//  Compute the weights and points.
//
  weight = new double[order_nd];

  webbur::product_mixed_growth_weight ( dim_num, order_1d, order_nd, rule, 
    alpha, beta, gw_compute_weights, weight );
//
//  Sum the weights.
//
  weight_sum = webbur::r8vec_sum ( order_nd, weight );

  weight_sum_error = webbur::r8_abs ( weight_sum - weight_sum_exact );

  std::cout << "\n";
  std::cout << "    Weight sum  Expected sum    Difference\n";
  std::cout << "\n";
  std::cout << "  " << std::setw(14) << weight_sum
            << "  " << std::setw(14) << weight_sum_exact
            << "  " << std::setw(14) << weight_sum_error << "\n";

  delete [] weight;

  return;
}
//****************************************************************************80

void sparse_grid_mixed_growth_index_tests ( double tol )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_INDEX_TESTS calls SPARSE_GRID_MIXED_GROWTH_INDEX_TEST.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    16 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, double TOL, a tolerance for point equality.
//    A value of sqrt ( eps ) is reasonable, and will allow the code to
//    consolidate points which are equal, or very nearly so.  A value of
//    -1.0, on the other hand, will force the code to use every point, 
//    regardless of duplication.
//
{
  double *alpha;
  double *beta;
  int dim_num;
  GWPointer *gw_compute_points;
  int level_max_max;
  int level_max_min;
  int *order_1d;
  int order_nd;
  int *rule;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_INDEX_TESTS\n";
  std::cout << "  Call SPARSE_GRID_MIXED_GROWTH_INDEX_TEST with various arguments.\n";
  std::cout << "\n";
  std::cout << "  All tests will use a point equality tolerance of " << tol << "\n";

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 1;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::clenshaw_curtis_compute_points;
  sparse_grid_mixed_growth_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 3;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::patterson_lookup_points;
  sparse_grid_mixed_growth_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 4;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  sparse_grid_mixed_growth_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 7;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::laguerre_compute_points;
  sparse_grid_mixed_growth_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 8;
  alpha[0] = 0.0;
  alpha[1] = 1.5;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::gen_laguerre_compute_points;
  sparse_grid_mixed_growth_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 2;
  rule[1] = 9;
  alpha[0] = 0.0;
  alpha[1] = 0.5;
  beta[0] = 0.0;
  beta[1] = 1.5;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::fejer2_compute_points;
  gw_compute_points[1] = webbur::jacobi_compute_points;
  sparse_grid_mixed_growth_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 6;
  rule[1] = 4;
  alpha[0] = 2.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::gen_hermite_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  sparse_grid_mixed_growth_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 3;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 4;
  rule[2] = 5;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  sparse_grid_mixed_growth_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;
//
//  Repeat, treating  rules #2 and #3 as Golub Welsch rules.
//
  dim_num = 3;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 10;
  rule[2] = 10;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  sparse_grid_mixed_growth_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  return;
}
//***************************************************************************80

void sparse_grid_mixed_growth_index_test ( int dim_num, int level_max_min, 
  int level_max_max, int rule[], double alpha[], double beta[], 
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  double tol )

//***************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_INDEX_TEST tests SPARSE_GRID_MIXED_INDEX.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    04 February 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX_MIN, LEVEL_MAX_MAX, the minimum and
//    maximum values of LEVEL_MAX.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, double ALPHA[DIM_NUM], BETA[DIM_NUM], parameters used for
//    Generalized Gauss Hermite, Generalized Gauss Laguerre, 
//    and Gauss Jacobi rules.
//
//    Input, void ( *GW_COMPUTE_POINTS[] ) ( int order, double alpha, double beta, double x[] ),
//    an array of pointers to functions which return the 1D quadrature points 
//    associated with each spatial dimension for which a Golub Welsch rule 
//    is used.
//
//    Input, double TOL, a tolerance for point equality.
//
{
  int dim;
  int level_max;
  int point;
  int point_num;
  int point_total_num;
  int *sparse_index;
  int *sparse_order;
  int *sparse_unique_index;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_INDEX_TEST\n";
  std::cout << "  SPARSE_GRID_MIXED_INDEX returns index and order vectors that\n";
  std::cout << "  identify each point in a multidimensional sparse grid \n";
  std::cout << "  with mixed factors.\n";
  std::cout << "\n";
  std::cout << "  Each sparse grid is of spatial dimension DIM_NUM,\n";
  std::cout << "  and is made up of product grids of levels up to LEVEL_MAX.\n";
  std::cout << "\n";
  std::cout << " Dimension      Rule  Growth rate       Alpha          Beta\n";
  std::cout << "\n";

  for ( dim = 0; dim < dim_num; dim++ )
  {
    if ( rule[dim] == 1 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim] 
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 2 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 3 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 4 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 5 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 6 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 7 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 8 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 9 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else if ( rule[dim] == 10 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else
    {
      std::cerr << "\n";
      std::cerr << "SPARSE_GRID_MIXED_INDEX_TEST - Fatal error!\n";
      std::cerr << "  Unexpected value of RULE = " << rule[dim] << "\n";
      std::exit ( 1 );
    }
  }

  for ( level_max = level_max_min; level_max <= level_max_max; level_max++ )
  {
    point_total_num = webbur::sparse_grid_mixed_growth_size_total ( dim_num, 
      level_max, rule, webbur::level_to_order_default );

    point_num = webbur::sparse_grid_mixed_growth_size ( dim_num, level_max, 
      rule, alpha, beta, gw_compute_points, tol, 
      webbur::level_to_order_default );

    sparse_unique_index = new int[point_total_num];

    webbur::sparse_grid_mixed_growth_unique_index ( dim_num, level_max, rule, 
      alpha, beta, gw_compute_points, tol, point_num, point_total_num, 
      webbur::level_to_order_default, sparse_unique_index );

    sparse_order = new int[dim_num*point_num];
    sparse_index = new int[dim_num*point_num];

    webbur::sparse_grid_mixed_growth_index ( dim_num, level_max, rule, 
      point_num, point_total_num, sparse_unique_index, 
      webbur::level_to_order_default, sparse_order, sparse_index );

    std::cout << "\n";
    std::cout << "  For LEVEL_MAX = " << level_max << "\n";
    std::cout << "\n";
    for ( point = 0; point < point_num; point++ )
    {
      std::cout << "  " << std::setw(4) << point << "  ";
      for ( dim = 0; dim < dim_num; dim++ )
      {
        std::cout << "  " << std::setw(3) << sparse_index[dim+point*dim_num]
             << " /" << std::setw(3) << sparse_order[dim+point*dim_num];
      }
      std::cout << "\n";
    }
    delete [] sparse_index;
    delete [] sparse_order;
    delete [] sparse_unique_index;
  }
  return;
}
//****************************************************************************80

void sparse_grid_mixed_growth_point_tests ( double tol )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_POINT_TESTS calls SPARSE_GRID_MIXED_GROWTH_POINT_TEST.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    01 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, double TOL, a tolerance for point equality.
//    A value of sqrt ( eps ) is reasonable, and will allow the code to
//    consolidate points which are equal, or very nearly so.  A value of
//    -1.0, on the other hand, will force the code to use every point, 
//    regardless of duplication.
//
{
  double *alpha;
  double *beta;
  int dim_num;
  GWPointer *gw_compute_points;
  int level_max_max;
  int level_max_min;
  int *order_1d;
  int order_nd;
  int *rule;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_POINT_TESTS\n";
  std::cout << "  Call SPARSE_GRID_MIXED_GROWTH_POINT_TEST with various arguments..\n";
  std::cout << "\n";
  std::cout << "  All tests will use a point equality tolerance of " << tol << "\n";

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 1;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::clenshaw_curtis_compute_points;
  sparse_grid_mixed_growth_point_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 3;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::patterson_lookup_points;
  sparse_grid_mixed_growth_point_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 4;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  sparse_grid_mixed_growth_point_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 7;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::laguerre_compute_points;
  sparse_grid_mixed_growth_point_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 1.5;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 8;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::gen_laguerre_compute_points;
  sparse_grid_mixed_growth_point_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.5;
  beta[0] = 0.0;
  beta[1] = 1.5;
  rule[0] = 2;
  rule[1] = 9;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::fejer2_compute_points;
  gw_compute_points[1] = webbur::jacobi_compute_points;
  sparse_grid_mixed_growth_point_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 2.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 6;
  rule[1] = 4;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::gen_hermite_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  sparse_grid_mixed_growth_point_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 3;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  rule[0] = 1;
  rule[1] = 4;
  rule[2] = 5;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  sparse_grid_mixed_growth_point_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;
//
//  Repeat, treating  rules #2 and #3 as Golub Welsch rules.
//
  dim_num = 3;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  rule[0] = 1;
  rule[1] = 10;
  rule[2] = 10;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  sparse_grid_mixed_growth_point_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  return;
}
//***************************************************************************80

void sparse_grid_mixed_growth_point_test ( int dim_num, int level_max_min, 
  int level_max_max, int rule[], double alpha[], double beta[], 
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  double tol )

//***************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_POINT_TEST tests SPARSE_GRID_MIXED_GROWTH_POINT.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    01 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX_MIN, LEVEL_MAX_MAX, the minimum and
//    maximum values of LEVEL_MAX.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, double ALPHA[DIM_NUM], BETA[DIM_NUM], parameters used for
//    Generalized Gauss Hermite, Generalized Gauss Laguerre, 
//    and Gauss Jacobi rules.
//
//    Input, void ( *GW_COMPUTE_POINTS[] ) ( int order, double alpha, double beta, double x[] ),
//    an array of pointers to functions which return the 1D quadrature points 
//    associated with each spatial dimension for which a Golub Welsch rule 
//    is used.
//
//    Input, double TOL, a tolerance for point equality.
//
{
  int dim;
  int level_max;
  int point;
  int point_num;
  int point_total_num;
  int *sparse_index;
  int *sparse_order;
  double *sparse_point;
  int *sparse_unique_index;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_POINT_TEST\n";
  std::cout << "  SPARSE_GRID_MIXED_GROWTH_POINT returns an array of the points\n";
  std::cout << "  forming a multidimensional sparse grid with mixed factors.\n";
  std::cout << "\n";
  std::cout << "  Each sparse grid is of spatial dimension DIM_NUM,\n";
  std::cout << "  and is made up of product grids of levels up to LEVEL_MAX.\n";
  std::cout << "\n";
  std::cout << " Dimension      Rule  Growth rate       Alpha          Beta\n";
  std::cout << "\n";

  for ( dim = 0; dim < dim_num; dim++ )
  {
    if ( rule[dim] == 1 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim] 
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 2 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 3 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 4 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 5 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 6 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 7 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 8 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 9 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else if ( rule[dim] == 10 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else
    {
      std::cerr << "\n";
      std::cerr << "SPARSE_GRID_MIXED_POINT_TEST - Fatal error!\n";
      std::cerr << "  Unexpected value of RULE = " << rule[dim] << "\n";
      std::exit ( 1 );
    }
  }

  for ( level_max = level_max_min; level_max <= level_max_max; level_max++ )
  {
    point_total_num = webbur::sparse_grid_mixed_growth_size_total ( dim_num, 
      level_max, rule, webbur::level_to_order_default );

    point_num = webbur::sparse_grid_mixed_growth_size ( dim_num, level_max, 
      rule, alpha, beta, gw_compute_points, tol, 
      webbur::level_to_order_default );

    sparse_unique_index = new int[point_total_num];

    webbur::sparse_grid_mixed_growth_unique_index ( dim_num, level_max, rule, 
      alpha, beta, gw_compute_points, tol, point_num, point_total_num, 
      webbur::level_to_order_default, sparse_unique_index );

    sparse_order = new int[dim_num*point_num];
    sparse_index = new int[dim_num*point_num];

    webbur::sparse_grid_mixed_growth_index ( dim_num, level_max, rule, 
      point_num, point_total_num, sparse_unique_index, 
      webbur::level_to_order_default, sparse_order, sparse_index );

    sparse_point = new double [ dim_num * point_num ];

    webbur::sparse_grid_mixed_growth_point ( dim_num, level_max, rule, alpha, 
      beta, gw_compute_points, point_num, sparse_order, sparse_index, 
      webbur::level_to_order_default, sparse_point );

    std::cout << "\n";
    std::cout << "  For LEVEL_MAX = " << level_max << "\n";
    std::cout << "\n";
    for ( point = 0; point < point_num; point++ )
    {
      std::cout << "  " << std::setw(4) << point << "  ";
      for ( dim = 0; dim < dim_num; dim++ )
      {
        std::cout << "  " << std::setw(10) << sparse_point[dim+point*dim_num];
      }
      std::cout << "\n";
    }
    delete [] sparse_index;
    delete [] sparse_order;
    delete [] sparse_point;
    delete [] sparse_unique_index;
  }
  return;
}
//****************************************************************************80

void sparse_grid_mixed_growth_size_tests ( double tol )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_SIZE_TESTS calls SPARSE_GRID_MIXED_GROWTH_SIZE_TEST.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    16 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, double TOL, a tolerance for point equality.
//    A value of sqrt ( eps ) is reasonable, and will allow the code to
//    consolidate points which are equal, or very nearly so.  A value of
//    -1.0, on the other hand, will force the code to use every point, 
//    regardless of duplication.
//
{
  double *alpha;
  double *beta;
  int dim_num;
  GWPointer *gw_compute_points;
  int level_max_max;
  int level_max_min;
  int *order_1d;
  int order_nd;
  int *rule;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_SIZE_TESTS\n";
  std::cout << "  Call SPARSE_GRID_MIXED_GROWTH_SIZE_TEST with various arguments.\n";
  std::cout << "\n";
  std::cout << "  All tests will use a point equality tolerance of " << tol << "\n";

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 1;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::clenshaw_curtis_compute_points;
  sparse_grid_mixed_growth_size_test ( dim_num, level_max_min, level_max_max,
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 3;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::patterson_lookup_points;
  sparse_grid_mixed_growth_size_test ( dim_num, level_max_min, level_max_max,
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 4;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  sparse_grid_mixed_growth_size_test ( dim_num, level_max_min, level_max_max,
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 7;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::laguerre_compute_points;
  sparse_grid_mixed_growth_size_test ( dim_num, level_max_min, level_max_max,
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 8;
  alpha[0] = 0.0;
  alpha[1] = 1.5;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::gen_laguerre_compute_points;
  sparse_grid_mixed_growth_size_test ( dim_num, level_max_min, level_max_max,
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 2;
  rule[1] = 9;
  alpha[0] = 0.0;
  alpha[1] = 0.5;
  beta[0] = 0.0;
  beta[1] = 1.5;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::fejer2_compute_points;
  gw_compute_points[1] = webbur::jacobi_compute_points;
  sparse_grid_mixed_growth_size_test ( dim_num, level_max_min, level_max_max,
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 6;
  rule[1] = 4;
  alpha[0] = 2.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::gen_hermite_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  sparse_grid_mixed_growth_size_test ( dim_num, level_max_min, level_max_max,
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 3;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 4;
  rule[2] = 5;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  sparse_grid_mixed_growth_size_test ( dim_num, level_max_min, level_max_max,
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;
//
//  Repeat, treating  rules #2 and #3 as Golub Welsch rules.
//
  dim_num = 3;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  rule[0] = 1;
  rule[1] = 10;
  rule[2] = 10;
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  sparse_grid_mixed_growth_size_test ( dim_num, level_max_min, level_max_max,
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  return;
}
//***************************************************************************80

void sparse_grid_mixed_growth_size_test ( int dim_num, int level_max_min, 
  int level_max_max, int rule[], double alpha[], double beta[], 
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  double tol )

//***************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_SIZE_TEST tests SGMG_SIZE, SGMG_SIZE_TOTAL.
//
//  Discussion:
//
//    The "level_to_order" argument for the SGMG_SIZE and SGMG_SIZE_TOTAL
//    functions is set to "LEVEL_TO_ORDER_DEFAULT", which uses exponential
//    growth for fully nested rules, and linear otherwise.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    01 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX_MIN, LEVEL_MAX_MAX, the minimum and
//    maximum values of LEVEL_MAX.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, double ALPHA[DIM_NUM], BETA[DIM_NUM], parameters used for
//    Generalized Gauss Hermite, Generalized Gauss Laguerre, 
//    and Gauss Jacobi rules.
//
//    Input, void ( *GW_COMPUTE_POINTS[] ) ( int order, double alpha, double beta, double x[] ),
//    an array of pointers to functions which return the 1D quadrature points 
//    associated with each spatial dimension for which a Golub Welsch rule 
//    is used.
//
//    Input, double TOL, a tolerance for point equality.
//
{
  int dim;
  int level_max;
  int level_min;
  int point_num;
  int point_total_num;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_SIZE_TEST\n";
  std::cout << "  SPARSE_GRID_MIXED_GROWTH_SIZE returns the number of distinct\n";
  std::cout << "  points in a multidimensional sparse grid with mixed factors.\n";
  std::cout << "\n";
  std::cout << "  SPARSE_GRID_MIXED_GROWTH_SIZE_TOTAL returns the TOTAL number of\n";
  std::cout << "  points in a multidimensional sparse grid with mixed factors,\n";
  std::cout << "  without checking for duplication.\n";
  std::cout << "\n";
  std::cout << "  Each sparse grid is of spatial dimension DIM_NUM,\n";
  std::cout << "  and is made up of product grids of levels up to LEVEL_MAX.\n";
  std::cout << "\n";
  std::cout << " Dimension      Rule  Growth rate       Alpha          Beta\n";
  std::cout << "\n";

  for ( dim = 0; dim < dim_num; dim++ )
  {
    if ( rule[dim] == 1 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim] 
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 2 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 3 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 4 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 5 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 6 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 7 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 8 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 9 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else if ( rule[dim] == 10 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else
    {
      std::cerr << "\n";
      std::cerr << "SPARSE_GRID_MIXED_GROWTH_SIZE_TEST - Fatal error!\n";
      std::cerr << "  Unexpected value of RULE = " << rule[dim] << "\n";
      std::exit ( 1 );
    }
  }

  std::cout << "\n";
  std::cout << " LEVEL_MIN LEVEL_MAX POINT_NUM POINT_NUM\n";
  std::cout << "                        Unique     Total\n";
  std::cout << "\n";

  for ( level_max = level_max_min; level_max <= level_max_max; level_max++ )
  {
    point_total_num = webbur::sparse_grid_mixed_growth_size_total ( dim_num, 
      level_max, rule, webbur::level_to_order_default );

    point_num = webbur::sparse_grid_mixed_growth_size ( dim_num, level_max, 
      rule, alpha, beta, gw_compute_points, tol, 
      webbur::level_to_order_default );

    level_min = webbur::i4_max ( 0, level_max + 1 - dim_num );

    std::cout << "  " << std::setw(8) << level_min
         << "  " << std::setw(8) << level_max
         << "  " << std::setw(8) << point_num
         << "  " << std::setw(8) << point_total_num << "\n";
  }

  return;
}
//****************************************************************************80

void sparse_grid_mixed_growth_unique_index_tests ( double tol )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_UNIQUE_INDEX_TESTS calls SPARSE_GRID_MIXED_GROWTH_UNIQUE_INDEX_TEST.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    16 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, double TOL, a tolerance for point equality.
//    A value of sqrt ( eps ) is reasonable, and will allow the code to
//    consolidate points which are equal, or very nearly so.  A value of
//    -1.0, on the other hand, will force the code to use every point, 
//    regardless of duplication.
//
{
  double *alpha;
  double *beta;
  int dim_num;
  GWPointer *gw_compute_points;
  int level_max_max;
  int level_max_min;
  int *order_1d;
  int order_nd;
  int *rule;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_UNIQUE_INDEX_TESTS\n";
  std::cout << "  Call SPARSE_GRID_MIXED_GROWTH_UNIQUE_INDEX_TEST with various arguments\n";
  std::cout << "\n";
  std::cout << "  All tests will use a point equality tolerance of " << tol << "\n";

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 1;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::clenshaw_curtis_compute_points;
  sparse_grid_mixed_growth_unique_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 3;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::patterson_lookup_points;
  sparse_grid_mixed_growth_unique_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 4;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  sparse_grid_mixed_growth_unique_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 7;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::laguerre_compute_points;
  sparse_grid_mixed_growth_unique_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 1.5;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 8;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::gen_laguerre_compute_points;
  sparse_grid_mixed_growth_unique_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.5;
  beta[0] = 0.0;
  beta[1] = 1.5;
  rule[0] = 2;
  rule[1] = 9;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::fejer2_compute_points;
  gw_compute_points[1] = webbur::jacobi_compute_points;
  sparse_grid_mixed_growth_unique_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 2.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 6;
  rule[1] = 4;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::gen_hermite_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  sparse_grid_mixed_growth_unique_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  dim_num = 3;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  rule[0] = 1;
  rule[1] = 4;
  rule[2] = 5;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  sparse_grid_mixed_growth_unique_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;
//
//  Repeat, treating  rules #2 and #3 as Golub Welsch rules.
//
  dim_num = 3;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  rule[0] = 1;
  rule[1] = 10;
  rule[2] = 10;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  sparse_grid_mixed_growth_unique_index_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] rule;

  return;
}
//***************************************************************************80

void sparse_grid_mixed_growth_unique_index_test ( int dim_num, int level_max_min, 
  int level_max_max, int rule[], double alpha[], double beta[], 
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  double tol )

//***************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_UNIQUE_INDEX_TEST tests SPARSE_GRID_MIXED_GROWTH_UNIQUE_INDEX.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    01 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX_MIN, LEVEL_MAX_MAX, the minimum and
//    maximum values of LEVEL_MAX.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, double ALPHA[DIM_NUM], BETA[DIM_NUM], parameters used for
//    Generalized Gauss Hermite, Generalized Gauss Laguerre, 
//    and Gauss Jacobi rules.
//
//    Input, void ( *GW_COMPUTE_POINTS[] ) ( int order, double alpha, double beta, double x[] ),
//    an array of pointers to functions which return the 1D quadrature points 
//    associated with each spatial dimension for which a Golub Welsch rule 
//    is used.
//
//    Input, double TOL, a tolerance for point equality.
//
{
  int dim;
  int level_max;
  int level_min;
  int point;
  int point_num;
  int point_total_num;
  int *sparse_unique_index;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_UNIQUE_INDEX_TEST\n";
  std::cout << "  SPARSE_GRID_MIXED_GROWTH_UNIQUE_INDEX returns a mapping between\n";
  std::cout << "  the nonunique and unique points in a sparse grid.\n";
  std::cout << "\n";
  std::cout << " Dimension      Rule  Growth rate      Alpha          Beta\n";
  std::cout << "\n";

  for ( dim = 0; dim < dim_num; dim++ )
  {
    if ( rule[dim] == 1 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 2 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 3 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 4 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     " << "\n";
    }
    else if ( rule[dim] == 5 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     " << "\n";
    }
    else if ( rule[dim] == 6 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 7 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     " << "\n";
    }
    else if ( rule[dim] == 8 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 9 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else if ( rule[dim] == 10 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else
    {
      std::cerr << "\n";
      std::cerr << "SPARSE_GRID_MIXED_GROWTH_UNIQUE_INDEX_TEST - Fatal error!\n";
      std::cerr << "  Unexpected value of RULE = " << rule[dim] << "\n";
      std::exit ( 1 );
    }
  }

  for ( level_max = level_max_min; level_max <= level_max_max; level_max++ )
  {
    point_total_num = webbur::sparse_grid_mixed_growth_size_total ( dim_num, 
      level_max, rule, webbur::level_to_order_default );

    point_num = webbur::sparse_grid_mixed_growth_size ( dim_num, level_max, 
      rule, alpha, beta, gw_compute_points, tol, 
      webbur::level_to_order_default );

    std::cout << "\n";
    std::cout << " LEVEL_MIN LEVEL_MAX POINT_NUM POINT_NUM\n";
    std::cout << "                        Unique     Total\n";

    level_min = webbur::i4_max ( 0, level_max + 1 - dim_num );

    std::cout << "\n";
    std::cout << "  " << std::setw(8) << level_min
         << "  " << std::setw(8) << level_max
         << "  " << std::setw(8) << point_num
         << "  " << std::setw(8) << point_total_num << "\n";

    sparse_unique_index = new int[point_total_num];

    webbur::sparse_grid_mixed_growth_unique_index ( dim_num, level_max, rule, 
      alpha, beta, gw_compute_points, tol, point_num, point_total_num, 
      webbur::level_to_order_default, sparse_unique_index );

    std::cout << "\n";
    std::cout << "     POINT    UNIQUE\n";
    std::cout << "\n";
    for ( point = 0; point < point_total_num; point++ )
    {
      std::cout << "  " << std::setw(8) << point
           << "  " << std::setw(8) << sparse_unique_index[point] << "\n";
    }
    delete [] sparse_unique_index;
  }
  return;
}
//****************************************************************************80

void sparse_grid_mixed_growth_weight_tests ( double tol )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_WEIGHT_TESTS calls SPARSE_GRID_MIXED_GROWTH_WEIGHT_TEST.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    16 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, double TOL, a tolerance for point equality.
//    A value of sqrt ( eps ) is reasonable, and will allow the code to
//    consolidate points which are equal, or very nearly so.  A value of
//    -1.0, on the other hand, will force the code to use every point, 
//    regardless of duplication.
//
{
  double *alpha;
  double *beta;
  int dim_num;
  GWPointer *gw_compute_points;
  GWPointer *gw_compute_weights;
  int level_max_max;
  int level_max_min;
  int *order_1d;
  int order_nd;
  int *rule;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_WEIGHT_TESTS\n";
  std::cout << "  Call SPARSE_GRID_MIXED_GROWTH_WEIGHT_TEST with various arguments.\n";
  std::cout << "\n";
  std::cout << "  All tests will use a point equality tolerance of " << tol << "\n";

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 1;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::clenshaw_curtis_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::clenshaw_curtis_compute_weights;
  sparse_grid_mixed_growth_weight_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, gw_compute_weights, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 3;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::fejer2_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::fejer2_compute_weights;
  sparse_grid_mixed_growth_weight_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, gw_compute_weights, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 4;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::legendre_compute_weights;
  sparse_grid_mixed_growth_weight_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, gw_compute_weights, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 7;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::laguerre_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::laguerre_compute_weights;
  sparse_grid_mixed_growth_weight_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, gw_compute_weights, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 1.5;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 8;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::gen_laguerre_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::gen_laguerre_compute_weights;
  sparse_grid_mixed_growth_weight_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, gw_compute_weights, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.5;
  beta[0] = 0.0;
  beta[1] = 1.5;
  rule[0] = 2;
  rule[1] = 9;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::fejer2_compute_points;
  gw_compute_points[1] = webbur::jacobi_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::fejer2_compute_weights;
  gw_compute_weights[1] = webbur::jacobi_compute_weights;
  sparse_grid_mixed_growth_weight_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, gw_compute_weights, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 2.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 6;
  rule[1] = 4;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::gen_hermite_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::gen_hermite_compute_weights;
  gw_compute_weights[1] = webbur::legendre_compute_weights;
  sparse_grid_mixed_growth_weight_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, gw_compute_weights, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 3;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  rule[0] = 1;
  rule[1] = 4;
  rule[2] = 5;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::legendre_compute_weights;
  gw_compute_weights[2] = webbur::hermite_compute_weights;
  sparse_grid_mixed_growth_weight_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, gw_compute_weights, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;
//
//  Repeat, treating  rules #2 and #3 as Golub Welsch rules.
//
  dim_num = 3;
  level_max_min = 0;
  level_max_max = 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  rule[0] = 1;
  rule[1] = 10;
  rule[2] = 10;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::legendre_compute_weights;
  gw_compute_weights[2] = webbur::hermite_compute_weights;
  sparse_grid_mixed_growth_weight_test ( dim_num, level_max_min, level_max_max, 
    rule, alpha, beta, gw_compute_points, gw_compute_weights, tol );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  return;
}
//***************************************************************************80

void sparse_grid_mixed_growth_weight_test ( int dim_num, int level_max_min, 
  int level_max_max, int rule[], double alpha[], double beta[],
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  void ( *gw_compute_weights[] ) ( int order, double alpha, double beta, double w[] ),
  double tol )

//***************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_WEIGHT_TEST checks the sum of the quadrature weights.
//
//  Discussion:
//
//    If any component rule is of Golub-Welsch type, we cannot compute
//    the exact weight sum, which we set, instead, to zero.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    01 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int LEVEL_MAX_MIN, LEVEL_MAX_MAX, the minimum and
//    maximum values of LEVEL_MAX.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, double ALPHA[DIM_NUM], BETA[DIM_NUM], parameters used for
//    Generalized Gauss Hermite, Generalized Gauss Laguerre, 
//    and Gauss Jacobi rules.
//
//    Input, void ( *GW_COMPUTE_POINTS[] ) ( int order, double alpha, double beta, double x[] ),
//    an array of pointers to functions which return the 1D quadrature points 
//    associated with each spatial dimension for which a Golub Welsch rule 
//    is used.
//
//    Input, void ( *GW_COMPUTE_WEIGHTS[] ) ( int order, double alpha, double beta, double w[] ),
//    an array of pointers to functions which return the 1D quadrature weights 
//    associated with each spatial dimension for which a Golub Welsch rule 
//    is used.
//
//    Input, double TOL, a tolerance for point equality.
//
{
  double arg1;
  double arg2;
  double arg3;
  double arg4;
  int dim;
  int level_max;
  double pi = 3.141592653589793;
  int point;
  int point_num;
  int point_total_num;
  int *sparse_unique_index;
  double *sparse_weight;
  double value1;
  double value2;
  double weight_sum;
  double weight_sum_error;
  double weight_sum_exact;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_WEIGHT_TEST\n";
  std::cout << "  Compute the weights of a sparse grid.\n";
  std::cout << "\n";
  std::cout << "  Each sparse grid is of spatial dimension DIM_NUM,\n";
  std::cout << "  and is made up of product grids of levels up to LEVEL_MAX.\n";
  std::cout << "\n";
  std::cout << " Dimension      Rule  Growth rate       Alpha          Beta\n";
  std::cout << "\n";

  for ( dim = 0; dim < dim_num; dim++ )
  {
    if ( rule[dim] == 1 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim] 
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 2 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 3 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Exponential" << "\n";
    }
    else if ( rule[dim] == 4 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 5 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 6 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 7 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear" << "\n";
    }
    else if ( rule[dim] == 8 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] << "\n";
    }
    else if ( rule[dim] == 9 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else if ( rule[dim] == 10 )
    {
      std::cout << "  " << std::setw(8) << dim
           << "  " << std::setw(8) << rule[dim]
           << "  " << "Linear     "
           << "  " << std::setw(14) << alpha[dim] 
           << "  " << std::setw(14) << beta[dim] << "\n";
    }
    else
    {
      std::cerr << "\n";
      std::cerr << "SPARSE_GRID_MIXED_WEIGHT_TEST - Fatal error!\n";
      std::cerr << "  Unexpected value of RULE = " << rule[dim] << "\n";
      std::exit ( 1 );
    }
  }

  weight_sum_exact = 1.0;

  for ( dim = 0; dim < dim_num; dim++ )
  {
    if ( rule[dim] == 1 )
    {
      weight_sum_exact = weight_sum_exact * 2.0;
    }
    else if ( rule[dim] == 2 )
    {
      weight_sum_exact = weight_sum_exact * 2.0;
    }
    else if ( rule[dim] == 3 )
    {
      weight_sum_exact = weight_sum_exact * 2.0;
    }
    else if ( rule[dim] == 4 )
    {
      weight_sum_exact = weight_sum_exact * 2.0;
    }
    else if ( rule[dim] == 5 )
    {
      weight_sum_exact = weight_sum_exact * std::sqrt ( pi );
    }
    else if ( rule[dim] == 6 )
    {
      weight_sum_exact = weight_sum_exact 
        * webbur::r8_gamma ( 0.5 * ( alpha[dim] + 1.0 ) );
    }
    else if ( rule[dim] == 7 )
    {
      weight_sum_exact = weight_sum_exact * 1.0;
    }
    else if ( rule[dim] == 8 )
    {
      weight_sum_exact = weight_sum_exact 
        * webbur::r8_gamma ( alpha[dim] + 1.0 );
    }
    else if ( rule[dim] == 9 )
    {
      arg1 = - alpha[dim];
      arg2 = 1.0;
      arg3 = beta[dim] + 2.0;
      arg4 = - 1.0;
      value1 = webbur::r8_hyper_2f1 ( arg1, arg2, arg3, arg4 );
      arg1 = - beta[dim];
      arg2 = 1.0;
      arg3 = alpha[dim] + 2.0;
      arg4 = - 1.0;
      value2 = webbur::r8_hyper_2f1 ( arg1, arg2, arg3, arg4 );
      weight_sum_exact = weight_sum_exact * ( 
        value1 / ( beta[dim] + 1.0 ) + value2 / ( alpha[dim] + 1.0 ) );
    }
    else if ( rule[dim] == 10 )
    {
      weight_sum_exact = 0.0;
    }
    else
    {
      std::cerr << "\n";
      std::cerr << "SPARSE_GRID_MIXED_WEIGHT_TEST - Fatal error!\n";
      std::cerr << "  Unexpected value of RULE[" << dim << "] = " 
                << rule[dim] << ".\n";
      std::exit ( 1 );
    }
  }

  if ( weight_sum_exact == 0.0 )
  {
    std::cout << "\n";
    std::cout << "  Because this rule includes Golub-Welsch components,\n";
    std::cout << "  we do not try to compute the exact weight sum.\n";
  }
  else
  {
    std::cout << "\n";
    std::cout << "  As a simple test, sum these weights.\n";
    std::cout << "  They should sum to exactly " << weight_sum_exact << "\n";
  }

  std::cout << "\n";
  std::cout << "     Level      Weight sum  Expected sum    Difference\n";
  std::cout << "\n";

  for ( level_max = level_max_min; level_max <= level_max_max; level_max++ )
  {
    point_total_num = webbur::sparse_grid_mixed_growth_size_total ( dim_num, 
      level_max, rule, webbur::level_to_order_default );

    point_num = webbur::sparse_grid_mixed_growth_size ( dim_num, level_max, 
      rule, alpha, beta, gw_compute_points, tol, 
      webbur::level_to_order_default );

    sparse_unique_index = new int[point_total_num];

    webbur::sparse_grid_mixed_growth_unique_index ( dim_num, level_max, rule, 
      alpha, beta, gw_compute_points, tol, point_num, point_total_num, 
      webbur::level_to_order_default, sparse_unique_index );

    sparse_weight = new double[point_num];

    webbur::sparse_grid_mixed_growth_weight ( dim_num, level_max, rule, alpha, 
      beta, gw_compute_weights, point_num, point_total_num, sparse_unique_index, 
      webbur::level_to_order_default, sparse_weight );

    weight_sum = webbur::r8vec_sum ( point_num, sparse_weight );

    weight_sum_error = webbur::r8_abs ( weight_sum - weight_sum_exact );

    std::cout << "  " << std::setw(8)  << level_max
         << "  " << std::setw(14) << weight_sum
         << "  " << std::setw(14) << weight_sum_exact
         << "  " << std::setw(14) << weight_sum_error << "\n";

    delete [] sparse_unique_index;
    delete [] sparse_weight;
  }
  return;
}
//****************************************************************************80

void sparse_grid_mixed_growth_write_tests ( double tol )

//****************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_WRITE_TESTS calls SPARSE_GRID_MIXED_GROWTH_WRITE_TEST.
//
//  Discussion:
//  
//    We can't test Golub-Welsch rules in this routine, because the program
//    that writes out the files needs to know the integration region for each
//    component, and we have not specified how that would be done with 
//    Golub Welsch rules.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    16 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, double TOL, a tolerance for point equality.
//    A value of sqrt ( eps ) is reasonable, and will allow the code to
//    consolidate points which are equal, or very nearly so.  A value of
//    -1.0, on the other hand, will force the code to use every point, 
//    regardless of duplication.
//
{
  double *alpha;
  double *beta;
  int dim_num;
  std::string file_name;
  GWPointer *gw_compute_points;
  GWPointer *gw_compute_weights;
  int level_max;
  int level_max_max;
  int level_max_min;
  int *order_1d;
  int order_nd;
  int *rule;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_WRITE_TESTS\n";
  std::cout << "  Call SPARSE_GRID_MIXED_GROWTH_WRITE_TEST with various arguments.\n";
  std::cout << "\n";
  std::cout << "  All tests will use a point equality tolerance of " << tol << "\n";

  dim_num = 2;
  level_max= 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 1;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::clenshaw_curtis_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::clenshaw_curtis_compute_weights;
  file_name = "sparse_grid_mixed_growth_d2_l2_ccxcc";
  sparse_grid_mixed_growth_write_test ( dim_num, level_max, rule, alpha, 
    beta, gw_compute_points, gw_compute_weights, tol, file_name );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max= 3;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 3;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::patterson_lookup_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::patterson_lookup_weights;
  file_name = "sparse_grid_mixed_growth_d2_l3_ccxgp";
  sparse_grid_mixed_growth_write_test ( dim_num, level_max, rule, alpha, 
    beta, gw_compute_points, gw_compute_weights, tol, file_name );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max= 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 4;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::legendre_compute_weights;
  file_name = "sparse_grid_mixed_growth_d2_l2_ccxgl";
  sparse_grid_mixed_growth_write_test ( dim_num, level_max, rule, alpha, 
    beta, gw_compute_points, gw_compute_weights, tol, file_name );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max= 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 7;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::laguerre_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::laguerre_compute_weights;
  file_name = "sparse_grid_mixed_growth_d2_l2_ccxlg";
  sparse_grid_mixed_growth_write_test ( dim_num, level_max, rule, alpha, 
    beta, gw_compute_points, gw_compute_weights, tol, file_name );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max= 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 1.5;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 1;
  rule[1] = 8;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::gen_laguerre_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::gen_laguerre_compute_weights;
  file_name = "sparse_grid_mixed_growth_d2_l2_ccxglg";
  sparse_grid_mixed_growth_write_test ( dim_num, level_max, rule, alpha, 
    beta, gw_compute_points, gw_compute_weights, tol, file_name );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max= 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.5;
  beta[0] = 0.0;
  beta[1] = 1.5;
  rule[0] = 2;
  rule[1] = 9;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::fejer2_compute_points;
  gw_compute_points[1] = webbur::jacobi_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::fejer2_compute_weights;
  gw_compute_weights[1] = webbur::jacobi_compute_weights;
  file_name = "sparse_grid_mixed_growth_d2_l2_f2xgj";
  sparse_grid_mixed_growth_write_test ( dim_num, level_max, rule, alpha, 
    beta, gw_compute_points, gw_compute_weights, tol, file_name );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 2;
  level_max= 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 2.0;
  alpha[1] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  rule[0] = 6;
  rule[1] = 4;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::gen_hermite_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::gen_hermite_compute_weights;
  gw_compute_weights[1] = webbur::legendre_compute_weights;
  file_name = "sparse_grid_mixed_growth_d2_l2_gghxgl";
  sparse_grid_mixed_growth_write_test ( dim_num, level_max, rule, alpha, 
    beta, gw_compute_points, gw_compute_weights, tol, file_name );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  dim_num = 3;
  level_max= 2;
  alpha = new double[dim_num];
  beta = new double[dim_num];
  rule = new int[dim_num];
  alpha[0] = 0.0;
  alpha[1] = 0.0;
  alpha[2] = 0.0;
  beta[0] = 0.0;
  beta[1] = 0.0;
  beta[2] = 0.0;
  rule[0] = 1;
  rule[1] = 4;
  rule[2] = 5;
  gw_compute_points = new GWPointer[dim_num];
  gw_compute_points[0] = webbur::clenshaw_curtis_compute_points;
  gw_compute_points[1] = webbur::legendre_compute_points;
  gw_compute_points[2] = webbur::hermite_compute_points;
  gw_compute_weights = new GWPointer[dim_num];
  gw_compute_weights[0] = webbur::clenshaw_curtis_compute_weights;
  gw_compute_weights[1] = webbur::legendre_compute_weights;
  gw_compute_weights[2] = webbur::hermite_compute_weights;
  file_name = "sparse_grid_mixed_growth_d3_l2_ccxglxgh";
  sparse_grid_mixed_growth_write_test ( dim_num, level_max, rule, alpha, 
    beta, gw_compute_points, gw_compute_weights, tol, file_name );
  delete [] alpha;
  delete [] beta;
  delete [] gw_compute_points;
  delete [] gw_compute_weights;
  delete [] rule;

  return;
}
//***************************************************************************80

void sparse_grid_mixed_growth_write_test ( int dim_num, int level_max, 
  int rule[], double alpha[], double beta[], 
  void ( *gw_compute_points[] ) ( int order, double alpha, double beta, double w[] ),
  void ( *gw_compute_weights[] ) ( int order, double alpha, double beta, double w[] ),
  double tol, std::string file_name )

//***************************************************************************80
//
//  Purpose:
//
//    SPARSE_GRID_MIXED_GROWTH_WRITE_TEST tests SPARSE_GRID_MIXED_GROWTH_WRITE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    01 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, integer DIM_NUM, the spatial dimension.
//
//    Input, integer LEVEL_MAX, the level that defines the grid.
//
//    Input, int RULE[DIM_NUM], the rule in each dimension.
//     1, "CC",  Clenshaw Curtis, Closed Fully Nested rule.
//     2, "F2",  Fejer Type 2, Open Fully Nested rule.
//     3, "GP",  Gauss Patterson, Open Fully Nested rule.
//     4, "GL",  Gauss Legendre, Open Weakly Nested rule.
//     5, "GH",  Gauss Hermite, Open Weakly Nested rule.
//     6, "GGH", Generalized Gauss Hermite, Open Weakly Nested rule.
//     7, "LG",  Gauss Laguerre, Open Non Nested rule.
//     8, "GLG", Generalized Gauss Laguerre, Open Non Nested rule.
//     9, "GJ",  Gauss Jacobi, Open Non Nested rule.
//    10, "GW",  Golub Welsch, (presumed) Open Non Nested rule.
//
//    Input, double ALPHA[DIM_NUM], BETA[DIM_NUM], parameters used for
//    Generalized Gauss Hermite, Generalized Gauss Laguerre, 
//    and Gauss Jacobi rules.
//
//    Input, void ( *GW_COMPUTE_POINTS[] ) ( int order, double alpha, double beta, double x[] ),
//    an array of pointers to functions which return the 1D quadrature points 
//    associated with each spatial dimension for which a Golub Welsch rule 
//    is used.
//
//    Input, void ( *GW_COMPUTE_WEIGHTS[] ) ( int order, double alpha, double beta, double w[] ),
//    an array of pointers to functions which return the 1D quadrature weights 
//    associated with each spatial dimension for which a Golub Welsch rule 
//    is used.
//
//    Input, double TOL, a tolerance for point equality.
//
//    Input, string FILE_NAME, the main name of the output files.
//
{
  int point_num;
  int point_total_num;
  int *sparse_index;
  int *sparse_order;
  double *sparse_point;
  int *sparse_unique_index;
  double *sparse_weight;

  std::cout << "\n";
  std::cout << "SPARSE_GRID_MIXED_GROWTH_WRITE_TEST\n";
  std::cout << "  SPARSE_GRID_MIXED_GROWTH_WRITE writes a sparse grid rule\n";
  std::cout << "  to X, W and R files.\n";
//
//  Compute necessary data.
//
  point_total_num = webbur::sparse_grid_mixed_growth_size_total ( dim_num, 
    level_max, rule, webbur::level_to_order_default );

  point_num = webbur::sparse_grid_mixed_growth_size ( dim_num, level_max, 
    rule, alpha, beta, gw_compute_points, tol, 
    webbur::level_to_order_default );

  sparse_unique_index = new int[point_total_num];

  webbur::sparse_grid_mixed_growth_unique_index ( dim_num, level_max, rule, 
    alpha, beta, gw_compute_points, tol, point_num, point_total_num, 
    webbur::level_to_order_default, sparse_unique_index );

  sparse_order = new int[dim_num*point_num];
  sparse_index = new int[dim_num*point_num];

  webbur::sparse_grid_mixed_growth_index ( dim_num, level_max, rule, point_num, 
    point_total_num, sparse_unique_index, webbur::level_to_order_default,
    sparse_order, sparse_index );
//
//  Compute points and weights.
//
  sparse_point = new double [ dim_num * point_num ];

  webbur::sparse_grid_mixed_growth_point ( dim_num, level_max, rule, alpha, 
    beta, gw_compute_points, point_num, sparse_order, sparse_index, 
    webbur::level_to_order_default, sparse_point );

  sparse_weight = new double[point_num];

  webbur::sparse_grid_mixed_growth_weight ( dim_num, level_max, rule, alpha, 
    beta, gw_compute_weights, point_num, point_total_num, sparse_unique_index, 
    webbur::level_to_order_default, sparse_weight );
//
//  Write points and weights to files.
//
  webbur::sparse_grid_mixed_growth_write ( dim_num, rule, alpha, beta, 
    point_num, sparse_weight, sparse_point, file_name );

  delete [] sparse_index;
  delete [] sparse_order;
  delete [] sparse_point;
  delete [] sparse_unique_index;
  delete [] sparse_weight;

  return;
}
