// $Id: MEDIATOR_Main_Serial.cpp,v 1.3 2008/06/10 13:08:03 jgriffi Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/MEDIATOR_Main_Serial.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file MEDIATOR_Main_Serial.cpp
  \brief The main for a serial version of appspack based on a simple system call.
  \todo Josh.  Fix comments.
*/

#include "APPSPACK_Common.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Utils.hpp"
#include "APPSPACK_Evaluator_SystemCall.hpp"
#include "APPSPACK_Executor_Serial.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_Mediator.hpp"
#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Citizen_Generator.hpp"

//! Main subroutine for default serial version of APPSPACK

string IntToString(int n)
{
  ostringstream ss;
  ss << n;
  return ss.str();
}

int main(int argc, char* argv[])
{
  
  if (argc < 2)
  {
    cout << "Usage: " << argv[0] << " <input file>" << endl;
    return 1;
  }

  // Parse parameters.
  APPSPACK::Parameter::List params;
  bool retCode = APPSPACK::parseTextInputFile(argv[1], params);
  if (retCode == false)
    return 1;

 
  // Create executor needed by Conveyor.
  APPSPACK::Evaluator::SystemCall evaluator(params.sublist("Evaluator"));
  APPSPACK::Executor::Serial executor(evaluator);
    
  // Create linear constraints object to determine scaling vector needed by conveyor.
  APPSPACK::Constraints::Linear linear(params.sublist("Linear"));
    
  // Create conveyor used by mediator to perform function evaluations in parallel.
  APPSPACK::Conveyor conveyor(params.sublist("Mediator"), linear.getScaling(), executor);
  APPSPACK::Mediator mediator(params.sublist("Mediator"), conveyor);
  APPSPACK::Combiner::Generic combiner;
  APPSPACK::Citizen::Generator generator;
  APPSPACK::Citizen::Interface* ci;
  vector<APPSPACK::Citizen::Interface*> citizenList;


  int nCitizens = params.sublist("Mediator").getParameter("Citizen Count", -1);
  if (nCitizens < 1)
  {
    cout << "Number of Citizens must be greater than 0." << endl;;
    return 1;
  }

  vector<int> workerRank;  // For serial always empty.
  for (int i=1; i<= nCitizens; i++)
  {
    // Must have nCitizens defined starting at 1.
    string name = "Citizen " + IntToString(i);
    if (!params.isParameter(name))
    { 
      cout << "Error: \"Citizen " << IntToString(i) << "\" must be defined." << endl;
      return 1;
    }

    // Each Citizen must specify it's type.
    APPSPACK::Parameter::List& citizenParams = params.sublist(name);
    if (!citizenParams.isParameter("Type"))
    {
      cout << "Error: \"Citizen " <<  IntToString(i) <<  "\" must define \"Type\"." << endl;
      return 1;
    }

    string type=citizenParams.getParameter("Type", "Undefined");

    // Citizens may not request multiple processors.
    int np = citizenParams.getParameter("Processors", 0);
    if (np > 0)
    {
      cout << "Error: \"Citizen " <<  IntToString(i) <<  "\" cannot request "
	   << "worker processors in serial.\n" << endl;
      return 1;
    }

    // Citizen appears to check out.
    ci = generator.newCitizenType(params, linear, combiner,
                                  type, name, workerRank);
    mediator.push(*ci);
    citizenList.push_back(ci);
  }
  
  mediator.mediate();

  for (int i=0; i<citizenList.size(); i++)
    delete citizenList[i];
    
  APPSPACK::Point::printMemoryStats();
}

