// $Id: APPSPACK_Mediator.hpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Mediator.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Mediator.hpp
  \brief Class definition for APPSPACK::Mediator
*/
#ifndef APPSPACK_MEDIATOR_HPP
#define APPSPACK_MEDIATOR_HPP

#include "APPSPACK_Conveyor.hpp"
#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Matrix.hpp"

namespace APPSPACK
{

//! A class that holds a list of Citizen, mediating point evaluations.
/*! \ref pageMediator */
class Mediator
{
public:

  //! Constructor 
  Mediator(Parameter::List& params, Conveyor& conveyor);


  //! Destructor 
  ~Mediator();

  //! Adds citizen to #town
  void push(Citizen::Interface& citizen);

  //! Loops while town is active, processing point evaluations.
  /*!
    This method repeats the following steps while the town remains
    active:

    <ul>

    <li> A call to citizenExhange() is first made where Citizens may 
    add points to the current queue and view evaluated points. 

    <li> After all Citizens have had a chance to view the evaluated
    points list, these points are deleted as there is a record of these
    evaluations stored in the cache owned by the Conveyor.

    <li> After all Citizens have had the chance to add points to the
    current queue lists, a call to citizenSelectOrder() is made, where Citizens
    are allowed to vote on the order of the queue (but not alter it).
    The Mediator then polls all votes and orders the queue accordingly.

    <li> A call to conveyorExchange() is made where unevaluated points
    are traded for evaluated points.  On exit from this call, the
    mediator will retain ownership of all unevaluated points that are
    not pending evaluation and evaluated points.  The Conveyor retains
    ownership of all points pending evaluation.
    
    </ul>
    
    \note The Mediator retains ownership of all points in queue and
    evaluated lists.
  */
  void mediate();

private:

  //! Returns true if town wishes to continue point evaluations.
  /*!  
    <ul> 

    <li> A Citizen in #town that is also a Citizen::Mayor, may force
    everyone to continue to exchange points, keeping the town active.
    
    <li> A Citizen in #town that is also a Citizen::Mayor, may force
    everyone to stop exchanging points, and halt the mediation
    process.

    <li> The mediation process will continue until either a Mayor
    signals a Citizen::MustStop flag or at least one Citizen signals
    Citizen::PassiveContinue or Citizen::MustContinue.

    \note For regular citizens, states Citizen::PassiveContinue and
    Citizen::MustContinue are treated identically.

    </ul>
  */
  bool townActive();

  //! Permits Citizens to see new evaluations and add to the queue.
  /*!
    This method calls APPSPACK::Citizen::Interface::exchange()
    for all Citizen's that are not currently retired.  This method
    permits citizens to do the following:
    <ul>
    
    <li> Peak at the state of the current queue list.  

    <li> View (but not gain ownership of) all evaluations that have
    completed since last call to citizenExchange().

    <li> Add points to an empty queue list which will then be
    absolved into the current queue list.  This ensures that
    a citizen does not modify the global queue list.

    <li> Citizens are also provided with the #ownermap so that
    they can determine who owns which points.
    
    </ul>
  */ 
  void citizenExchange(ConveyorList& W, const ConveyorList& R);

  //! Citizens may vote on queue order.
  /*!  Get votes from each citizen to determine queue order.  A
    Citizen does not necessarily need to vote on all tags in W.  After
    all votes have been recieved, votes for each point tag in W are
    tallied.  Point's that have a higher number of votes are placed
    earlier in the queue.  No specific order is placed on points whose
    votes tie.  If a point recieves zero votes, it is pruned from the
    evalaution queue.

    \note Citizens may not vote on points outside the queue.
  */
  void citizenSelectOrder(ConveyorList& W);
  
  //! Exchange point list with Conveyor.
  /*!
    \param W (input/output) On entry W stores a list of points the
    Mediator wishes to be evaluated in FIFO order.  On exit, W retains
    ownership of unevaluated points that are not pending.

    \param R (output) On exits R stores the list of newly evaluated points
    returned from the Conveyor.

    \see APPSPACK::Conveyor::exchange()
  */ 
  void conveyorExchange(ConveyorList& W, ConveyorList& R);

  //! Prune R and update ownermap.
  /*!
    All Citizens have had a chance to look at the evaluated list R.  As the
    Conveyor stores a cache of all point evaluations, R is no longer needed.
    Thus we empty this list, deleting the corresponding points.  Method
    deleteProcessedTags() is also call to ensure ownermap is up to date
    with respect to points currently owned by the Mediator.    
  */
  void pruneExchangeList(ConveyorList& R);

  //! Call preprocess methods for all Citizens. 
  /*!
    This method is called once at the beginning of the mediation
    process and allows Citizens a final chance to do any necessary
    preprocessing.  Preprocessing may simply be printing initial 
    information like the parameter list.
  */
  void preprocess();

  //! Call postprocess methods for all Citizens. 
  /*!
    This method is called once on conclusion of the mediation
    process and allows Citizens a lkast chance to do any necessary
    preprocessing.  Preprocessing may simply be print final
    solution state for your particular algorithm to a file.
  */
  void postprocess();

  //! Insert tags contained in P to #ownermap corresponding to Citizen name.
  /*! Essentially, tags of points in P are added to list of tags
    stored in ownermap[name].
  */
  void updateOwnerMap(const string& name, const ConveyorList& P);
  
  //! Get queue order map from each Citizen in clist.
  int getTagOrderFromCitizenList(const ConveyorList& W, const vector<int>& clist, 
                                 map<int, vector<int> >& cmap);

  //! Adds tags in clist to queue, top down order.
  void spliceTags(const vector<int>& clist, map<int, vector<int> >& cmap,
                  vector<int>& newq);

  //! Deletes points from owner map corresponding to tags in tagslist. 
  void deleteProcessedTags(const vector<int>& taglist);

private:
  //! Parameters
  Parameter::List& params;
  
  //! Trial Point Evaluator
  Conveyor& conveyor; 

  //! Container for reference to citizens.
  vector<Citizen::Interface*> town;

  //! Contains record of point ownership.
  map<string, vector<int> >  ownermap;

  //! How big the queue is allowed to be after a prune (set according to "Max Queue Size" parameter)
  int maxQueue;

  //! Function evaluation budget
  int maxEvaluations;

  //! Priority level.
  vector<int> priority;

  //! Debug level for printing mediator information.
  int debug;

  //! Records number of points evaluated by each citizen.
  map<string, int> record;

  //! Rank of mediator.
  int rank;

};

}

//! Prints the contents of a vector of ints.
ostream& operator<<(ostream& stream, const vector<int>& v);

//! Prints the contents of a vector of ints.
ostream& operator<<(ostream& stream, const map<string, vector<int> >& m);

#endif
