//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_DIRECT_Wrappers.cpp
  \brief Implementation of functions declared in APPSPACK_DIRECT_Wrappers.hpp.
*/

#include "APPSPACK_DIRECT_Wrappers.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Cache_Manager.hpp"
#include "APPSPACK_Executor_Oracle.hpp"

int writeDIRECTPoints(int *n, double c[], double l[], double u[],
		      int point[], int* maxI, int* start, int* maxfunc,
                      double fvec[], int iidata[], int *iisize, double ddata[], 
		      int* idsize, char cdata[], int* icsize)
{
  // Dimension of x.
  int nx=*n;
  vector<APPSPACK::Vector> X;
  if (*start == 1)
  {
    // In this case, c is just a vector.
    APPSPACK::Vector p(nx);
    // evaluating initial points.
    for (int i=0; i<nx; i++)
      p[i] = (c[i]+u[i])*l[i];
    X.push_back(p);
  }
  else
  {
    // Number of trial points.
    int np=*maxI*2;
    // Starting point in Fortran matrix c.
    int pos=*start-1;
    for (int j=0; j<np; j++)
    {    
      APPSPACK::Vector p(nx);
      for (int i=0; i<nx; i++)
      {
	// c[pos+i*maxfunc] = c(pos,i) in Fortran.
	double ci=c[pos+i*(*maxfunc)];
	p[i] = (ci + u[i])*l[i];
      }
      pos = point[pos]-1;
      X.push_back(p);
    }
  }
  
  vector<APPSPACK::Vector> F;  
  bool directContinue=exchangePoints(X, F);
  
  if (directContinue == false)
    iidata[0]=0;  // Signal a stop to direct optimization.

  int cnt=*start-1;
  for (int i=0; i<F.size(); i++)
  {
    // Now set f(xi);    
    fvec[cnt]=F[i][0];
    // Now set the flag;
    fvec[cnt+(*maxfunc)-1]=0;
    cnt++;
  }

  return 0;
}

void callDIRECT(const APPSPACK::Vector& xlow,
		const APPSPACK::Vector& xupp,
                string logfile,
		vector<int>& iidata, vector<double>& ddata,
		vector<char>& cdata)
{
  int state;
  string lname=logfile;
  char gname[200];

  int lunit=2;
  strcpy(gname,lname.c_str());
  f77open_(&lunit,gname,&state,lname.length());

  int n=xlow.size();
  vector<double> x(n,0);
  vector<double> l=xlow.getStlVector();
  vector<double> u=xupp.getStlVector();
  double eps=ddata[0];
  double fglper=ddata[1]; // Global percentage.
  double fglobal=ddata[2]; // Known global minimum.
  double volper=-1;
  double sigmaper=-1;

  int maxf=iidata[2];  // Max function evaluations.
  int maxT=iidata[3];  // Max iterations.
  double fmin=1e20;
  int algmethod=iidata[4];
  int Ierror=11111;
    
  int iisize=iidata.size();
  int icsize=ddata.size();
  int idsize=cdata.size();  

  direct_(writeDIRECTPoints, &x[0], 
	  &n, &eps, &maxf, &maxT, &fmin,
          &l[0], &u[0], &algmethod, &Ierror, &lunit,
          &fglobal, &fglper, &volper, &sigmaper,
          &iidata[0], &iisize, &ddata[0], &idsize, 
	  &cdata[0], &icsize, icsize);
  
  f77close_(&lunit);
  
  cout << "DIRECT call complete." << endl;

  // Communicate to Citizen that file exists.
  APPSPACK::GCI::initSend();
  string exit_status="DIRECT Finished";
  APPSPACK::GCI::pack(exit_status);
  APPSPACK::GCI::send(APPSPACK::Executor::Oracle::Generate, 0);  

}

bool exchangePoints(const vector<APPSPACK::Vector>& T, vector<APPSPACK::Vector>& F)
{
  // Communicate to Citizen that file exists.

  APPSPACK::GCI::initSend();
  string state="New Points";
  APPSPACK::GCI::pack(state);
  APPSPACK::GCI::pack(T);
  APPSPACK::GCI::send(APPSPACK::Executor::Oracle::Generate, 0);  
  
  // Wait until points are evaluated.
  int msgtag, junk, taskid;
  vector<int> Xtags;

  // X may have different order from T.
  vector<APPSPACK::Vector> X;
  // Function F ordered by X values.
  vector<APPSPACK::Vector> FX;
  APPSPACK::GCI::recv();
  APPSPACK::GCI::bufinfo(msgtag, junk);

  // Check for termination.
  if (msgtag == APPSPACK::Executor::Oracle::Terminate)
  {
    cout << "DIRECT should exit now." << endl;
    return false;
  }

  APPSPACK::GCI::unpack(state);
  APPSPACK::GCI::unpack(FX);
  APPSPACK::GCI::unpack(X);
  APPSPACK::GCI::unpack(Xtags);

  // Order objective values according to increasing tag values.
  map<int,APPSPACK::Vector> orderMap;
  for (int i=0; i<Xtags.size(); i++)
    orderMap[Xtags[i]] = FX[i];
  
  // Clear all F to be safe.
  F.clear();
  map<int,APPSPACK::Vector>::iterator itr;
  for (itr = orderMap.begin(); itr != orderMap.end(); itr++)
    F.push_back(itr->second);

  return true;
}
