// $Id: APPSPACK_Citizen_Random.cpp,v 1.3 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_Random.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_Random.cpp
  \brief Implements APPSPACK::Citizen_Random
*/

#include "APPSPACK_Citizen_Random.hpp"
#include <time.h>

APPSPACK::Citizen::Random::Random(Parameter::List& params_in,
				  const Constraints::Linear& constraints_in,
				  Combiner::Generic& combiner_in,
				  const string name_in) :
  params(params_in),
  constraints(constraints_in),
  directions(params, constraints),
  bestPointPtr(NULL),
  name(name_in)
{    
  srand(time(NULL));
}

APPSPACK::Citizen::Random::~Random()
{
  exchangeList.prune();
}

void APPSPACK::Citizen::Random::preprocess()
{
}

void APPSPACK::Citizen::Random::postprocess()
{
}

void APPSPACK::Citizen::Random::exchange(const APPSPACK::ConveyorList& R,
					 APPSPACK::ConveyorList& Wnew,
                                         const map<string, vector<int> >& ownermap)
{
  const vector<int>& citizen_tags = ownermap.find(name)->second;
  if (R.isEmpty())
    return;

  // Update new best point.
  List newPoints;
  newPoints.copyFrom(R);
  bool computeRandom = false;  // Only compute random when new points are found.
  if (bestPointPtr == NULL)
  {
    bestPointPtr = newPoints.popBest();
    computeRandom = true;
  }
  else if (newPoints.best() < *bestPointPtr)
  {
    delete bestPointPtr;
    bestPointPtr = newPoints.popBest();
    computeRandom = true;
  }
   
  // If bestPointPtr is still null return.
  if ( (bestPointPtr == NULL) || (computeRandom == false) )
    return;
  
  // Now compute feasible directions.
  directions.computeNewDirections(*bestPointPtr);

  Wnew.push(getRandomPoint());

  newPoints.prune();
}

APPSPACK::Point* APPSPACK::Citizen::Random::getRandomPoint()
{
  // Local references
  const Vector& parentX = bestPointPtr->getX();
  int n = parentX.size();

  // Create random directions.
  Vector drand(n, 0.0);
  vector<int> indices;
  directions.getDirectionIndices(indices);
  for (int i = 0; i < indices.size(); i ++)
  {
    // Get the a search direction and compute max feasible step.
    const Vector& direction = directions.getDirection(indices[i]);
    const double step = directions.getStep(indices[i]);
    double maxStep = constraints.maxStep(parentX, direction, step);
  
    // wi denotes maximum step times random number between 0 and 1.
    double wi = maxStep*((double) rand()) / ((double) RAND_MAX);

    // Now add this step to random direction.
    for (int j = 0; j < n; j++)
      drand[j] = drand[j] + wi*direction[j];
  } 
  
  // Now drand denotes our random direction.

  // Take max step along random direction.
  Vector x(n);
  double maxStep = constraints.maxStep(parentX, drand, drand.norm());
  for (int j = 0; j < n; j++)
    x[j] = parentX[j] + maxStep*drand[j];

  constraints.assertFeasible(x);  
  Point p(x, maxStep, *bestPointPtr, -1);

  // Return an orphan point so we do not mess up APPSPACK.
  return new Point(p,-1);  
}

void APPSPACK::Citizen::Random::vote(const APPSPACK::ConveyorList& W,
				     const map<string, vector<int> >& ownermap,
				     vector<int>& tagOrder)
{  
  tagOrder.clear();

  // Get list of tags in the queue W.
  vector<int> wtags;
  W.getTagList(wtags);

  // Get list of tags that this owns.
  const vector<int>& random_tags = ownermap.find(name)->second;

  // Give all points that we own a vote of one.
  for (int i=0; i<random_tags.size(); i++)
  {
    int tag = random_tags[i];
    if (find(wtags.begin(), wtags.end(), tag) != wtags.end())
      tagOrder.push_back(tag);
  }

}

APPSPACK::Citizen::State APPSPACK::Citizen::Random::getState()
{
  // Random wishes to stop when everyone is ready.
  return APPSPACK::Citizen::StopRequest;
}

const string& APPSPACK::Citizen::Random::getName() const
{
  return name;
}
