// $Id: APPSPACK_Citizen_LHS.hpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_LHS.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_LHS.hpp
  \brief Class definition for APPSPACK::Citizen_LHS

  Options (and defaults) for the LHS Citizen:
  - "Executable Name" string "a.out"
  - "Sample Size" int 100
  - "LHS File" string "LHS.txt"
  - "Default Step" double 1.0
*/

#ifndef APPSPACK_CITIZEN_LHS_HPP
#define APPSPACK_CITIZEN_LHS_HPP

#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Constraints_Linear.hpp"

namespace APPSPACK
{

namespace Citizen
{

//! Class for submitting LHS trial points.
class LHS : public Interface
{
public:

  //! Constructor 
  LHS(Parameter::List& params_in,
      const Constraints::Linear& constraints_in,
      Combiner::Generic& combiner_in, 
      const string name = "");
  
  //! Destructor
  ~LHS();

  //! Prints initial state of citizen.
  virtual void preprocess();

  //! Prints final state of citizen.
  virtual void postprocess();
  
  //! Take in list of evaluated appspacks. Give list of new points.
  virtual void exchange(const ConveyorList& R, ConveyorList& Wnew,
                        const map<string, vector<int> >& ownermap);

  //! Vote on points to be evaluated.
  virtual void vote(const ConveyorList& W, 
		    const map<string, vector<int> >& ownermap,
		    vector<int>& tagOrder);

  //! Return Citizen's state.
  virtual Citizen::State getState();

  //! Returns Citizen's name.
  virtual const string& getName() const;
  
  //! Call's tgp oracle.
  virtual bool worker();

private:

  //! Converts an int to a string.
  string IntToString(int n) const;

  //! Reads numPoints from output file
  void readOutputFile(Matrix& TP);

  //! Converts a string of form line = "1 2 3" to a vector x =<1 2 3>.
  void StringToVector(const string& line, Vector& x) const;

  //! Adds rows of TP as points to Wnew list.
  void addTrialPoints(ConveyorList& Wnew, const Matrix& TP);

private:

  //! Name of citizen.
  const string name;

  //! Parameters
  Parameter::List& params;
  
  //! Combiner reference needed when generating points
  Combiner::Generic& combiner;

  //! Constraints
  const Constraints::Linear& constraints;

  //! Number of samples.
  int sampleSize;
  
  //! File where TGP writes LHS points.
  string samplesFile;
  
  //! Tag referencing individual call to TGP.
  string fileTag;

  //! Name of executable that generates points.
  string executableName;

  //! Default step size.
  double defaultStep;

  //! Number of times this particular method has been called.
  int callCount;

  //! Counts number of times an LHS routine has been called.
  static int staticCallCount;
};

}
}
#endif
