//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*! 
  \file NAPPSPACK_NLSolver.cpp
  \brief Implementation of NAPPSPACK::NLSolver
*/

#include "NAPPSPACK_NLSolver.hpp"
#include "APPSPACK_Float.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Executor_Serial.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "NAPPSPACK_Method_Smooth.hpp"

NAPPSPACK::NLSolver::NLSolver(APPSPACK::Parameter::List& params_in, 
			      APPSPACK::Executor::Interface& executor_in, 
			      const APPSPACK::Constraints::Linear& constraints_in) :
  params(params_in),
  executor(executor_in),
  constraints(constraints_in),
  cache(params.sublist("Solver"), constraints.getScaling()),
  nCached(0),
  nEvaluated(0)
{
  /*
  cout << "\n"
       << "-----------------------------------------------------\n"
       << "NAPPSPACK: Nonlinear Asynchronous Parallel Pattern Search\n"
       << "Sandia National Labs\n"
       << "For more information visit \n"
       << "http://software.sandia.gov/appspack\n"
       << "-----------------------------------------------------\n"
       << "\n";
  */

  method = (Method::Generic*)(new Method::Smooth(params));

  maxTotalEvaluations=params.sublist("Nonlinear").getParameter("Maximum Total Evaluations", 100000);
  maxMajorEvaluations=params.sublist("Nonlinear").getParameter("Maximum Major Evaluations", 1000);
  maxMajorIterations=params.sublist("Nonlinear").getParameter("Maximum Major Iterations", 1000);

  // Get number of nonlinear constraints.
  nlec = params.sublist("Nonlinear").getParameter("Number Equality", 0);
  nlic = params.sublist("Nonlinear").getParameter("Number Inequality", 0);

  cout << "=====================================================\n";
  cout << "Nonlinear/linear constraint summary.\n";
  cout << "=====================================================";
  constraints.print();
  cout << "Number of nonlinear equality: " << nlec << endl;
  cout << "Number of nonlinear inequality: " << nlic << endl;
}

NAPPSPACK::NLSolver::~NLSolver()
{
  delete method;
}

//! Return the x-vector corresponding to the best point
const vector<double>& NAPPSPACK::NLSolver::getBestX() const
{
  return xBest.getStlVector();
}

//! Returns instance of class Value associated with current best point.
const vector<double>& NAPPSPACK::NLSolver::getBestFC() const
{
  return fcBest.getStlVector();
}

// Returns total number of points evaluated.
int  NAPPSPACK::NLSolver::getNumEvaluations() const
{
  return nEvaluated;
}

// Returns total number of cached evaluations.
int  NAPPSPACK::NLSolver::getNumCached() const
{
  return nCached;
}

void NAPPSPACK::NLSolver::solveSubproblem()
{
  // Solve subproblem.
  maxMajorEvaluations=min(maxTotalEvaluations - nEvaluated, maxMajorEvaluations);
  params.sublist("Solver").setParameter("Maximum Evaluations", maxMajorEvaluations);
  APPSPACK::Solver solver(params.sublist("Solver"), executor, 
			  constraints, method->getCombiner(),
			  cache);
  minorState = solver.solve();

  cout << "MinorState = " << minorState << endl;
  // Update initial best x, f(x), and c(x) and add to parameter list
  // for next solve.
  xBest=solver.getBestX();
  fcBest=solver.getBestVecF();
  params.sublist("Solver").setParameter("Initial X", xBest);
  params.sublist("Solver").setParameter("Initial F", fcBest); 
  nCached += solver.getNumCached();
  nEvaluated += solver.getNumEvaluations();
  method->update(xBest, fcBest, minorState);
}


NAPPSPACK::NLSolver::State NAPPSPACK::NLSolver::solve()
{
  int oldEval=0; 
  int oldCach=0; 
  string success="f";
  majorState=NAPPSPACK::NLSolver::MaximumMajors;
  cout << "=====================================================\n" 
       << "Solving with " << method->getName() << ":\n" 
       << "=====================================================\n";
  for (int k = 1; k <= maxMajorIterations; k++)
  {
    solveSubproblem();
    
    cout << "Major: " << k << ", Evals = " << nEvaluated << ", Cache = " << nCached
	 << ", dEvals = " << nEvaluated - oldEval << ", dCache = " << nCached - oldCach << endl;
    oldEval = nEvaluated;
    oldCach = nCached;
    method->printMajor(xBest,fcBest);    

    if (method->isOptimal(fcBest) && minorState == APPSPACK::Solver::StepConverged)
    {
      cout << "NLP Converged!" << endl;  
      success="s";
      majorState=NAPPSPACK::NLSolver::SuccessfulExit;
      break;
    }
    if (nEvaluated > maxTotalEvaluations)
    {
      cout << "Max Evaluations" << endl;
      majorState=NAPPSPACK::NLSolver::MaximumEvals;
      success="m";
      break;
    }
  }
  cout.precision(12);
  cout << "(f[0], cinf, eval, cache, minorState, nvar, nlec, nlic) = " << fcBest[0] << ", " << method->infnorm(fcBest) 
       << ", " << nEvaluated << ", " << nCached << ", " << success 
       << ", " << xBest.size() << ", " << nlec << ", " << nlic << endl;

  cout << "(Use method " << method->getName() << ")" << endl;

  /*
  cout << "\n"
       << "-----------------------------------------------------\n"
       << "NAPPSPACK: Nonlinear Asynchronous Parallel Pattern Search\n"
       << "Sandia National Labs\n"
       << "For more information visit \n"
       << "http://software.sandia.gov/appspack\n"
       << "-----------------------------------------------------\n"
       << "\n";
  */

  return majorState;
}


