//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*! 
  \file NAPPSPACK_Filter.cpp
  \brief Implementation of NAPPSPACK::Filter
*/

#include "NAPPSPACK_Filter.hpp"
#include "APPSPACK_Print.hpp"
#include <cassert>

NAPPSPACK::Filter::Filter(int esmall, int ebig) :
  es(esmall), eb(ebig)
{
}

void NAPPSPACK::Filter::insert(const APPSPACK::Vector& x, const APPSPACK::Vector& fc)
{
  if (fc.size() == 0)
    return;

  double norm = onenorm(fc);
  for (int i = es; i<eb; i++)
  {
    if (norm < pow(10.0,i)) //Only update if norm is in filter[i]'s range.
    {
      if (filter.find(i) == filter.end()) //Always add pair if filter[i] is empty.
	filter[i] = make_pair(x,fc);
      else if (fc[0] < filter[i].second[0]) // if fnew(x) < fold(x) update.
	filter[i] = make_pair(x,fc);
    }
  }
}

double NAPPSPACK::Filter::onenorm(const APPSPACK::Vector& fc)
{
  // Determine one norm.
  double norm = 0;
  for (int i=1; i<fc.size();i++)
    norm += fabs(fc[i]);
  
  return norm;
}

void NAPPSPACK::Filter::print()
{
  cout << "Filter is ..." << endl;
  for (iter=filter.begin(); iter != filter.end(); iter++)
  {
    const APPSPACK::Vector& fc = (iter->second).second;
    const APPSPACK::Vector& x = (iter->second).first;
    cout << "d= " << setw(3) << iter->first;
    cout.precision(2);
    cout.setf(ios::scientific);
    cout << ", f1 = " << setw(5) << fc[0] << ", c1 = " << setw(5) 
	 << onenorm(fc) << ", x = " << x << ", fc = " << fc << endl;
  }
}

bool NAPPSPACK::Filter::getBest(int d, APPSPACK::Vector& xbest, APPSPACK::Vector& fcbest)
{
  assert( d <= eb && d >= es);
  if (filter.find(d) != filter.end())
  {
    xbest = filter[d].first;
    fcbest = filter[d].second;
    return true;
  }
  return false;
}
