// $Id: APPSPACK_ConveyorList.hpp,v 1.4 2008/05/02 00:57:22 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src-conveyor/APPSPACK_ConveyorList.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_ConveyorList.hpp
  \brief Class definition for APPSPACK::ConveyorList
*/
#ifndef APPSPACK_CONVEYORLIST_HPP
#define APPSPACK_CONVEYORLIST_HPP

#include "APPSPACK_Point.hpp"

namespace APPSPACK
{

/*!  
  \brief  Manages a list of Points to communicate with APPSPACK::Conveyor.
  
  
  In contrast to APPSPACK::List, the primary purpose of this class to
  support communication with the Conveyor and hence does not provide a
  mechanism for popping the best point from the list.  Rather all
  operations are dedicated to maintaining (or intentionally
  manipulating) the order of the list.  Note that in
  APPSPACK::List::best(), APPSPACK::List::popBest(), the order of the
  list may change as the best point is moved to the front of the list.
  This class provides for the ability to perform "masked" deep copies
  of a list of of points, flagging certain points as being parentless.
  This provide a mechanism for signaling to APPSPACK::Iterator which
  points were externally generated and which points the Iterator itself
  generated.
*/
class ConveyorList
{
public:

  //! Constructor 
  ConveyorList();

  //! Constructor.  Performs deep copy of all points in source. 
  /*! All points not listed in taglist are created with parentTag of -1. */
  ConveyorList(const ConveyorList& source, const vector<int>& taglist);

  //! Destructor: deletes the contents of the current list 
  ~ConveyorList();

  //! Return true of the list is empty 
  bool isEmpty() const;

  //! Return true of the list is \b not empty 
  bool isNotEmpty() const;

  //! Returns iterator to the first element of stored list 
  list<Point*>::const_iterator begin() const;  

  //! The end() function returns an iterator just past the end of the list.
  list<Point*>::const_iterator end() const;

  //! Returns reverse iterator to the last element of stored list 
  list<Point*>::const_reverse_iterator rbegin() const;  

  //! The end() function returns reverse iterator just past the beginning of the list.
  list<Point*>::const_reverse_iterator rend() const;

  //! Return size 
  int size() const;

  //! All points not listed in taglist are created with parentTag of -1. 
  void copyFrom(const ConveyorList& source, const vector<int>& taglist);

  //! Returns list of tags of all points in list
  void getTagList(vector<int>& tagList) const;

  //! Prune the list except for the most recently added n points.    
  /*!  The most recently added points are at the front of the list. */
  void prune(int n = 0);

  //! Pop the next Point from the list.
  /*! If the list is empty, returns NULL. Otherwise, pops a Point off
    the \e end of the list. Ownership of the pointer is passed on to
    the calling object.
  */
  Point* pop();
  
  //! Push the given Point onto the list.
  /*! The Point is pushed onto the \e front of the list. The list takes
    ownership for the given pointer. 
  */
  void push(Point* trialPointPtr);

  //! Pop the Point with the given tag.
  /*! If the Point with the given tag is not in the list, returns
    NULL. Otherwise, it pops the Point with the given tag off the
    list. Ownership of the pointer is passed on to the calling object.
  */
  Point* pop(int tag);

  //! Insert the source list into this list and empty the source list.
  /*!
    Insert the given source list at the \e front of this list. This
    list takes ownership of all the pointers in the source list, and
    the source list is emptied. 
  */
  void prependList(ConveyorList& source);
   
  //!  Insert the source list into this list and empty the source list.
  /*!
    Appends the given source list to the \e end of this list. This
    list takes ownership of all the pointers in the source list, and
    the source list is emptied. 
  */
  void appendList(ConveyorList& source);

  //! Returns true if w equals a point in this list.
  /*!  \param w (input) Method returns true if a point z is found in
    this last that is \e cache \e equivalent to w.  \param tag
    (output) Is either assigned the value of the corresponding
    \e cache \e equivalent point z in this list, or -1.  
    
    \note Two points are considered \e cache \e equivalent if the
    satisfy the same criteria for equality used for
    APPSPACK::Cache::Point.
    
  \see pointsEqual().
  */
  bool contains(const Point *w, int& tag) const;

  //! Compares points for equality within a given tolerance.
  /*!    Let w denote the vector corresponding to this object and
    z denote the vector corresponding to the incoming
    Point. Then this operator returns false if
    \f[
    \frac{1}{s_i}\left|{w_i} - {z_i}\right| > \tau \mbox{ for any } i = 1,\dots,n.
    \f]
    
    Here \f$s_i\f$ denotes the i-th entry of Cache::Point::scaling and \f$\tau\f$
    represents the Cache::Point::tolerance.
    
    Otherwise, returns true. 

    \note that this test is essentially the same test made by
    APPSPACK::Cache::Point comparison operators.
  */
  bool pointsEqual(const Point *w, const Point *z) const;

  //! Prints contents of list 
  void print(const string label) const;
    
private:

  //! A list of pointers to trial points
  typedef list<Point*> TPL;

  //! List of points.
  TPL tpl;
};


}
#endif
