// $Id: APPSPACK_Solver.cpp,v 1.4 2008/05/03 16:13:25 jgriffi Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src-appspack/APPSPACK_Solver.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Solver.cpp
  \brief Implementation of APPSPACK::Solver
*/

#include "APPSPACK_Solver.hpp"
#include "APPSPACK_Print.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_Float.hpp"
#include "APPSPACK_LAPACK_Wrappers.hpp"
#include <algorithm>

APPSPACK::Solver::Solver(const Parameter::List& params_in,
			 Executor::Interface& executor_in,
			 const Constraints::Linear& constraints_in,
			 Combiner::Generic& combiner_in,
			 Cache::Manager& cache_in) :
  params(params_in),
  conveyor(params, constraints_in.getScaling(), executor_in, cache_in), // modifies params
  gss_iterator(params, constraints_in, combiner_in),
  maxQueue(params.getParameter("Max Queue Size", 0)),
  isMaxEvaluations(false),
  state(Continue)
{
  if (params.isParameter("Maximum Evaluations"))
  {
    isMaxEvaluations = true;
    maxEvaluations = params.getParameter("Maximum Evaluations", 0);
  }
}

APPSPACK::Solver::Solver(const Parameter::List& params_in,
			 Executor::Interface& executor_in,
			 const Constraints::Linear& constraints_in) :
  params(params_in),
  conveyor(params, constraints_in.getScaling(), executor_in), // modifies params
  gss_iterator(params, constraints_in),
  maxQueue(params.getParameter("Max Queue Size", 0)),
  isMaxEvaluations(false),
  state(Continue)
{
  if (params.isParameter("Maximum Evaluations"))
  {
    isMaxEvaluations = true;
    maxEvaluations = params.getParameter("Maximum Evaluations", 0);
  }
}

APPSPACK::Solver::~Solver()
{
  queueList.prune();
  exchangeList.prune();
}

const vector<double>& APPSPACK::Solver::getBestX() const
{
  return gss_iterator.getBestX();
}

double APPSPACK::Solver::getBestF() const
{
  return gss_iterator.getBestF(); 
}

const vector<double>& APPSPACK::Solver::getBestVecF() const
{
  return gss_iterator.getBestVecF(); 
}

int APPSPACK::Solver::getNumEvaluations() const
{
  return conveyor.getCounter().getNumEvaluations();
}

int APPSPACK::Solver::getNumCached() const
{
  return conveyor.getCounter().getNumCached();
}

APPSPACK::Solver::State APPSPACK::Solver::solve()
{
  // Print initial information according to current debug setting.
  if (Print::doPrint(Print::InitialData))
    gss_iterator.printInitializationInformation();
  
  while (state == Continue)
  {    
    // Perform a single APPSPACK GSS iteration.
    bool doPrune = gss_iterator.pointExchange(exchangeList);
    state = getIteratorState();

    // Print diagnostic information.
    if (Print::doPrint(Print::Directions))
      gss_iterator.printDirections("Directions After Trial Point Generation");
    
    if (Print::doPrint(Print::UnevaluatedPoints))
      exchangeList.print("Trial Points Before Conveyor");
    
    if (doPrune)
    {
      // New best point has been found.  In this case we prune
      // according to parameter maxQueue.
      prune();

      // Print diagnostic information.
      if (Print::doPrint(Print::NewBestPoint))	
	gss_iterator.printBestPoint("New Min");
      if (Print::doPrint(Print::NewBestDirections))
	gss_iterator.printDirections("New Directions");
    }
    
    // Add the new points to the queue.  Has the side effect of emptying
    // the exchangeList.
    queueList.insertList(exchangeList);
    
    // Add points to conveyor queue lists.
    ConveyorList Q;
    queueList.copyTo(Q);
    queueList.prune();
    ConveyorList R;
    exchangeList.copyTo(R);
    exchangeList.prune();

    // Get newly evaluated points.    
    conveyor.exchange(Q, R);

    queueList.copyFrom(Q);
    Q.prune();
    exchangeList.copyFrom(R);
    R.prune();

    // Print dagnostic information.
    if (Print::doPrint(Print::EvaluatedPoints))
      exchangeList.print("Trial Points After Conveyor");
  }
  
  // Print final information once solve has completed.
  printFinalInformation();

  // Returns the state of the solver.
  return state;
}

void APPSPACK::Solver::printInitializationInformation() const
{
  /*
  cout << "\n"
       << "-----------------------------------------------------\n"
       << "APPSPACK: Asynchronous Parallel Pattern Search\n"
       << "Written by T. G. Kolda et al., Sandia National Labs\n"
       << "For more information visit \n"
       << "http://software.sandia.gov/appspack\n"
       << "-----------------------------------------------------\n"
       << "\n";
  */
  
  if (Print::doPrint(Print::InitialData))
  {      
    cout << "\n";
    cout << "###########################################" << "\n";
    cout << "###   APPSPACK Initialization Results   ###" << endl;
    cout << "\n";
  
    gss_iterator.printInitializationInformation();
    
    cout << "\n" << "*** Conveyor ***" << "\n";
    conveyor.print();
    
    cout << "\n";
    cout << "### End APPSPACK Initialization Results ###" << endl;
    cout << "###########################################" << "\n";
  }

  if (Print::doPrint(Print::NewBestPoint))	
    gss_iterator.printBestPoint("New Min");
  if (Print::doPrint(Print::NewBestDirections))
    gss_iterator.printDirections("New Directions");
}

void APPSPACK::Solver::printFinalInformation() const
{
  // Print the final solution
  if (Print::doPrint(Print::FinalSolution))
  {
    cout << "\nFinal State: " << state << "\n";
    gss_iterator.printBestPoint("Final Min");
    conveyor.getCounter().print();
    gss_iterator.printDirections("Final Directions");
  }
  
  // Print solution file if specified by user.
  string solfile;
  solfile = params.getParameter("Solution File", solfile);
  if (!solfile.empty())
    gss_iterator.writeSolutionFile(solfile);
  
  // Print the final solution
  if (Print::doPrint(Print::FinalSolution))
  {  
    /*
    cout << "\n"
	 << "-----------------------------------------------------\n"
	 << "Thank you!\n"
	 << "APPSPACK: Asynchronous Parallel Pattern Search\n"
	 << "Written by T. G. Kolda et al., Sandia National Labs\n"
	 << "For more information visit \n"
	 << "http://software.sandia.gov/appspack\n"
	 << "-----------------------------------------------------\n"
	 << "\n";
    */
  }
}

void APPSPACK::Solver::prune()
{
  queueList.prune(maxQueue);
}

APPSPACK::Solver::State APPSPACK::Solver::getIteratorState()
{
  // Check for number of evaluations
  if ((state == Continue) && (isMaxEvaluations) 
      && (conveyor.getCounter().getNumEvaluations() >= maxEvaluations))
    return EvaluationsExhausted; 

  switch (gss_iterator.getState())
  {
  case APPSPACK::Iterator::Continue:
    return APPSPACK::Solver::Continue;
    break;
  case APPSPACK::Iterator::StepConverged:
    return APPSPACK::Solver::StepConverged;
    break;
  case APPSPACK::Iterator::FunctionConverged:
    return APPSPACK::Solver::FunctionConverged;
    break;
  }  
}

ostream& operator<<(ostream& stream, APPSPACK::Solver::State state) 
{
  switch (state)
  {
  case APPSPACK::Solver::Continue:
    stream << "Continue";
    break;
  case APPSPACK::Solver::StepConverged:
    stream << "Step Converged";
    break;
  case APPSPACK::Solver::FunctionConverged:
    stream << "Function Converged";
    break;
  case APPSPACK::Solver::EvaluationsExhausted:
    stream << "Evaluations Exhausted";
    break;
  }

  return stream;
}
