/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

//
// RandomMOO.cpp
//
// Randomized MOO
//

#include <acro_config.h>
#include <utilib/_math.h>
#include <utilib/stl_auxillary.h>
#include <coliny/Factory.h>
#include <coliny/RandomMOO.h>

using namespace std;

namespace coliny {

RandomMOO::RandomMOO()
 : batch_evaluator_t(&problem)
{
opt_name="RandomMoo";
}


void RandomMOO::reset()
{
if (!problem) return;

colin::StdOptSolver<BasicArray<double>,response_t>::reset();

if (!rng)
   EXCEPTION_MNGR(runtime_error,"RandomMOO::reset - undefined random number generator");

urnd.generator(&rng);
nrnd.generator(&rng);
batch_evaluator_t::reset();
colin::AppResponseAnalysis::initialize(problem.numNonlinearIneqConstraints(),constraint_tolerance);
}


void RandomMOO::minimize()
{
//
// Misc initialization of the optimizer
//
opt_init();
if (opt_response().point.size() == 0) {
   opt_response().termination_info = "No-Real-Params";
   return;
   }
if (!(problem.has_all_bounds())) {
   opt_response().termination_info = "Missing-Bound-Constraints";
   return;
   }
//
// Get bounds
//
BasicArray<double> x_lower,x_upper;
if (problem.enforcing_bounds())
   problem.get_real_bounds(x_lower,x_upper);

//
//
//
unsigned int num_iters;
if (max_iters <= 0)
   num_iters = MAXINT;
else
   num_iters = curr_iter + max_iters;
//
// This is a hack to make the default number of iterations be small.
// In general, this will not be necessary...
//
num_iters = std::min(static_cast<unsigned int>(10),num_iters);

NumArray<double> tmp_pt(problem.num_real_params());
real constraint_value=0.0;
//best().value() = real :: positive_infinity;
response_t tmp_response;
BasicArray<colin::real> values;

debug_io(ucout);
for (curr_iter++; curr_iter <= num_iters;  curr_iter++) {
  //
  // Determine if the algorithm is finished
  //
  if (check_convergence())
     break;
  //
  // Generate a random point
  //
  for (unsigned int i=0; i<tmp_pt.size(); i++)
    tmp_pt[i] = urnd(x_lower[i],x_upper[i]);
  problem.EvalF(tmp_pt,values);
  //perform_evaluation(tmp_pt, tmp_response, values, constraint_value);
  bool inserted = cache.insert(tmp_pt,values);
  DEBUGPR(100,
  ucout << "Trying New Point" << endl;
  ucout << tmp_pt << " / " << values << endl;
  ucout << cache;
  ucout << "Inserted: " << inserted << endl;
  );
  //
  // Perform random perturbation of a random cached point
  //
  do {
     tmp_pt << cache.get_random(urnd);
     for (unsigned int i=0; i<tmp_pt.size(); i++)
       tmp_pt[i] += nrnd();
     } while (!problem.test_feasibility(tmp_pt));
  problem.EvalF(tmp_pt,values);
  //perform_evaluation(tmp_pt, tmp_response, values, constraint_value);
  inserted = cache.insert(tmp_pt,values);
  DEBUGPR(100,
  ucout << "Trying New Point" << endl;
  ucout << tmp_pt << " / " << values << endl;
  ucout << cache;
  ucout << "Inserted: " << inserted << endl;
  );
  //
  // Debugging IO
  // 
  debug_io(ucout);
  }

debug_io(ucout,true);

ucout << "Multi-Objective Cache (Point/Values)" << endl;
ucout << "------------------------------------" << endl;
ucout << cache;
ucout << utilib::Flush;
}


} // namespace coliny

typedef colin::OptSolver<utilib::BasicArray<double>,colin::                     AppResponse_Utilib> base_t;
FACTORY_REGISTER(rmoo, base_t*, return new coliny::RandomMOO();, "An alias to RandomMOO")
FACTORY_REGISTER(RandomMOO, base_t*, return new coliny::RandomMOO();, "A random multi-objective optimizer")

