/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2007, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

/**
 * \file MultiStatePS.h
 *
 * Defines the coliny::MultiStatePS class.
 */

#ifndef coliny_MultiStatePS_h
#define coliny_MultiStatePS_h

#include <acro_config.h>
#include <utilib/default_rng.h>
#include <utilib/Normal.h>
#include <colin/StdOptSolver.h>
#include <colin/AsyncEvaluator.h>
#include <colin/ResponseSet.h>

#define DEBUG_MULTISTATE_PS 0

namespace coliny {

using colin::real;
using utilib::BasicArray;
using utilib::NumArray;


/** An implemention of a "stateless" pattern seatch algorithm that
 *  leverages the colin::AsyncEvaluator class.
 *
 *  NOTE: Currently, the MultiStatePS has NO termination criteria.  It
 *  is STRONGLY RECOMMENDED that you set max_nevals.
 */
class MultiStatePS :
    public colin::StdOptSolver<
        BasicArray<double>, 
        colin::AppResponse_Utilib>, 
    protected colin::AsyncEvaluator<
        colin::OptProblem<BasicArray<double>, colin::AppResponse_Utilib>,
        BasicArray<double> >
  {
  public:
    typedef BasicArray<double>                 domain_t;
    typedef colin::AppResponse_Utilib          response_t;
    typedef colin::ResponseSet<domain_t, response_t>  responseSet_t;

    typedef colin::AsyncEvaluator<
      colin::OptProblem<BasicArray<double>, MultiStatePS::response_t>, 
      MultiStatePS::domain_t
    > async_evaluator_t;

    ///
    MultiStatePS();

    ///
    virtual void reset();

    ///
    virtual void minimize();

    ///
    virtual void write(std::ostream& os) const;

  protected:

    ///
    virtual void initialize_best_point()
      {
      if (( best().point.size() > 0 ) &&
          ( best().point.size() != problem.num_real_params() ))
        {
        EXCEPTION_MNGR
            ( std::runtime_error, "initialize_best_point - user-provided "
              "best point has length " << best().point.size() << 
              " but the problem size is " << problem.num_real_params() << 
              std::endl );
        }

      best().point.resize(problem.num_real_params());
      }

    double m_contraction_factor;
    double m_delta_init;
    double m_delta_thresh;
    double m_expansion_factor;
    double m_sufficient_decrease_coef;
    int    m_max_success;
    BasicArray<double> m_sigma;


  private:
    struct PSState {
      PSState()
        : improving_found(false),
          improving_count(0),
          pending(0),
          id(nextStateID())
        {}

      bool   improving_found;
      int    improving_count;
      int    pending;
      long   id;
      double step;
      responseSet_t  base;
      std::list<responseSet_t>  children;
      };

    /// The master list of all pattern search states
    std::list<PSState> m_states;
    /// The list of all pending function evaluations
    std::map< async_evaluator_t::solverID_t, 
              std::map< async_evaluator_t::evalID_t, 
                        std::list<std::list<PSState>::iterator> > > m_pending;
    /// My little home-built response cache
    std::map<double, std::list<responseSet_t> > m_cache;


    static long nextStateID()
      {
      static long lastStateID = 0;
      return ++lastStateID;
      }
    /// Return a response set for the given state (point)
    responseSet_t checkCache(domain_t &point);
    /// Explore around a given point and return the new pattern state
    virtual std::list<PSState>::iterator explore( responseSet_t  newCenter, 
                                                  double         step, 
                                                  bool           improving );
  };

} // namespace coliny

#endif // coliny_MultiStatePS_h
