/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

/**
 * \file APPS.h
 *
 * Defines the coliny::APPS class.
 */

#ifndef coliny_APPS_h
#define coliny_APPS_h

#include <acro_config.h>
#ifdef ACRO_USING_APPSPACK

namespace APPSPACK {
namespace Parameter {
class List;
}
}
 
#include <utilib/std_headers.h>
#include <colin/colin.h>
#include <colin/AppResponseAnalysis.h>


namespace coliny {

class APPS;

using colin::real;

//! Namespace for the Feval Manager used by AppsOptimizer
namespace ColinyAPPS {

class TaskStatus
{
public:

  TaskStatus() : free(true), fval(0.0), failed(false), id(-1), tag(-1),
			response(0) {}

  ~TaskStatus()
	{if (response) delete response;}

  bool free;

  real fval;

  bool failed;

  int id;

  int tag;

  //! COLIN response object used for constrained problems
  colin::AppResponse<>* response;
};


class FevalMgr;

} // namespace ColinyAPPS




/** COLIN wrapper class for asynchronous parallel pattern search class.

TODO: describe APPS and cite papers from Kolda, Hough and Torczon.
*/

class APPS : virtual public colin::OptSolver<std::vector<double> >, 
					protected colin::AppResponseAnalysis
{

  friend class ColinyAPPS::FevalMgr;

public:

  /// Constructor
  APPS();

  /// Destructor
  virtual ~APPS();

  /// Setup APPS for optimization.  Call immediately before \ref{minimize}
  void reset();

  /// Perform minimization
  void minimize();

  /// Write out APPS parameter information
  void write(std::ostream& os) const;

protected:

  ///
  void initialize_best_point()
	{
	if ((best().point.size() > 0) &&
	    (best().point.size() != problem.num_real_params()))
	   EXCEPTION_MNGR(std::runtime_error, "initialize_best_point - user-provided best point has length " << best().point.size() << " but the problem size is " << problem.num_real_params() << std::endl);
        best().point.resize(problem.num_real_params());
	}

  ///
  bool auto_rescale;

  ///
  double step_tolerance;

  ///
  double alpha;

  ///
  std::string pattern;

  /// 
  int num_search_directions;

  /// 
  int num_active_directions;

  ///
  int profile;

  ///
  bool inc_flag;

  ///
  double contraction_factor;

  ///
  double initial_step;

  ///
  double min_step_allowed;

  ///
  double max_step_allowed;

  ///
  int arg_count;

  ///
  std::vector<std::string> args;

  ///
  char** args_;

  ///
  ColinyAPPS::FevalMgr* feval_mgr;

  ///
  APPSPACK::Parameter::List* params;

  ///
  bool synch_flag;

  ///
  std::string batch_str;
};

} // namespace coliny 
#endif

#endif
