/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

//
// OptSolver test
//
// Illustrates what needs to be done to derive an optimizer.
// Illustrates how an optimizer can be applied to a function with a 
//		different domain.
//

#include <acro_config.h>
#include <utilib/stl_auxillary.h>
#include <colin/colin.h>

using namespace colin;
using namespace std;


//
// Define a generic datatype that contains real parameters
//
class A {
public:
  vector<double> vec;

  operator const vector<double>&() const
	{return vec;}
};

//
// Define necessary operator functions
//
ostream& operator<<(ostream& os, const A& a)
{ os << a.vec; return os; }

A& operator&=(A& a1, A& a2)
{ a1.vec &= a2.vec; return a1; }

istream& operator>>(istream& is, A& a)
{ is >> a.vec; return is; }

namespace colin {

template <>
void map_domain(A& a1, const utilib::pvector<double>& a2)
{ a1.vec &= a2; }
}

SetOptDomainTrait(A,reals,true);
SetOptDomainTrait(A,boundconstraints,true);


//
// Define an objective function over A.
//
double func(A& point)
{
double val;
for (unsigned int i=0; i<point.vec.size(); i++)
  val = (i+1)*point.vec[i];
return val;
}



//
// Define an optimizer over vector<double>'s
//
class TestOptSolver : public OptSolver<utilib::pvector<double> >
{
public:

  void minimize();

  void set_initial_point(utilib::pvector<double>& pt)
		{curr = pt;}

protected:

  void initialize_best_point() {best().point.resize(problem.num_real_params());}

  utilib::pvector<double> curr;
};


void TestOptSolver::minimize()
{
real tmp;
problem.EvalF(curr, tmp);
best().value() = tmp;

int j=0;
while (best().value() > accuracy) {
  for (unsigned int i=0; i<curr.size(); i++)
    curr[i] -= 0.0001;
  problem.EvalF(curr,tmp);
  best().value() = tmp;
  cout << j << " " << best().value() << " " << curr << endl;
  j++;
  }
}



int main()
{
//
// Define the OptProblem
//
OptProblem<A > prob;
OptSetup(prob,func);
//
// We can set real bounds, because we've define the DomainTraits
//
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
utilib::pvector<real> lbounds(3), ubounds(3);
#else
BasicArray<real> lbounds(3), ubounds(3);
#endif
lbounds << static_cast<real>(-1.0);
ubounds << static_cast<real>(1.0);
prob.set_real_bounds(lbounds,ubounds);
//
// Define an OptSolver<vector<double> > optimizer
//
TestOptSolver opt;
//
// Set the initial point (which is not put in the OptProblem!)
//
utilib::pvector<double> point(4);
point << 10.0;
opt.set_initial_point(point);
//
// Set the OptProblem for the optimizer
//
opt.set_problem(prob);
//
// Set other options and reset
//
opt.set_parameter("accuracy",38.0);
opt.reset();
//
// Perform minimization
//
opt.minimize();

return 0;
}
