/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptSolverBase.h
 *
 * Defines the colin::OptSolverBase class.
 */

#ifndef colin_OptSolverBase_h
#define colin_OptSolverBase_h

#include <acro_config.h>
#include <utilib/ParameterList.h>
#include <utilib/AnyRNG.h>
#include <utilib/ParameterSet.h>
#include <utilib/CommonIO.h>
#include <utilib/seconds.h>
#include <utilib/BasicArray.h>
#include <colin/real.h>
#include <colin/ColinGlobals.h>
#ifdef ACRO_HAVE_MPI
#include <mpi.h>
#endif

namespace colin {

using utilib::AnyRNG;
using utilib::CommonIO;


/**
 * Domain-independent base class for optimization solvers
 */
class OptSolverBase : public virtual utilib::ParameterSet, public virtual utilib::CommonIO
{
public:

  /// Constructor
  OptSolverBase();

  /// Virtual destructor
  ~OptSolverBase() {}

  /// Sets the random number generator
  template <class RNGT>
  void set_rng(RNGT* rng_)
	{
	if (rng_ && (rng != rng_)) {
	   rng = rng_;
           }
	}

  /// Prepares the optimizer for minimization
  virtual void reset();

  /// Perform minimization.
  virtual void minimize() = 0;

  /// The value of the current iteration
  virtual unsigned int current_iter() const
		{return curr_iter;}

  /// Returns the number of evaluations
  virtual int neval() const = 0;

  /// Print information about the optimizer
  virtual void write(std::ostream& output) const {}

  ///
  virtual void print_stats(std::ostream& output) const = 0;

  /// Read information about the optimizer
  virtual void read(std::istream& )
		{ EXCEPTION_MNGR(std::runtime_error, "OptSolver::read - undefined."); }

#ifdef ACRO_HAVE_MPI
  #ifdef UTILIB_MPI_COMM_IS_POINTER
  /// MPI communicator
  void* mpicomm;
  #else
  /// MPI communicator
  MPI_Comm mpicomm;
  #endif
#endif

  /// Set sub-solver parameters.
  virtual void set_subsolver_parameters(utilib::BasicArray<utilib::ParameterList>& plist) = 0;

  /// Write parameter values to a stream
  virtual void write_parameter_values(std::ostream& os, const char* opt_label="") const = 0;

  /// Pack parameter values
  virtual void write_parameter_values(utilib::PackBuffer& os) const = 0;

  /// Unpack parameter values
  virtual void read_parameter_values(utilib::UnPackBuffer& is) = 0;

  /// Read parameter values from a stream
  virtual void read_parameter_values(std::istream& is, const std::string& terminator) = 0;

  /// Read parameter values from a stream
  virtual void read_parameter_values(std::istream& is) = 0;

  /// Write parameter info.
  void help_parameters(std::ostream& os) const
	{
	write_parameters(os);
	write_parameter_values(os);
	}

protected:

  /**@name General Information */
  //@{
  /** A routine used to setup the optimizer each time it gets called.
    * The \c reset method is designed to initialize an optimizer
    * before optimization has begun.  The \c opt_init method
    * prepares an optimizer each time it is called.  This enables
    * an optimizer to be run multiple times, in an interupted 
    * fashion.
    */
  virtual void opt_init();

  /// The name of the solver
  std::string opt_name;

  /// The random number generator.
  AnyRNG rng;
  //@}


  /**@name Termination Controls */
  //@{
  /// Initialize the timing routines.
  /// By default, this relies on the UTILIB timing routines.
  virtual void InitializeTiming()
		{::InitializeTiming();}

  /** Return the number of CPU seconds since a given point in time.
    * By default, this relies on the UTILIB timing routines.
    */
  virtual double CPUSeconds()
		{return ::CPUSeconds();}

  /// The current time
  double time_curr;

  /// The time optimization was begin
  double time_start;

  /// The average time for each evaluation
  double time_eval;

  /// The current iteration
  unsigned int curr_iter;

  /// The number of function evaluations at the start of \c minimize
  int neval_start;
  //@}


  /**@name Debugging Controls */
  //@{
  /// The output level (for user output)
  std::string output_levelstr;

  /// The output level for header
  std::string output_headerstr;

  /// The output level (for user output)
  int output_level;

  /// If true, then flush output
  bool Output_flush;

  /// If true, then this is the final output
  bool output_final;

  /// If true, then dynamically control output.
  /// Only iterations with new solution values are
  /// printed.
  bool output_dynamic;

  /// The frequency of output.
  int Output_freq;

  ///
  virtual void virt_debug_io(std::ostream& /*os*/, 
			const bool /*finishing*/,
                        const int /*output_level*/) {}
  //@}


  /**@name Control Options */
  //@{
  /** The maximum number of iterations allowed before termination.
    * If the optimizer is run multiple times, this provides a limit on the
    * total amount of work that is allowed.
    */
  unsigned int max_iters;

  /// The maximum number of function evaluations allowed.
  int max_neval;

  /** The maximum number of function evaluations allowed in the current 
    * execution of the optimizer.
    */
  int max_neval_curr;

  /// The maximum number of CPU seconds allowed before termination.
  double max_time;

  /// The minimum function value allowed before termination.
  double accuracy;

  /// The function tolerance required by the algorithm.
  double ftol;

  /// Tolerance used to determine if a constraint is violated
  double constraint_tolerance;

  ///
  int print_precision;
  //@}

};



//============================================================================
//
//
inline OptSolverBase :: OptSolverBase()
{
debug = 0;
Output_freq=1;
Output_flush = false;
output_dynamic=false;
output_final=false;
output_level = 2;
max_neval=0;
max_neval_curr=0;
accuracy=-MAXDOUBLE;
ftol=0.0;
max_iters=0;
max_time=0.0; 
constraint_tolerance=1e-8;
output_levelstr = "normal";
output_headerstr = "normal";
#ifdef ACRO_HAVE_MPI
#ifdef UTILIB_MPI_COMM_IS_POINTER
mpicomm=0;
#else
mpicomm=MPI_COMM_WORLD;
#endif
#endif

curr_iter=0;

/// Termination parameters

create_categorized_parameter("max_iterations",max_iters,
	"<unsigned int>","0 (no limit)",
	"Maximum # of iterations before termination",
	"Termination");
alias_parameter("max_iterations","max_iters");

create_categorized_parameter("max_func_evaluations",max_neval,
	"<int>","0 (no limit)",
	"Max # of fevals allowed before termination (total ever allowed)",
	"Termination");
alias_parameter("max_func_evaluations","max_neval");
alias_parameter("max_func_evaluations","max_function_evaluations");

create_categorized_parameter("max_func_evaluations_this_trial",max_neval_curr,
	"<int>","0 (no limit)",
	"Max # of fevals allowed in this minimization trial",
	"Termination");
alias_parameter("max_func_evaluations_this_trial","max_neval_curr");
alias_parameter("max_func_evaluations_this_trial","max_function_evaluations_this_trial");

create_categorized_parameter("max_time",max_time,
	"<double>","0.0 (no limit)",
	"Max time allowed before termination",
	"Termination");

create_categorized_parameter("min_function_value",accuracy,
	"<double>","-MAXDOUBLE",
	"Minimum solution value allowed before termination",
	"Termination");
alias_parameter("min_function_value","accuracy");

create_categorized_parameter("function_value_tolerance",ftol,
	"<double>","0.0",
	"A function tolerance used for termination",
	"Termination");
alias_parameter("function_value_tolerance","ftol");

/// General parameters

create_parameter("constraint_tolerance",constraint_tolerance,
	"<double>","1e-8",
	"Tolerance used to determine whether a constraint is violated");

print_precision=10;
create_parameter("precision",print_precision,
	"<int>","10",
	"Used to control the precision of I/O");

/// Debugging parameters

create_categorized_parameter("debug",debug,
	"<int>","0",
	"General debugging level (>0)",
	"Debugging");

create_categorized_parameter("output_header",output_headerstr,
	"<string>","normal",
	"Output level for the solver header:\n"
"\t  none    - No header output\n"
"\t  normal  - Simple header output\n"
"\t  verbose - Simple header output plus parameter information",
	"Debugging"
	);

create_categorized_parameter("output_level",output_levelstr,
	"<string>","normal",
	"Output level for debugging:\n"
"\t  none    - No debugging output\n"
"\t  normal  - Normal I/O\n"
"\t  summary - Summary debugging information\n"
"\t  verbose - All debugging information",
	"Debugging");

create_categorized_parameter("output_frequency",Output_freq,
	"<int>","1",
	"The frequency with which output is printed.",
	"Debugging");
alias_parameter("output_frequency","output_freq");

create_categorized_parameter("output_dynamic",output_dynamic,
	"<bool>","false",
	"If true, then only print output when an improving value is found",
	"Debugging");

create_categorized_parameter("output_final",output_final,
	"<bool>","false",
	"If true, then only print output when finished with optimization",
	"Debugging");

create_categorized_parameter("output_flush",Output_flush,
	"<bool>","false",
	"If true, then flush after every time debugging IO is generated",
	"Debugging");
}


//============================================================================
//
//
inline void OptSolverBase :: reset()
{
if (Output_freq < 0)
   Output_freq = 0;

ucout.precision(print_precision);
std::cout.precision(print_precision);

curr_iter=0;

if (output_levelstr == "none")
   output_level = 0;
else if (output_levelstr == "summary")
   output_level = 1;
else if (output_levelstr == "normal")
   output_level = 2;
else if (output_levelstr == "verbose")
   output_level = 3;
else
   {
   EXCEPTION_MNGR(std::runtime_error, "OptSolver::reset - unknown output level \"" << output_levelstr << "\"\n\t\tOptions: none, summary, normal, verbose");
   }

if ((output_dynamic) && (output_level > 1)) {
   std::cerr << "Can only use dynamic output with output level=\"summary\"\n\tReseting output level to \"summary\"";
   output_level = 1;
   }
}


//============================================================================
//
//
inline void OptSolverBase ::opt_init()
{
InitializeTiming();
time_start = CPUSeconds();
neval_start = neval();

if ((output_headerstr == "normal") || (output_headerstr == "verbose")) {
   ucout << std::endl;
ucout << "*****************************************************************************" << std::endl;
ucout << "*****************************************************************************" << std::endl;
   ucout << "****** Coliny Solver: " << opt_name << std::endl;
ucout << "*****************************************************************************" << std::endl;
ucout << "*****************************************************************************" << std::endl;
   }

if (output_headerstr == "verbose") {
   ucout << std::endl << "Solver Parameters:" << std::endl;
   write_parameters(ucout);
   ucout << std::endl << "Solver Parameter Values:" << std::endl;
   write_parameter_values(ucout);
   }
//
//
//
#if !(defined(COUGAR) || defined(TFLOPS))
ucout << utilib::Flush;
#endif
}

}

#endif
