/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptSolver.h
 *
 * Defines the colin::OptSolver class.
 */

#ifndef colin_OptSolver_h
#define colin_OptSolver_h

#include <acro_config.h>
#include <colin/GenericOptSolver.h>
#include <colin/OptSolverHandle.h>
#include <colin/OptSolverBase.h>
#include <colin/OptSolverWrapperDerived.h>

namespace colin {

using utilib::BasicArray;

/**
 * The abstract base class for all optimization classes.
 * Upon termination from the \c minimization() method, the optimizer
 * contains a copy of the best solution, which can be retrieved with a
 * call to the \c best_point method.  The stream
 * operators for IO are defined to use the \c  write and \c read
 * methods, so subclasses of colin::OptSolver do not need to define these
 * stream operators.  The header file for colin::BaseOptimizer also does a few
 * things that make development of the optimization classes easier.
 * 
 * CommonIO.h is included to allow for easy control of all of the IO in optimization classes.
 * 
 */
template <class DomainT, class ResponseT=AppResponse<> >
class OptSolver : public OptSolverBase, public GenericOptSolver<DomainT,ResponseT>
{
public:

  /**@name General Information */
  //@{
  /// Constructor
  OptSolver() {}

  /// Virtual destructor
  ~OptSolver()
		{
		for (unsigned int i=0; i<colin_solver.size(); i++) {
		  delete colin_solver[i];
		  delete generic_solver[i];
		  }
		}

  ///
  virtual OptProblem<DomainT,ResponseT> & get_problem()
		{return problem;}

  ///
  virtual const OptProblem<DomainT,ResponseT> & get_problem() const {return problem;}

  ///
  template <class LDomainT, class LResponseT>
  void set_problem( OptProblem<LDomainT,LResponseT> & problem_) 
		{problem &= problem_;}

  /// Sets the initial point
  virtual void set_initial_point(DomainT& pt)
		{ GenericOptSolver<DomainT,ResponseT>::set_initial_point(pt); }

  ///
  template <class ParamT>
  void set_initial_point(ParamT& pt)
		{
		map_domain(this->initial_point,pt);
		this->initial_point_flag=true;
		}

  ///
  template <class LDomainT, class LResponseT>
  int set_solver(OptSolver<LDomainT,LResponseT>* solver, bool own=false)
		{
		if (generic_solver.size() > colin_solver.size()) {
		   EXCEPTION_MNGR(std::runtime_error, "OptSolver::set_solver - colin::OptSolver objects cannot be inserted after colin::GenericOptSolver objects when setting up sub_solvers!");
		   }
		int id = generic_solver.size();
		solver->set_parameter("output_level","none");
		solver->set_parameter("output_header","none");
		OptSolverWrapper<DomainT,ResponseT>* tmp = new OptSolverWrapperDerived<DomainT,ResponseT,LDomainT,LResponseT>(solver,own);
		colin_solver.push_back(tmp);
		generic_solver.push_back( new OptSolverHandle<DomainT,ResponseT>(tmp,false));
		return id;
		}

  ///
  template <class LDomainT, class LResponseT>
  int set_solver(GenericOptSolver<LDomainT,LResponseT>* solver)
		{
		int id = generic_solver.size();
		generic_solver.push_back(solver);
		return id;
		}

  /// The set of solvers that can be called by this optimizer
  BasicArray<OptSolverWrapper<DomainT,ResponseT>*> colin_solver;
  
  /// The set of generic solvers that can be called by this optimizer
  /// Every colin_solver has a corresponding generic_solver, but not visa-versa
  BasicArray<GenericOptSolver<DomainT,ResponseT>*> generic_solver;
	
  /// A convenience function, that refers to the information about the 'best' point, as well as
  /// the status of the model.
  virtual OptResponse<DomainT,ResponseT>& best()
                {return this->opt_response();}

  /// A convenience function, that refers to the information about the 'best' point, as well as
  /// the status of the model.
  virtual const OptResponse<DomainT,ResponseT>& best() const
                {return this->opt_response();}

  /* THE REMAINING PUBLIC METHODS ARE DEFINED IN OptSolverBase */

  ///
  void reset()
		{
		best().reset();
		for (unsigned int i=0; i<colin_solver.size(); i++)
		  colin_solver[i]->base().set_rng(&rng);
		OptSolverBase :: reset();
		GenericOptSolver<DomainT,ResponseT> :: reset();
		if (problem) {
		   this->opt_response().response.resize(problem.numObjectives(),
				problem.numNonlinearConstraints(),
				problem.num_real_params());
		  for (unsigned int i=0; i<colin_solver.size(); i++) {
		     colin_solver[i]->set_problem(problem);
		     colin_solver[i]->base().reset();
		     }
		   }
		}

  ///
  int neval() const
		{return problem ? problem.neval() : 0;}

  ///
  void write(std::ostream& output) const;

  ///
  virtual void print_stats(std::ostream& output) const
		{
		GenericOptSolver<DomainT,ResponseT>::print_stats(output);
		ucout << "Final-Point: " << best().point << std::endl;
     		ucout << "Final-Stats: ";
     		ucout.precision(12);
     		ucout << " neval " << this->neval();
     		ucout << " constraint_violation ";
     		ucout.precision(12);
     		ucout << best().constraint_violation << std::endl;
		ucout << "Termination: " << best().termination_info << std::endl;
		//ucout << "ModelStatus: " << best().model_status << std::endl;
		//ucout << "SolverStatus: " << best().solver_status << std::endl;
		}

  //@}

  /// Set parameters for sub solvers
  void set_subsolver_parameters(BasicArray<utilib::ParameterList>& plist)
		{
		if (plist.size() > colin_solver.size()) {
		   EXCEPTION_MNGR(std::runtime_error, "Bad number of sub_solver parameters: Expected " << colin_solver.size() << " but got " << plist.size() << " solver parameter sets");
		   }
		for (unsigned int i=0; i<plist.size(); i++)
		  colin_solver[i]->base().set_parameters(plist[i],false);
		}

  /// Write parameter values to a stream
  void write_parameter_values(std::ostream& os, const char* opt_label="") const
		{
		ParameterSet::write_parameter_values(os,opt_label);
		for (unsigned int i=0; i<colin_solver.size(); i++) {
		  os << "END-SOLVER" << std::endl;
		  os << "# " << i << std::endl;
		  colin_solver[i]->base().write_parameter_values(os);
           	  }
		}

  /// Pack parameter values
  virtual void write_parameter_values(utilib::PackBuffer& os) const
		{ParameterSet::write_parameter_values(os);}

  /// Unpack parameter values
  void read_parameter_values(utilib::UnPackBuffer& is)
		{ParameterSet::read_parameter_values(is);}

  /// Read parameter values from a stream
  void read_parameter_values(std::istream& is, const std::string& terminator)
		{
		if (colin_solver.size() > 0) {
		   ParameterSet::read_parameter_values(is,"END-SOLVER");
		   if (colin_solver.size() > 1) {
		      for (unsigned int i=0; i<colin_solver.size()-1; i++)
		        colin_solver[i]->base().read_parameter_values(is,"END-SOLVER");
		      }
		   colin_solver[colin_solver.size()-1]->base().read_parameter_values(is,terminator);
		   }
		else
		   ParameterSet::read_parameter_values(is,terminator);
		}

  /// Read parameter values from a stream
  void read_parameter_values(std::istream& is)
                {read_parameter_values(is,"");}

  ///
  virtual void get_final_points(std::vector<DomainT>& points)
        {
        points.resize(1);
        points[0] = this->best().point;
        }

protected:

  /// Initialize the optimizer state at the beginning of optimization
  void opt_init();

  /// The problem that this solver optimizes
  OptProblem<DomainT,ResponseT> problem;

  /// Perform convergence checks with the \c OptSolver termination controls
  virtual bool check_convergence(real& fret);

};





//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptSolver<DomainT,ResponseT> ::write(std::ostream& os) const
{
os << "##------------------------------------------------------------------------------" << std::endl;
os << "##------------------------------------------------------------------------------" << std::endl;
os << "##" << std::endl;
os << "## OptSolver Information: " << opt_name << std::endl;
os << "##" << std::endl;
os << "##------------------------------------------------------------------------------" << std::endl;
os << "##------------------------------------------------------------------------------" << std::endl;

os << std::endl;
if (problem)
   problem.write(os);
else {
   os << "#" << std::endl;
   os << "# Empty OptProblem" << std::endl;
   os << "#" << std::endl;
   }
os << std::endl;

os << "##------------------------------------------------------------------------------" << std::endl;
os << "## OptSolver Parameters" << std::endl;
os << "##------------------------------------------------------------------------------" << std::endl;
os << "##" << std::endl;
os << "## Base OptSolver" << std::endl;
os << "##" << std::endl;
 
os << "#" << std::endl;
os << "# Optimizer Status Info" << std::endl;
os << "#" << std::endl;
os << "curr_iter      " << curr_iter << std::endl;
os << "rng ";
if (rng)
   os << rng;
else
   os << "-- NULL\n";

os << "#" << std::endl;
os << "# Generic Termination Controls" << std::endl;
os << "#" << std::endl;
os << "max_iters      " << max_iters << std::endl;
os << "max_neval_curr " << max_neval_curr << std::endl;
os << "max_neval      " << max_neval << std::endl;
os << "max_time       " << max_time << std::endl;
os << "accuracy       " << accuracy << std::endl;
os << "ftol           " << ftol << std::endl;
os << "debug           " << debug << std::endl;

os << "#" << std::endl;
os << "# Generic I/O controls" << std::endl;
os << "#" << std::endl;
os << "Output_level     ";
switch (output_level) {
  case 0:	os << "none" << std::endl; break;
  case 1:	os << "summary" << std::endl; break;
  case 2:	os << "normal" << std::endl; break;
  case 3:	os << "verbose" << std::endl; break;
  };
os << "Output_flush     " << Output_flush << std::endl;
if (output_dynamic)
   os << "Output      dynamic" << std::endl;
else if (output_final)
   os << "Output      final" << std::endl;
else {
   os << "Output      freq=" << Output_freq << std::endl;
   os << "debug            " << debug << std::endl;
   }
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
void OptSolver<DomainT,ResponseT> ::opt_init()
{
if (!problem)
   EXCEPTION_MNGR(std::runtime_error, "OptSolver::opt_init - no problem specified");
OptSolverBase::opt_init();
}


//============================================================================
//
//
template <class DomainT, class ResponseT>
bool OptSolver<DomainT,ResponseT> :: check_convergence(real& fret)
{
time_curr = CPUSeconds();

if ((max_time > 0.0) && ((time_curr-time_start) >= max_time)) {
   best().termination_info = "Time-Limit";
   return true;
   }

if ((max_iters > 0) && (curr_iter > max_iters)) {
   std::stringstream tmp;
   tmp << "Max-Num-Iterations (" << curr_iter << ">" << max_iters << ")";
   best().termination_info = tmp.str();
   return true;
   }

if (fret <= accuracy) {
   std::stringstream tmp;
   tmp << "Accuracy (" << fret << "<=" << accuracy << ")";
   best().termination_info = tmp.str();
   return true;
   }

if ((max_neval > 0) && (neval() >= max_neval)) {
   std::stringstream tmp;
   tmp << "Max-Num-Evals (" << max_neval << "<=" << neval() << ")";
   best().termination_info = tmp.str();
   return true;
   }

if ( (max_neval_curr > 0) && ((neval() - neval_start) >= max_neval_curr)) {
   std::stringstream tmp;
   tmp << "Max-Num-Evals-Curr (" << max_neval_curr << "<=" << (neval() -neval_start) << ")";
   best().termination_info = tmp.str();
   return true;
   } 

return false;
}

} // namespace colin

//============================================================================
//
//
/// Write a colin::OptSolver object.
template <class DomainT, class ResponseT>
std::ostream& operator<<(std::ostream& output, const colin:: OptSolver<DomainT,ResponseT> & opt_class)
{ opt_class.write(output); return(output); }

//============================================================================
//
//
/// Read a colin::OptSolver object.
template <class DomainT, class ResponseT>
std::istream& operator>>(std::istream& input, colin:: OptSolver<DomainT,ResponseT> & opt_class)
{ opt_class.read(input); return(input); }


#endif
