/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file AppResponse.h
 *
 * Defines the colin::AppResponse class.
 */

#ifndef colin_AppResponse_h
#define colin_AppResponse_h

#include <acro_config.h>
#include <utilib/std_headers.h>
#include <utilib/stl_auxillary.h>
#include <utilib/SmartPtr.h>
#include <utilib/traits.h>
#include <utilib/PackObject.h>
#include <utilib/CachedAllocator.h>
#include <colin/AppResponseTypes.h>
#include <colin/AppResponseLValues.h>

namespace colin {

/**
 * \class AppResponse
 *
 * This contains the response of a colin::Application calculation.  By default, 
 * this class stores this data in an AppResponseValues class.  However, this
 * user can pass in objects that are populated with the response values.
 */
template <class TypesT=AppResponseTypes<> >
class AppResponse : public utilib::CachedAllocatorObject< AppResponse<TypesT> >, public utilib::PackObject
{
public:

  #if !defined(DOXYGEN)
  typedef TypesT types_t;
  typedef typename TypesT::vector_t vector_t;
  typedef typename TypesT::matrix_t matrix_t;
  typedef typename TypesT::realarray_t realarray_t;
  typedef typename TypesT::intarray_t intarray_t;
  typedef typename TypesT::vectorarray_t vectorarray_t;
  typedef typename TypesT::matrixarray_t matrixarray_t;
  #endif

  /// Constructor
  #if !defined(ACRO_SOLARIS_CC)
  explicit
  #endif
  AppResponse(int num_functions=1, int num_constraints=0, 
				int num_real_params=0,
				bool using_gradients=false,
				bool using_hessians=false);

  /// Constructor
  AppResponse(AppResponseInfo* info_, 
			AppResponseValues<TypesT>* values_);

  /// Copy constructor
  AppResponse(const AppResponse<TypesT>& response)
	{ *this = response; }

  ///
  virtual ~AppResponse() {}

  /// Copy operator=
  AppResponse<TypesT>& operator=(const AppResponse<TypesT>& response)
	{
	allocate_exec();
	resize(response.info->num_functions,
		response.info->num_constraints,
		response.info->num_real_params,
		response.info->using_gradients,
		response.info->using_hessians);
	reset();
        copy(response);
	return *this;
	}

  /// Copy one response from another
  void copy(const AppResponse<TypesT>& response)
  	{
	*info << *(response.info);
	*values << *(response.values);
  	}

  /// Reset the state of the response object with the provided
  /// information and value data structures.
  template <class InfoT>
  void reset(InfoT& info_, AppResponseValues<TypesT>* values_)
	{
	info = info_;
	values = values_;
	values->resize(info->num_functions,info->num_constraints,
		info->num_real_params,
		info->using_gradients, info->using_hessians);
	}

  /// Resize the local data structures.
  void resize(unsigned int num_functions_, unsigned int num_constraints_=0, 
				unsigned int num_real_params=0,
				bool using_gradients=false,
				bool using_hessians=false);

  /// Set a response buffer for a single function value
  void response_bufferF(real* value);

  /// Set a response buffer for an array of function values
  void response_bufferF(realarray_t* value);

  /// Set a response buffer for a single gradient value
  void response_bufferG(vector_t* value);

  /// Set a response buffer for an array of gradient values
  void response_bufferG(vectorarray_t* value);

  /// Set a response buffer for a single Hessian
  void response_bufferH(matrix_t* value);

  /// Set a response buffer for an array of Hessians
  void response_bufferH(matrixarray_t* value);

  /// Set a response buffer for an array of constraint values
  void response_bufferCF(vector_t* value);

  /// Set a response buffer for an array of constraint gradients
  void response_bufferCG(vectorarray_t* value);

  /// Set a response buffer for an array of constrain Hessians
  void response_bufferCH(matrixarray_t* value);

  /// Return the ASV request vector
  std::vector<int>& request_vector()
		{ return info->requestASV; }

  /// Get the response (active-set) vector.
  std::vector<int>& response_vector()
		{ return info->responseASV; }

  /// Get summary statistics about the ASV
  void get_request_summary(bool& f_flag, bool& cf_flag) const
	{
	bool g_flag, h_flag, cg_flag, ch_flag;
	g_flag = h_flag = cg_flag = ch_flag=false;
	f_flag = cf_flag = true;
	info->get_requests(f_flag,g_flag,h_flag,cf_flag,cg_flag,ch_flag);
	}

  /// Get summary statistics about the ASV
  void get_request_summary(bool& f_flag, bool& g_flag, 
					bool& cf_flag, bool& cg_flag) const
	{
	bool h_flag, ch_flag;
	h_flag = ch_flag=false;
	f_flag = cf_flag = g_flag = cg_flag = true;
	info->get_requests(f_flag,g_flag,h_flag,cf_flag,cg_flag,ch_flag);
	}

  /// Get summary statistics about the ASV
  void get_request_summary(bool& f_flag, bool& g_flag, bool& h_flag,
			bool& cf_flag, bool& cg_flag, bool& ch_flag) const
	{
	f_flag = cf_flag = g_flag = cg_flag = h_flag = ch_flag = true;
	info->get_requests(f_flag,g_flag,h_flag,cf_flag,cg_flag,ch_flag);
	}

  /// Get the first function value
  void augmented_function_value(real& val) const
		{
		if (values->aug_functionValuePtr) 
		   val = *(values->aug_functionValuePtr);
		else
		   val = (*(values->aug_functionValuesPtr))[0];
		}

  /// Return a function value
  real augmented_function_value(int i=0) const
		{
		if (values->aug_functionValuePtr) {
		   if (i==0)
		      return *(values->aug_functionValuePtr);
		   else
		      EXCEPTION_MNGR(std::runtime_error, "AppResponse::aug_function_value - i<>0 but using a buffered function value pointer");
		   }
		return (*(values->aug_functionValuesPtr))[i];
		}
  
  /// Return a function value
  real& augmented_function_value(int i=0)
		{
		if (values->aug_functionValuePtr) {
		   if (i==0)
		      return *(values->aug_functionValuePtr);
		   else
		      EXCEPTION_MNGR(std::runtime_error, "AppResponse::aug_function_value - i<>0 but using a buffered function value pointer");
		   }
		return (*(values->aug_functionValuesPtr))[i];
		}
  
  /// Fill an array with the function values
  void augmented_function_values(realarray_t& vals) const
		{
		vals.resize(info->num_functions);
		if (values->aug_functionValuePtr) 
		   vals[0] = *(values->aug_functionValuePtr);
		else
		   vals = *(values->aug_functionValuesPtr);
		}

  /// Returns the value of the constraint violation
  real& l2_constraint_violation() 
		{ return values->l2_cviolation; }
  
  /// Returns the value of the constraint violation
  real l2_constraint_violation() const
		{ return values->l2_cviolation; }
  
  /// Get the first function value
  void function_value(real& val) const
		{
		if (values->functionValuePtr) 
		   val = *(values->functionValuePtr);
		else
		   val = (*(values->functionValuesPtr))[0];
		}

  /// Return a function value
  real function_value(int i=0) const
		{
		if (values->functionValuePtr) {
		   if (i==0)
		      return *(values->functionValuePtr);
		   else
		      EXCEPTION_MNGR(std::runtime_error, "AppResponse::function_value - i<>0 but using a buffered function value pointer");
		   }
		return (*(values->functionValuesPtr))[i];
		}
  
  /// Return a function value
  real& function_value(int i=0)
		{
		if (values->functionValuePtr) {
		   if (i==0)
		      return *(values->functionValuePtr);
		   else
		      EXCEPTION_MNGR(std::runtime_error, "AppResponse::function_value - i<>0 but using a buffered function value pointer");
		   }
		return (*(values->functionValuesPtr))[i];
		}
  
  /// Fill an array with the function values
  void function_values(realarray_t& vals) const
		{
		vals.resize(info->num_functions);
		if (values->functionValuePtr) 
		   vals[0] = *(values->functionValuePtr);
		else
		   vals = *(values->functionValuesPtr);
		}
  
  /// Get the first gradient value
  void function_gradient(vector_t& val) const
		{
		if (values->functionGradientPtr)
		   val << *(values->functionGradientPtr);
		else
		   val << (*(values->functionGradientsPtr))[0];
		}

  /// Return a reference to the i-th gradient value
  const vector_t& function_gradient(unsigned int i=0) const
		{
		if (values->functionGradientPtr) {
		   if (i==0)
		      return *(values->functionGradientPtr);
		   else
		      EXCEPTION_MNGR(std::runtime_error, "AppResponse::gradient_value - i<>0 but using a buffered gradientfunction value pointer");
		   }
		return (*(values->functionGradientsPtr))[i];
		}

  /// Return a reference to the i-th gradient value
  vector_t& function_gradient(unsigned int i=0)
		{
		if (values->functionGradientPtr) {
		   if (i==0)
		      return *(values->functionGradientPtr);
		   else
		      EXCEPTION_MNGR(std::runtime_error, "AppResponse::gradient_value - i<>0 but using a buffered gradientfunction value pointer");
		   }
		return (*(values->functionGradientsPtr))[i];
		}

  /// Fill an array with the gradient values
  void function_gradients(vectorarray_t & grads) const
		{
		grads.resize(this->num_functions);
		if (values->functionGradientPtr)
		   grads[0] << *(values->functionGradientPtr);
		else
		   grads << *(values->functionGradients);
		}

  /// Get the first hessian value
  void function_hessian(matrix_t& val) const
		{
		if (values->functionGradientPtr)
		   val << *(values->functionGradientPtr);
		else
		   val << (*(values->functionGradientsPtr))[0];
		}

  /// Return a reference to the i-th hessian value
  const matrix_t& function_hessian(unsigned int i=0) const
		{
		if (values->functionHessianPtr) {
		   if (i==0)
		      return *(values->functionHessianPtr);
		   else
		      EXCEPTION_MNGR(std::runtime_error, "AppResponse::hessian - i<>0 but using a buffered hessian value pointer");
		   }
		return (*(values->functionHessiansPtr))[i];
		}

  /// Return a reference to the i-th hessian value
  matrix_t& function_hessian(unsigned int i=0)
		{
		if (values->functionHessianPtr) {
		   if (i==0)
		      return *(values->functionHessianPtr);
		   else
		      EXCEPTION_MNGR(std::runtime_error, "AppResponse::hessian - i<>0 but using a buffered hessian value pointer");
		   }
		return (*(values->functionHessiansPtr))[i];
		}

  /// Fill an array with the hessian values
  void function_hessian(matrixarray_t & hessians) const
		{
		hessians.resize(this->num_functions);
		if (values->functionHessianPtr)
		   hessians[0] << *(values->functionHessianPtr);
		else
		   hessians << *(values->functionHessiansPtr);
		}

  /// Get the constraint values
  void constraint_values(vector_t& vals) const
		{vals << *(values->constraintValuesPtr);}

  /// Get the constraint values
  vector_t& constraint_values()
		{return *(values->constraintValuesPtr);}
 
  /// Get the constraint values
  const vector_t& constraint_values() const
		{return *(values->constraintValuesPtr);}
 
  /// Get the constraint gradients (aka the Jacobian)
  void constraint_gradients(matrix_t& grads) const
		{grads << *(values->constraintGradientsPtr);}

  /// Get the constraint gradients (aka the Jacobian)
  matrix_t& constraint_gradients()
		{return *(values->constraintGradientsPtr);}

  /// Get the constraint gradients (aka the Jacobian)
  const matrix_t& constraint_gradients() const
		{return *(values->constraintGradientsPtr);}

  /// Get the constraint Hessians
  void constraint_hessians(matrixarray_t & hessians) const
		{hessians = *(values->constraintHessiansPtr);}

  /// Get the constraint Hessians
  matrixarray_t & constraint_hessians()
		{return *(values->constraintHessiansPtr);}

  /// Get the constraint Hessians
  const matrixarray_t & constraint_hessians() const
		{return *(values->constraintHessiansPtr);}

  /// Write to an output stream
  void write(std::ostream& os) const;

  /// Read from an input stream
  void read(std::istream& is);

  /// Write to a pack buffer
  void write(utilib::PackBuffer& os) const;

  /// Read from a pack buffer
  void read(utilib::UnPackBuffer& is);

  /// Initialize the values in this response.
  void init()
		{values->init(info);}

  /// Reset the response ASV.
  void reset()
		{ info->responseASV << 0; }

  /// The number of real-valued parameters.
  unsigned int get_num_real_params() const
		{return info->num_real_params;}

  /// The number of functions
  unsigned int get_num_functions() const
		{return info->num_functions;}

  /// The number of constraints
  unsigned int get_num_constraints() const
		{return info->num_constraints;}

  /// The default storage for response values.
  utilib::SmartPtr< AppResponseValues<TypesT> > values;
    
  /// Core information for an AppResponse object
  utilib::SmartPtr<AppResponseInfo> info;

  /// Allocator used for cached evals.
  void allocate_exec()
	{
	info = utilib::CachedAllocator<AppResponseInfo>::allocate();;
	values = utilib::CachedAllocator< AppResponseValues<TypesT> >::allocate();
	}

  /// Deallocator used for cached evals.
  void deallocate_exec()
	{ values.reset(); info.reset(); }

  /// Deallocator used for cached evals.
  void deallocate_derived()
        { utilib::CachedAllocator< AppResponse<TypesT> > :: deallocate_derived(this); }

};


template <class TypesT>
AppResponse<TypesT>::AppResponse(int num_functions_, 
			int num_constraints_, int num_real_params_,
			bool using_gradients, bool using_hessians)
{
info = utilib::CachedAllocator<AppResponseInfo>::allocate();
info->using_gradients=using_gradients;
info->using_hessians=using_hessians;

values = utilib::CachedAllocator< AppResponseValues<TypesT> >::allocate();

resize(num_functions_,num_constraints_,num_real_params_, 
		using_gradients, using_hessians);
}

template <class TypesT>
AppResponse<TypesT>::AppResponse(
			AppResponseInfo* info_,
			AppResponseValues<TypesT>* values_)
{
reset(info_,values_);
}


template <class TypesT>
void AppResponse<TypesT>::resize(unsigned int num_functions_, 
				unsigned int num_constraints_,
				unsigned int num_real_params_,
				bool using_gradients,
				bool using_hessians)
{
if (!((info->num_functions == num_functions_) && 
    (info->num_constraints == num_constraints_) &&
    (info->num_real_params == num_real_params_))) {
   info->num_functions=num_functions_;
   info->num_constraints=num_constraints_;
   info->num_real_params=num_real_params_;

   info->requestASV.resize(info->num_functions+info->num_constraints);
   info->responseASV.resize(info->num_functions+info->num_constraints);
   values->resize(info->num_functions,info->num_constraints,
		   info->num_real_params, using_gradients, using_hessians);
   }

info->requestASV << 0;
info->responseASV << 0;
}


template <class TypesT>
void AppResponse<TypesT>::response_bufferF(real* val)         
{
values->functionValuePtr = val;
info->mode |= mode_f;
}


template <class TypesT>
void AppResponse<TypesT>::response_bufferF(realarray_t * val)
{
values->functionValuesPtr = val;
info->mode |= mode_f;
}


template <class TypesT>
void AppResponse<TypesT>::response_bufferG(vector_t* val)   
{
values->functionGradientPtr = val;
info->mode |= mode_g;
}


template <class TypesT>
void AppResponse<TypesT>::response_bufferG( vectorarray_t * val)   
{
values->functionGradientsPtr = val;
info->mode |= mode_g;
}


template <class TypesT>
void AppResponse<TypesT>::response_bufferH(matrix_t* val)   
{
values->functionHessianPtr = val;
info->mode |= mode_h;
}
 

template <class TypesT>
void AppResponse<TypesT>::response_bufferH( matrixarray_t * val)   
{
values->functionHessiansPtr = val;
info->mode |= mode_h;
}
 

template <class TypesT>
void AppResponse<TypesT>::response_bufferCF(vector_t* val) 
{
values->constraintValuesPtr = val;
info->mode |= mode_cf;
}


template <class TypesT>
void AppResponse<TypesT>::response_bufferCG(vectorarray_t * val) 
{
values->constraintGradientsPtr = val;
info->mode |= mode_cg;
}


template <class TypesT>
void AppResponse<TypesT>::response_bufferCH( matrixarray_t * val) 
{
values->constraintHessiansPtr = val;
info->mode |= mode_ch;
}


template <class TypesT>
void AppResponse<TypesT>::write(std::ostream& s) const
{
s << "Mode   = " << info->mode << std::endl;
s << "ID     = " << info->id << std::endl;
s << "ID_GEN = " << info->id_generate << std::endl;

//
// Print ASV information (currently printed as concatenated integer string)
//
s << "Response ASV = { ";
int asv_length  = info->responseASV.size();
for (int i=0; i<asv_length; i++)
  s << info->responseASV[i] << ' ';
s << '}' << std::endl;

s << "Request ASV= { ";
asv_length  = info->requestASV.size();
for (int i=0; i<asv_length; i++)
  s << info->requestASV[i] << ' ';
s << '}' << std::endl;
s.flush();

//
// Print the function values if present
//
if (info->mode & mode_f) {
  for (unsigned int i=0; i<info->num_functions; i++) {
    if (info->responseASV[i] & mode_f)
      s << "                     " << std::setw(17) <<  function_value(i)
	<< " f" << i << "\n";
    }
  }

//
// Print the augmented function values if present
//
if (info->mode & mode_f) {
  for (unsigned int i=0; i<info->num_functions; i++) {
    if (info->responseASV[i] & mode_f)
      s << "                     " << std::setw(17) <<  augmented_function_value(i)
	<< " f" << i << "\n";
    }
  }

//
// Print the function gradients if present
//
if (info->mode & mode_g) {
   for (unsigned int i=0; i<info->num_functions; i++) {
     if (info->responseASV[i] & mode_g) {
        s << " [";
        for (unsigned int j=0; j<info->num_real_params; j++) {
          s << ' ' << std::setw(17) <<  function_gradient(i)[j];
 	  if ((j+1)%4 == 0)
 	     s << "\n  "; // Output 4 gradient components per line
          }
        s << "  ] df" << i << "/dx\n";
	}
     }
   }

//
// Print the function Hessians if present
//
if (info->mode & mode_h) {
  for (unsigned int i=0; i<info->num_functions; i++) {
    if (info->responseASV[i] & mode_h) {
       const matrix_t& mat = function_hessian(i);
       if (info->responseASV[i] & mode_h) {
          s << "[[ ";
          for (unsigned int j=0; j<info->num_real_params; j++) {
	    for (unsigned int k=0; k<info->num_real_params; k++)
              s << std::setw(17) <<  mat[j][k] << ' ';
	    if (j!=info->num_real_params-1)
	      s << "\n   ";
            }
          s << "]] d^2f" << i << "/dx^2\n";
          }
       }
    }
  s << std::endl;
  }

//
// Print the constraint values if present
//
if (info->mode & mode_cf) {
  for (unsigned int i=0; i<info->num_constraints; i++) {
    if (info->responseASV[i+info->num_functions] & mode_f)
      s << "                     " << std::setw(17) <<  constraint_values()[i]
	<< " cf" << i << "\n";
    }
  }

//
// Print the constraint gradients if present
//
if (info->mode & mode_cg) {
   for (unsigned int i=0; i<info->num_constraints; i++) {
     if (info->responseASV[i+info->num_functions] & mode_cg) {
        s << " [";
        for (unsigned int j=0; j<info->num_real_params; j++) {
          s << ' ' << std::setw(17) <<  constraint_gradients()[i][j];
 	  if ((j+1)%4 == 0)
 	     s << "\n  "; // Output 4 gradient components per line
          }
        s << "  ] dcf" << i << "/dx\n";
	}
     }
   }

//
// Print the constraint Hessians if present
//
if (info->mode & mode_ch) {
  for (unsigned int i=0; i<info->num_constraints; i++) {
    if (info->responseASV[i+info->num_functions] & mode_ch) {
       s << "[[ ";
       for (unsigned int j=0; j<info->num_real_params; j++) {
	 for (unsigned int k=0; k<info->num_real_params; k++)
           s << std::setw(17) <<  constraint_hessians()[i][j][k] << ' ';
	 if (j!=info->num_real_params-1)
	   s << "\n   ";
         }
       s << "]] d^2cf" << i << "/dx^2\n";
       }
    }
  s << std::endl;
  }
}


template <class TypesT>
void AppResponse<TypesT>::read(std::istream& s)
{
std::string ignore_text;
char l_bracket, r_bracket; // eat white space and grab 1 character
char l_brackets[2], r_brackets[2]; // eat white space and grab 2 characters

// l_bracket = '[' = 1 character; l_brackets = "[[" =2 characters
size_t asv_length  = info->responseASV.size();

// Get fn. values as governed by ASV requests
unsigned int i=0;
for (; i<info->num_functions; i++) {
  if (info->responseASV[i] & mode_f) {
     s >> function_value(i) >> ignore_text;
     s >> augmented_function_value(i) >> ignore_text;
     }
  else if (info->requestASV[i] & mode_f) {
     s >> function_value(i) >> ignore_text;
     s >> augmented_function_value(i) >> ignore_text;
     info->requestASV[i] ^= mode_f;
     info->responseASV[i] |= mode_f;
     }
  }

for (unsigned int j=0; i<info->num_functions+info->num_constraints; i++,j++) {
  if (info->responseASV[i] & mode_f)
     s >> constraint_values()[j] >> ignore_text;
  else if (info->requestASV[i] & mode_f) {
     s >> constraint_values()[j] >> ignore_text;
     info->requestASV[i] ^= mode_f;
     info->responseASV[i] |= mode_f;
     }
  }

  // Get function gradients as governed by ASV requests
  for (size_t i=0; i<asv_length; i++) {
    if ((info->responseASV[i] & mode_g) || 
	(info->requestASV[i] & mode_g)) {
       s >> l_bracket;
       for (size_t j=0; j<info->num_functions; j++)
         s >> function_gradient(i)[j];
       s >> r_bracket;
       if (l_bracket != '[' || r_bracket != ']')
          EXCEPTION_MNGR(std::runtime_error,"AppResponse::read : Response recovery format error with function_gradient() " << i);
       if (info->requestASV[i] & mode_g) {
          info->requestASV[i] ^= mode_g;
          info->responseASV[i] |= mode_g;
          }
       }
  }

  // Get function Hessians as governed by ASV requests
  for (size_t i=0; i<asv_length; i++) {
    if ((info->responseASV[i] & mode_h) || 
	(info->requestASV[i] & mode_g)) {
      s >> l_brackets;
      for (size_t j=0; j<info->num_functions; j++)
	for (size_t k=0; k<info->num_functions; k++)
          s >> function_hessian(i)[j][k];
      s >> r_brackets;
      if ((l_brackets[0] != '[' || l_brackets[1] != '[')  ||
	  (r_brackets[0] != ']' || r_brackets[1] != ']'))
         EXCEPTION_MNGR(std::runtime_error,"AppResponse::read : Response recovery format error with functionHessian " << i);
      if (info->requestASV[i] & mode_h) {
         info->requestASV[i] ^= mode_h;
         info->responseASV[i] |= mode_h;
         }
    }
  }

values->init(info);
}


template <class TypesT>
void AppResponse<TypesT>::read(utilib::UnPackBuffer& s)
{
  // PackBuffer version
  
  //
  // Store general information
  //
  int nfunc, nconstr, nparam;
  s >> info->mode >> nfunc >> nconstr >> nparam >> info->id >> info->id_generate;
  resize(nfunc,nconstr,nparam);

  // Print ASV information (currently printed as concatenated integer string)
  s >> info->responseASV;
  s >> info->requestASV;

  if (info->mode & mode_f) {
     for (unsigned int i=0; i<info->num_functions; i++)
       s >> function_value(i);
     for (unsigned int i=0; i<info->num_functions; i++)
       s >> augmented_function_value(i);
     }
  if (info->mode & mode_g)
     for (unsigned int i=0; i<info->num_functions; i++)
       s >> function_gradient(i);
  if (info->mode & mode_h)
     for (unsigned int i=0; i<info->num_functions; i++)
       s >> function_hessian(i);

  if (info->mode & mode_cf)
     for (unsigned int i=0; i<info->num_constraints; i++)
       s >> constraint_values()[i];
  if (info->mode & mode_cg)
     for (unsigned int i=0; i<info->num_constraints; i++)
       s >> constraint_gradients()[i];
  if (info->mode & mode_ch)
     for (unsigned int i=0; i<info->num_constraints; i++)
       s >> constraint_hessians()[i];
}

template <class TypesT>
void AppResponse<TypesT>::write(utilib::PackBuffer& s) const
{
  // PackBuffer version
  
  //
  // Store general information
  //
  s << info->mode << info->num_functions << info->num_constraints << info->num_real_params << info->id << info->id_generate;

  // Print ASV information (currently printed as concatenated integer string)
  s << info->responseASV;
  s << info->requestASV;

  if (info->mode & mode_f) {
     for (unsigned int i=0; i<info->num_functions; i++)
       s << function_value(i);
     for (unsigned int i=0; i<info->num_functions; i++)
       s << augmented_function_value(i);
     }
  if (info->mode & mode_g)
     for (unsigned int i=0; i<info->num_functions; i++)
       s << function_gradient(i);
  if (info->mode & mode_h)
     for (unsigned int i=0; i<info->num_functions; i++)
       s << function_hessian(i);

  if (info->mode & mode_cf)
     for (unsigned int i=0; i<info->num_constraints; i++)
       s << constraint_values()[i];
  if (info->mode & mode_cg)
     for (unsigned int i=0; i<info->num_constraints; i++)
       s << constraint_gradients()[i];
  if (info->mode & mode_ch)
     for (unsigned int i=0; i<info->num_constraints; i++)
       s << constraint_hessians()[i];
}



} // namespace colin


/// Copy operator.
template <class TypesT>
inline colin:: AppResponse<TypesT>& operator<<( colin:: AppResponse<TypesT>& x, const  colin:: AppResponse<TypesT>& info)
{ x.copy(info); return x; }

#endif
