/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class GeneticAlgorithmEvaluator

    NOTES:

        See notes of GeneticAlgorithmEvaluator.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Tue May 20 10:41:52 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the GeneticAlgorithmEvaluator class.
 */






/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <list>
#include <limits>
#include <numeric>
#include <algorithm>
#include <utilities/include/Math.hpp>
#include <GeneticAlgorithmEvaluator.hpp>
#include <../Utilities/include/Logging.hpp>
#include <../Utilities/include/DesignGroup.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/ParameterExtractor.hpp>

#ifdef JEGA_THREADSAFE
#include <threads/include/mutex_lock.hpp>
#endif

// Not all implementations of pthreads define this constant.  So use a default
// large value if not.
#ifndef PTHREAD_THREADS_MAX
#define PTHREAD_THREADS_MAX 1024
#endif


/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA;

#ifdef JEGA_THREADSAFE
using namespace eddy::threads;
#endif

using namespace JEGA::Logging;
using namespace JEGA::Utilities;
using namespace eddy::utilities;






/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Algorithms {




/*
================================================================================
Nested Utility Class Implementations
================================================================================
*/
class GeneticAlgorithmEvaluator::ThreadSafeQueueIterator
{
    /*
    ===========================================================================
    Class Scope Typedefs
    ===========================================================================
    */
    public:

        /// A shorthand of the container type used as a job queue.
        typedef
        list<GeneticAlgorithmEvaluator::EvaluationJob>
        JobQueue;

    /*
    ===========================================================================
    Member Data Declarations
    ===========================================================================
    */
    private:

        /// The current iterator represented by this object.
        JobQueue::iterator _curr;

        /// The end of the job queue stored for the is_end method.
        const JobQueue::iterator _end;

        /// The mutex used to protect \a _curr in threadsafe mode.
        EDDY_DECLARE_MUTABLE_MUTEX(_mutex)

    /*
    ===========================================================================
    Accessors
    ===========================================================================
    */
    public:

        /// Tests to see if this iterator has reached the end of the queue.
        /**
         * \return True if this iterator has reached the end of the job queue
         *         into which it iterates and false otherwise.
         */
        inline
        bool
        is_end(
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            return this->_curr == this->_end;
        }

        /// Locks this iterator's mutex for synchronized access to it.
        /**
         * A call to this method must always be paired with a call to unlock.
         */
        inline
        void
        lock(
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            JEGA_IF_THREADSAFE(this->_mutex.lock();)
        }

        /// Unlocks this iterator's mutex to allow subsequent thread access.
        /**
         * A call to this method must always be issued after a call to lock.
         */
        inline
        void
        unlock(
            ) const
        {
            EDDY_FUNC_DEBUGSCOPE
            JEGA_IF_THREADSAFE(this->_mutex.unlock();)
        }

        /// Does pre-increment by 1 of a ThreadSafeQueueIterator
        /**
         * \return This iterator after incrementation.
         */
        inline
        ThreadSafeQueueIterator&
        operator ++(
            )
        {
            EDDY_FUNC_DEBUGSCOPE
            ++this->_curr;
            return *this;
        }

        /// Dereferences this iterator.
        /**
         * \return The EvaluationJob currently pointed to by this iterator
         *         in the job queue.
         */
        inline
        JobQueue::reference
        operator *(
            )
        {
            EDDY_FUNC_DEBUGSCOPE
            return *this->_curr;
        }

    /*
    ===========================================================================
    Structors
    ===========================================================================
    */
    public:

        /**
         * \brief Constructs a ThreadSafeQueueIterator to traverse the supplied
         *        range.
         *
         * \a begin and \a end must come from the same job queue instance.
         *
         * \param begin An iterator to the first EvaluationJob to do.
         * \param end An iterator to one past the last EvaluationJob to do.
         */
        ThreadSafeQueueIterator(
            const JobQueue::iterator& begin,
            const JobQueue::iterator& end
            ) :
                _curr(begin),
                _end(end) EDDY_COMMA_IF_THREADSAFE
                EDDY_INIT_MUTEX(_mutex, PTHREAD_MUTEX_RECURSIVE)
        {
            EDDY_FUNC_DEBUGSCOPE
        }

}; // GeneticAlgorithmEvaluator::ThreadSafeQueueIterator


bool
GeneticAlgorithmEvaluator::EvaluationJob::GetResult(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(this->_toEval.IsEvaluated())
    assert(this->_toEval.IsEvaluated());
    return this->_result;
}

GeneticAlgorithmEvaluator::EvaluationJob::EvaluationJob(
    JEGA::Utilities::Design& toEval,
    GeneticAlgorithmEvaluator& evaler,
    std::size_t evalNum
    ) :
        _toEval(toEval),
        _evaler(evaler),
        _evalNum(evalNum),
        _result(true)
{
    EDDY_FUNC_DEBUGSCOPE
}


/*
================================================================================
Static Member Data Definitions
================================================================================
*/

const std::size_t GeneticAlgorithmEvaluator::DEFAULT_MAX_EVALS(
    numeric_limits<std::size_t>::max()
    );

const std::size_t GeneticAlgorithmEvaluator::DEFAULT_EVAL_CONCUR(1);

const std::size_t GeneticAlgorithmEvaluator::MAX_EVAL_CONCUR(
    JEGA_IF_THREADSAFE(PTHREAD_THREADS_MAX) JEGA_IF_NO_THREADSAFE(1)
    );



/*
================================================================================
Mutators
================================================================================
*/
void
GeneticAlgorithmEvaluator::SetMaxEvaluations(
    std::size_t maxEvals
    )
{
    EDDY_FUNC_DEBUGSCOPE
    this->_maxEvals = maxEvals;

    JEGALOG_II(this->GetLogger(), lverbose(), this,
        ostream_entry(lverbose(),
            this->GetName() + ": Maximum allowable evaluations now = "
            ) << this->_maxEvals
        )
}

void
GeneticAlgorithmEvaluator::SetEvaluationConcurrency(
    std::size_t ec
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // If the evaluation concurrency is not changing, then do nothing.
    if(ec == this->_evalConcur) return;

    JEGAIFLOG_CF_II(ec < 1, this->GetLogger(), lquiet(), this,
        text_entry(lquiet(), this->GetName() + ": Attempt to assign the "
            "evaluation concurrency to 0.  1 is the minimum evaluation "
            "concurency.  Using the value of 1 instead.")
        )

    JEGAIFLOG_CF_II(ec > MAX_EVAL_CONCUR, this->GetLogger(), lquiet(), this,
        ostream_entry(lquiet(), this->GetName() + ": Attempt to assign the "
            "evalauation concurrency to a value of ") << ec << " which is "
            "greater than the maximum allowable value of " << MAX_EVAL_CONCUR
            << ". Using the max value."
        )

    this->_evalConcur =
        Math::Min<size_t>(Math::Max<size_t>(ec, 1), MAX_EVAL_CONCUR);

    JEGALOG_II(this->GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), this->GetName() + ": The evaluation "
            "concurrency is now ") << this->_evalConcur
        )


#ifdef JEGA_THREADSAFE
    
    // The evaluation concurrency changed.  We need to correct the number of
    // running threads.  The most sure-fire way to do this is to let them all
    // exit and start new ones.  Let all existing exit.
    size_t newConcur = this->_evalConcur;
    this->_evalConcur = 1;

    this->_jobsCond.notify_all();

    for(ThreadList::iterator it(this->_threads.begin());
        it!=this->_threads.end(); ++it) thread::join(*it);

    this->_threads.clear();

    if((this->_evalConcur = newConcur) > 1)
    {
        for(size_t i=0; i<this->_evalConcur; ++i)
            this->_threads.push_back(thread(&EvaluateMain, this));
    }

#endif
}


void
GeneticAlgorithmEvaluator::SetNumberEvaluations(
    std::size_t num
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_SCOPEDLOCK(l, this->_numEvalsMutex)
    this->_numEvals = num;
}

void
GeneticAlgorithmEvaluator::IncrementNumberEvaluations(
    std::size_t by
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_SCOPEDLOCK(l, this->_numEvalsMutex)
    this->_numEvals += by;
}








/*
================================================================================
Accessors
================================================================================
*/

std::size_t
GeneticAlgorithmEvaluator::GetNumberEvaluations(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_SCOPEDLOCK(l, this->_numEvalsMutex)
    return this->_numEvals;
}








/*
================================================================================
Public Methods
================================================================================
*/

bool
GeneticAlgorithmEvaluator::IsMaxEvalsExceeded(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_SCOPEDLOCK(l, this->_numEvalsMutex)
    return (this->_numEvals > this->_maxEvals);
}

bool
GeneticAlgorithmEvaluator::IsMaxEvalsReached(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_SCOPEDLOCK(l, this->_numEvalsMutex)
    return (this->_numEvals >= this->_maxEvals);
}








/*
================================================================================
Subclass Visible Methods
================================================================================
*/


std::size_t
GeneticAlgorithmEvaluator::ResolveClones(
    const DesignGroup& in
    )
{
    EDDY_FUNC_DEBUGSCOPE
    std::size_t ret = 0;
    for(DesignDVSortSet::const_iterator it(in.BeginDV()); it!=in.EndDV(); ++it)
        ret += ResolveClone(**it) ? 1 : 0;
    return ret;
}

bool
GeneticAlgorithmEvaluator::ResolveClone(
    Design& des
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // This method does not do any clone detection, only resolution
    // which means that it copies response values from an evaluated
    // clone if it can find one.

    // if this design is already evaluated or isn't a clone, stop.
    if(des.IsEvaluated() || !des.IsCloned()) return false;

    // this will be the pointer to a clone if one is found.
    Design* clone = 0x0;

    // look beyond "des" in the clone list for an evaluated Design
    for(clone=des.GetNextClone();
        (clone!=0x0) && !clone->IsEvaluated();
        clone = clone->GetNextClone());

    // if we didn't make it to the end, we found a clone.
    // record the values, mark "des" evaluated, and get out.
    if(clone != 0x0)
    {
        CopyResponses(*clone, des);
        des.SetEvaluated(true);
        return true;
    }

    // look before "des" in the clone list for an evaluated Design
    for(clone=des.GetPreviousClone();
        (clone!=0x0) && !clone->IsEvaluated();
        clone = clone->GetPreviousClone());

    // if we didn't make it to the end, we found a clone.
    // record the values, mark "des" evaluated, and get out.
    if(clone != 0x0)
    {
        CopyResponses(*clone, des);
        des.SetEvaluated(true);
        return true;
    }

    // if we make it here,
    // des could not be resolved this way.
    return false;
}

void
GeneticAlgorithmEvaluator::CopyResponses(
    const Design& from,
    Design& into
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(&from.GetDesignTarget() == &into.GetDesignTarget())

    const size_t nof = from.GetDesignTarget().GetNOF();
    const size_t ncn = from.GetDesignTarget().GetNCN();

    for(size_t i=0; i<nof; ++i) into.SetObjective(i, from.GetObjective(i));
    for(size_t i=0; i<ncn; ++i) into.SetConstraint(i, from.GetConstraint(i));
}

/*
================================================================================
Subclass Overridable Methods
================================================================================
*/
bool
GeneticAlgorithmEvaluator::Evaluate(
    Design& des
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // Create a job info for the current evaluation and evaluate it.
    // When this method is called, we will do the single evaluation from
    // this thread no matter what.
    EvaluationJob eJob(des, *this, this->GetNumberEvaluations());
    return this->Evaluate(eJob);
}


bool
GeneticAlgorithmEvaluator::Evaluate(
    DesignGroup& group
    )
{
    EDDY_FUNC_DEBUGSCOPE

    JEGALOG_II(this->GetLogger(), ldebug(), this,
        text_entry(ldebug(), this->GetName() + ": Group evaluator in use.")
        )

    bool ret = true;

    // first resolve clones
    JEGA_LOGGING_IF_ON(size_t nres =) this->ResolveClones(group);

    JEGAIFLOG_CF_II(nres > 0, this->GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), this->GetName() + ": able to avoid ") << nres
            << " evaluations by copying responses from cloned designs."
        )

    typedef std::list<EvaluationJob> JobQueue;

    JEGA_LOGGING_IF_ON(size_t nskipped = 0;)

    // Now create the job queue and send it off for evaluation.  This is
    // carried out no matter whether we are in threadsafe mode, have a >1
    // evaluation concurrency or not, etc.
    JobQueue evalJobQueue;

    // Also keep a tally of the evaluation numbers, max allowable, etc.
    size_t maxEvals = this->GetMaxEvaluations();
    size_t evalNum = this->GetNumberEvaluations();

    for(DesignDVSortSet::const_iterator it(group.BeginDV());
        it!=group.EndDV(); ++it)
    {
        if((*it)->IsEvaluated()) continue;

        // If we've reached our maximum allowable number of
        // evaluations, tag remaining as evaluated and illconditioned.
        // By doing so, they will be flushed from the algorithm.
        if(evalNum >= maxEvals)
        {
            JEGA_LOGGING_IF_ON(++nskipped;)
            (*it)->SetEvaluated(true);
            (*it)->SetIllconditioned(true);
            continue;
        }

        //if(!Evaluate(*(*it))) ret = false;
        evalJobQueue.push_back(EvaluationJob(**it, *this, evalNum++));
    }

    // now that we have our queue, get them evaluated.
    delete this->_qit;

    this->_qit = new ThreadSafeQueueIterator(
        evalJobQueue.begin(), evalJobQueue.end()
        );

#ifdef JEGA_THREADSAFE
    // we have our queue, now create our threads and get them started.
    const size_t ec = this->GetEvaluationConcurrency();

    if(ec > 1)
    {
        mutex_lock l(this->_resumeMutex);
        this->_jobsCond.notify_all();
        this->_resumeCond.wait(l);
    }
    else
    {
        this->NoThreadEvaluate();
    }
#else
    this->NoThreadEvaluate();
#endif

    for(JobQueue::const_iterator it(evalJobQueue.begin());
        it!=evalJobQueue.end() && ret; ++it) ret &= (*it).GetResult();

    JEGAIFLOG_CF_II(nskipped > 0, this->GetLogger(), lquiet(), this,
        ostream_entry(lquiet(), this->GetName() + ": skipped evaluation of ")
            << nskipped << " designs because the maximum number of "
            "evaluations was reached.  They were marked illconditioned."
        )

    return ret;
}

bool
GeneticAlgorithmEvaluator::PollForParameters(
    const JEGA::Utilities::ParameterDatabase& db
    )
{
    EDDY_FUNC_DEBUGSCOPE

    bool success = ParameterExtractor::GetSizeTypeFromDB(
        db, "method.max_function_evaluations", this->_maxEvals
        );

    // If we did not find the max iterations, warn about it and use the default
    // value.  Note that if !success, then maxGens is still equal to _maxGens
    JEGAIFLOG_CF_II(!success, this->GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), this->GetName() + ": The maximum allowable "
            "number of evaluations was not found in the parameter "
            "database.  Using the current value of ") << this->_maxEvals
        )

    // now go ahead and set it.
    this->SetMaxEvaluations(this->_maxEvals);

    size_t evalConcurr = this->_evalConcur;

    success = ParameterExtractor::GetSizeTypeFromDB(
        db, "method.jega.eval_concurrency", evalConcurr
        );

    JEGAIFLOG_CF_II(!success, this->GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), this->GetName() + ": The evaluation "
            "concurrency was not found in the parameter database.  Using the "
            "current value of ") << this->_evalConcur
        )

    this->SetEvaluationConcurrency(evalConcurr);

    return true;
}

string
GeneticAlgorithmEvaluator::GetType(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return "Evaluator";
}

bool
GeneticAlgorithmEvaluator::PostEvaluate(
    JEGA::Utilities::Design& des
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // need the target to get the design variable information objects.
    const DesignTarget& target = des.GetDesignTarget();

    des.SetEvaluated(true);

    if(!des.IsIllconditioned())
    {
        target.CheckFeasibility(des);
        target.RecordAllConstraintViolations(des);
    }

    this->IncrementNumberEvaluations();

    return !des.IsIllconditioned();
}


/*
================================================================================
Private Methods
================================================================================
*/

void
GeneticAlgorithmEvaluator::NoThreadEvaluate(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    for(; !this->_qit->is_end(); ++(*this->_qit))
    {
        EvaluationJob& job = **this->_qit;
        job.SetResult(job.GetEvaluator().Evaluate(job));
    }
}

void*
GeneticAlgorithmEvaluator::EvaluateMain(
    void* data
    )
{
    EDDY_FUNC_DEBUGSCOPE

    GeneticAlgorithmEvaluator* me = 
        static_cast<GeneticAlgorithmEvaluator*>(data);

    EDDY_ASSERT(me != 0x0);

    while(me->GetEvaluationConcurrency() > 1)
    {
        while(me->_qit != 0x0)
        {
            me->_qit->lock();
            if(me->_qit->is_end()) { me->_qit->unlock(); break; }
            EvaluationJob& job = **me->_qit;
            ++(*me->_qit);
            me->_qit->unlock();
            job.SetResult(job.GetEvaluator().Evaluate(job));
        }

        me->IncrementWaitCount();
        EDDY_SCOPEDLOCK(l, me->_jobsMutex);
        EDDY_IF_THREADSAFE(me->_jobsCond.wait(l);)
    }

    return data;
};

void
GeneticAlgorithmEvaluator::IncrementWaitCount(
    )
{
    EDDY_FUNC_DEBUGSCOPE

#ifdef JEGA_THREADSAFE

    this->_resumeMutex.lock();
    if((++this->_waitCt) == this->_evalConcur)
    {
        this->_waitCt = 0;
        this->_resumeMutex.unlock();
        this->_resumeCond.notify_all();
    }
    else
    {
        this->_resumeMutex.unlock();
    }

#endif
}



#define ECIT EDDY_COMMA_IF_THREADSAFE

/*
================================================================================
Structors
================================================================================
*/
GeneticAlgorithmEvaluator::GeneticAlgorithmEvaluator(
    GeneticAlgorithm& algorithm
    ) :
        GeneticAlgorithmOperator(algorithm),
        _numEvals(0),
        _maxEvals(DEFAULT_MAX_EVALS),
        _evalConcur(DEFAULT_EVAL_CONCUR),
        _qit(0x0),
        _waitCt(0) ECIT
        EDDY_INIT_MUTEX(_jobsMutex, PTHREAD_MUTEX_RECURSIVE) ECIT
        EDDY_IF_THREADSAFE(_jobsCond()) ECIT
        EDDY_INIT_MUTEX(_resumeMutex, PTHREAD_MUTEX_RECURSIVE) ECIT
        EDDY_IF_THREADSAFE(_resumeCond()) ECIT
        EDDY_INIT_MUTEX(_numEvalsMutex, PTHREAD_MUTEX_RECURSIVE)
{
    EDDY_FUNC_DEBUGSCOPE
}

GeneticAlgorithmEvaluator::GeneticAlgorithmEvaluator(
    const GeneticAlgorithmEvaluator& copy
    ) :
        GeneticAlgorithmOperator(copy),
        _numEvals(copy._numEvals),
        _maxEvals(copy._maxEvals),
        _evalConcur(copy._evalConcur),
        _qit(0x0),
        _waitCt(0) ECIT
        EDDY_INIT_MUTEX(_jobsMutex, PTHREAD_MUTEX_RECURSIVE) ECIT
        EDDY_IF_THREADSAFE(_jobsCond()) ECIT
        EDDY_INIT_MUTEX(_resumeMutex, PTHREAD_MUTEX_RECURSIVE) ECIT
        EDDY_IF_THREADSAFE(_resumeCond()) ECIT
        EDDY_INIT_MUTEX(_numEvalsMutex, PTHREAD_MUTEX_RECURSIVE)
{
    EDDY_FUNC_DEBUGSCOPE
}

GeneticAlgorithmEvaluator::GeneticAlgorithmEvaluator(
    const GeneticAlgorithmEvaluator& copy,
    GeneticAlgorithm& algorithm
    ) :
        GeneticAlgorithmOperator(copy, algorithm),
        _numEvals(copy._numEvals),
        _maxEvals(copy._maxEvals),
        _evalConcur(copy._evalConcur),
        _qit(0x0),
        _waitCt(0) ECIT
        EDDY_INIT_MUTEX(_jobsMutex, PTHREAD_MUTEX_RECURSIVE) ECIT
        EDDY_IF_THREADSAFE(_jobsCond()) ECIT
        EDDY_INIT_MUTEX(_resumeMutex, PTHREAD_MUTEX_RECURSIVE) ECIT
        EDDY_IF_THREADSAFE(_resumeCond()) ECIT
        EDDY_INIT_MUTEX(_numEvalsMutex, PTHREAD_MUTEX_RECURSIVE)
{
    EDDY_FUNC_DEBUGSCOPE
}

GeneticAlgorithmEvaluator::~GeneticAlgorithmEvaluator(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    delete this->_qit;
}


#undef ECIT

/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Algorithms
} // namespace JEGA
