/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class UnconstrainedStatistician.

    NOTES:

        See notes of UnconstrainedStatistician.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Thu May 22 08:05:53 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the UnconstrainedStatistician class.
 */




/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <cfloat>
#include <utilities/include/Math.hpp>
#include <../Utilities/include/Logging.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/DesignMultiSet.hpp>
#include <../Utilities/include/DesignValueMap.hpp>
#include <../Utilities/include/ConstraintInfo.hpp>
#include <../Utilities/include/UnconstrainedStatistician.hpp>


/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA::Logging;
using namespace eddy::utilities;






/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {






/*
================================================================================
Utility Class Definitions
================================================================================
*/
bool
UnconstrainedStatistician::Penalizer::CheckDesign(
    const Design* des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    JEGAIFLOG_CF_II_G(!des->IsEvaluated(), lquiet(), this,
        text_entry(lquiet(), "Applying penalty to non-evaluated Design.  "
            "Result may be nonsensical.")
        )

    JEGAIFLOG_CF_II_G(des->IsIllconditioned(), lquiet(), this,
        text_entry(lquiet(), "Applying penalty to illconditioned Design.  "
            "Result may be nonsensical.")
        )

    return des->IsEvaluated() && !des->IsIllconditioned();
}


double
UnconstrainedStatistician::BoundsPenalizer::operator()(
    const Design* des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    CheckDesign(des);
    const DesignVariableInfoVector& infos =
        des->GetDesignTarget().GetDesignVariableInfos();

    double pen2Add = 0.0;

    // iterate the dv infos and tack on penalty for any violation
    DesignVariableInfoVector::const_iterator it(infos.begin());
    std::size_t dv = 0;

    for(; it!=infos.end(); ++it, ++dv)
    {
        double above = (*it)->GetMaxDoubleRep() - des->GetVariableRep(dv);
        double below = des->GetVariableRep(dv) - (*it)->GetMinDoubleRep();

        double viol = (above < 0.0) ? above :
                      (below < 0.0) ? below :
                       0.0;
        pen2Add += Math::Pow(viol, 2);
    }

    return pen2Add;
}

double
UnconstrainedStatistician::ExteriorPenalizer::operator()(
    const Design* des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    CheckDesign(des);
    const ConstraintInfoVector& infos =
        des->GetDesignTarget().GetConstraintInfos();

    double penalty = 0.0;
    for(ConstraintInfoVector::const_iterator it(infos.begin());
        it!=infos.end(); ++it)
            penalty += Math::Pow((*it)->GetViolationAmount(*des), 2);

    return _multiplier * penalty;
}

UnconstrainedStatistician::ExteriorPenalizer::ExteriorPenalizer(
    double multiplier
    ) :
        _multiplier(multiplier)
{
}





/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/

DesignDoubleMap
UnconstrainedStatistician::ApplyExteriorPenalty(
    const DesignGroup& designs,
    double multiplier
    )
{
    EDDY_FUNC_DEBUGSCOPE

    return ApplyPenalty<ExteriorPenalizer>(
        designs, ExteriorPenalizer(multiplier)
        );
}

DesignDoubleMap
UnconstrainedStatistician::ApplyExteriorPenalty(
    const DesignGroupVector& designs,
    double multiplier
    )
{
    EDDY_FUNC_DEBUGSCOPE

    return ApplyPenalty<ExteriorPenalizer>(
        designs, ExteriorPenalizer(multiplier)
        );
}

DesignDoubleMap
UnconstrainedStatistician::ApplyQuadraticPenalty(
    const DesignGroup& designs,
    double multiplier
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return ApplyExteriorPenalty(designs, multiplier);
}

DesignDoubleMap
UnconstrainedStatistician::ApplyQuadraticPenalty(
    const DesignGroupVector& designs,
    double multiplier
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return ApplyExteriorPenalty(designs, multiplier);
}

const DesignDoubleMap&
UnconstrainedStatistician::AddOutOfBoundsPenalties(
    const DesignGroup& designs,
    DesignDoubleMap& to
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return AddPenalty<BoundsPenalizer>(designs, BoundsPenalizer(), to);
}

const DesignDoubleMap&
UnconstrainedStatistician::AddOutOfBoundsPenalties(
    const DesignGroupVector& designs,
    DesignDoubleMap& to
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return AddPenalty<BoundsPenalizer>(designs, BoundsPenalizer(), to);
}






/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/




/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/





/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA
