/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class ObjectiveFunctionInfo.

    NOTES:

        See notes of ObjectiveFunctionInfo.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Mon Sep 08 12:19:58 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the ObjectiveFunctionInfo class.
 */




/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <../Utilities/include/Design.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/ObjectiveFunctionInfo.hpp>
#include <../Utilities/include/ObjectiveFunctionTypeBase.hpp>
#include <../Utilities/include/MinimizeObjectiveFunctionType.hpp>


/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA::Utilities;







/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {





/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/


void
ObjectiveFunctionInfo::SetType(
    ObjectiveFunctionTypeBase* type
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(type != 0x0);

    delete _type;
    _type = (type == 0x0) ? new MinimizeObjectiveFunctionType(*this) : type;

} // ObjectiveFunctionInfo::SetType






/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/

double
ObjectiveFunctionInfo::Which(
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return des.GetObjective(GetNumber());

} // ObjectiveFunctionInfo::Which(Design)

double
ObjectiveFunctionInfo::WhichForMinimization(
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().GetValueForMinimization(Which(des));

} // ObjectiveFunctionInfo::WhichForMinimization(Design)

eddy::utilities::int8_t
ObjectiveFunctionInfo::PreferComp(
    double val1,
    double val2
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    val1 = GetType().GetValueForMinimization(val1);
    val2 = GetType().GetValueForMinimization(val2);

    return (val1 < val2 ) ? -1 : (val1 > val2 ) ? 1 : 0;
}

const Design*
ObjectiveFunctionInfo::GetPreferredDesign(
    const Design& des1,
    const Design& des2
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    eddy::utilities::int8_t pref = PreferComp(Which(des1), Which(des2));

    return (pref == -1) ? &des1 : (pref ==  1) ? &des2 : 0x0;

} // ObjectiveFunctionInfo::GetPreferredDesign(Designs)

double
ObjectiveFunctionInfo::GetPreferredAmount(
    const Design& des1,
    const Design& des2
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().GetPreferredAmount(Which(des1), Which(des2));

} // ObjectiveFunctionInfo::GetPreferredAmount(Designs)

string
ObjectiveFunctionInfo::GetTypeString(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().ToString();

} // ObjectiveFunctionInfo::GetTypeString


string
ObjectiveFunctionInfo::GetNatureString(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().GetNatureString();

} // ObjectiveFunctionInfo::GetNatureString


bool
ObjectiveFunctionInfo::SetNature(
    ObjectiveFunctionNatureBase* nature
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().SetNature(nature);

} // ObjectiveFunctionInfo::SetNature


ObjectiveFunctionNatureBase&
ObjectiveFunctionInfo::GetNature(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().GetNature();

} // ObjectiveFunctionInfo::GetNature



/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/





/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/


ObjectiveFunctionInfo::ObjectiveFunctionInfo(
    DesignTarget& target
    ) :
        InfoBase(target),
        _type(0x0)
{
    EDDY_FUNC_DEBUGSCOPE
    _type = new MinimizeObjectiveFunctionType(*this);

} // ObjectiveFunctionInfo::ObjectiveFunctionInfo

ObjectiveFunctionInfo::ObjectiveFunctionInfo(
    const ObjectiveFunctionInfo& copy,
    DesignTarget& target
    ) :
        InfoBase(copy, target),
        _type(0x0)
{
    EDDY_FUNC_DEBUGSCOPE
    _type = copy.GetType().Clone(*this);

} // ObjectiveFunctionInfo::ObjectiveFunctionInfo

ObjectiveFunctionInfo::ObjectiveFunctionInfo(
    const ObjectiveFunctionInfo& copy
    ) :
        InfoBase(copy),
        _type(0x0)
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(copy._type != 0x0);

    _type = copy.GetType().Clone(*this);

} // ObjectiveFunctionInfo::ObjectiveFunctionInfo

ObjectiveFunctionInfo::~ObjectiveFunctionInfo(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    delete _type;

} // ObjectiveFunctionInfo::~ObjectiveFunctionInfo






/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA

