/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class DesignVariableInfo.

    NOTES:

        See notes of DesignVariableInfo.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Tue Jun 03 07:34:36 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the DesignVariableInfo class.
 */




/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <../Utilities/include/Design.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/DesignVariableInfo.hpp>
#include <../Utilities/include/DesignVariableTypeBase.hpp>
#include <../Utilities/include/RealDesignVariableType.hpp>


/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;








/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {







/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/
void
DesignVariableInfo::SetType(
    DesignVariableTypeBase* type
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(type != 0x0);

    delete _type;
    _type = (type == 0x0) ? new RealDesignVariableType(*this) : type;
}







/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/


bool
DesignVariableInfo::AddDiscreteValues(
    const JEGA::DoubleVector& values
    )
{
    EDDY_FUNC_DEBUGSCOPE
    bool ret = true;

    for(JEGA::DoubleVector::const_iterator it(values.begin());
        it!=values.end(); ++it)
            ret &= AddDiscreteValue(*it);

    return ret;
}


double
DesignVariableInfo::GetRepBoundViolation(
    double rep
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return (rep < GetMinDoubleRep()) ? GetMinDoubleRep() - rep :
           (rep > GetMaxDoubleRep()) ? rep - GetMaxDoubleRep() :
           0.0;
}

double
DesignVariableInfo::GetRepBoundViolation(
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetRepBoundViolation(WhichDoubleRep(des));
}

double
DesignVariableInfo::GetValueBoundViolation(
    double val
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return (val < GetMinValue()) ? GetMinValue() - val :
           (val > GetMaxValue()) ? val - GetMaxValue() :
           0.0;
}

double
DesignVariableInfo::GetValueBoundViolation(
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetValueBoundViolation(WhichValue(des));
}

double
DesignVariableInfo::WhichValue(
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return des.GetVariableValue(GetNumber());
}

double
DesignVariableInfo::WhichDoubleRep(
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return des.GetVariableRep(GetNumber());
}

double
DesignVariableInfo::AssertPrecision(
    double val
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->AssertPrecision(val);
}

double
DesignVariableInfo::GetDefaultValue(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetDefaultValue();
}

double
DesignVariableInfo::GetMaxValue(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetMaxValue();
}

double
DesignVariableInfo::GetMinValue(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetMinValue();
}

double
DesignVariableInfo::GetValueRange(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    return GetMaxValue() - GetMinValue();
}

double
DesignVariableInfo::GetValueOf(
    double rep
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetValueOf(rep);
}

double
DesignVariableInfo::GetRandomValue(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetRandomValue();
}

double
DesignVariableInfo::GetDefaultDoubleRep(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetDefaultDoubleRep();
}

double
DesignVariableInfo::GetMaxDoubleRep(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetMaxDoubleRep();
}

double
DesignVariableInfo::GetMinDoubleRep(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetMinDoubleRep();
}

double
DesignVariableInfo::GetDoubleRepRange(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    return GetMaxDoubleRep() - GetMinDoubleRep();
}

double
DesignVariableInfo::GetNearestValidValue(
    double value
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetNearestValidValue(value);
}

double
DesignVariableInfo::GetNearestValidDoubleRep(
    double rep
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetNearestValidDoubleRep(rep);
}

double
DesignVariableInfo::GetDoubleRepOf(
    double value
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetDoubleRepOf(value);
}

double
DesignVariableInfo::GetRandomDoubleRep(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetRandomDoubleRep();
}

double
DesignVariableInfo::GetRandomDoubleRep(
    const RegionOfSpace& within
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetRandomDoubleRep(within);
}

double
DesignVariableInfo::GetDistanceBetweenDoubleReps(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetDistanceBetweenDoubleReps();
}

bool
DesignVariableInfo::IsDiscreteValueLocked(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->IsDiscreteValueLocked();
}

bool
DesignVariableInfo::IsValueInBounds(
    double value
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->IsValueInBounds(value);
}

bool
DesignVariableInfo::IsRepInBounds(
    double rep
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->IsRepInBounds(rep);
}

bool
DesignVariableInfo::IsNatureLocked(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->IsNatureLocked();
}

bool
DesignVariableInfo::IsOutOfBoundsDefined(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->IsOutOfBoundsDefined();
}

bool
DesignVariableInfo::IsPrecisionLocked(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->IsPrecisionLocked();
}

bool
DesignVariableInfo::IsValidValue(
    double value
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->IsValidValue(value);
}

bool
DesignVariableInfo::IsValidDoubleRep(
    double rep
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->IsValidDoubleRep(rep);
}

string
DesignVariableInfo::GetTypeString(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->ToString();
}

string
DesignVariableInfo::GetNatureString(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetNatureString();
}

bool
DesignVariableInfo::AddDiscreteValue(
    double value
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->AddDiscreteValue(value);
}

void
DesignVariableInfo::ClearDiscreteValues(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    _type->ClearDiscreteValues();
}

bool
DesignVariableInfo::RemoveDiscreteValue(
    double value
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->RemoveDiscreteValue(value);
}

void
DesignVariableInfo::SetMinValue(
    double value
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    _type->SetMinValue(value);
}

void
DesignVariableInfo::SetMaxValue(
    double value
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    _type->SetMaxValue(value);
}

bool
DesignVariableInfo::SetPrecision(
    eddy::utilities::int16_t prec
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->SetPrecision(prec);
}

eddy::utilities::int16_t
DesignVariableInfo::GetPrecision(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->GetPrecision();
}

bool
DesignVariableInfo::SetNature(
    DesignVariableNatureBase* nature
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->SetNature(nature);
}

DesignVariableNatureBase&
DesignVariableInfo::GetNature(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().GetNature();
}

const DesignVariableInfo&
DesignVariableInfo::operator = (
    const DesignVariableInfo &right
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(right._type != 0x0);
    EDDY_ASSERT(&GetDesignTarget() == &right.GetDesignTarget());

    if(this == &right) return *this;
    SetType(right._type->Clone(*this));
    return *this;
}

bool
DesignVariableInfo::IsDiscrete(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->IsDiscrete();
}

bool
DesignVariableInfo::IsContinuum(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(_type != 0x0);

    return _type->IsContinuum();
}




/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/








/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/
DesignVariableInfo::DesignVariableInfo(
    DesignTarget& target
    ) :
        InfoBase(target),
        _type(0x0)
{
    EDDY_FUNC_DEBUGSCOPE
    _type = new RealDesignVariableType(*this);
}

DesignVariableInfo::DesignVariableInfo(
    const DesignVariableInfo& copy,
    DesignTarget& target
    ) :
        InfoBase(copy, target),
        _type(0x0)
{
    EDDY_FUNC_DEBUGSCOPE
    _type = copy.GetType().Clone(*this);

} // DesignVariableInfo::DesignVariableInfo

DesignVariableInfo::DesignVariableInfo(
    const DesignVariableInfo& copy
    ) :
        InfoBase(copy),
        _type(0x0)
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(copy._type != 0x0);

    _type = copy._type->Clone(*this);
}

DesignVariableInfo::~DesignVariableInfo(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    delete _type;
}







/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA
