/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Inline methods of class DesignValueMap.

    NOTES:

        See notes of DesignValueMap.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        2.0.0

    CHANGES:

        Tue Dec 20 12:33:44 2005 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the inline methods of the DesignValueMap class.
 */




/*
================================================================================
Includes
================================================================================
*/
#include <utilities/include/Math.hpp>
#include <utilities/include/numeric_limits.hpp>





/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {




/*
================================================================================
Static Member Initialization
================================================================================
*/
template <typename VT>
const VT DesignValueMap<VT>::MAX_POSSIBLE =
    eddy::utilities::numeric_limits<VT>::max();

template <typename VT>
const VT DesignValueMap<VT>::MIN_POSSIBLE =
    eddy::utilities::numeric_limits<VT>::smallest();



/*
================================================================================
Inline Mutators
================================================================================
*/








/*
================================================================================
Inline Accessors
================================================================================
*/
template<typename VT>
inline
const typename DesignValueMap<VT>::value_type&
DesignValueMap<VT>::GetMaxValue(
    ) const
{
    return this->_maxValue;
}

template<typename VT>
inline
const typename DesignValueMap<VT>::value_type&
DesignValueMap<VT>::GetMinValue(
    ) const
{
    return this->_minValue;
}

template<typename VT>
inline
const typename DesignValueMap<VT>::value_type&
DesignValueMap<VT>::GetTotalValue(
    ) const
{
    return this->_totalValue;
}







/*
================================================================================
Inline Public Methods
================================================================================
*/
template<typename VT>
inline
typename DesignValueMap<VT>::value_type
DesignValueMap<VT>::GetValue(
    const Design* of
    ) const
{
    typename base_type::const_iterator it(this->find(of));
    return (it == this->end()) ?
        std::numeric_limits<value_type>::max() : (*it).second;
}

template<typename VT>
inline
typename DesignValueMap<VT>::value_type
DesignValueMap<VT>::GetValue(
    const Design& of
    ) const
{
    return this->GetValue(&of);
}

template<typename VT>
inline
bool
DesignValueMap<VT>::AddValue(
    const Design& of,
    const value_type& val
    )
{
    return this->AddValue(&of, val);
}

template<typename VT>
inline
const typename DesignValueMap<VT>::value_type&
DesignValueMap<VT>::AddToValue(
    const Design& of,
    const value_type& amt
    )
{
    return this->AddToValue(&of, amt);
}

template<typename VT>
inline
bool
DesignValueMap<VT>::RemoveValue(
    const Design& of
    )
{
    return this->RemoveValue(&of);
}


template<typename VT>
bool
DesignValueMap<VT>::AddValue(
    const Design* of,
    const value_type& val
    )
{
    if(this->insert(typename base_type::value_type(of, val)).second)
    {
        this->_minValue = eddy::utilities::Math::Min(this->_minValue, val);
        this->_maxValue = eddy::utilities::Math::Max(this->_maxValue, val);
        this->_totalValue += val;
        return true;
    }
    return false;
}

template<typename VT>
typename DesignValueMap<VT>::value_type
DesignValueMap<VT>::GetAverageValue(
    ) const
{
    return this->empty() ?
        std::numeric_limits<value_type>::max() :
        this->_totalValue/this->size();
}

template<typename VT>
const typename DesignValueMap<VT>::value_type&
DesignValueMap<VT>::AddToValue(
    const Design* of,
    const value_type& amt
    )
{
    typename base_type::iterator it(this->find(of));
    if(it == this->end())
    {
        this->AddValue(of, amt);
        return amt;
    }
    else if(amt == value_type(0))
    {
        return (*it).second;
    }
    else
    {
        bool wasMax = ((*it).second == this->_maxValue);
        bool wasMin = ((*it).second == this->_minValue);

        (*it).second += amt;
        this->_totalValue += amt;

        // If it was the max and we are adding a positive value, then we don't
        // need to update totals.
        if(wasMax && (amt > 0))
        {
            this->_maxValue = (*it).second;
        }
        // Similar logic works for the min.
        if(wasMin && (amt < 0))
        {
            this->_minValue = (*it).second;
        }
        // If it was the max or min but we reduce/increased it respectively,
        // then we don't know where the new max and min are and must do a full
        // update of the statistics.
        else if(wasMax || wasMin)
        {
            this->UpdateTotals();
        }
        // If it wasn't the max or min then we can just see if it is now and
        // we are all set.
        else
        {
            // don't need to check for empty here b/c if it were empty
            // we wouldn't be here.
            this->_minValue =
                eddy::utilities::Math::Min(this->_minValue, (*it).second);
            this->_maxValue =
                eddy::utilities::Math::Max(this->_maxValue, (*it).second);
        }
        return (*it).second;
    }
}

template<typename VT>
bool
DesignValueMap<VT>::RemoveValue(
    const Design* of
    )
{
    typename base_type::iterator it(this->find(of));
    if(it == this->end()) return false;

    double val = (*it).second;
    bool wasMax = (val == this->_maxValue);
    bool wasMin = (val == this->_minValue);
    this->erase(it);

    if(wasMax || wasMin) this->UpdateTotals();
    else this->_totalValue -= val;
    return true;
}



/*
================================================================================
Inline Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Inline Private Methods
================================================================================
*/

template <typename VT>
void
DesignValueMap<VT>::UpdateTotals(
    )
{
    this->_maxValue = MIN_POSSIBLE;
    this->_minValue = MAX_POSSIBLE;
    this->_totalValue = value_type(0);
    for(typename base_type::const_iterator it(this->begin());
        it!=this->end(); ++it)
    {
        this->_minValue =
            eddy::utilities::Math::Min(this->_minValue, (*it).second);
        this->_maxValue =
            eddy::utilities::Math::Max(this->_maxValue, (*it).second);
        this->_totalValue += (*it).second;
    }
}







/*
================================================================================
Inline Structors
================================================================================
*/


template <typename VT>
DesignValueMap<VT>::DesignValueMap(
    ) :
        base_type(),
        _minValue(MAX_POSSIBLE),
        _maxValue(MIN_POSSIBLE),
        _totalValue(value_type(0))
{
}

template <typename VT>
DesignValueMap<VT>::DesignValueMap(
    const my_type& copy
    ) :
        base_type(copy),
        _minValue(copy._minValue),
        _maxValue(copy._maxValue),
        _totalValue(copy._totalValue)
{
}

template <typename VT>
DesignValueMap<VT>::~DesignValueMap(
    )
{
}






/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA

