#!/bin/csh -f
#
# A csh script to create a platform-specific binary tar file
# Dakota_${tar_tag}_${distro_name}.tar for distribution, as well as docs if
# requested
#
# --- external arguments/variables used ---
# $1  name of source tarball (full path or local) or directory (full path)
# $2  (optional) tar_tag version for binary tarball created (e.g., 4_0, votd)
# $3  (optional) distro_name (int_full, ext_web)
# $*  (optional) remaining are make_script.csh options
#     (must specify tar_tag and distro_name if using make_script options)
#
# $UNAME      output of uname (environment.csh)
# $TEST_SQA   script directory (passed to top-level scripts)
# $BUILD_JOBS number of parallel jobs (environment.csh)
# -----------------------------------------

# check command syntax
if ( $#argv < 1 ) then
  echo "Usage:   $0 src_tar_filename version_tag disto_name make_options"
  echo " -or-    $0 src_directory    version_tag disto_name make_options"
  exit
endif

# TODO: more comprehensive gzip option
echo "Running $0 on `uname -n` at `date`"

# default structure is 
#   $TEST_DIR_FQ/Dakota
#   $TEST_DIR_FQ/bin_tars/$distro_name
if ( ${?TEST_DIR_FQ} == 0 ) then
  set TEST_DIR_FQ = $cwd
endif

# DAKOTA source tarball or build directory tree
set dakota_src = $1
shift

# version tag for binary tarball
if ( $#argv > 0 ) then
  set tar_tag = $1
  shift
else
  set tar_tag = votd
endif

# distribution scope string
if ( $#argv > 0 ) then
  set distro_name = $1
  shift
else
  set distro_name = ext_web
endif

# installation prefix
set prefix = $TEST_DIR_FQ/bin_tars/$distro_name

# grab any remaining options; default configuration is external web
if ( $#argv > 0 ) then
  set make_script_opts = "$*"
else
  set make_script_opts = "--without-nlpql --without-dot --without-npsol --prefix=$prefix"
endif


# -----------------------
# extract tar, set DAKOTA
# -----------------------

# if given a tarball: untar, configure, make, optionally run tests
# if given a directory: configure, make, optionally run tests
# always install in $prefix

# DAKOTA is needed by make_script
if ( `file $dakota_src | grep -c "tar archive"` > 0 ) then
  cd $TEST_DIR_FQ
  tar xf $dakota_src
  setenv DAKOTA $cwd/Dakota
  echo "${0}: using tarball $dakota_src"
else if ( `file $dakota_src | grep -c "gzip compressed"` > 0 ) then
  cd $TEST_DIR_FQ
  gunzip -c $dakota_src | tar xf -
  setenv DAKOTA $cwd/Dakota
  echo "${0}: using tarball $dakota_src"
else if ( -d $dakota_src ) then
  # make in directory defined by $dakota_src
  setenv DAKOTA $dakota_src
  echo "${0}: using directory $dakota_src"
else
  # perhaps default to $TEST_DIR_FQ/Dakota?
  echo "${0}: error finding sources for Dakota build; exiting."
  exit
endif

echo "${0}: building in directory $DAKOTA for binaries in $prefix"


# ------------------
# configure and make
# ------------------
cd $DAKOTA
$TEST_SQA/make_script.csh $make_script_opts

# ----------------
# check for errors
# ----------------
# archive build records up one level
foreach f ( reconfig_dakota.err config_dakota.out config_dakota.err \
            make_dakota.out make_dakota.err )
  if ( -e $DAKOTA/$f ) then
    cp -f $DAKOTA/$f $TEST_DIR_FQ/${distro_name}_$f
  endif
end
# only proceed in this script if make succeeded
if ( -e $DAKOTA/config_dakota.err || -e $DAKOTA/make_dakota.err ) then
  echo "${0}: make error creating binary tar file for $tar_tag, $distro_name"
  exit
endif

# -------------
# documentation
# -------------

# Create documentation (invoke Developers twice to update tag file cross-refs)
if ( `echo $make_script_opts | grep -c "\-\-enable-docs"` > 0 ) then
  if ( $tar_tag == 'votd' ) then 
    set doc_tag = _votd
  else
    set doc_tag = `echo $tar_tag | tr '_' '.'`
  endif
  cd docs
  doxygen Doxyfile
  doxygen Doxyfile-ref
  doxygen Doxyfile
  # to do -- get this to tar to html-dev$DOC_TAG directory
  # KH - we want docs to be file type *tgz, for now create the same
  #      as before and change tar.gz to tgz
  tar cf $TEST_DIR_FQ/bin_tars/html-dev$doc_tag.tar html-dev
  gzip -f $TEST_DIR_FQ/bin_tars/html-dev$doc_tag.tar
  mv $TEST_DIR_FQ/bin_tars/html-dev$doc_tag.tar.gz $TEST_DIR_FQ/bin_tars/html-dev$doc_tag.tgz
  tar cf $TEST_DIR_FQ/bin_tars/html-ref$doc_tag.tar html-ref
  gzip -f $TEST_DIR_FQ/bin_tars/html-ref$doc_tag.tar
  mv $TEST_DIR_FQ/bin_tars/html-ref$doc_tag.tar.gz $TEST_DIR_FQ/bin_tars/html-ref$doc_tag.tgz
  # make pdf versions of reference and developers for non-votd
  if ( $tar_tag != 'votd' ) then
    ./latex_modify.perl
    cd latex-ref
    make pdf
    cp refman.pdf $TEST_DIR_FQ/bin_tars/Reference${doc_tag}.pdf
    rm -f *.aux *.brf *.idx *.log *.out *.toc
    cd ../latex-dev
    make pdf
    cp refman.pdf $TEST_DIR_FQ/bin_tars/Developers${doc_tag}.pdf
    rm -f *.aux *.brf *.idx *.log *.out *.toc
    cd ../
  endif
  cd latex-user
  make pdf
  cp Users_Main.pdf $TEST_DIR_FQ/bin_tars/Users${doc_tag}.pdf
  rm -f *.aux *.brf *.idx *.log *.out *.toc
endif


# ------------------
# install to $prefix
# ------------------

# copy binaries, headers and libraries
cd $DAKOTA
make install

# TODO: Do not symlink binaries in Windows distributions
if ( $UNAME =~ *{CYGWIN,Cygwin,cygwin}* ) then
  echo "Copying DAKOTA's CYGWIN DLL dependencies to $prefix/bin"
  foreach f ( `cygcheck ./src/dakota.exe | grep cyg` )
    cp `cygpath $f` $prefix/bin
  end
endif 

# eventually do the following in a loop over a read file?
# excerpts from extract_src.csh
foreach f ( INSTALL \
            LICENSE \
	    COPYRIGHT \
	    README \
	    test )
  cp -rf $f $prefix
end
# handle applications separately since need subdirs
set at_pre = "examples/script_interfaces"
foreach f ( examples/linked_interfaces \
	    examples/methods \
	    examples/parallelism \
 	    examples/platforms \
            examples/README \
	    $at_pre/Abaqus \
	    $at_pre/Cask/filters \
	    $at_pre/Nastran/cylinder \
	    $at_pre/Qtran/scripts \
	    $at_pre/Salinas/aff_all_fns.C \
	    $at_pre/Salinas/dakota_aff_lhs.in \
	    $at_pre/Salinas/salinas_script_asv \
	    $at_pre/Rsmas \
            $at_pre/generic \
            examples/tutorial )
  tar cf - $f | (cd $prefix; tar xf -)
end

# removals from extract_bin.csh
cd $prefix
foreach d ( test examples )
  # TODO: this doesn't seem to work correctly on Solaris (sass9000), nor AIX
  #foreach f ( Makefile\* config\* \*.\[Cc\] \*.\[Ff\] \*.m4 \*.o )
  #  find $d -name "$f" -exec rm -f '{}' \;
  #end
  find $d -name 'Makefile*' -exec rm -f '{}' \;
  find $d -name 'config*' -exec rm -f '{}' \;
  find $d -name '*.[Cc]' -exec rm -f '{}' \;
  find $d -name '*.[Ff]' -exec rm -f '{}' \;
  find $d -name '*.m4' -exec rm -f '{}' \;
  find $d -name '*.o' -exec rm -f '{}' \;
end

# repair soft links broken by moving src to bin during make install
# TODO: use $EXEEXT for Cygwin where appropriate
set ln = /bin/ln

cd $prefix/test
rm -rf .deps .libs
rm -f plxtnd5.fnt dakota dakota_library_mode dakota_restart_util
if ( -e  ../share/plplot5.5.2/plxtnd5.fnt ) then
  $ln -s ../share/plplot5.5.2/plxtnd5.fnt plxtnd5.fnt
endif
$ln -s ../bin/dakota dakota
$ln -s ../bin/dakota_restart_util dakota_restart_util
# dakota_library_mode only built in int_full for now
if ( -e ../bin/dakota_library_mode ) then 
  $ln -s ../bin/dakota_library_mode dakota_library_mode
endif

# only need to fix dakota and dakota_restart_util below, be cautious for now
cd $prefix/examples/tutorial
rm -rf .deps .libs
rm -f dakota rosenbrock text_book
$ln -s ../../bin/dakota dakota
$ln -s ../../test/rosenbrock rosenbrock
$ln -s ../../test/text_book text_book

cd $prefix/examples/methods
rm -rf .deps .libs
rm -f dakota
$ln -s ../../bin/dakota dakota

cd $prefix/examples/script_interfaces/generic
rm -rf .deps .libs
rm -f dakota rosenbrock 
$ln -s ../../../bin/dakota dakota
$ln -s ../../../test/rosenbrock rosenbrock

# Limit read access permissions for certain binaries and libraries 
cd $prefix/bin
chmod -f o-r dakota dakota_restart_util lssol nlssol npsol
cd $prefix/lib
chmod -f o-r libdakota.a libdakota.la libdot.a libdot.la libnpsol.a libnpsol.la
 

# ----------
# create tar
# TODO: need option to not make tar (in case direct install to target)
# ----------

cd $TEST_DIR_FQ/bin_tars
# we always want the tar to contain Dakota top-level directory
rm -rf Dakota
mv $distro_name Dakota
rm -f Dakota_${tar_tag}_${distro_name}.t*
tar cf Dakota_${tar_tag}_${distro_name}.tar Dakota
gzip Dakota_${tar_tag}_${distro_name}.tar
mv Dakota $distro_name


# Don't remove the Dakota working directory in this case since we may 
# want to run some verification tests or perform another build
echo "${0}: Finished creating binary tar file Dakota_${tar_tag}_${distro_name}.tar"
