/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Description:  This file contains code related to data utilities that should
//-               be compiled, rather than inlined in data_util.h.
//-
//- Owner:        Mike Eldred
//- Version: $Id: data_util.C 5792 2009-03-30 18:22:20Z mseldre $

#include "data_types.h"
#ifdef DAKOTA_NEWMAT
#include "newmat.h"
#endif // DAKOTA_NEWMAT
#ifdef DAKOTA_DDACE
#include "DDaceSamplePoint.h"
#endif // DAKOTA_DDACE


namespace Dakota {

// ------------
// == operators
// ------------

bool operator==(const RealVector& drv1, const RealVector& drv2)
{
  // Check for equality in array lengths
  size_t len = drv1.length();
  if ( drv2.length() != len )
    return false;

  // Check each value (labels are ignored!)
  size_t i;
  for (i=0; i<len; i++) {
    const Real& val1 = drv1[i];
    const Real& val2 = drv2[i];
 
    if (val1 == 0.0) { // prevent division by 0
      if (val2 != 0.0)
        return false;
    }
    else if ( fabs(1.0 - val2/val1) > DBL_EPSILON ) // normalization needed
      // since val1 & val2 could be small (see p. 258 in Kernighan & Ritchie 
      // for defn. of DBL_EPSILON).  NOTE: a scalar factor on DBL_EPSILON could
      // be used to account for compounding roundoff, e.g. 10*DBL_EPSILON ??
      return false;
  }

  return true;
}

bool operator==(const IntVector& div1, const IntVector& div2)
{
  // Check for equality in array lengths
  size_t len = div1.length();
  if ( div2.length() != len )
    return false;

  // Check each value (labels are ignored!)
  size_t i;
  for (i=0; i<len; i++)
    if ( div2[i] != div1[i] )
      return false;

  return true;
}

bool operator==(const IntArray& dia1, const IntArray& dia2)
{
  // Check for equality in array lengths
  size_t len = dia1.length();
  if ( dia2.length() != len )
    return false;

  // Check each value
  size_t i;
  for (i=0; i<len; i++)
    if ( dia2[i] != dia1[i] )
      return false;

  return true;
}

bool operator==(const ShortArray& dsa1, const ShortArray& dsa2)
{
  // Check for equality in array lengths
  size_t len = dsa1.length();
  if ( dsa2.length() != len )
    return false;

  // Check each value
  size_t i;
  for (i=0; i<len; i++)
    if ( dsa2[i] != dsa1[i] )
      return false;

  return true;
}

bool operator==(const RealMatrix& drm1, const RealMatrix& drm2)
{
  // Check for equality in matrix dimensions
  size_t nr = drm1.num_rows();
  size_t nc = drm1.num_columns();
  if ( drm2.num_rows() != nr || drm2.num_columns() != nc )
    return false;

  // Check each value
  size_t i, j;
  for (i=0; i<nr; i++) {
    for (j=0; j<nc; j++) {
      const Real& val1 = drm1[i][j];
      const Real& val2 = drm2[i][j];
      if (val1==0.0) { // prevent division by 0
        if (val2!=0.0)
          return false;
      }
      else if ( fabs(1.0 - val2/val1) > DBL_EPSILON ) // see note above
        return false;
    }
  }
  return true;
}

bool operator==(const RealMatrixArray& drma1, const RealMatrixArray& drma2)
{
  // Check for equality in array length
  size_t len = drma1.length();
  if ( drma2.length() != len )
    return false;
  size_t i, j, k;
  for (i=0; i<len; i++) {
    
    // Check for equality in matrix dimensions
    size_t nr  = drma1[i].num_rows();
    size_t nc  = drma1[i].num_columns();
    if ( drma2[i].num_rows() != nr || drma2[i].num_columns() != nc )
      return false;

    // Check each value
    for (j=0; j<nr; j++) {
      for (k=0; k<nc; k++) {
        const Real& val1 = drma1[i][j][k];
        const Real& val2 = drma2[i][j][k];
        if (val1==0.0) { // prevent division by 0
          if (val2!=0.0)
            return false;
        }
        else if ( fabs(1.0 - val2/val1) > DBL_EPSILON ) // see note above
          return false;
      }
    }
  }
  return true;
}

bool operator==(const StringArray& dsa1, const StringArray& dsa2)
{
  // Check for equality in array lengths
  size_t len = dsa1.length();
  if ( dsa2.length() != len )
    return false;

  // Check each string
  size_t i;
  for (i=0; i<len; i++)
    if ( dsa2[i] != dsa1[i] )
      return false;

  return true;
}

bool operator==(const UIntArray& ua, UIntMultiArrayConstView umav)
{
  // Check for equality in array lengths
  size_t len = ua.length();
  if ( umav.size() != len )
    return false;

  // Check each unsigned integer
  size_t i;
  for (i=0; i<len; i++)
    if ( umav[i] != ua[i] )
      return false;

  return true;
}


// ----------------------------
// non-templated copy functions
// ----------------------------
// Note: retaining these functions in data_types.C instead of inlining them
// in data_*.h avoids collisions between NEWMAT and other headers.

#ifdef DAKOTA_NEWMAT
/// copy NEWMAT::ColumnVector to RealBaseVector
void copy_data(const NEWMAT::ColumnVector& cv, RealBaseVector& drbv)
{
  int size_cv = cv.Nrows();
  if (size_cv != drbv.length())
    drbv.reshape(size_cv);
  for (size_t i=0; i<size_cv; i++)
    drbv[i] = cv(i+1);
}

/// copy RealBaseVector to NEWMAT::ColumnVector
void copy_data(const RealBaseVector& drbv, NEWMAT::ColumnVector& cv)
{
  size_t size_drbv = drbv.length();
  if (cv.Nrows() != size_drbv)
    cv.ReSize(size_drbv);
  for (size_t i=0; i<size_drbv; i++)
    cv(i+1) = drbv[i];
}

/// copy RealArray to NEWMAT::ColumnVector
void copy_data(const RealArray& dra, NEWMAT::ColumnVector& cv)
{
  size_t size_dra = dra.length();
  if (cv.Nrows() != size_dra)
    cv.ReSize(size_dra);
  for (size_t i=0; i<size_dra; i++)
    cv(i+1) = dra[i];
}

/// copy RealMatrix to NEWMAT::SymmetricMatrix
void copy_data(const RealSymDenseMatrix& rsdm, NEWMAT::SymmetricMatrix& sm)
{
  // SymmetricMatrix = symmetric and square, but Dakota::Matrix can be general
  // (e.g., functionGradients = numFns x numVars).  Therefore, have to verify
  // sanity of the copy.  Could copy square submatrix of rsdm into sm, but 
  // aborting with an error seems better since this should only currently be
  // used for copying Hessian matrices.
  size_t nr = rsdm.numRows(), nc = rsdm.numCols();
  if (nr != nc) {
    Cerr << "Error: copy_data(const Dakota::RealSymDenseMatrix& rsdm, "
	 << "SymmetricMatrix& sm) called with nonsquare rsdm." << endl;
    abort_handler(-1);
  }
  if (sm.Nrows() != nr) // sm = symmetric & square -> only 1 dimension needed
    sm.ReSize(nr);
  for (size_t i=0; i<nr; i++)
    for (size_t j=0; j<nr; j++)
      sm(i+1,j+1) = rsdm(i,j);
}

/// copy RealMatrix to NEWMAT::Matrix
void copy_data(const RealDenseMatrix& rdm, NEWMAT::Matrix& m)
{
  // Newmat Matrix and Dakota::RealMatrix are both general rectangular matrices.
  int nr = rdm.numRows(), nc = rdm.numCols();
  if (m.Nrows() != nr || m.Ncols() != nc)
    m.ReSize(nr, nc);
  for (size_t i=0; i<nr; i++)
    for (size_t j=0; j<nc; j++)
      m(i+1,j+1) = rdm(i,j);
}

/// copy NEWMAT::ColumnVector to RealDenseVector
void copy_data(const NEWMAT::ColumnVector& cv, RealDenseVector& rdv)
{
  int size_cv = cv.Nrows();
  if (rdv.length() != size_cv)
    rdv.sizeUninitialized(size_cv);
  for (size_t i=0; i<size_cv; i++)
    rdv[i] = cv(i+1);
}

/// copy RealDenseVector to NEWMAT::ColumnVector
void copy_data(const RealDenseVector& rdv, NEWMAT::ColumnVector& cv)
{
  int size_rdv = rdv.length();
  if (cv.Nrows() != size_rdv)
    cv.ReSize(size_rdv);
  for (size_t i=0; i<size_rdv; i++)
    cv(i+1) = rdv[i];
}
#endif // DAKOTA_NEWMAT

#ifdef DAKOTA_DDACE
/// copy DDACE point to RealDenseVector
void copy_data(const DDaceSamplePoint& dsp, RealDenseVector& rdv)
{
  int j, vec_len = dsp.length();
  if (rdv.length() != vec_len)
    rdv.sizeUninitialized(vec_len);
  for (j=0; j<vec_len; j++)
    rdv[j] = dsp[j];
}

/// copy DDACE point array to RealDenseVectorArray
void copy_data(const std::vector<DDaceSamplePoint>& dspa,
	       RealDenseVectorArray& rdva)
{
  int i, num_vec = dspa.size();
  if (rdva.length() != num_vec)
    rdva.reshape(num_vec);
  for (i=0; i<num_vec; i++)
    copy_data(dspa[i], rdva[i]);
}

/// copy DDACE point array to Real*
void copy_data(const std::vector<DDaceSamplePoint>& dspa, Real* ptr,
	       const int ptr_len)
{
  int i, j, num_vec = dspa.size(), total_len = 0, cntr = 0;
  for (i=0; i<num_vec; i++)
    total_len += dspa[i].length();
  if (total_len != ptr_len) {
    Cerr << "Error: pointer allocation (" << ptr_len << ") does not equal "
	 << "total Array<DDaceSamplePoint> length (" << total_len << ") in "
	 << "copy_data(Array<DDaceSamplePoint>, Real*)." << endl;
    abort_handler(-1);
  }
  for (i=0; i<num_vec; i++) {
    int vec_len = dspa[i].length();
    for (j=0; j<vec_len; j++)
      ptr[cntr++] = dspa[i][j];
  }
}
#endif // DAKOTA_DDACE

} // namespace Dakota
