/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

#ifndef DATA_TYPES_H
#define DATA_TYPES_H

#include "DakotaString.H"
#include "DakotaList.H"
#include "DakotaArray.H"
#include "DakotaVector.H"
#include "DakotaMatrix.H"

#include "Teuchos_SerialDenseVector.hpp"
#ifndef DLSfuncs_H
#include "Teuchos_SerialDenseSolver.hpp"
#include "Teuchos_SerialSpdDenseSolver.hpp"
#endif

#ifdef HAVE_BOOST
#include "boost/multi_array.hpp"
#endif // HAVE_BOOST

namespace Dakota {

// avoid problems with circular dependencies by using fwd declarations
class Iterator;
class Model;
class Variables;
class Interface;
class Response;
class ActiveSet;
class ParamResponsePair;
class ParallelLevel;
class ParallelConfiguration;


// -----------------------------------
// Aliases for fundamental data types:
// -----------------------------------
typedef double Real;


// -----------------------------------
// Numerical arrays (serial/parallel):
// -----------------------------------
// numerical with STL: (deprecated)
typedef Vector<Real>               RealVector;
typedef Vector<int>                IntVector;
typedef BaseVector<Real>           RealBaseVector;
typedef Matrix<Real>               RealMatrix;
typedef Matrix<int>                IntMatrix;

// numerical with Teuchos
typedef Teuchos::SerialDenseVector<int, Real>    RealDenseVector;
typedef Teuchos::SerialDenseVector<int, int>     IntDenseVector;
typedef Teuchos::SerialDenseMatrix<int, Real>    RealDenseMatrix;
typedef Teuchos::SerialSymDenseMatrix<int, Real> RealSymDenseMatrix;
//typedef Tpetra::CrsMatrix<int, Real>             RealSparseMatrix
//typedef Tpetra::Vector<int, Real>                RealParallelVector
//typedef Tpetra::MultiVector<int, Real>           RealParallelMatrix


// ---------------------------------
// Numerical solvers (serial dense):
// ---------------------------------
#ifndef DLSfuncs_H
typedef Teuchos::SerialDenseSolver<int, Real>    RealDenseSolver;
typedef Teuchos::SerialSpdDenseSolver<int, Real> RealSpdDenseSolver;
#endif


// ---------------------------------------
// Admin/bookkeeping arrays (serial only):
// ---------------------------------------
typedef std::deque<bool>           BoolDeque; // See Effective STL (Meyers), #18
typedef Array<BoolDeque>           BoolDequeArray;
typedef Array<Real>                RealArray;
//typedef Array<RealArray>         Real2DArray;
//typedef Array<Real2DArray>       Real3DArray;
typedef Array<int>                 IntArray;
typedef Array<IntArray>            Int2DArray;
typedef Array<unsigned int>        UIntArray;
typedef Array<short>               ShortArray;
typedef Array<unsigned short>      UShortArray;
typedef Array<UShortArray>         UShort2DArray;
typedef Array<UShort2DArray>       UShort3DArray;
typedef Array<size_t>              SizetArray;
typedef Array<SizetArray>          Sizet2DArray;
typedef Array<String>              StringArray;
typedef Array<StringArray>         String2DArray;
#ifdef HAVE_BOOST
typedef boost::multi_array_types::index_range       idx_range;
typedef boost::multi_array<String, 1>               StringMultiArray;
typedef StringMultiArray::array_view<1>::type       StringMultiArrayView;
typedef StringMultiArray::const_array_view<1>::type StringMultiArrayConstView;
//typedef boost::multi_array<String, 2>             StringMulti2DArray;
//typedef StringMultiArray::array_view<2>::type     StringMulti2DArrayView;
typedef boost::multi_array<unsigned int, 1>         UIntMultiArray;
typedef UIntMultiArray::array_view<1>::type         UIntMultiArrayView;
typedef UIntMultiArray::const_array_view<1>::type   UIntMultiArrayConstView;
#endif // HAVE_BOOST
typedef Array<RealVector>           RealVectorArray;
typedef Array<RealVectorArray>      RealVector2DArray;
typedef Array<RealBaseVector>       RealBaseVectorArray;
typedef Array<RealMatrix>           RealMatrixArray;
typedef Array<RealDenseVector>      RealDenseVectorArray;
typedef Array<RealDenseVectorArray> RealDenseVector2DArray;
typedef Array<RealDenseMatrix>      RealDenseMatrixArray;
typedef Array<RealSymDenseMatrix>   RealSymDenseMatrixArray;
typedef Array<Variables>            VariablesArray;
typedef Array<Response>             ResponseArray;
typedef Array<ParamResponsePair>    PRPArray;
typedef Array<Model>                ModelArray;
typedef Array<Iterator>             IteratorArray;

typedef List<bool>                  BoolList;
typedef List<int>                   IntList;
typedef List<size_t>                SizetList;
typedef List<Real>                  RealList;
typedef List<String>                StringList;
typedef List<RealVector>            RealVectorList;
typedef List<Variables>             VariablesList;
typedef List<Interface>             InterfaceList;
typedef List<Response>              ResponseList;
typedef List<Model>                 ModelList;
typedef List<Iterator>              IteratorList;
//typedef List<ParallelLevel>         ParLevList;
//typedef List<ParallelConfiguration> ParConfigList;

typedef std::pair<int, String>         IntStringPair;
typedef std::set<int>                  IntSet;
typedef std::set<Real>                 RealSet;
typedef std::map<int, int>             IntIntMap;
typedef std::map<int, short>           IntShortMap;
typedef std::map<int, RealVector>      IntRealVectorMap;
typedef std::map<int, RealDenseVector> IntRealDenseVectorMap;
typedef std::map<int, ActiveSet>       IntActiveSetMap;
typedef std::map<int, Variables>       IntVariablesMap;
typedef std::map<int, Response>        IntResponseMap;
typedef std::map<IntArray, size_t>     IntArraySizetMap;


// ---------
// Iterators
// ---------
typedef IntList::iterator                      ILIter;
typedef IntList::const_iterator                ILCIter;
typedef SizetList::iterator                    StLIter;
typedef SizetList::const_iterator              StLCIter;
typedef RealList::iterator                     RLIter;
typedef RealList::const_iterator               RLCIter;
typedef StringList::iterator                   StringLIter;
typedef StringList::const_iterator             StringLCIter;
typedef RealVectorList::iterator               RVLIter;
typedef RealVectorList::const_iterator         RVLCIter;
typedef VariablesList::iterator                VarsLIter;
typedef InterfaceList::iterator                InterfLIter;
typedef ResponseList::iterator                 RespLIter;
typedef ModelList::iterator                    ModelLIter;
typedef IteratorList::iterator                 IterLIter;
typedef List<ParallelLevel>::iterator          ParLevLIter;
typedef List<ParallelConfiguration>::iterator  ParConfigLIter;

typedef IntSet::iterator                       ISIter;
typedef IntSet::const_iterator                 ISCIter;
typedef IntIntMap::iterator                    IntIntMIter;
typedef IntIntMap::const_iterator              IntIntMCIter;
typedef IntShortMap::iterator                  IntShMIter;
typedef IntRealVectorMap::iterator             IntRVMIter;
typedef IntRealVectorMap::const_iterator       IntRVMCIter;
typedef IntRealDenseVectorMap::iterator        IntRDVMIter;
typedef IntRealDenseVectorMap::const_iterator  IntRDVMCIter;
typedef IntActiveSetMap::iterator              IntASMIter;
typedef IntVariablesMap::iterator              IntVarsMIter;
typedef IntVariablesMap::const_iterator        IntVarsMCIter;
typedef IntResponseMap::iterator               IntRespMIter;
typedef IntResponseMap::const_iterator         IntRespMCIter;

} // namespace Dakota

#endif // DATA_TYPES_H
