/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2001, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDQuadrature
//- Description: Projects 1-D Gaussian quadratures in a tensor-product approach
//- Owner:       Mike Eldred
//- Revised by:  
//- Version:

#ifndef NOND_QUADRATURE_H
#define NOND_QUADRATURE_H

#include "data_types.h"
#include "NonDIntegration.H"

namespace Dakota {


/// Derived nondeterministic class that generates N-dimensional
/// numerical quadrature points for evaluation of expectation
/// integrals over uncorrelated standard
/// normals/uniforms/exponentials/betas/gammas.

/** This class is used by NonDPolynomialChaos, but could also be used
    for general numerical integration of moments.  It employs
    Gauss-Hermite, Gauss-Legendre, Gauss-Laguerre, Gauss-Jacobi and
    generalized Gauss-Laguerre quadrature for use with normal,
    uniform, exponential, beta, and gamma density functions and
    integration bounds.  The abscissas and weights for one-dimensional
    integration are extracted from the appropriate
    OrthogonalPolynomial class and are extended to n-dimensions using
    a tensor product approach. */

class NonDQuadrature: public NonDIntegration
{
public:

  //
  //- Heading: Constructors and destructor
  //

  // alternate constructor for instantiations "on the fly"
  NonDQuadrature(Model& model, const UShortArray& order);

  //
  //- Heading: Member functions
  //

  /// return quadOrder
  const UShortArray& quadrature_order()    const;

protected:

  //
  //- Heading: Constructors and destructor
  //

  NonDQuadrature(Model& model); ///< constructor

  ~NonDQuadrature();            ///< destructor

  //
  //- Heading: Virtual function redefinitions
  //

  void get_parameter_sets(const Model& model);

  void check_input();

  void sampling_reset(int min_samples, int rec_samples, bool all_data_flag,
		      bool stats_flag);

private:

  //
  //- Heading: Data
  //

  /// the user specification for the number of Gauss points per dimension
  UShortArray quadOrderSpec;
  /// the actual number of Gauss points per dimension, as modified by
  /// external requirements communicated through sampling_reset()
  UShortArray quadOrder;
};


inline const UShortArray& NonDQuadrature::quadrature_order() const
{ return quadOrder; }


/** used by DataFitSurrModel::build_global() to publish the minimum
    number of points needed from the quadrature routine in order to
    build a particular global approximation. */
inline void NonDQuadrature::
sampling_reset(int min_samples, int rec_samples, bool all_data_flag,
	       bool stats_flag)
{
  // quadrature order may be increased or decreased to provide at least
  // min_samples, but the original user specification (quadOrderSpec) is
  // a hard lower bound.  maxConcurrency must not be updated since parallel
  // config management depends on having the same value at ctor/run/dtor times.
  unsigned short min_order = (unsigned short)ceil(
    pow((Real)min_samples, 1./(Real)numUncertainVars) - 1.e-10);
  if (quadOrderSpec.length() == 1) {
    unsigned short order_spec = quadOrderSpec[0];
    quadOrder = (min_order > order_spec) ? min_order : order_spec;
  }
  else if (quadOrderSpec.length() == numUncertainVars)
    for (size_t i=0; i<numUncertainVars; i++)
      quadOrder[i]
	= (min_order > quadOrderSpec[i]) ? min_order : quadOrderSpec[i];

  // not currently used by this class:
  //allDataFlag = all_data_flag;
  //statsFlag   = stats_flag;
}

} // namespace Dakota

#endif
