/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       NCSUOptimizer
//- Description: Implementation code for the NCSUOptimizer class
//- Owner:       Barron J Bichon, Vanderbilt University

#include "DakotaModel.H"
#include "DakotaResponse.H"
#include "NCSUOptimizer.H"
#include "ProblemDescDB.H"

#define NCSU_DIRECT_F77 F77_FUNC_(ncsu_direct,NCSU_DIRECT)

extern "C" {

void NCSU_DIRECT_F77(int (*objfun)(int *n, double c[], double l[], double u[],
				   int point[], int *maxI, int *start,
				   int *maxfunc, double fvec[], int iidata[],
				   int *iisize, double ddata[], int *idsize,
				   char cdata[], int *icsize),
		     double* x, int& n, double& eps, int& maxf, int& maxT,
		     double& fmin, double* l, double* u, int& algmethod,
		     int& ierror, int& logfile, double& fglobal, double& fglper,
		     double& volper, double& sigmaper, int* idata, int& isize, 
		     double* ddata, int& dsize, char* cdata, int& csize,
		     int& quiet_flag);

}

namespace Dakota {

// define values for setup type (DAKOTA Model or user-supplied function)
enum { SETUP_MODEL, SETUP_USERFUNC };

NCSUOptimizer* NCSUOptimizer::ncsudirectInstance(NULL);


/** This is the standard constructor with method specification support. */ 
NCSUOptimizer::NCSUOptimizer(Model& model): Optimizer(model),
  setUpType(SETUP_MODEL),
  minBoxSize(probDescDB.get_real("method.min_boxsize_limit")), 
  volBoxSize(probDescDB.get_real("method.volume_boxsize_limit")),
  solutionAccuracy(probDescDB.get_real("method.solution_accuracy")),
  userObjectiveEval(NULL)
{ initialize(); }


/** This is an alternate constructor for instantiations on the fly
    using a Model but no ProblemDescDB. */
NCSUOptimizer::
NCSUOptimizer(Model& model, const int& max_iter, const int& max_eval) :
  Optimizer(NoDBBaseConstructor(), model), setUpType(SETUP_MODEL),
  minBoxSize(1.e-15), volBoxSize(1.e-15), // tight conv for surrogate opt
  solutionAccuracy(-DBL_MAX), userObjectiveEval(NULL)
{ maxIterations = max_iter; maxFunctionEvals = max_eval; initialize(); }


/** This is an alternate constructor for Iterator instantiations by name
    using a Model but no ProblemDescDB. */
NCSUOptimizer::
NCSUOptimizer(NoDBBaseConstructor, Model& model):
  Optimizer(NoDBBaseConstructor(), model), setUpType(SETUP_MODEL),
  minBoxSize(1.e-15), volBoxSize(1.e-15), // tight conv for surrogate opt
  solutionAccuracy(-DBL_MAX), userObjectiveEval(NULL)
{ initialize(); }


/** This is an alternate constructor for performing an optimization using
    the passed in objective function pointer. */
NCSUOptimizer::
NCSUOptimizer(const RealDenseVector& var_l_bnds,
	      const RealDenseVector& var_u_bnds, const int& max_iter,
	      const int& max_eval,
	      double (*user_obj_eval) (const RealDenseVector &x) ) :
  Optimizer(NoDBBaseConstructor(), var_l_bnds.length(), 0, 0, 0, 0, 0),
  setUpType(SETUP_USERFUNC), minBoxSize(-1.), volBoxSize(-1.),// *** GP setting?
  solutionAccuracy(-DBL_MAX), lowerBounds(var_l_bnds), upperBounds(var_u_bnds),
  userObjectiveEval(user_obj_eval)
{ maxIterations = max_iter; maxFunctionEvals = max_eval; }


void NCSUOptimizer::initialize()
{
  // Prevent nesting of an instance of a Fortran iterator within another
  // instance of the same iterator (which would result in data clashes since
  // Fortran does not support object independence).  Recurse through all
  // sub-models and test each sub-iterator for NCSU presence.
  Iterator sub_iterator = iteratedModel.subordinate_iterator();
  if (!sub_iterator.is_null() && 
       ( sub_iterator.method_name().begins("ncsu_") ||
	 sub_iterator.uses_method().begins("ncsu_") ) )
    sub_iterator.method_recourse();
  ModelList& sub_models = iteratedModel.subordinate_models();
  for (ModelLIter ml_iter = sub_models.begin();
       ml_iter != sub_models.end(); ml_iter++) {
    sub_iterator = ml_iter->subordinate_iterator();
    if (!sub_iterator.is_null() && 
	 ( sub_iterator.method_name().begins("ncsu_") ||
	   sub_iterator.uses_method().begins("ncsu_") ) )
      sub_iterator.method_recourse();
  }
}


NCSUOptimizer::~NCSUOptimizer() 
{ }


/// Modified batch evaluator that accepts multiple points and returns
/// corresponding vector of functions in fvec.  Must be used with modified
/// DIRECT src (DIRbatch.f).
int NCSUOptimizer::
objective_eval(int *n, double c[], double l[], double u[], int point[],
	       int *maxI, int *start, int *maxfunc, double fvec[], int iidata[],
	       int *iisize, double ddata[], int *idsize, char cdata[],
	       int *icsize)
{
  int cnt = *start-1; // starting index into fvec
  int nx  = *n;       // dimension of design vector x.
  
  // number of trial points to evaluate
  // if initial point, we have a single point to evaluate
  int np = (*start == 1) ? 1 : *maxI*2;

  // loop over trial points, lift scaling, and either submit for asynch
  // evaluation or compute synchronously
  RealDenseVector local_des_vars(nx, false);
  int pos = *start-1; // only used for second eval and beyond
  for (int j=0; j<np; j++) {

    if (*start == 1)
      for (int i=0; i<nx; i++)
	local_des_vars[i] = (c[i]+u[i])*l[i];
    else {
      for (int i=0; i<nx; i++) {
	// c[pos+i*maxfunc] = c(pos,i) in Fortran.
	double ci=c[pos+i*(*maxfunc)];
	local_des_vars[i] = (ci + u[i])*l[i];
      }
      pos = point[pos]-1;
    }

    // below, use default ASV (function values, no gradients or hessians)
    if (ncsudirectInstance->setUpType == SETUP_MODEL) {

      ncsudirectInstance->iteratedModel.continuous_variables(local_des_vars);

      // request the evaluation in synchronous or asynchronous mode
      if (ncsudirectInstance->iteratedModel.asynch_flag())
	ncsudirectInstance->iteratedModel.asynch_compute_response();
      else {
	ncsudirectInstance->iteratedModel.compute_response();
	// record the response in the function vector
	const Response& local_response 
	  = ncsudirectInstance->iteratedModel.current_response();
	fvec[cnt+j] = local_response.function_values()[0];
	fvec[cnt+(*maxfunc)-1+j] = 0; // TODO: flag if error in eval
      }

    }
    else {
      fvec[cnt+j] = ncsudirectInstance->userObjectiveEval(local_des_vars);
      fvec[cnt+(*maxfunc)-1+j] = 0; // TODO: flag if error in eval
    }

  } // end evaluation loop over points

  // If using model and evaluations performed asynchronously, need to record
  // the results now, after blocking until evaluation completion 
  if (ncsudirectInstance->setUpType == SETUP_MODEL &&
      ncsudirectInstance->iteratedModel.asynch_flag()) { 
      
    // block and wait for the responses
    const IntResponseMap& response_map
      = ncsudirectInstance->iteratedModel.synchronize();
    // record the responses in the function vector
    IntRespMCIter r_cit = response_map.begin();
    for (int j=0; j<np; ++j, ++r_cit) {
      fvec[cnt+j] = r_cit->second.function_values()[0];
      fvec[cnt+(*maxfunc)-1+j] = 0; // TODO: flag if error in eval
    }
  }

  return 0;
}


void NCSUOptimizer::find_optimum()
{
  //------------------------------------------------------------------
  //     Solve the problem.
  //------------------------------------------------------------------

  // set the object instance pointers for use within the static member fns
  NCSUOptimizer* prev_instance = ncsudirectInstance;
  ncsudirectInstance = this;

  int ierror, num_cv = numContinuousVars, algmethod = 1, logfile = 13,
      quiet_flag  = 1;
  double fmin = 0., eps = 1.e-4;
  // convergence tolerance to known global solution
  double fglper   = (solutionAccuracy > -DBL_MAX) ? convergenceTol : 0.;
  // terminate when volume of box w/ f_min < volper*volume of orig box
  double volper   = (volBoxSize >= 0.) ? volBoxSize : 1.e-6;
  // terminate when size of box  w/ f_min < sigmaper*size of orig box
  double sigmaper = (minBoxSize >= 0.) ? minBoxSize : 1.e-4;

  // for passing additional data to objective_eval()
  int isize = 0, dsize = 0, csize = 0;
  int*    idata = NULL;
  double* ddata = NULL;
  char*   cdata = NULL;

  RealDenseVector local_des_vars;
  if (setUpType == SETUP_MODEL) {
    // initialize local_des_vars with DB initial point.  Variables are updated 
    // in constraint_eval/objective_eval
    copy_data(iteratedModel.continuous_variables(), local_des_vars);
    copy_data(iteratedModel.continuous_lower_bounds(), lowerBounds);
    copy_data(iteratedModel.continuous_upper_bounds(), upperBounds);
  } 
  else
    local_des_vars.size(num_cv);

  NCSU_DIRECT_F77(objective_eval, local_des_vars.values(), num_cv, eps,
		  maxFunctionEvals, maxIterations, fmin, lowerBounds.values(),
		  upperBounds.values(), algmethod, ierror, logfile,
		  solutionAccuracy, fglper, volper, sigmaper, idata, isize,
		  ddata, dsize, cdata, csize, quiet_flag);

  // NCSU DIRECT completed. 
  // Do post-processing/output of final info and data:
  //Cout << "\nNCSU DIRECT exits with Ierror code = " << ierror;
  //Cout << "\n(see \"Calling DIRECT\" section in NCSU DIRECT manual)\n";

  //delete [] idata;
  //delete [] ddata;
  //delete [] cdata;

  // Set bestVariables and bestResponse for use by strategy level.
  // local_des_vars, fmin contain the optimal design 
  bestVariables.continuous_variables(local_des_vars);
  if (!multiObjFlag) { // else multi_objective_retrieve() is used in
                       // Optimizer::derived_post_run()
    RealDenseVector best_fns(numFunctions);
    best_fns[0] = fmin;
    bestResponse.function_values(best_fns);
  }

  // restore in case of recursion
  ncsudirectInstance = prev_instance;
}

} // namespace Dakota
