/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        InterpPolyApproximation
//- Description:  Class for Lagrange Interpolation Polynomial Approximation
//-               
//- Owner:        Mike Eldred

#ifndef INTERP_POLY_APPROXIMATION_H
#define INTERP_POLY_APPROXIMATION_H

#include "BasisPolyApproximation.H"
#include "LagrangeInterpPolynomial.H"
#include "DataModel.H"


namespace Dakota {

class ProblemDescDB;


/// Derived approximation class for interpolation polynomials (global
/// approximation).

/** The InterpPolyApproximation class provides a global approximation
    based on interpolation polynomials.  It is used primarily for
    stochastic collocation approaches to uncertainty quantification. */

class InterpPolyApproximation: public BasisPolyApproximation
{
public:

  //
  //- Heading: Constructor and destructor
  //

  /// default constructor
  InterpPolyApproximation();
  // ProblemDescDB-based constructor
  //InterpPolyApproximation(const ProblemDescDB& problem_db,
  //			    const size_t& num_acv);
  /// destructor
  ~InterpPolyApproximation();

protected:

  //
  //- Heading: Virtual function redefinitions
  //

  int min_coefficients() const;

  /// find the coefficients for the expansion of multivariate
  /// interpolation polynomials
  void find_coefficients();

  /// retrieve the response expansion value for a given parameter vector
  const Real& get_value(const RealDenseVector& x);
  /// retrieve the response expansion gradient for a given parameter vector
  /// and default DVV
  const RealBaseVector& get_gradient(const RealDenseVector& x);
  /// retrieve the response expansion gradient for a given parameter vector
  /// and given DVV
  const RealBaseVector& get_gradient(const RealDenseVector& x,
				     const UIntArray& dvv);

  /// return the mean of the expansion, treating all variables as random
  const Real& get_mean();
  /// return the mean of the expansion for a given parameter vector,
  /// treating a subset of the variables as random
  const Real& get_mean(const RealDenseVector& x);
  /// return the gradient of the expansion mean for a given parameter vector,
  /// treating all variables as random
  const RealBaseVector& get_mean_gradient();
  /// return the gradient of the expansion mean for a given parameter vector
  /// and given DVV, treating a subset of the variables as random
  const RealBaseVector& get_mean_gradient(const RealDenseVector& x,
					  const UIntArray& dvv);

  /// return the variance of the expansion, treating all variables as random
  const Real& get_variance();
  /// return the variance of the expansion for a given parameter vector,
  /// treating a subset of the variables as random
  const Real& get_variance(const RealDenseVector& x);
  /// return the gradient of the expansion variance for a given parameter
  /// vector, treating all variables as random
  const RealBaseVector& get_variance_gradient();
  /// return the gradient of the expansion variance for a given parameter
  /// vector and given DVV, treating a subset of the variables as random
  const RealBaseVector& get_variance_gradient(const RealDenseVector& x,
					      const UIntArray& dvv);

private:

  //
  //- Heading: Convenience functions
  //

  /// compute the value of an interpolant on an isotropic/anisotropic
  /// tensor-product grid; contributes to get_value(x)
  const Real& tensor_product_value(const RealDenseVector& x, size_t tp_index);
  /// compute the gradient of an interpolant on an isotropic/anisotropic
  /// tensor-product grid; contributes to get_gradient(x)
  const RealBaseVector& tensor_product_gradient(const RealDenseVector& x,
    size_t tp_index);
  /// compute the gradient of an interpolant on an isotropic/anisotropic
  /// tensor-product grid for given DVV; contributes to get_gradient(x, dvv)
  const RealBaseVector& tensor_product_gradient(const RealDenseVector& x,
    size_t tp_index, const UIntArray& dvv);
  /// compute the mean of an interpolant on an isotropic/anisotropic
  /// tensor-product grid; contributes to get_mean(x)
  const Real& tensor_product_mean(const RealDenseVector& x, size_t tp_index);
  /// compute the mean of an interpolant on an isotropic/anisotropic
  /// tensor-product grid; contributes to get_mean(x)
  const RealBaseVector& tensor_product_mean_gradient(const RealDenseVector& x,
    size_t tp_index, const UIntArray& dvv);

  /// compute the variance of an interpolant on an isotropic/anisotropic
  /// tensor-product grid; contributes to get_variance(x)
  const Real& tensor_product_variance(const RealDenseVector& x,
    size_t tp_index);
  /// compute the variance of an interpolant on an isotropic/anisotropic
  /// tensor-product grid; contributes to get_variance(x)
  const RealBaseVector& tensor_product_variance_gradient(
    const RealDenseVector& x, size_t tp_index, const UIntArray& dvv);

  //
  //- Heading: Data
  //

  /// 2D array of one-dimensional basis polynomial objects which are used in
  /// constructing the multivariate orthogonal/interpolation polynomials.
  /** Each variable (outer array size = numVars) may have multiple integration
      orders associated with it (inner array size = num_levels_per_var = 1 for
      quadrature, w + numVars for sparse grid). */
  Array< Array< BasisPolynomial > > polynomialBasis;

  /// total number of collocation points = number of terms in interpolation
  /// expansion (length of expansionCoeffs)
  int numCollocPts;

  /// numSmolyakIndices-by-numVars array for identifying the index to use
  /// within the polynomialBasis for a particular variable
  /** The index sets correspond to j (0-based) for use as indices, which are
      offset from the i indices (1-based) normally used in the Smolyak
      expressions.  For quadrature, the indices are zero (irrespective of
      integration order) since there is one polynomialBasis per variable; for
      sparse grid, the index corresponds to level - 1 within each anisotropic
      tensor-product integration of a Smolyak recursion. */
  UShort2DArray smolyakMultiIndex;
  /// precomputed array of Smolyak combinatorial coefficients
  RealArray smolyakCoeffs;
  /// numSmolyakIndices-by-numTensorProductPts-by-numVars array for identifying
  /// the 1-D interpolant indices for sets of tensor-product collocation points.
  UShort3DArray collocKey;
  /// numSmolyakIndices-by-numTensorProductPts array for linking the
  /// set of tensor products to the expansionCoeffs array.
  Sizet2DArray expansionCoeffIndices;

  /// the value of a tensor-product interpolant; a contributor to approxValue
  Real tpValue;
  /// the gradient of a tensor-product interpolant; a contributor to
  /// approxGradient
  RealBaseVector tpGradient;
  /// the mean of a tensor-product interpolant; a contributor to expansionMean
  Real tpMean;
  /// the gradient of the mean of a tensor-product interpolant; a
  /// contributor to expansionMeanGrad
  RealBaseVector tpMeanGrad;
  /// the variance of a tensor-product interpolant; a contributor to
  /// expansionVariance
  Real tpVariance;
  /// the gradient of the variance of a tensor-product interpolant; a
  /// contributor to expansionVarianceGrad 
  RealBaseVector tpVarianceGrad;
};


inline InterpPolyApproximation::InterpPolyApproximation(): numCollocPts(0)
{ }


inline InterpPolyApproximation::~InterpPolyApproximation()
{ }

} // namespace Dakota

#endif
