/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        ForkAnalysisCode
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "ForkAnalysisCode.H"
#include "ProblemDescDB.H"
#include "system_defs.h"
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

static const char rcsId[]="@(#) $Id: ForkAnalysisCode.C 5384 2008-10-17 22:00:51Z dmgay $";

extern "C" const char** arg_list_adjust(const char **, void **);

namespace Dakota {

ForkAnalysisCode::ForkAnalysisCode(const ProblemDescDB& problem_db): 
  AnalysisCode(problem_db)
{ argList.reshape(3); }

pid_t ForkAnalysisCode::fork_program(const bool block_flag)
{
  pid_t pid = 0;

  // vfork() should be used here since there is an immediate execvp(). This 
  // conserves memory over fork().  If some platforms have problems with a
  // hybrid fork/vfork approach, add #ifdef's but make vfork the default.
#if defined(HAVE_WORKING_VFORK) && !defined(__USE_STD_IOSTREAM)
  // We have vfork and are not using the cxx C++ compiler on OSF/Tru64.
  pid = vfork(); // replicate this process
#elif defined(HAVE_WORKING_FORK)
  pid = fork(); // replicate this process
#else
  Cerr << "Error: fork/vfork are not supported under this OS." << endl;
  abort_handler(-1);
#endif

#if defined(HAVE_UNISTD_H) && defined(HAVE_SYS_WAIT_H)
  if (pid == 0) { // this is the child: replace process & execute application

    // Convert argList StringArray to an array of const char*'s.  An arg_list
    // entry is passed as the first argument, and the entire arg_list is cast
    // as the second argument.
    const char* arg_list[4];
    arg_list[0] = argList[0].c_str();
    if (commandLineArgs) {
      for (int i=1; i<3; i++)
	arg_list[i] = argList[i].c_str();
      arg_list[3] = (const char *)NULL;
    }
    else
      arg_list[1] = (const char *)NULL;

    const char **argL = arg_list_adjust(arg_list, 0);

    // replace the child process with the fork target defined in arg_list
    int return_code = execvp(argL[0], (char* const *)argL);
    // if execvp returns then it failed, exit gracefully. 
    _exit(return_code); // since this is a copy of the parent, use _exit
    // so that the parent i/o stream is not prematurely flushed and closed.
  }
  else if (block_flag) { // parent: wait for completion
    int status;
    waitpid(pid, &status, 0); // be explicit about waiting on the pid created
    // above (use waitpid instead of wait) so that this blocking fork works
    // properly in the possible presence of other nonblocking fork pid's 
    // (required by failure capturing routine, but also good form in general).
    check_status(status); // check the exit status
  }
#endif // HAVE_UNISTD_H && HAVE_SYS_WAIT_H

  return(pid);
}


/** Check to see if the process terminated abnormally (WIFEXITED(status)==0)
    or if either execvp or the application returned a status code of -1
    (WIFEXITED(status)!=0 && (signed char)WEXITSTATUS(status)==-1).  If one
    of these conditions is detected, output a failure message and abort.
    Note: the application code should not return a status code of -1 unless an
    immediate abort of dakota is wanted.  If for instance, failure capturing
    is to be used, the application code should write the word "FAIL" to the
    appropriate results file and return a status code of 0 through exit(). */
void ForkAnalysisCode::check_status(const int status)
{
#ifdef HAVE_SYS_WAIT_H
  // checks for an abnormal exit status and, if present, abort 
  if ( (WIFEXITED(status) != 0 && (signed char)WEXITSTATUS(status) == -1)
     || WIFEXITED(status) == 0 ) {
    Cerr << "Fork application failure, aborting.\nSystem error message: "
         << strerror(errno) << '\n';
    // strerror() returns the system error message associated with errno (a
    // system constant defined in errno.h containing the number of the last
    // observed error).  Note: If it becomes an issue, it would also be possible
    // to kill simulation drivers or intermediate processes for the three-piece
    // interface here, based on pid's obtained from any asynch forks. This would
    // be beneficial on systems that do not perform a complete cleanup when one
    // (asynch) child process dies.
    abort_handler(-1);
  }
#endif // HAVE_SYS_WAIT_H
}


void ForkAnalysisCode::driver_argument_list(const int analysis_id)
{
  argList[0] = programNames[analysis_id-1];
  char tag_str[16];
  if (multipleParamsFiles || numPrograms > 1)
    sprintf(tag_str, ".%d", analysis_id);
  argList[1] = (multipleParamsFiles) ? paramsFileName+tag_str  : paramsFileName;
  argList[2] = (numPrograms > 1) ?    resultsFileName+tag_str : resultsFileName;
#ifdef DEBUG
  Cout << "argList: " << argList[0] << ' ' << argList[1] << ' ' << argList[2]
       << endl;
#endif
}

} // namespace Dakota
