/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        ActiveSet
//- Description:  Container class for active set tracking data.
//-
//- Owner:        Mike Eldred
//- Version: $Id: DakotaActiveSet.H 5792 2009-03-30 18:22:20Z mseldre $

#ifndef DAKOTA_ACTIVE_SET_H
#define DAKOTA_ACTIVE_SET_H

#include "data_util.h"

namespace Dakota {


/// Container class for active set tracking information.  Contains the
/// active set request vector and the derivative variables vector.

/** The ActiveSet class is a small class whose initial design function
    is to avoid having to pass the ASV and DVV separately.  It is not
    part of a class hierarchy and does not employ reference-counting/
    representation-sharing idioms (e.g., handle-body). */

class ActiveSet
{
  //
  //- Heading: Friends
  //

  /// equality operator
  friend bool operator==(const ActiveSet& set1, const ActiveSet& set2);
  /// inequality operator
  friend bool operator!=(const ActiveSet& set1, const ActiveSet& set2);

public:

  //
  //- Heading: Constructors and destructor
  //

  ActiveSet();                                      ///< default constructor
  ActiveSet(size_t num_fns, size_t num_deriv_vars); ///< standard constructor
  ActiveSet(const ActiveSet& set);                  ///< copy constructor
  ~ActiveSet();                                     ///< destructor

  //
  //- Heading: operators
  //

  /// assignment operator
  ActiveSet& operator=(const ActiveSet& set);

  //
  //- Heading: Member functions
  //

  /// reshape requestVector and derivVarsVector
  void reshape(size_t num_fns, size_t num_deriv_vars);

  /// return the request vector
  const ShortArray& request_vector() const;
  /// set the request vector
  void request_vector(const ShortArray& rv);
  /// set all request vector values
  void request_values(const short rv_val);
  /// set the value of an entry in the request vector
  void request_value(const size_t index, const short rv_val);

  /// return the derivative variables vector
  const UIntArray& derivative_vector() const;
  /// set the derivative variables vector from a UIntArray
  void derivative_vector(const UIntArray& dvv);
  /// set the derivative variables vector from a UIntMultiArrayConstView
  void derivative_vector(UIntMultiArrayConstView dvv);
  /// set the derivative variables vector values
  void derivative_start_value(const unsigned int dvv_start_val);

  /// read an active set object from an istream
  void read(istream& s);
  /// write an active set object to an ostream
  void write(ostream& s) const;
  /// write an active set object to an ostream in annotated format
  void write_annotated(ostream& s) const;

  /// read an active set object from the binary restart stream
  void read(BiStream& s);
  /// write an active set object to the binary restart stream
  void write(BoStream& s) const;

  /// read an active set object from a packed MPI buffer
  void read(MPIUnpackBuffer& s);
  /// write an active set object to a packed MPI buffer
  void write(MPIPackBuffer& s) const;

private:

  //
  //- Heading: Data
  //
 
  /// the vector of response requests
  /** It uses a 0 value for inactive functions and sums 1 (value),
      2 (gradient), and 4 (Hessian) for active functions. */
  ShortArray requestVector;
  /// the vector of variable ids used for computing derivatives
  /** These ids will generally identify either the active continuous
      variables or the inactive continuous variables. */
  UIntArray derivVarsVector;
};


inline ActiveSet::ActiveSet()
{ }


inline ActiveSet::ActiveSet(const ActiveSet& set)
{ requestVector = set.requestVector; derivVarsVector = set.derivVarsVector; }


inline ActiveSet::~ActiveSet()
{ }


inline ActiveSet& ActiveSet::operator=(const ActiveSet& set)
{
  requestVector   = set.requestVector;
  derivVarsVector = set.derivVarsVector;
  return *this;
}


//inline bool operator==(const ActiveSet& set1, const ActiveSet& set2)
//{
//  return (set1.requestVector   == set2.requestVector &&
//	  set1.derivVarsVector == set2.derivVarsVector);
//}


//inline bool operator!=(const ActiveSet& set1, const ActiveSet& set2)
//{ return !(set1 == set2); }


inline void ActiveSet::reshape(size_t num_fns, size_t num_deriv_vars)
{ requestVector.reshape(num_fns); derivVarsVector.reshape(num_deriv_vars); }


inline const ShortArray& ActiveSet::request_vector() const
{ return requestVector; }


inline void ActiveSet::request_vector(const ShortArray& rv)
{ requestVector = rv; }


inline void ActiveSet::request_values(const short rv_val)
{ requestVector = rv_val; }


inline void ActiveSet::request_value(const size_t index, const short rv_val)
{ requestVector[index] = rv_val; }


inline const UIntArray& ActiveSet::derivative_vector() const
{ return derivVarsVector; }


inline void ActiveSet::derivative_vector(const UIntArray& dvv)
{ derivVarsVector = dvv; }


inline void ActiveSet::derivative_vector(UIntMultiArrayConstView dvv)
{ copy_data(dvv, derivVarsVector); }


inline void ActiveSet::derivative_start_value(const unsigned int dvv_start_val)
{
  size_t dvv_len = derivVarsVector.length();
  for (size_t i=0; i<dvv_len; i++)
    derivVarsVector[i] = dvv_start_val+i;
}


inline void ActiveSet::read(istream& s)
{ s >> requestVector >> derivVarsVector; }


inline void ActiveSet::write(ostream& s) const
{ s << requestVector << derivVarsVector; }


inline void ActiveSet::write_annotated(ostream& s) const
{
  requestVector.write_annotated(s, false);
  derivVarsVector.write_annotated(s, false);
}


inline void ActiveSet::read(BiStream& s)
{ s >> requestVector >> derivVarsVector; }


inline void ActiveSet::write(BoStream& s) const
{ s << requestVector << derivVarsVector; }


inline void ActiveSet::read(MPIUnpackBuffer& s)
{ s >> requestVector >> derivVarsVector; }


inline void ActiveSet::write(MPIPackBuffer& s) const
{ s << requestVector << derivVarsVector; }


/// istream extraction operator for ActiveSet.  Calls read(istream&).
inline istream& operator>>(istream& s, ActiveSet& set)
{ set.read(s); return s; }


/// ostream insertion operator for ActiveSet.  Calls write(istream&).
inline ostream& operator<<(ostream& s, const ActiveSet& set)
{ set.write(s); return s; }


/// BiStream extraction operator for ActiveSet.  Calls read(BiStream&).
inline BiStream& operator>>(BiStream& s, ActiveSet& set)
{ set.read(s); return s; }


/// BoStream insertion operator for ActiveSet.  Calls write(BoStream&).
inline BoStream& operator<<(BoStream& s, const ActiveSet& set)
{ set.write(s); return s; }


/// MPIUnpackBuffer extraction operator for ActiveSet.
/// Calls read(MPIUnpackBuffer&).
inline MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, ActiveSet& set)
{ set.read(s); return s; }


/// MPIPackBuffer insertion operator for ActiveSet. Calls write(MPIPackBuffer&).
inline MPIPackBuffer& operator<<(MPIPackBuffer& s, const ActiveSet& set)
{ set.write(s); return s; }


/// inequality operator for ActiveSet
inline bool operator!=(const ActiveSet& set1, const ActiveSet& set2)
{ return !(set1 == set2); }

} // namespace Dakota

#endif
