/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        AnalysisCode
//- Description:  Abstract base class for the analysis code simulators used
//-               by the interface classes to perform function evaluations.
//- Owner:        Mike Eldred
//- Version: $Id: AnalysisCode.H 5356 2008-10-14 18:17:23Z mseldre $

#ifndef ANALYSIS_CODE_H
#define ANALYSIS_CODE_H

#include "data_types.h"


namespace Dakota {

class Variables;
class ActiveSet;
class Response;
class ProblemDescDB;
class ParallelLibrary;


/// Base class providing common functionality for derived classes
/// (SysCallAnalysisCode and ForkAnalysisCode) which spawn separate
/// processes for managing simulations.

/** The AnalysisCode class hierarchy provides simulation spawning
    services for ApplicationInterface derived classes and alleviates
    these classes of some of the specifics of simulation code
    management.  The hierarchy does not employ the letter-envelope
    technique since the ApplicationInterface derived classes
    instantiate the appropriate derived AnalysisCode class directly. */

class AnalysisCode
{
public:

  //
  //- Heading: Methods
  //

  /// define modified filenames from user input by handling Unix temp
  /// file and tagging options
  void define_filenames(const int id);

  /// write the parameters data and response request data to one or
  /// more parameters files (using one or more invocations of
  /// write_parameters_file()) in either standard or aprepro format
  void write_parameters_files(const Variables& vars, const ActiveSet& set,
			      const int id);

  /// read the response object from one or more results files
  void read_results_files(Response& response, const int id);

  //
  //- Heading: Set and Inquire functions
  //

  //void io_filters(IOFilter& ifilter, IOFilter& ofilter);

  /// return programNames
  const StringArray& program_names()   const;
  /// return iFilterName
  const String& input_filter_name()    const;
  /// return oFilterName
  const String& output_filter_name()   const;

  /// return paramsFileName
  const String& parameters_filename()  const;
  /// return resultsFileName
  const String& results_filename()     const;
  /// return the results filename entry in fileNameMap corresponding to id
  const String& results_filename(const int id);

  /// set suppressOutputFlag
  void suppress_output_flag(const bool flag);
  /// return suppressOutputFlag
  bool suppress_output_flag() 	       const;

  /// return commandLineArgs
  bool command_line_arguments()        const;

  /// return multipleParamsFiles
  bool multiple_parameters_filenames() const;

protected:

  //
  //- Heading: Constructors and destructor (protected since only derived class
  //           objects should be instantiated)
  //

  AnalysisCode(const ProblemDescDB& problem_db); ///< constructor
  virtual ~AnalysisCode();                       ///< destructor

  //
  //- Heading: Data
  //

  /// flag set by master processor to suppress output from slave processors
  bool suppressOutputFlag;
  /// output verbosity level: {SILENT,QUIET,NORMAL,VERBOSE,DEBUG}_OUTPUT
  short outputLevel;
  /// flags tagging of parameter/results files
  bool fileTagFlag;
  /// flags retention of parameter/results files
  bool fileSaveFlag;
  /// flag indicating use of passing of filenames as command line arguments to
  /// the analysis drivers and input/output filters
  bool commandLineArgs;
  /// flag indicating use of the APREPRO (the Sandia "A PRE PROcessor" utility)
  /// format for parameter files
  bool apreproFlag;
  /// flag indicating the need for separate parameters files for multiple
  /// analysis drivers
  bool multipleParamsFiles;

  /// the name of the input filter (input_filter user specification)
  String iFilterName;
  /// the name of the output filter (output_filter user specification)
  String oFilterName;
  /// the names of the analysis code programs (analysis_drivers user
  /// specification)
  StringArray programNames;
  /// the number of analysis code programs (length of programNames)
  size_t numPrograms;
  /// the name of the parameters file from user specification
  String specifiedParamsFileName;
  /// the parameters file name actually used (modified with tagging or
  /// temp files)
  String paramsFileName;
  /// the name of the results file from user specification
  String specifiedResultsFileName;
  /// the results file name actually used (modified with tagging or temp files)
  String resultsFileName;

  /// stores parameters and results file names used in spawning function
  /// evaluations.  Map key is the function evaluation identifier.
  map<int, pair<String, String> > fileNameMap;

  //IOFilter inputFilter;
  //IOFilter outputFilter;

private:

  //
  //- Heading: Convenience functions
  //

  /// write the variables, active set vector, derivative variables vector,
  /// and analysis components to the specified parameters file in either
  /// standard or aprepro format
  void write_parameters_file(const Variables& vars, const ActiveSet& set,
			     const StringArray& an_comps,
			     const String& params_fname);

  //
  //- Heading: Data
  //

  /// reference to the ParallelLibrary object.  Used in define_filenames().
  ParallelLibrary& parallelLib;

  /// the set of optional analysis components used by the analysis drivers
  /// (from the analysis_components interface specification)
  String2DArray analysisComponents;
};


inline AnalysisCode::~AnalysisCode() { }

inline const StringArray& AnalysisCode::program_names() const
{ return programNames; }

inline const String& AnalysisCode::input_filter_name() const
{ return iFilterName; }

inline const String& AnalysisCode::output_filter_name() const
{ return oFilterName; }

inline const String& AnalysisCode::parameters_filename() const
{ return paramsFileName; }

inline const String& AnalysisCode::results_filename() const
{ return resultsFileName; }

inline const String& AnalysisCode::results_filename(const int id)
{ return fileNameMap[id].second; }

inline void AnalysisCode::suppress_output_flag(const bool flag)
{ suppressOutputFlag = flag; }

inline bool AnalysisCode::suppress_output_flag() const
{ return suppressOutputFlag; }

inline bool AnalysisCode::command_line_arguments() const
{ return commandLineArgs; }

inline bool AnalysisCode::multiple_parameters_filenames() const
{ return multipleParamsFiles; }

//inline void AnalysisCode::io_filters(IOFilter& ifilter, IOFilter& ofilter)
//{ inputFilter = ifilter; outputFilter = ofilter; }

} // namespace Dakota

#endif
