/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        AllConstraints
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "AllConstraints.H"
#include "ProblemDescDB.H"
#include "data_io.h"
#include "data_util.h"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, the all data approach (design, uncertain, and state
    types are combined) is used.  Iterators/strategies which use this
    class include: parameter studies, dace, and nond_sampling in
    all_variables mode.  Extract fundamental lower and upper bounds
    and combine them into allContinuousLowerBnds, allContinuousUpperBnds,
    allDiscreteLowerBnds, and allDiscreteUpperBnds. */
AllConstraints::
AllConstraints(const ProblemDescDB& problem_db, const pair<short,short>& view,
	       const Sizet2DArray& vars_comps):
  Constraints(BaseConstructor(), problem_db, view, vars_comps)
{
  // Initialize derived arrays for this class.
  aggregate_data(
    problem_db.get_rdv("variables.continuous_design.lower_bounds"),
    problem_db.get_rdv("variables.uncertain.lower_bounds"),
    problem_db.get_rdv("variables.continuous_state.lower_bounds"),
    allContinuousLowerBnds);
  aggregate_data(
    problem_db.get_rdv("variables.continuous_design.upper_bounds"),
    problem_db.get_rdv("variables.uncertain.upper_bounds"),
    problem_db.get_rdv("variables.continuous_state.upper_bounds"),
    allContinuousUpperBnds);

  aggregate_data(
    problem_db.get_idv("variables.discrete_design.lower_bounds"),
    problem_db.get_idv("variables.discrete_state.lower_bounds"),
    allDiscreteLowerBnds);
  aggregate_data(
    problem_db.get_idv("variables.discrete_design.upper_bounds"),
    problem_db.get_idv("variables.discrete_state.upper_bounds"),
    allDiscreteUpperBnds);

  // construct active/inactive views of all arrays
  build_views();

  // Manage linear constraints.
  manage_linear_constraints(problem_db);

#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif
}


void AllConstraints::reshape_rep()
{
  size_t num_acv = (*variablesComponents)[0][1] + (*variablesComponents)[1][0] +
    (*variablesComponents)[2][0] + (*variablesComponents)[3][1],
    num_adv = (*variablesComponents)[0][2] + (*variablesComponents)[3][2];
  allContinuousLowerBnds.resize(num_acv);
  allContinuousUpperBnds.resize(num_acv);
  allDiscreteLowerBnds.resize(num_adv);
  allDiscreteUpperBnds.resize(num_adv);

  build_views();
}


void AllConstraints::build_active_views()
{
  size_t num_cv = allContinuousLowerBnds.length(),
    num_dv = allDiscreteLowerBnds.length();
  if (num_cv) {
    continuousLowerBnds = RealDenseVector(Teuchos::View,
      allContinuousLowerBnds.values(), num_cv);
    continuousUpperBnds = RealDenseVector(Teuchos::View,
      allContinuousUpperBnds.values(), num_cv);
  }
  if (num_dv) {
    discreteLowerBnds = IntDenseVector(Teuchos::View,
      allDiscreteLowerBnds.values(), num_dv);
    discreteUpperBnds = IntDenseVector(Teuchos::View,
      allDiscreteUpperBnds.values(), num_dv);
  }
}


void AllConstraints::read(istream& s)
{
  size_t num_cdv = (*variablesComponents)[0][1],
    num_ddv      = (*variablesComponents)[0][2],
    num_uv_csv   = allContinuousLowerBnds.length() - num_cdv,
    num_dsv      = allDiscreteLowerBnds.length()   - num_ddv;
  read_data_partial(s, 0, num_cdv, allContinuousLowerBnds);
  read_data_partial(s, 0, num_ddv, allDiscreteLowerBnds);
  read_data_partial(s, num_cdv, num_uv_csv, allContinuousLowerBnds);
  read_data_partial(s, num_ddv, num_dsv, allDiscreteLowerBnds);

  read_data_partial(s, 0, num_cdv, allContinuousUpperBnds);
  read_data_partial(s, 0, num_ddv, allDiscreteUpperBnds);
  read_data_partial(s, num_cdv, num_uv_csv, allContinuousUpperBnds);
  read_data_partial(s, num_ddv, num_dsv, allDiscreteUpperBnds);
}


void AllConstraints::write(ostream& s) const
{
  size_t num_cdv = (*variablesComponents)[0][1],
    num_ddv      = (*variablesComponents)[0][2],
    num_uv_csv   = allContinuousLowerBnds.length() - num_cdv,
    num_dsv      = allDiscreteLowerBnds.length()   - num_ddv;
  write_data_partial(s, 0, num_cdv, allContinuousLowerBnds);
  write_data_partial(s, 0, num_ddv, allDiscreteLowerBnds);
  write_data_partial(s, num_cdv, num_uv_csv, allContinuousLowerBnds);
  write_data_partial(s, num_ddv, num_dsv, allDiscreteLowerBnds);

  write_data_partial(s, 0, num_cdv, allContinuousUpperBnds);
  write_data_partial(s, 0, num_ddv, allDiscreteUpperBnds);
  write_data_partial(s, num_cdv, num_uv_csv, allContinuousUpperBnds);
  write_data_partial(s, num_ddv, num_dsv, allDiscreteUpperBnds);
}

} // namespace Dakota
