# include "sandia_rules.H"

# include <cstdlib>
# include <iomanip>
# include <iostream>
# include <cmath>

namespace webbur 
{

//****************************************************************************80

void dtable_close_write ( std::ofstream &output )

//****************************************************************************80
//
//  Purpose:
//
//    DTABLE_CLOSE_WRITE closes a file to which a DTABLE was to be written.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    27 September 2006
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, ofstream output, the output file stream.
//
{
  output.close ( );

  return;
}
//****************************************************************************80

void dtable_data_write ( std::ofstream &output, int m, int n, double table[] )

//****************************************************************************80
//
//  Purpose:
//
//    DTABLE_DATA_WRITE writes data to a DTABLE file.
//
//  Discussion:
//
//    The file should already be open.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    31 January 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, ofstream &OUTPUT, a pointer to the output stream.
//
//    Input, int M, the spatial dimension.
//
//    Input, int N, the number of points.
//
//    Input, double TABLE[M*N], the table data.
//
{
  int i;
  int j;

  for ( j = 0; j < n; j++ )
  {
    for ( i = 0; i < m; i++ )
    {
      output << std::setprecision(16) << std::setw(24) << table[i+j*m] << "  ";
    }
    output << "\n";
  }

  return;
}
//****************************************************************************80
 
void dtable_header_write ( std::string output_filename, std::ofstream &output, 
  int m, int n )
 
//****************************************************************************80
//
//  Purpose:
//
//    DTABLE_HEADER_WRITE writes the header of a DTABLE file.
//
//  Discussion:
//
//    The file should already be open.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    04 February 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, string OUTPUT_FILENAME, the output filename.
//
//    Input, ofstream &OUTPUT, the output stream.
//
//    Input, int M, the spatial dimension.
//
//    Input, int N, the number of points.
//
{
  char *s;

  s = webbur::timestring ( );

  output << "#  " << output_filename << "\n";
  output << "#  created by dtable_write in table_io.C" << "\n";
  output << "#  at " << s << "\n";
  output << "#\n";
  output << "#  Spatial dimension M = " << m << "\n";
  output << "#  Number of points N =  " << n << "\n";
  output << "#  EPSILON (unit roundoff) = " << webbur::r8_epsilon ( ) << "\n";
  output << "#\n";

  delete [] s;

  return;
}
//****************************************************************************80

void dtable_write ( std::string output_filename, int m, int n, double table[], 
  bool header )

//****************************************************************************80
//
//  Purpose:
//
//    DTABLE_WRITE writes information to a DTABLE file.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    04 February 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, string OUTPUT_FILENAME, the output filename.
//
//    Input, int M, the spatial dimension.
//
//    Input, int N, the number of points.
//
//    Input, double TABLE[M*N], the table data.
//
//    Input, bool HEADER, is TRUE if the header is to be included.
//
{
  std::ofstream output;

  output.open ( output_filename.c_str ( ) );

  if ( !output )
  {
    std::cerr << "\n";
    std::cerr << "DTABLE_WRITE - Fatal error!\n";
    std::cerr << "  Could not open the output file.\n";
    return;
  }

  if ( header )
  {
    webbur::dtable_header_write ( output_filename, output, m, n );
  }

  webbur::dtable_data_write ( output, m, n, table );

  webbur::dtable_close_write ( output );

  return;
}
//****************************************************************************80

void chebyshev1_compute ( int order, double alpha, double beta, double x[], 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    CHEBYSHEV1_COMPUTE computes a Chebyshev type 1 quadrature rule.
//
//  Discussion:
//
//    The integration interval is [ -1, 1 ].
//
//    The weight function is w(x) = 1.0 / sqrt ( 1 - x^2 ).
//
//    The integral to approximate:
//
//      Integral ( -1 <= X <= 1 ) F(X) / sqrt ( 1 - x^2 ) dX
//
//    The quadrature rule:
//
//      Sum ( 1 <= I <= ORDER ) W(I) * F ( X(I) )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Philip Davis, Philip Rabinowitz,
//    Methods of Numerical Integration,
//    Second Edition,
//    Dover, 2007,
//    ISBN: 0486453391,
//    LC: QA299.3.D28.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
//    Output, double W[ORDER], the weights.
//
{
  int i;
  double pi = 3.141592653589793;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "CHEBYSHEV1_COMPUTE - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  for ( i = 0; i < order; i++ )
  {
    w[i] = pi / ( double ) ( order );
  }
  for ( i = 0; i < order; i++ )
  {
    x[i] = std::cos ( pi * ( double ) ( 2 * order - 1 - 2 * i ) 
                         / ( double ) ( 2 * order ) );
  }
  if ( ( order % 2 ) == 1 )
  {
    x[(order-1)/2] = 0.0;
  }

  return;
}
//****************************************************************************80

void chebyshev1_compute_points ( int order, double alpha, double beta, 
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    CHEBYSHEV1_COMPUTE_POINTS computes Chebyshev type 1 quadrature points.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Philip Davis, Philip Rabinowitz,
//    Methods of Numerical Integration,
//    Second Edition,
//    Dover, 2007,
//    ISBN: 0486453391,
//    LC: QA299.3.D28.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  int i;
  double pi = 3.141592653589793;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "CHEBYSHEV1_COMPUTE_POINTS - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  for ( i = 0; i < order; i++ )
  {
    x[i] =  std::cos ( pi * ( double ) ( 2 * order - 1 - 2 * i ) 
                          / ( double ) ( 2 * order ) );
  }
  if ( ( order % 2 ) == 1 )
  {
    x[(order-1)/2] = 0.0;
  }

  return;
}
//****************************************************************************80

void chebyshev1_compute_weights ( int order, double alpha, double beta, 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    CHEBYSHEV1_COMPUTE_WEIGHTS computes Chebyshev type 1 quadrature weights.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Philip Davis, Philip Rabinowitz,
//    Methods of Numerical Integration,
//    Second Edition,
//    Dover, 2007,
//    ISBN: 0486453391,
//    LC: QA299.3.D28.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  int i;
  double pi = 3.141592653589793;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "CHEBYSHEV1_COMPUTE_WEIGHTS - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  for ( i = 0; i < order; i++ )
  {
    w[i] = pi / ( double ) ( order );
  }

  return;
}
//****************************************************************************80

double chebyshev1_integral ( int expon )

//****************************************************************************80
//
//  Purpose:
//
//    CHEBYSHEV1_INTEGRAL evaluates a monomial Chebyshev type 1 integral.
//
//  Discussion:
//
//    To test a Chebyshev type 1 quadrature rule, we use it to approximate the
//    integral of a monomial:
//
//      integral ( -1 <= x <= +1 ) x^n / sqrt ( 1 - x^2 ) dx
//
//    This routine is given the value of the exponent, and returns the
//    exact value of the integral.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int EXPON, the exponent.
//
//    Output, double CHEBYSHEV1_INTEGRAL, the value of the exact integral.
//
{
  double bot;
  double exact;
  int i;
  double pi = 3.141592653589793;
  double top;
//
//  Get the exact value of the integral.
//
  if ( ( expon % 2 ) == 0 )
  {
    top = 1;
    bot = 1;
    for ( i = 2; i <= expon; i = i + 2 )
    {
      top = top * ( i - 1 );
      bot = bot *   i;
    }
	
    exact = pi * ( double ) ( top ) / ( double ) ( bot );
  }
  else
  {
    exact = 0.0;	
  }

  return exact;
}
//****************************************************************************80

void chebyshev2_compute ( int order, double alpha, double beta, double x[], 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    CHEBYSHEV2_COMPUTE computes a Chebyshev type 2 quadrature rule.
//
//  Discussion:
//
//    The integration interval is [ -1, 1 ].
//
//    The weight function is w(x) = sqrt ( 1 - x^2 ).
//
//    The integral to approximate:
//
//      Integral ( -1 <= X <= 1 ) F(X)  sqrt ( 1 - x^2 )  dX
//
//    The quadrature rule:
//
//      Sum ( 1 <= I <= ORDER ) W(I) * F ( X(I) )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Philip Davis, Philip Rabinowitz,
//    Methods of Numerical Integration,
//    Second Edition,
//    Dover, 2007,
//    ISBN: 0486453391,
//    LC: QA299.3.D28.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
//    Output, double W[ORDER], the weights.
//
{
  double angle;
  int i;
  double pi = 3.141592653589793;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "CHEBYSHEV2_COMPUTE - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  for ( i = 0; i < order; i++ )
  {
    angle = pi * ( double ) ( order - i ) / ( double ) ( order + 1 );
    w[i] = pi / ( double ) ( order + 1 ) * std::pow ( std::sin ( angle ), 2 );
    x[i] =  std::cos ( angle );
  }

  if ( ( order % 2 ) == 1 )
  {
    x[(order-1)/2] = 0.0;
  }

  return;
}
//****************************************************************************80

void chebyshev2_compute_points ( int order, double alpha, double beta, 
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    CHEBYSHEV2_COMPUTE_POINTS computes Chebyshev type 2 quadrature points.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Philip Davis, Philip Rabinowitz,
//    Methods of Numerical Integration,
//    Second Edition,
//    Dover, 2007,
//    ISBN: 0486453391,
//    LC: QA299.3.D28.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  double angle;
  int i;
  double pi = 3.141592653589793;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "CHEBYSHEV2_COMPUTE_POINTS - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  for ( i = 0; i < order; i++ )
  {
    angle = pi * ( double ) ( order - i ) / ( double ) ( order + 1 );
    x[i] =  std::cos ( angle );
  }

  if ( ( order % 2 ) == 1 )
  {
    x[(order-1)/2] = 0.0;
  }

  return;
}
//****************************************************************************80

void chebyshev2_compute_weights ( int order, double alpha, double beta, 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    CHEBYSHEV2_COMPUTE_WEIGHTS computes Chebyshev type 2 quadrature weights.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Philip Davis, Philip Rabinowitz,
//    Methods of Numerical Integration,
//    Second Edition,
//    Dover, 2007,
//    ISBN: 0486453391,
//    LC: QA299.3.D28.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  double angle;
  int i;
  double pi = 3.141592653589793;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "CHEBYSHEV2_COMPUTE_WEIGHTS - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  for ( i = 0; i < order; i++ )
  {
    angle = pi * ( double ) ( order - i ) / ( double ) ( order + 1 );
    w[i] = pi / ( double ) ( order + 1 ) * std::pow ( std::sin ( angle ), 2 );
  }

  return;
}
//****************************************************************************80

double chebyshev2_integral ( int expon )

//****************************************************************************80
//
//  Purpose:
//
//    CHEBYSHEV2_INTEGRAL evaluates a monomial Chebyshev type 2 integral.
//
//  Discussion:
//
//    To test a Chebyshev type 2 quadrature rule, we use it to approximate the
//    integral of a monomial:
//
//      integral ( -1 <= x <= +1 ) x^n * sqrt ( 1 - x^2 ) dx
//
//    This routine is given the value of the exponent, and returns the
//    exact value of the integral.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int EXPON, the exponent.
//
//    Output, double CHEBYSHEV2_INTEGRAL, the value of the exact integral.
//
{
  double bot;
  double exact;
  int i;
  double pi = 3.141592653589793;
  double top;
//
//  Get the exact value of the integral.
//
  if ( ( expon % 2 ) == 0 )
  {
    top = 1;
    bot = 1;
    for ( i = 2; i <= expon; i = i + 2 )
    {
      top = top * ( i - 1 );
      bot = bot *   i;
    }

	bot = bot * ( double ) ( expon + 2 );

    exact = pi * ( double ) ( top ) / ( double ) ( bot );
  }
  else
  {
    exact = 0.0;
  }
  return exact;
}
//****************************************************************************80

void clenshaw_curtis_compute ( int order, double alpha, double beta, double x[], 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    CLENSHAW_CURTIS_COMPUTE computes a Clenshaw Curtis quadrature rule.
//
//  Discussion:
//
//    The integration interval is [ -1, 1 ].
//
//    The weight function is w(x) = 1.0.
//
//    The integral to approximate:
//
//      Integral ( -1 <= X <= 1 ) F(X) dX
//
//    The quadrature rule:
//
//      Sum ( 1 <= I <= ORDER ) W(I) * F ( X(I) )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
//    Output, double W[ORDER], the weights.
//
{
  double b;
  int i;
  int j;
  double pi = 3.141592653589793;
  double theta;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "CLENSHAW_CURTIS_COMPUTE - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }
  else if ( order == 1 )
  {
    x[0] = 0.0;
    w[0] = 2.0;
  }
  else
  {
    for ( i = 0; i < order; i++ )
    {
      x[i] =  std::cos ( ( double ) ( order - 1 - i ) * pi 
                       / ( double ) ( order - 1     ) );
    }
    x[0] = -1.0;
    if ( ( order % 2 ) == 1 )
    {
      x[(order-1)/2] = 0.0;
    }
    x[order-1] = +1.0;

    for ( i = 0; i < order; i++ )
    {
      theta = ( double ) ( i ) * pi / ( double ) ( order - 1 );

      w[i] = 1.0;

      for ( j = 1; j <= ( order - 1 ) / 2; j++ )
      {
        if ( 2 * j == ( order - 1 ) )
        {
          b = 1.0;
        }
        else
        {
          b = 2.0;
        }

        w[i] = w[i] - b *  std::cos ( 2.0 * ( double ) ( j ) * theta ) 
          / ( double ) ( 4 * j * j - 1 );
      }
    }

    w[0] = w[0] / ( double ) ( order - 1 );
    for ( i = 1; i < order - 1; i++ )
    {
      w[i] = 2.0 * w[i] / ( double ) ( order - 1 );
    }
    w[order-1] = w[order-1] / ( double ) ( order - 1 );
  }

  return;
}
//****************************************************************************80

void clenshaw_curtis_compute_points ( int order, double alpha, double beta, 
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    CLENSHAW_CURTIS_COMPUTE_POINTS computes Clenshaw Curtis quadrature points.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    This rule is defined on [-1,1].
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  int index;
  double pi = 3.141592653589793;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "CLENSHAW_CURTIS_COMPUTE_POINTS - Fatal error!\n";
    std::cerr << "  ORDER < 1.\n";
    std::exit ( 1 );
  }
  else if ( order == 1 )
  {
    x[0] = 0.0;
  }
  else
  {
    for ( index = 1; index <= order; index++ )
    {
      x[index-1] =  std::cos ( ( double ) ( order - index ) * pi 
                             / ( double ) ( order - 1     ) );
    }
    x[0] = -1.0;
    if ( ( order % 2 ) == 1 )
    {
      x[(order-1)/2] = 0.0;
    }
    x[order-1] = +1.0;
  }
  return;
}
//****************************************************************************80

void clenshaw_curtis_compute_weights ( int order, double alpha, double beta,
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    CLENSHAW_CURTIS_COMPUTE_WEIGHTS computes Clenshaw Curtis quadrature weights.
//
//  Discussion:
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Charles Clenshaw, Alan Curtis,
//    A Method for Numerical Integration on an Automatic Computer,
//    Numerische Mathematik,
//    Volume 2, Number 1, December 1960, pages 197-205.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights of the rule.
//
{
  double b;
  int i;
  int j;
  double pi = 3.141592653589793;
  double theta;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "CLENSHAW_CURTIS_COMPUTE_WEIGHTS - Fatal error!\n";
    std::cerr << "  ORDER < 1.\n";
    std::exit ( 1 );
  }
  else if ( order == 1 )
  {
    w[0] = 2.0;
    return;
  }

  for ( i = 1; i <= order; i++ )
  {
    theta = ( double ) ( i - 1 ) * pi / ( double ) ( order - 1 );

    w[i-1] = 1.0;

    for ( j = 1; j <= ( order - 1 ) / 2; j++ )
    {
      if ( 2 * j == ( order - 1 ) )
      {
        b = 1.0;
      }
      else
      {
        b = 2.0;
      }

      w[i-1] = w[i-1] - b *  std::cos ( 2.0 * ( double ) ( j ) * theta ) 
           / ( double ) ( 4 * j * j - 1 );
    }
  }

  w[0] = w[0] / ( double ) ( order - 1 );
  for ( i = 1; i < order - 1; i++ )
  {
    w[i] = 2.0 * w[i] / ( double ) ( order - 1 );
  }
  w[order-1] = w[order-1] / ( double ) ( order - 1 );

  return;
}
//****************************************************************************80

void comp_next ( int n, int k, int a[], bool *more, int *h, int *t )

//****************************************************************************80
//
//  Purpose:
//
//    COMP_NEXT computes the compositions of the integer N into K parts.
//
//  Discussion:
//
//    A composition of the integer N into K parts is an ordered sequence
//    of K nonnegative integers which sum to N.  The compositions (1,2,1)
//    and (1,1,2) are considered to be distinct.
//
//    The routine computes one composition on each call until there are no more.
//    For instance, one composition of 6 into 3 parts is
//    3+2+1, another would be 6+0+0.
//
//    On the first call to this routine, set MORE = FALSE.  The routine
//    will compute the first element in the sequence of compositions, and
//    return it, as well as setting MORE = TRUE.  If more compositions
//    are desired, call again, and again.  Each time, the routine will
//    return with a new composition.
//
//    However, when the LAST composition in the sequence is computed 
//    and returned, the routine will reset MORE to FALSE, signaling that
//    the end of the sequence has been reached.
//
//    This routine originally used a SAVE statement to maintain the
//    variables H and T.  I have decided that it is safer
//    to pass these variables as arguments, even though the user should
//    never alter them.  This allows this routine to safely shuffle
//    between several ongoing calculations.
//
//
//    There are 28 compositions of 6 into three parts.  This routine will
//    produce those compositions in the following order:
//
//     I         A
//     -     ---------
//     1     6   0   0
//     2     5   1   0
//     3     4   2   0
//     4     3   3   0
//     5     2   4   0
//     6     1   5   0
//     7     0   6   0
//     8     5   0   1
//     9     4   1   1
//    10     3   2   1
//    11     2   3   1
//    12     1   4   1
//    13     0   5   1
//    14     4   0   2
//    15     3   1   2
//    16     2   2   2
//    17     1   3   2
//    18     0   4   2
//    19     3   0   3
//    20     2   1   3
//    21     1   2   3
//    22     0   3   3
//    23     2   0   4
//    24     1   1   4
//    25     0   2   4
//    26     1   0   5
//    27     0   1   5
//    28     0   0   6
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    02 July 2008
//
//  Author:
//
//    Original FORTRAN77 version by Albert Nijenhuis, Herbert Wilf.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Albert Nijenhuis, Herbert Wilf,
//    Combinatorial Algorithms for Computers and Calculators,
//    Second Edition,
//    Academic Press, 1978,
//    ISBN: 0-12-519260-6,
//    LC: QA164.N54.
//
//  Parameters:
//
//    Input, int N, the integer whose compositions are desired.
//
//    Input, int K, the number of parts in the composition.
//
//    Input/output, int A[K], the parts of the composition.
//
//    Input/output, bool *MORE.
//    Set MORE = FALSE on first call.  It will be reset to TRUE on return
//    with a new composition.  Each new call returns another composition until
//    MORE is set to FALSE when the last composition has been computed
//    and returned.
//
//    Input/output, int *H, *T, two internal parameters needed for the
//    computation.  The user should allocate space for these in the calling
//    program, include them in the calling sequence, but never alter them!
//
{
  int i;

  if ( !( *more ) )
  {
    *t = n;
    *h = 0;
    a[0] = n;
    for ( i = 1; i < k; i++ )
    {
       a[i] = 0;
    }
  }
  else
  {
    if ( 1 < *t )
    {
      *h = 0;
    }
    *h = *h + 1;
    *t = a[*h-1];
    a[*h-1] = 0;
    a[0] = *t - 1;
    a[*h] = a[*h] + 1;
  }

  *more = ( a[k-1] != n );

  return;
}
//****************************************************************************80

void fejer2_compute ( int order, double alpha, double beta, double x[], 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    FEJER2_COMPUTE computes a Fejer type 2 rule.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on [-1,1].
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the Fejer type 2 rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
//    Output, double W[ORDER], the weights.
//
{
  int i;
  int j;
  double p;
  double pi = 3.141592653589793;
  double theta;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "FEJER2_COMPUTE - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }
  else if ( order == 1 )
  {
    x[0] = 0.0;
    w[0] = 2.0;
    return;
  }

  for ( i = 0; i < order; i++ )
  {
    x[i] =  std::cos ( ( double ) ( order - i ) * pi 
                     / ( double ) ( order + 1 ) );
  }
  if ( ( order % 2 ) == 1 )
  {
    x[(order-1)/2] = 0.0;
  }

  if ( order == 2 )
  {
    w[0] = 1.0;
    w[1] = 1.0;
  }
  else
  {
    for ( i = 0; i < order; i++ )
    {
      theta = ( double ) ( order - i ) * pi 
            / ( double ) ( order + 1 );

      w[i] = 1.0;

      for ( j = 1; j <= ( ( order - 1 ) / 2 ); j++ )
      {
        w[i] = w[i] - 2.0 *  std::cos ( 2.0 * ( double ) ( j ) * theta ) 
          / ( double ) ( 4 * j * j - 1 );
      }
      p = 2.0 * ( double ) ( ( ( order + 1 ) / 2 ) ) - 1.0;
      w[i] = w[i] -  std::cos ( ( p + 1.0 ) * theta ) / p;
    }
    for ( i = 0; i < order; i++ )
    {
      w[i] = 2.0 * w[i] / ( double ) ( order + 1 );
    }
  }

  return;
}
//****************************************************************************80

void fejer2_compute_points ( int order, double alpha, double beta, double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    FEJER2_COMPUTE_POINTS computes Fejer type 2 quadrature points.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on [-1,1].
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the Fejer type 2 rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  int index;
  double pi = 3.141592653589793;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "FEJER2_COMPUTE_POINTS - Fatal error!\n";
    std::cerr << "  ORDER < 1.\n";
    std::exit ( 1 );
  }
  else if ( order == 1 )
  {
    x[0] = 0.0;
  }
  else
  {
    for ( index = 1; index <= order; index++ )
    {
      x[index-1] =  std::cos ( ( double ) ( order + 1 - index ) * pi 
                             / ( double ) ( order + 1 ) );
    }
    if ( ( order % 2 ) == 1 )
    {
      x[(order-1)/2] = 0.0;
    }
  }
  return;
}
//****************************************************************************80

void fejer2_compute_weights ( int order, double alpha, double beta, double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    FEJER2_COMPUTE_WEIGHTS computes Fejer type 2 quadrature weights.
//
//  Discussion:
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Philip Davis, Philip Rabinowitz,
//    Methods of Numerical Integration,
//    Second Edition,
//    Dover, 2007,
//    ISBN: 0486453391,
//    LC: QA299.3.D28.
//
//    Walter Gautschi,
//    Numerical Quadrature in the Presence of a Singularity,
//    SIAM Journal on Numerical Analysis,
//    Volume 4, Number 3, 1967, pages 357-362.
//
//    Joerg Waldvogel,
//    Fast Construction of the Fejer and Clenshaw-Curtis Quadrature Rules,
//    BIT Numerical Mathematics,
//    Volume 43, Number 1, 2003, pages 1-18.
//
//  Parameters:
//
//    Input, int ORDER, the order.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  int i;
  int j;
  double p;
  double pi = 3.141592653589793;
  double theta;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "FEJER2_COMPUTE_WEIGHTS - Fatal error!\n";
    std::cerr << "  ORDER < 1.\n";
    std::exit ( 1 );
  }
  else if ( order == 1 )
  {
    w[0] = 2.0;
  }
  else if ( order == 2 )
  {
    w[0] = 1.0;
    w[1] = 1.0;
  }
  else
  {
    for ( i = 1; i <= order; i++ )
    {
      theta = ( double ) ( order + 1 - i ) * pi 
            / ( double ) ( order + 1 );

      w[i-1] = 1.0;

      for ( j = 1; j <= ( ( order - 1 ) / 2 ); j++ )
      {
        w[i-1] = w[i-1] - 2.0 *  std::cos ( 2.0 * ( double ) ( j ) * theta ) 
          / ( double ) ( 4 * j * j - 1 );
      }
      p = 2.0 * ( double ) ( ( ( order + 1 ) / 2 ) ) - 1.0;
      w[i-1] = w[i-1] -  std::cos ( ( p + 1.0 ) * theta ) / p;
    }
    for ( i = 0; i < order; i++ )
    {
      w[i] = 2.0 * w[i] / ( double ) ( order + 1 );
    }
  }
  return;
}
//****************************************************************************80

void gegenbauer_compute ( int order, double alpha, double beta, double x[], 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEGENBAUER_COMPUTE computes a Gegenbauer quadrature rule.
//
//  Discussion:
//
//    The integral to approximate:
//
//      Integral ( -1 <= X <= 1 ) (1-X^2)^ALPHA * F(X) dX
//
//    The quadrature rule:
//
//      Sum ( 1 <= I <= ORDER ) W(I) * F ( X(I) )
//
//    Thanks to Janiki Raman for pointing out a problem in an earlier
//    version of the code that occurred when ALPHA was -0.5.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, the exponent of (1-X^2).  -1.0 < ALPHA is required.
//
//    Input, double BETA, a parameter which is not needed by this
//    function, but which is needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
//    Output, double W[ORDER], the weights.
//
{
  double an;
  double *c;
  double cc;
  double delta;
  double dp2;
  int i;
  double p1;
  double prod;
  double r1;
  double r2;
  double r3;
  double temp;
  double x0;
//
//  Check ORDER.
//
  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "GEGENBAUER_COMPUTE - Fatal error!\n";
    std::cerr << "  1 <= ORDER is required.\n";
    std::exit ( 1 );
  }
  c = new double[order];
//
//  Check ALPHA.
//
  if ( alpha <= -1.0 )
  {
    std::cerr << "\n";
    std::cerr << "GEGENBAUER_COMPUTE - Fatal error!\n";
    std::cerr << "  -1.0 < ALPHA is required.\n";
    std::exit ( 1 );
  }
//
//  Set the recursion coefficients.
//
  c[0] = 0.0;
  if ( 2 <= order )
  {
    c[1] = 1.0 / ( 2.0 * alpha + 3.0 );
  }

  for ( i = 3; i <= order; i++ )
  {
    c[i-1] = ( double ) ( i - 1 ) 
          * ( alpha + alpha + ( double ) ( i - 1 ) ) / 
          ( ( alpha + alpha + ( double ) ( 2 * i - 1 ) ) 
          * ( alpha + alpha + ( double ) ( 2 * i - 3 ) ) );
  }

  delta = webbur::r8_gamma ( alpha         + 1.0 ) 
        * webbur::r8_gamma (         alpha + 1.0 ) 
        / webbur::r8_gamma ( alpha + alpha + 2.0 );

  prod = 1.0;
  for ( i = 2; i <= order; i++ )
  {
    prod = prod * c[i-1];
  }
  cc = delta * std::pow ( 2.0, alpha + alpha + 1.0 ) * prod;

  for ( i = 1; i <= order; i++ )
  {
    if ( i == 1 )
    {
      an = alpha / ( double ) ( order );

      r1 = ( 1.0 + alpha ) 
        * ( 2.78 / ( 4.0 + ( double ) ( order * order ) ) 
        + 0.768 * an / ( double ) ( order ) );

      r2 = 1.0 + 2.44 * an + 1.282 * an * an;

      x0 = ( r2 - r1 ) / r2;
    }
    else if ( i == 2 )
    {
      r1 = ( 4.1 + alpha ) / 
        ( ( 1.0 + alpha ) * ( 1.0 + 0.156 * alpha ) );

      r2 = 1.0 + 0.06 * ( ( double ) ( order ) - 8.0 ) * 
        ( 1.0 + 0.12 * alpha ) / ( double ) ( order );

      r3 = 1.0 + 0.012 * alpha * 
        ( 1.0 + 0.25 * r8_abs ( alpha ) ) / ( double ) ( order );

      x0 = x0 - r1 * r2 * r3 * ( 1.0 - x0 );
    }
    else if ( i == 3 )
    {
      r1 = ( 1.67 + 0.28 * alpha ) / ( 1.0 + 0.37 * alpha );

      r2 = 1.0 + 0.22 * ( ( double ) ( order ) - 8.0 ) 
        / ( double ) ( order );

      r3 = 1.0 + 8.0 * alpha / 
        ( ( 6.28 + alpha ) * ( double ) ( order * order ) );

      x0 = x0 - r1 * r2 * r3 * ( x[0] - x0 );
    }
    else if ( i < order - 1 )
    {
      x0 = 3.0 * x[i-2] - 3.0 * x[i-3] + x[i-4];
    }
    else if ( i == order - 1 )
    {
      r1 = ( 1.0 + 0.235 * alpha ) / ( 0.766 + 0.119 * alpha );

      r2 = 1.0 / ( 1.0 + 0.639 
        * ( ( double ) ( order ) - 4.0 ) 
        / ( 1.0 + 0.71 * ( ( double ) ( order ) - 4.0 ) ) );

      r3 = 1.0 / ( 1.0 + 20.0 * alpha / ( ( 7.5 + alpha ) * 
        ( double ) ( order * order ) ) );

      x0 = x0 + r1 * r2 * r3 * ( x0 - x[i-3] );
    }
    else if ( i == order )
    {
      r1 = ( 1.0 + 0.37 * alpha ) / ( 1.67 + 0.28 * alpha );

      r2 = 1.0 / 
        ( 1.0 + 0.22 * ( ( double ) ( order ) - 8.0 ) 
        / ( double ) ( order ) );

      r3 = 1.0 / ( 1.0 + 8.0 * alpha / 
        ( ( 6.28 + alpha ) * ( double ) ( order * order ) ) );

      x0 = x0 + r1 * r2 * r3 * ( x0 - x[i-3] );
    }

    webbur::gegenbauer_root ( &x0, order, alpha, &dp2, &p1, c );

    x[i-1] = x0;
    w[i-1] = cc / ( dp2 * p1 );
  }
//
//  Reverse the order of the values.
//
  for ( i = 1; i <= order/2; i++ )
  {
    temp       = x[i-1];
    x[i-1]     = x[order-i];
    x[order-i] = temp;
  }

  for ( i = 1; i <=order/2; i++ )
  {
    temp       = w[i-1];
    w[i-1]     = w[order-i];
    w[order-i] = temp;
  }

  delete [] c;

  return;
}
//****************************************************************************80

void gegenbauer_compute_points ( int order, double alpha, double beta, 
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEGENBAUER_COMPUTE_POINTS computes Gegenbauer quadrature points.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, the exponent of (1-X^2).  -1.0 < ALPHA is required.
//
//    Input, double BETA, a parameter which is not needed by this
//    function, but which is needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  double *w;

  w = new double[order];

  gegenbauer_compute ( order, alpha, beta, x, w );

  delete [] w;

  return;
}
//****************************************************************************80

void gegenbauer_compute_weights ( int order, double alpha, double beta,
   double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEGENBAUER_COMPUTE_WEIGHTS computes Gegenbauer quadrature weights.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, the exponent of (1-X^2).  -1.0 < ALPHA is required.
//
//    Input, double BETA, a parameter which is not needed by this
//    function, but which is needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  double *x;

  x = new double[order];

  gegenbauer_compute ( order, alpha, beta, x, w );

  delete [] x;

  return;
}
//****************************************************************************80

double gegenbauer_integral ( int expon, double alpha )

//****************************************************************************80
//
//  Purpose:
//
//    GEGENBAUER_INTEGRAL integrates a monomial with Gegenbauer weight.
//
//  Discussion:
//
//    VALUE = Integral ( -1 <= X <= +1 ) x^EXPON (1-x^2)^ALPHA dx
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int EXPON, the exponent.
//
//    Input, double ALPHA, the exponent of (1-X^2) in the weight factor.
//
//    Output, double GEGENBAUER_INTEGRAL, the value of the integral.
//
{
  double arg1;
  double arg2;
  double arg3;
  double arg4;
  double c;
  double s;
  double value;
  double value1;

  if ( ( expon % 2 ) == 1 )
  {
    value = 0.0;
    return value;
  }

  c = ( double ) ( expon );

  arg1 = - alpha;
  arg2 =   1.0 + c;
  arg3 =   2.0 + alpha + c;
  arg4 = - 1.0;

  value1 = webbur::r8_hyper_2f1 ( arg1, arg2, arg3, arg4 );

  value = webbur::r8_gamma ( 1.0 + c ) * 2.0 
    * webbur::r8_gamma ( 1.0 + alpha  ) * value1 
    / webbur::r8_gamma ( 2.0 + alpha  + c );

  return value;
}
//****************************************************************************80

void gegenbauer_recur ( double *p2, double *dp2, double *p1, double x, 
  int order, double alpha, double c[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEGENBAUER_RECUR evaluates a Gegenbauer polynomial.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Output, double *P2, the value of J(ORDER)(X).
//
//    Output, double *DP2, the value of J'(ORDER)(X).
//
//    Output, double *P1, the value of J(ORDER-1)(X).
//
//    Input, double X, the point at which polynomials are evaluated.
//
//    Input, int ORDER, the order of the polynomial to be computed.
//
//    Input, double ALPHA, the exponents of (1-X^2).
//
//    Input, double C[ORDER], the recursion coefficients.
//
{
  double dp0;
  double dp1;
  int i;
  double p0;

  *p1 = 1.0;
  dp1 = 0.0;

  *p2 = x;
  *dp2 = 1.0;

  for ( i = 2; i <= order; i++ )
  {
    p0 = *p1;
    dp0 = dp1;

    *p1 = *p2;
    dp1 = *dp2;

    *p2 = x *  ( *p1 ) - c[i-1] * p0;
    *dp2 = x * dp1 + ( *p1 ) - c[i-1] * dp0;
  }
  return;
}
//****************************************************************************80

void gegenbauer_root ( double *x, int order, double alpha,  double *dp2, 
  double *p1, double c[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEGENBAUER_ROOT improves an approximate root of a Gegenbauer polynomial.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input/output, double *X, the approximate root, which
//    should be improved on output.
//
//    Input, int ORDER, the order of the polynomial to be computed.
//
//    Input, double ALPHA, the exponents of (1-X^2).
//
//    Output, double *DP2, the value of J'(ORDER)(X).
//
//    Output, double *P1, the value of J(ORDER-1)(X).
//
//    Input, double C[ORDER], the recursion coefficients.
//
{
  double d;
  double eps;
  double p2;
  int step;
  int step_max = 10;

  eps = webbur::r8_epsilon ( );

  for ( step = 1; step <= step_max; step++ )
  {
    webbur::gegenbauer_recur ( &p2, dp2, p1, *x, order, alpha, c );

    d = p2 / ( *dp2 );
    *x = *x - d;

    if ( webbur::r8_abs ( d ) <= eps * ( webbur::r8_abs ( *x ) + 1.0 ) )
    {
      return;
    }
  }
  return;
}
//****************************************************************************80

void gen_hermite_compute ( int order, double alpha, double beta, double x[], 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEN_HERMITE_COMPUTE computes a Generalized Hermite quadrature rule.
//
//  Discussion:
//
//    The integral to be approximated has the form:
//
//      Integral ( -oo < x < +oo ) x^ALPHA exp(-x^2) f(x) dx
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Philip Davis, Philip Rabinowitz,
//    Methods of Numerical Integration,
//    Second Edition,
//    Dover, 2007,
//    ISBN: 0486453391,
//    LC: QA299.3.D28.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, the exponent of the X factor.
//    -1.0 < ALPHA.
//
//    Input, double BETA, a parameter which is not needed by this
//    function, but which is needed by certain quadrature rules.
//
//    Output, double X[ORDER], W[ORDER], the abscissas and weights of the rule.
//
{
  double alpha_laguerre;
  double arg;
  int i;
  int order_laguerre;
  double *w_laguerre;
  double *x_laguerre;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "GEN_HERMITE_COMPUTE - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  if ( order == 1 )
  {
    arg = ( alpha + 1.0 ) / 2.0;
    x[0] = 0.0;
    w[0] = webbur::r8_gamma ( arg );
    return;
  }

  if ( ( order % 2 ) == 0 ) 
  {
    order_laguerre = order / 2;
    alpha_laguerre = ( alpha - 1.0 ) / 2.0;
  }
  else
  {
    order_laguerre = ( order - 1 ) / 2;
    alpha_laguerre = ( alpha + 1.0 ) / 2.0;
  }
  
  w_laguerre = new double[order_laguerre];
  x_laguerre = new double[order_laguerre];

  webbur::gen_laguerre_compute ( order_laguerre, alpha_laguerre, beta,
    x_laguerre, w_laguerre );

  if ( ( order % 2 ) == 0 )
  {
    for ( i = 0; i < order_laguerre; i++ )
    {
      x[i] = - std::sqrt ( x_laguerre[order_laguerre-1-i] );
    }
    for ( i = 0; i < order_laguerre; i++ )
    {
      x[order_laguerre+i] = std::sqrt ( x_laguerre[i] );
	}
    for ( i = 0; i < order_laguerre; i++ )
    {
      w[i] = 0.5 * w_laguerre[order_laguerre-1-i];
    }
    for ( i = 0; i < order_laguerre; i++ )
    {
      w[order_laguerre+i] = 0.5 * w_laguerre[i];
    }
  }
  else if ( ( order % 2 ) == 1 )
  {
    for ( i = 0; i < order_laguerre; i++ )
    {
      x[i] = - std::sqrt ( x_laguerre[order_laguerre-1-i] );
    }
    x[order_laguerre] = 0.0;
    for ( i = 0; i < order_laguerre; i++ )
    {
      x[order_laguerre+1+i] = std::sqrt ( x_laguerre[i] );
	}
    for ( i = 0; i < order_laguerre; i++ )
    {
      w[i] = 0.5 * w_laguerre[order_laguerre-1-i] / x_laguerre[order_laguerre-1-i];
    }

    arg = ( alpha + 1.0 ) / 2.0;
    w[order_laguerre] = webbur::r8_gamma ( arg );
    for ( i = 0; i < order_laguerre; i++ )
    {
      w[order_laguerre] = w[order_laguerre] - w_laguerre[i] / x_laguerre[i];
    }

    for ( i = 0; i < order_laguerre; i++ )
    {
      w[order_laguerre+1+i] = 0.5 * w_laguerre[i] / x_laguerre[i];
    }
  }
  delete [] w_laguerre;
  delete [] x_laguerre;

  return;
}
//****************************************************************************80

void gen_hermite_compute_points ( int order, double alpha, double beta,
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEN_HERMITE_COMPUTE_POINTS computes Generalized Hermite quadrature points.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, the exponent of the X factor.
//    -1.0 < ALPHA.
//
//    Input, double BETA, a parameter which is not needed by this
//    function, but which is needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  double *w;

  w = new double[order];

  webbur::gen_hermite_compute ( order, alpha, beta, x, w );

  delete [] w;

  return;
}
//****************************************************************************80

void gen_hermite_compute_weights ( int order, double alpha, double beta,
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEN_HERMITE_COMPUTE_WEIGHTS computes Generalized Hermite quadrature weights.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, the exponent of the X factor.
//    -1.0 < ALPHA.
//
//    Input, double BETA, a parameter which is not needed by this
//    function, but which is needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  double *x;

  x = new double[order];

  webbur::gen_hermite_compute ( order, alpha, beta, x, w );

  delete [] x;

  return;
}
//****************************************************************************80

double gen_hermite_integral ( int expon, double alpha )

//****************************************************************************80
//
//  Purpose:
//
//    GEN_HERMITE_INTEGRAL evaluates a monomial Generalized Hermite integral.
//
//  Discussion:
//
//    H(n,alpha) = Integral ( -oo < x < +oo ) x^n |x|^alpha exp(-x^2) dx
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int EXPON, the exponent of the monomial.
//    0 <= EXPON.
//
//    Input, double ALPHA, the exponent of |X| in the weight function.
//    -1.0 < ALPHA.
//
//    Output, double GEN_HERMITE_INTEGRAL, the value of the integral.
//
{
  double a;
  double arg;
  double value;

  if ( ( expon % 2 ) == 1 )
  {
    value = 0.0;
  }
  else
  {
    a = alpha + ( double ) ( expon );
    if ( a <= - 1.0 )
    {
      value = - webbur::r8_huge ( );
    }
    else
    {
      arg = ( a + 1.0 ) / 2.0;
      value = webbur::r8_gamma ( arg );
    }
  }
  return value;
}
//****************************************************************************80

void gen_laguerre_compute ( int order, double alpha, double beta, double x[], 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEN_LAGUERRE_COMPUTE computes a Generalized Laguerre quadrature rule.
//
//  Discussion:
//
//    In the simplest case, ALPHA is 0, and we are approximating the
//    integral from 0 to +oo of exp(-X) * F(X).  When this is so,
//    it is easy to modify the rule to approximate the integral from
//    A to +oo as well.
//
//    If ALPHA is nonzero, then there is no simple way to extend the
//    rule to approximate the integral from A to +oo.  The simplest
//    procedures would be to approximate the integral from 0 to A.
//
//    The integration interval is [ A, +oo ) or [ 0, +oo ).
//
//    The weight function is w(x) = exp ( -x ) or exp ( -x ) * x^alpha.
//
//
//    If the integral to approximate is:
//
//        Integral ( A <= X < +oo ) exp ( - X ) * F(X) dX
//      or
//        Integral ( 0 <= X < +oo ) exp ( - X ) * X^ALPHA * F(X) dX
//
//    then the quadrature rule is:
//
//      exp ( - A ) * Sum ( 1 <= I <= ORDER ) W(I) * F ( A+X(I) )
//    or
//      Sum ( 1 <= I <= ORDER ) W(I) * F ( X(I) )
//
//
//    If the integral to approximate is:
//
//        Integral ( A <= X < +oo ) F(X) dX
//      or
//        Integral ( 0 <= X < +oo ) X^ALPHA * F(X) dX
//
//    then the quadrature rule is:
//
//      exp ( - A ) * Sum ( 1 <= I <= ORDER ) 
//        W(I) * exp(A+X(I)) * F ( A+X(I) )
//    or
//      Sum ( 1 <= I <= ORDER ) W(I) * exp(X(I)) * F ( X(I) )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the quadrature rule to be computed.
//    1 <= ORDER.
//
//    Input, double ALPHA, the exponent of the X factor.
//    Set ALPHA = 0.0 for the simplest rule.
//    ALPHA must be nonnegative.
//
//    Input, double BETA, a parameter which is not needed by this
//    function, but which is needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
//    Output, double W[ORDER], the weights.
//
{
  double *b;
  double *c;
  double cc;
  double dp2;
  int i;
  double p1;
  double prod;
  double r1;
  double r2;
  double ratio;
  double x0;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "GEN_LAGUERRE_COMPUTE - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  b = new double[order];
  c = new double[order];
//
//  Set the recursion coefficients.
//
  for ( i = 0; i < order; i++ )
  {
    b[i] = ( alpha + ( double ) ( 2 * i + 1 ) );
  }

  for ( i = 0; i < order; i++ )
  {
    c[i] = ( double ) ( i ) * ( alpha + ( double ) ( i ) );
  }
  prod = 1.0;
  for ( i = 1; i < order; i++ )
  {
    prod = prod * c[i];
  }
  cc = webbur::r8_gamma ( alpha + 1.0 ) * prod;

  for ( i = 0; i < order; i++ )
  {
//
//  Compute an estimate for the root.
//
    if ( i == 0 )
    {
      x0 = ( 1.0 + alpha ) * ( 3.0+ 0.92 * alpha ) / 
        ( 1.0 + 2.4 * ( double ) ( order ) + 1.8 * alpha );
    }
    else if ( i == 1 )
    {
      x0 = x0 + ( 15.0 + 6.25 * alpha ) / 
        ( 1.0 + 0.9 * alpha + 2.5 * ( double ) ( order ) );
    }
    else
    {
      r1 = ( 1.0 + 2.55 * ( double ) ( i - 1 ) ) 
        / ( 1.9 * ( double ) ( i - 1 ) );

      r2 = 1.26 * ( double ) ( i - 1 ) * alpha / 
        ( 1.0 + 3.5 * ( double ) ( i - 1 ) );

      ratio = ( r1 + r2 ) / ( 1.0 + 0.3 * alpha );

      x0 = x0 + ratio * ( x0 - x[i-2] );
    }
//
//  Use iteration to find the root.
//
    webbur::gen_laguerre_root ( &x0, order, alpha, &dp2, &p1, b, c );
//
//  Set the abscissa and weight.
//
    x[i] = x0;
    w[i] = ( cc / dp2 ) / p1;
  }

  delete [] b;
  delete [] c;

  return;
}
//****************************************************************************80

void gen_laguerre_compute_points ( int order, double alpha, double beta,
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEN_LAGUERRE_COMPUTE_POINTS: Generalized Laguerre quadrature points.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, the exponent of the X factor.
//    Set ALPHA = 0.0 for the simplest rule.
//    ALPHA must be nonnegative.
//
//    Input, double BETA, a parameter which is not needed by this
//    function, but which is needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  double *w;

  w = new double[order];

  webbur::gen_laguerre_compute ( order, alpha, beta, x, w );

  delete [] w;

  return;
}
//****************************************************************************80

void gen_laguerre_compute_weights ( int order, double alpha, double beta,
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEN_LAGUERRE_COMPUTE_WEIGHTS: Generalized Laguerre quadrature weights.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, the exponent of the X factor.
//    Set ALPHA = 0.0 for the simplest rule.
//    ALPHA must be nonnegative.
//
//    Input, double BETA, a parameter which is not needed by this
//    function, but which is needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  double *x;

  x = new double[order];

  webbur::gen_laguerre_compute ( order, alpha, beta, x, w );

  delete [] x;

  return;
}
//****************************************************************************80

double gen_laguerre_integral ( int expon, double alpha )

//****************************************************************************80
//
//  Purpose:
//
//    GEN_LAGUERRE_INTEGRAL evaluates a monomial Generalized Laguerre integral.
//
//  Discussion:
//
//    L(n,alpha) = Integral ( 0 <= x < +oo ) x^n * x^alpha exp(-x) dx
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    20 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int EXPON, the exponent of the monomial.
//    0 <= EXPON.
//
//    Input, double ALPHA, the exponent of X in the weight function.
//    -1.0 < ALPHA.
//
//    Output, double GEN_LAGUERRE_INTEGRAL, the value of the integral.
//
{
  double arg;
  double value;

  arg = alpha + ( double ) ( expon + 1.0 );
  value = webbur::r8_gamma ( arg );

  return value;
}
//****************************************************************************80

void gen_laguerre_recur ( double *p2, double *dp2, double *p1, double x, 
  int order, double alpha, double b[], double c[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEN_LAGUERRE_RECUR evaluates a Generalized Laguerre polynomial.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Output, double *P2, the value of L(ORDER)(X).
//
//    Output, double *DP2, the value of L'(ORDER)(X).
//
//    Output, double *P1, the value of L(ORDER-1)(X).
//
//    Input, double X, the point at which polynomials are evaluated.
//
//    Input, int ORDER, the order of the polynomial to be computed.
//
//    Input, double ALPHA, the exponent of the X factor in the
//    integrand.
//
//    Input, double B[ORDER], C[ORDER], the recursion coefficients.
//
{
  double dp0;
  double dp1;
  int i;
  double p0;

  *p1 = 1.0;
  dp1 = 0.0;

  *p2 = x - alpha - 1.0;
  *dp2 = 1.0;

  for ( i = 1; i < order; i++ )
  {
    p0 = *p1;
    dp0 = dp1;

    *p1 = *p2;
    dp1 = *dp2;

    *p2  = ( x - b[i] ) * ( *p1 ) - c[i] * p0;
    *dp2 = ( x - b[i] ) * dp1 + ( *p1 ) - c[i] * dp0;
  }

  return;
}
//****************************************************************************80

void gen_laguerre_root ( double *x, int order, double alpha, double *dp2, 
  double *p1, double b[], double c[] )

//****************************************************************************80
//
//  Purpose:
//
//    GEN_LAGUERRE_ROOT improves a root of a Generalized Laguerre polynomial.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input/output, double *X, the approximate root, which
//    should be improved on output.
//
//    Input, int ORDER, the order of the polynomial to be computed.
//
//    Input, double ALPHA, the exponent of the X factor.
//
//    Output, double *DP2, the value of L'(ORDER)(X).
//
//    Output, double *P1, the value of L(ORDER-1)(X).
//
//    Input, double B[ORDER], C[ORDER], the recursion coefficients.
//
{
  double d;
  double eps;
  double p2;
  int step;
  int step_max = 10;

  eps = webbur::r8_epsilon ( );

  for ( step = 1; step <= step_max; step++ )
  {
    webbur::gen_laguerre_recur ( &p2, dp2, p1, *x, order, alpha, b, c );

    d = p2 / ( *dp2 );
    *x = *x - d;

    if ( webbur::r8_abs ( d ) <= eps * ( webbur::r8_abs ( *x ) + 1.0 ) )
    {
      break;
    }
  }

  return;
}
//****************************************************************************80

void hermite_compute ( int order, double alpha, double beta, double x[], 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    HERMITE_COMPUTE computes a Hermite quadrature rule.
//
//  Discussion:
//
//    The abscissas are the zeros of the N-th order Hermite polynomial.
//
//    The integration interval is ( -oo, +oo ).
//
//    The weight function is w(x) = exp ( - x * x ).
//
//    The integral to approximate:
//
//      Integral ( -oo < X < +oo ) exp ( - X * X ) * F(X) dX
//
//    The quadrature rule:
//
//      Sum ( 1 <= I <= ORDER ) W(I) * F ( X(I) )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
//    Output, double W[ORDER], the weights.
//
{
  double cc;
  double dp2;
  int i;
  double p1;
  double s;
  double temp;
  double x0;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "HERMITE_COMPUTE - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  cc = 1.7724538509 * webbur::r8_gamma ( ( double ) ( order ) ) 
    / std::pow ( 2.0, order - 1 );

  s = std::pow ( 2.0 * ( double ) ( order ) + 1.0, 1.0 / 6.0 );

  for ( i = 0; i < ( order + 1 ) / 2; i++ )
  {
    if ( i == 0 )
    {
      x0 = s * s * s - 1.85575 / s;
    }
    else if ( i == 1 )
    {
      x0 = x0 - 1.14 * std::pow ( ( double ) ( order ), 0.426 ) / x0;
    }
    else if ( i == 2 )
    {
      x0 = 1.86 * x0 - 0.86 * x[0];
    }
    else if ( i == 3 )
    {
      x0 = 1.91 * x0 - 0.91 * x[1];
    }
    else
    {
      x0 = 2.0 * x0 - x[i-2];
    }

    webbur::hermite_root ( &x0, order, &dp2, &p1 );

    x[i] = x0;
    w[i] = ( cc / dp2 ) / p1;

    x[order-i-1] = -x0;
    w[order-i-1] = w[i];
  }
//
//  Reverse the order of the abscissas.
//
  for ( i = 1; i <= order/2; i++ )
  {
    temp       = x[i-1];
    x[i-1]     = x[order-i];
    x[order-i] = temp;
  }

  if ( ( order % 2 ) == 1 )
  {
    x[(order-1)/2] = 0.0;
  }

  return;
}
//****************************************************************************80

void hermite_compute_points ( int order, double alpha, double beta, 
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    HERMITE_COMPUTE_POINTS computes Hermite quadrature points.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  double *w;

  w = new double[order];

  webbur::hermite_compute ( order, alpha, beta, x, w );

  delete [] w;

  return;
}
//****************************************************************************80

void hermite_compute_weights ( int order, double alpha, double beta, 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    HERMITE_COMPUTE_WEIGHTS computes Hermite quadrature weights.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  double *x;

  x = new double[order];

  webbur::hermite_compute ( order, alpha, beta, x, w );

  delete [] x;

  return;
}
//****************************************************************************80

double hermite_integral ( int n )

//****************************************************************************80
//
//  Purpose:
//
//    HERMITE_INTEGRAL evaluates a monomial Hermite integral.
//
//  Discussion:
//
//    H(n) = Integral ( -oo < x < +oo ) x^n exp(-x^2) dx
//
//    H(n) is 0 for n odd.
//
//    H(n) = (n-1)!! * sqrt(pi) / 2^(n/2) for n even.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the order of the integral.  
//    0 <= N.
//
//    Output, double VALUE, the value of the integral.
//
{
  double pi = 3.141592653589793;
  double value;

  if ( n < 0 )
  {
    value = - webbur::r8_huge ( );
  }
  else if ( ( n % 2 ) == 1 )
  {
    value = 0.0;
  }
  else
  {
    value = webbur::r8_factorial2 ( n - 1 ) * std::sqrt ( pi ) 
      / std::pow ( 2.0, n / 2 );
  }

  return value;
}
//****************************************************************************80

void hermite_lookup_points ( int order, double alpha, double beta,
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    HERMITE_LOOKUP_POINTS looks up Hermite quadrature points.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on (-oo,+oo).
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    Legal values are 1, 3, 7, 15, 31, 63 and 127.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  static double x_001[1] =
  {
    0.0E+00
  };
  static double x_003[3] =
  {
   -0.122474487139158904909864203735E+01, 
    0.0E+00, 
    0.122474487139158904909864203735E+01
  };
  static double x_007[7] =
  {
   -0.265196135683523349244708200652E+01, 
   -0.167355162876747144503180139830E+01, 
   -0.816287882858964663038710959027E+00, 
    0.0E+00, 
    0.816287882858964663038710959027E+00, 
    0.167355162876747144503180139830E+01, 
    0.265196135683523349244708200652E+01
  };
  static double x_015[15] =
  {
   -0.449999070730939155366438053053E+01, 
   -0.366995037340445253472922383312E+01, 
   -0.296716692790560324848896036355E+01, 
   -0.232573248617385774545404479449E+01, 
   -0.171999257518648893241583152515E+01, 
   -0.113611558521092066631913490556E+01, 
   -0.565069583255575748526020337198E+00, 
    0.0E+00, 
    0.565069583255575748526020337198E+00, 
    0.113611558521092066631913490556E+01, 
    0.171999257518648893241583152515E+01, 
    0.232573248617385774545404479449E+01, 
    0.296716692790560324848896036355E+01, 
    0.366995037340445253472922383312E+01, 
    0.449999070730939155366438053053E+01
  };
  static double x_031[31] =
  {
   -6.9956801237185402753248521473232E+00, 
   -6.2750787049428601427036567812530E+00, 
   -5.6739614446185883296332558789276E+00, 
   -5.1335955771123807045862968913996E+00, 
   -4.6315595063128599420667997654336E+00, 
   -4.1562717558181451724831352315314E+00, 
   -3.7007434032314694224497164589673E+00, 
   -3.2603207323135408104645401509648E+00, 
   -2.8316804533902054557015640151425E+00, 
   -2.4123177054804201051740184582119E+00, 
   -2.0002585489356389657975562598571E+00, 
   -1.5938858604721398261388419455550E+00, 
   -1.1918269983500464260821358649242E+00, 
   -0.79287697691530893968593032998830E+00, 
   -0.39594273647142311094670041663436E+00, 
    0.0000000000000000000000000000000E+00, 
    0.39594273647142311094670041663436E+00, 
    0.79287697691530893968593032998830E+00, 
    1.1918269983500464260821358649242E+00, 
    1.5938858604721398261388419455550E+00, 
    2.0002585489356389657975562598571E+00, 
    2.4123177054804201051740184582119E+00, 
    2.8316804533902054557015640151425E+00, 
    3.2603207323135408104645401509648E+00, 
    3.7007434032314694224497164589673E+00, 
    4.1562717558181451724831352315314E+00, 
    4.6315595063128599420667997654336E+00, 
    5.1335955771123807045862968913996E+00, 
    5.6739614446185883296332558789276E+00, 
    6.2750787049428601427036567812530E+00, 
    6.9956801237185402753248521473232E+00
  };
  static double x_063[63] =
  {
   -10.435499877854168053468115427285E+00, 
   -9.8028759912974963635223935286507E+00, 
   -9.2792019543050391319404745506496E+00, 
   -8.8118581437284546442526628275570E+00, 
   -8.3807683451863219343010651043788E+00, 
   -7.9755950801420373181541806298501E+00, 
   -7.5901395198641066762479783194468E+00, 
   -7.2203167078889678461161324222529E+00, 
   -6.8632544331795368527353285876066E+00, 
   -6.5168348106821160605273395854042E+00, 
   -6.1794379922705969862418461787263E+00, 
   -5.8497884000810673462526582961482E+00, 
   -5.5268572526403031425047575122840E+00, 
   -5.2097979830408354861575136416263E+00, 
   -4.8979018644975742350745099214868E+00, 
   -4.5905665744435190229271294569091E+00, 
   -4.2872733352824404031727616199454E+00, 
   -3.9875699104197157485227052068068E+00, 
   -3.6910577000963465117322810559754E+00, 
   -3.3973817713303911852755941806287E+00, 
   -3.1062230279282566329138616746036E+00, 
   -2.8172919672837977750747135657355E+00, 
   -2.5303236304712010926855221718499E+00, 
   -2.2450734604812066298995918179330E+00, 
   -1.9613138583081485293922008411321E+00, 
   -1.6788312791720137520802800622638E+00, 
   -1.3974237486049625107570752063702E+00, 
   -1.1168987050996462690510970277840E+00, 
   -0.83707109558947615977737795461293E+00, 
   -0.55776166427908221668763665253822E+00, 
   -0.27879538567115223986687628627202E+00, 
    0.00000000000000000000000000000000E+00, 
    0.27879538567115223986687628627202E+00, 
    0.55776166427908221668763665253822E+00, 
    0.83707109558947615977737795461293E+00, 
    1.1168987050996462690510970277840E+00, 
    1.3974237486049625107570752063702E+00, 
    1.6788312791720137520802800622638E+00, 
    1.9613138583081485293922008411321E+00, 
    2.2450734604812066298995918179330E+00, 
    2.5303236304712010926855221718499E+00, 
    2.8172919672837977750747135657355E+00, 
    3.1062230279282566329138616746036E+00, 
    3.3973817713303911852755941806287E+00, 
    3.6910577000963465117322810559754E+00, 
    3.9875699104197157485227052068068E+00, 
    4.2872733352824404031727616199454E+00, 
    4.5905665744435190229271294569091E+00, 
    4.8979018644975742350745099214868E+00, 
    5.2097979830408354861575136416263E+00, 
    5.5268572526403031425047575122840E+00, 
    5.8497884000810673462526582961482E+00, 
    6.1794379922705969862418461787263E+00, 
    6.5168348106821160605273395854042E+00, 
    6.8632544331795368527353285876066E+00, 
    7.2203167078889678461161324222529E+00, 
    7.5901395198641066762479783194468E+00, 
    7.9755950801420373181541806298501E+00, 
    8.3807683451863219343010651043788E+00, 
    8.8118581437284546442526628275570E+00, 
    9.2792019543050391319404745506496E+00, 
    9.8028759912974963635223935286507E+00, 
    10.435499877854168053468115427285E+00
  };
  static double x_127[127] =
  {
   -15.228338148167350978246954433464E+00, 
   -14.669595158833972632746354112896E+00, 
   -14.209085995284870755168244250887E+00, 
   -13.799722290211676634645246746673E+00, 
   -13.423518590070950062438258321855E+00, 
   -13.071208660474601901583995439649E+00, 
   -12.737235652415686338138003924072E+00, 
   -12.417939378869715805445879624069E+00, 
   -12.110749020947747600132123508132E+00, 
   -11.813772198267727195134584136191E+00, 
   -11.525565112572696599167888588564E+00, 
   -11.244994583785543445194384194300E+00, 
   -10.971150569840247423423040263881E+00, 
   -10.703288201027481347670940744690E+00, 
   -10.440787957772772867742591798027E+00, 
   -10.183127473450343888624126450357E+00, 
   -9.9298610495114250736847004273684E+00, 
   -9.6806044412474728038150712732737E+00, 
   -9.4350233389881650135019598506287E+00, 
   -9.1928244988460305715774195052527E+00, 
   -8.9537488108565404323807890169970E+00, 
   -8.7175658087076307363833999548548E+00, 
   -8.4840692689832473326097180339984E+00, 
   -8.2530736454457156579694124243888E+00, 
   -8.0244111514703375578594739796798E+00, 
   -7.7979293513870105420829120455591E+00, 
   -7.5734891556083454022834960763301E+00, 
   -7.3509631392269052701961258043733E+00, 
   -7.1302341220350710668064025713431E+00, 
   -6.9111939615465713197465633109366E+00, 
   -6.6937425208758294190074417381666E+00, 
   -6.4777867811645365448144903821487E+00, 
   -6.2632400742737354345609723857092E+00, 
   -6.0500214161419845694465474482388E+00, 
   -5.8380549248774187386601690807757E+00, 
   -5.6272693105464816659423455794909E+00, 
   -5.4175974259243240722848425872924E+00, 
   -5.2089758693153983587570258372239E+00, 
   -5.0013446320386360038520809107373E+00, 
   -4.7946467843764925009748509930857E+00, 
   -4.5888281947698372951606485031212E+00, 
   -4.3838372778464736294253744407459E+00, 
   -4.1796247675352031349421189892408E+00, 
   -3.9761435120673355916035814195920E+00, 
   -3.7733482881250526721004678400057E+00, 
   -3.5711956317782180447199756485249E+00, 
   -3.3696436841717397896643629240035E+00, 
   -3.1686520501953630191857798261495E+00, 
   -2.9681816685955910267761649521505E+00, 
   -2.7681946921824058801226545958892E+00, 
   -2.5686543769473501723144013022363E+00, 
   -2.3695249790490401080012474645702E+00, 
   -2.1707716587411506879498498083695E+00, 
   -1.9723603904195020079324743227565E+00, 
   -1.7742578780516791584676442103681E+00, 
   -1.5764314753267801315519597621879E+00, 
   -1.3788491099261778091441557053728E+00, 
   -1.1814792113700685848678583598423E+00, 
   -0.98429064194027277726568984213773E+00, 
   -0.78725263021825034151596831878971E+00, 
   -0.59033470680942102142230439346102E+00, 
   -0.39350664185130136568037826200185E+00, 
   -0.19673838392423251964272239737078E+00, 
    0.0000000000000000000000000000000E+00, 
    0.19673838392423251964272239737078E+00, 
    0.39350664185130136568037826200185E+00, 
    0.59033470680942102142230439346102E+00, 
    0.78725263021825034151596831878971E+00, 
    0.98429064194027277726568984213773E+00, 
    1.1814792113700685848678583598423E+00, 
    1.3788491099261778091441557053728E+00, 
    1.5764314753267801315519597621879E+00, 
    1.7742578780516791584676442103681E+00, 
    1.9723603904195020079324743227565E+00, 
    2.1707716587411506879498498083695E+00, 
    2.3695249790490401080012474645702E+00, 
    2.5686543769473501723144013022363E+00, 
    2.7681946921824058801226545958892E+00, 
    2.9681816685955910267761649521505E+00, 
    3.1686520501953630191857798261495E+00, 
    3.3696436841717397896643629240035E+00, 
    3.5711956317782180447199756485249E+00, 
    3.7733482881250526721004678400057E+00, 
    3.9761435120673355916035814195920E+00, 
    4.1796247675352031349421189892408E+00, 
    4.3838372778464736294253744407459E+00, 
    4.5888281947698372951606485031212E+00, 
    4.7946467843764925009748509930857E+00, 
    5.0013446320386360038520809107373E+00, 
    5.2089758693153983587570258372239E+00, 
    5.4175974259243240722848425872924E+00, 
    5.6272693105464816659423455794909E+00, 
    5.8380549248774187386601690807757E+00, 
    6.0500214161419845694465474482388E+00, 
    6.2632400742737354345609723857092E+00, 
    6.4777867811645365448144903821487E+00, 
    6.6937425208758294190074417381666E+00, 
    6.9111939615465713197465633109366E+00, 
    7.1302341220350710668064025713431E+00, 
    7.3509631392269052701961258043733E+00, 
    7.5734891556083454022834960763301E+00, 
    7.7979293513870105420829120455591E+00, 
    8.0244111514703375578594739796798E+00, 
    8.2530736454457156579694124243888E+00, 
    8.4840692689832473326097180339984E+00, 
    8.7175658087076307363833999548548E+00, 
    8.9537488108565404323807890169970E+00, 
    9.1928244988460305715774195052527E+00, 
    9.4350233389881650135019598506287E+00, 
    9.6806044412474728038150712732737E+00, 
    9.9298610495114250736847004273684E+00, 
    10.183127473450343888624126450357E+00, 
    10.440787957772772867742591798027E+00, 
    10.703288201027481347670940744690E+00, 
    10.971150569840247423423040263881E+00, 
    11.244994583785543445194384194300E+00, 
    11.525565112572696599167888588564E+00, 
    11.813772198267727195134584136191E+00, 
    12.110749020947747600132123508132E+00, 
    12.417939378869715805445879624069E+00, 
    12.737235652415686338138003924072E+00, 
    13.071208660474601901583995439649E+00, 
    13.423518590070950062438258321855E+00, 
    13.799722290211676634645246746673E+00, 
    14.209085995284870755168244250887E+00, 
    14.669595158833972632746354112896E+00, 
    15.228338148167350978246954433464E+00
  };
 
  if ( order == 1 )
  {
    webbur::r8vec_copy ( order, x_001, x );
  }
  else if ( order == 3 )
  {
    webbur::r8vec_copy ( order, x_003, x );
  }
  else if ( order == 7 )
  {
    webbur::r8vec_copy ( order, x_007, x );
  }
  else if ( order == 15 )
  {
    webbur::r8vec_copy ( order, x_015, x );
  }
  else if ( order == 31 )
  {
    webbur::r8vec_copy ( order, x_031, x );
  }
  else if ( order == 63 )
  {
    webbur::r8vec_copy ( order, x_063, x );
  }
  else if ( order == 127 )
  {
    webbur::r8vec_copy ( order, x_127, x );
  }
  else
  {
    std::cerr << "\n";
    std::cerr << "HERMITE_LOOKUP_POINTS - Fatal error!\n";
    std::cerr << "  Unexpected value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }
  return;
}
//****************************************************************************80

void hermite_lookup_weights ( int order, double alpha, double beta,
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    HERMITE_LOOKUP_WEIGHTS looks up Hermite quadrature weights.
//
//  Discussion:
//
//    The allowed orders are 1, 3, 7, 15, 31, 63 and 127.
//
//    The weights are positive, symmetric and should sum to SQRT(PI).
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Milton Abramowitz, Irene Stegun,
//    Handbook of Mathematical Functions,
//    National Bureau of Standards, 1964,
//    ISBN: 0-486-61272-4,
//    LC: QA47.A34.
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    ORDER must be 1, 3, 7, 15, 31, 63 or 127.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  static double w_001[1] =
  {
    1.77245385090551602729816748334E+00
  };
  static double w_003[31] =
  {
    0.295408975150919337883027913890E+00,
    0.118163590060367735153211165556E+01,
    0.295408975150919337883027913890E+00
  };
  static double w_007[7] = 
  {
    0.971781245099519154149424255939E-03,
    0.545155828191270305921785688417E-01,
    0.425607252610127800520317466666E+00,
    0.810264617556807326764876563813E+00,
    0.425607252610127800520317466666E+00,
    0.545155828191270305921785688417E-01,
    0.971781245099519154149424255939E-03
  };
  static double w_015[15] =
  {
    0.152247580425351702016062666965E-08,
    0.105911554771106663577520791055E-05,
    0.100004441232499868127296736177E-03,
    0.277806884291277589607887049229E-02,
    0.307800338725460822286814158758E-01,
    0.158488915795935746883839384960E+00,
    0.412028687498898627025891079568E+00,
    0.564100308726417532852625797340E+00,
    0.412028687498898627025891079568E+00,
    0.158488915795935746883839384960E+00,
    0.307800338725460822286814158758E-01,
    0.277806884291277589607887049229E-02,
    0.100004441232499868127296736177E-03,
    0.105911554771106663577520791055E-05,
    0.152247580425351702016062666965E-08
  };
  static double w_031[31] = 
  {
    0.46189683944498305857470556847735E-21,
    0.51106090079112519643027197715274E-17,
    0.58995564987355133075257722133966E-14,
    0.18603735214463569590294465062239E-11,
    0.23524920032013205739850619940094E-09,
    0.14611988344865057576066495091513E-07,
    0.50437125589241034841778074689627E-06,
    0.10498602757642934202945441341697E-04,
    0.13952090395003623854995664958146E-03,
    0.12336833073030489880608311394968E-02,
    0.74827999140119116765002499116934E-02,
    0.31847230731201222775249585776902E-01,
    0.96717948160569462991143316029341E-01,
    0.21213278866810461318136114862419E+00,
    0.33877265789305344906000174083214E+00,
    0.39577855609737786462923720809676E+00,
    0.33877265789305344906000174083214E+00,
    0.21213278866810461318136114862419E+00,
    0.96717948160569462991143316029341E-01,
    0.31847230731201222775249585776902E-01,
    0.74827999140119116765002499116934E-02,
    0.12336833073030489880608311394968E-02,
    0.13952090395003623854995664958146E-03,
    0.10498602757642934202945441341697E-04,
    0.50437125589241034841778074689627E-06,
    0.14611988344865057576066495091513E-07,
    0.23524920032013205739850619940094E-09,
    0.18603735214463569590294465062239E-11,
    0.58995564987355133075257722133966E-14,
    0.51106090079112519643027197715274E-17,
    0.46189683944498305857470556847735E-21
  };
  static double w_063[63] = 
  {
    0.37099206434787551197827130470031E-47,
    0.10400778615192299534481914814892E-41,
    0.19796804708258311251124226474396E-37,
    0.84687478191640015120141181138947E-34,
    0.13071305930779945903630127634063E-30,
    0.93437837175367456929765381518998E-28,
    0.36027426635173044862245783257252E-25,
    0.82963863115951789374753323156164E-23,
    0.12266629909105281472971700203949E-20,
    0.12288435628797061539461585325494E-18,
    0.86925536958188009075932426691516E-17,
    0.44857058689176221240330804981619E-15,
    0.17335817955735154599902643794700E-13,
    0.51265062385038307838565047455223E-12,
    0.11808921844532942490513037158404E-10,
    0.21508698297808025739828859845140E-09,
    0.31371929535285447801497640621672E-08,
    0.37041625984781705796752840204084E-07,
    0.35734732949879669663960738150956E-06,
    0.28393114498380927832990899215541E-05,
    0.18709113003730498008961134765721E-04,
    0.10284880800653635546698378640623E-03,
    0.47411702610173128107201781718693E-03,
    0.18409222622384813438539657470055E-02,
    0.60436044551187631655712178246467E-02,
    0.16829299199599730926458559757600E-01,
    0.39858264027692992170237391875317E-01,
    0.80467087993950415219587554532823E-01,
    0.13871950817615293377792092082674E+00,
    0.20448695346833761570957197160475E+00,
    0.25799889943058042204920467417642E+00,
    0.27876694884838411919175686949858E+00,
    0.25799889943058042204920467417642E+00,
    0.20448695346833761570957197160475E+00,
    0.13871950817615293377792092082674E+00,
    0.80467087993950415219587554532823E-01,
    0.39858264027692992170237391875317E-01,
    0.16829299199599730926458559757600E-01,
    0.60436044551187631655712178246467E-02,
    0.18409222622384813438539657470055E-02,
    0.47411702610173128107201781718693E-03,
    0.10284880800653635546698378640623E-03,
    0.18709113003730498008961134765721E-04,
    0.28393114498380927832990899215541E-05,
    0.35734732949879669663960738150956E-06,
    0.37041625984781705796752840204084E-07,
    0.31371929535285447801497640621672E-08,
    0.21508698297808025739828859845140E-09,
    0.11808921844532942490513037158404E-10,
    0.51265062385038307838565047455223E-12,
    0.17335817955735154599902643794700E-13,
    0.44857058689176221240330804981619E-15,
    0.86925536958188009075932426691516E-17,
    0.12288435628797061539461585325494E-18,
    0.12266629909105281472971700203949E-20,
    0.82963863115951789374753323156164E-23,
    0.36027426635173044862245783257252E-25,
    0.93437837175367456929765381518998E-28,
    0.13071305930779945903630127634063E-30,
    0.84687478191640015120141181138947E-34,
    0.19796804708258311251124226474396E-37,
    0.10400778615192299534481914814892E-41,
    0.37099206434787551197827130470031E-47
  };
  static double w_127[127] = 
  {
    0.12504497577050595552677230002883E-10,
    0.17272798059419131415318615789672E-93,
    0.89321681571986548608031150791499E-88,
    0.77306185240893578449625186483810E-83,
    0.20143957652648255497735460506196E-78,
    0.21503714733610239701351039429345E-74,
    0.11341924208594594813715533569504E-70,
    0.33489139011795051950683388483136E-67,
    0.60486548964016681064424451668405E-64,
    0.71375092946352177824971347343892E-61,
    0.57884563374885556636801095624030E-58,
    0.33581166223858230300409326551248E-55,
    0.14394641949253923568603163698953E-52,
    0.46821808383216117724080263903889E-50,
    0.11817054440684264071348471955361E-47,
    0.23581659156008927203181682045005E-45,
    0.37814427940797540210712758405540E-43,
    0.49411031115771638145610738414006E-41,
    0.53255303775425059266087298458297E-39,
    0.47854390680131484999315199332765E-37,
    0.36191883445952356128627543209554E-35,
    0.23232083386343554805352497446119E-33,
    0.12753331411008716683688974281454E-31,
    0.60277753850758742112436095241270E-30,
    0.24679773241777200207460855084439E-28,
    0.88019567691698482573264198727415E-27,
    0.27482489212040561315005725890593E-25,
    0.75468218903085486125222816438456E-24,
    0.18303134636280466270545996891835E-22,
    0.39355990860860813085582448449811E-21,
    0.75293161638581191068419292570042E-20,
    0.12857997786722855037584105682618E-18,
    0.19659326888445857792541925311450E-17,
    0.26986511907214101894995783364250E-16,
    0.33344414303198856330118301113874E-15,
    0.37173303125150639885726463109574E-14,
    0.37473954472839737091885387788983E-13,
    0.34230094493397259538669512076007E-12,
    0.28385303724993373166810860630552E-11,
    0.21406920290454669208938772802828E-10,
    0.14706331273431716244229273183839E-09,
    0.92173940967434659264335883218167E-09,
    0.52781663936972714041837056042506E-08,
    0.27650497044951117835905283127679E-07,
    0.13267855842539464770913063113371E-06,
    0.58380944276113062188573331195042E-06,
    0.23581561724775629112332165335800E-05,
    0.87524468034280444703919485644809E-05,
    0.29876790535909012274846532159647E-04,
    0.93874435720072545206729594267039E-04,
    0.27170762627931172053444716883938E-03,
    0.72493929742498358979684249380921E-03,
    0.17841208326763432884316727108264E-02,
    0.40524855186046131499765636276283E-02,
    0.85000263041544110385806705526917E-02,
    0.16471142241609687824005585301760E-01,
    0.29499296248213632269675010319119E-01,
    0.48847387114300011006959603975676E-01,
    0.74807989768583731416517226905270E-01,
    0.10598520508090929403834368934301E+00,
    0.13893945309051540832066283010510E+00,
    0.16856236074207929740526975049765E+00,
    0.18927849580120432177170145550076E+00,
    0.19673340688823289786163676995151E+00,
    0.18927849580120432177170145550076E+00,
    0.16856236074207929740526975049765E+00,
    0.13893945309051540832066283010510E+00,
    0.10598520508090929403834368934301E+00,
    0.74807989768583731416517226905270E-01,
    0.48847387114300011006959603975676E-01,
    0.29499296248213632269675010319119E-01,
    0.16471142241609687824005585301760E-01,
    0.85000263041544110385806705526917E-02,
    0.40524855186046131499765636276283E-02,
    0.17841208326763432884316727108264E-02,
    0.72493929742498358979684249380921E-03,
    0.27170762627931172053444716883938E-03,
    0.93874435720072545206729594267039E-04,
    0.29876790535909012274846532159647E-04,
    0.87524468034280444703919485644809E-05,
    0.23581561724775629112332165335800E-05,
    0.58380944276113062188573331195042E-06,
    0.13267855842539464770913063113371E-06,
    0.27650497044951117835905283127679E-07,
    0.52781663936972714041837056042506E-08,
    0.92173940967434659264335883218167E-09,
    0.14706331273431716244229273183839E-09,
    0.21406920290454669208938772802828E-10,
    0.28385303724993373166810860630552E-11,
    0.34230094493397259538669512076007E-12,
    0.37473954472839737091885387788983E-13,
    0.37173303125150639885726463109574E-14,
    0.33344414303198856330118301113874E-15,
    0.26986511907214101894995783364250E-16,
    0.19659326888445857792541925311450E-17,
    0.12857997786722855037584105682618E-18,
    0.75293161638581191068419292570042E-20,
    0.39355990860860813085582448449811E-21,
    0.18303134636280466270545996891835E-22,
    0.75468218903085486125222816438456E-24,
    0.27482489212040561315005725890593E-25,
    0.88019567691698482573264198727415E-27,
    0.24679773241777200207460855084439E-28,
    0.60277753850758742112436095241270E-30,
    0.12753331411008716683688974281454E-31,
    0.23232083386343554805352497446119E-33,
    0.36191883445952356128627543209554E-35,
    0.47854390680131484999315199332765E-37,
    0.53255303775425059266087298458297E-39,
    0.49411031115771638145610738414006E-41,
    0.37814427940797540210712758405540E-43,
    0.23581659156008927203181682045005E-45,
    0.11817054440684264071348471955361E-47,
    0.46821808383216117724080263903889E-50,
    0.14394641949253923568603163698953E-52,
    0.33581166223858230300409326551248E-55,
    0.57884563374885556636801095624030E-58,
    0.71375092946352177824971347343892E-61,
    0.60486548964016681064424451668405E-64,
    0.33489139011795051950683388483136E-67,
    0.11341924208594594813715533569504E-70,
    0.21503714733610239701351039429345E-74,
    0.20143957652648255497735460506196E-78,
    0.77306185240893578449625186483810E-83,
    0.89321681571986548608031150791499E-88,
    0.17272798059419131415318615789672E-93,
    0.12504497577050595552677230002883E-100
  };
  if ( order == 1 )
  {
    webbur::r8vec_copy ( order, w_001, w );
  }
  else if ( order == 3 )
  {
    webbur::r8vec_copy ( order, w_003, w );
  }
  else if ( order == 7 )
  {
    webbur::r8vec_copy ( order, w_007, w );
  }
  else if ( order == 15 )
  {
    webbur::r8vec_copy ( order, w_015, w );
  }
  else if ( order == 31 )
  {
    webbur::r8vec_copy ( order, w_031, w );
  }
  else if ( order == 63 )
  {
    webbur::r8vec_copy ( order, w_063, w );
  }
  else if ( order == 127 )
  {
    webbur::r8vec_copy ( order, w_127, w );
  }
  else
  {
    std::cerr << "\n";
    std::cerr << "HERMITE_LOOKUP_WEIGHTS - Fatal error!\n";
    std::cerr << "  Unexpected value of ORDER = " << order << ".\n";
    std::exit ( 1 );
  }
  return;
}
//****************************************************************************80

void hermite_recur ( double *p2, double *dp2, double *p1, double x, int order )

//****************************************************************************80
//
//  Purpose:
//
//    HERMITE_RECUR finds the value and derivative of a Hermite polynomial.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Output, double *P2, the value of H(ORDER)(X).
//
//    Output, double *DP2, the value of H'(ORDER)(X).
//
//    Output, double *P1, the value of H(ORDER-1)(X).
//
//    Input, double X, the point at which polynomials are evaluated.
//
//    Input, int ORDER, the order of the polynomial to be computed.
//
{
  int i;
  double dq0;
  double dq1;
  double dq2;
  double q0;
  double q1;
  double q2;

  q1 = 1.0;
  dq1 = 0.0;

  q2 = x;
  dq2 = 1.0;

  for ( i = 2; i <= order; i++ )
  {
    q0 = q1;
    dq0 = dq1;

    q1 = q2;
    dq1 = dq2;

    q2  = x * q1 - 0.5 * ( ( double ) ( i ) - 1.0 ) * q0;
    dq2 = x * dq1 + q1 - 0.5 * ( ( double ) ( i ) - 1.0 ) * dq0;
  }

  *p2 = q2;
  *dp2 = dq2;
  *p1 = q1;

  return;
}
//****************************************************************************80

void hermite_root ( double *x, int order, double *dp2, double *p1 )

//****************************************************************************80
//
//  Purpose:
//
//    HERMITE_ROOT improves an approximate root of a Hermite polynomial.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input/output, double *X, the approximate root, which
//    should be improved on output.
//
//    Input, int ORDER, the order of the Hermite polynomial.
//
//    Output, double *DP2, the value of H'(ORDER)(X).
//
//    Output, double *P1, the value of H(ORDER-1)(X).
//
{
  double d;
  double eps;
  double p2;
  int step;
  int step_max = 10;

  eps = webbur::r8_epsilon ( );

  for ( step = 1; step <= step_max; step++ )
  {
    webbur::hermite_recur ( &p2, dp2, p1, *x, order );

    d = p2 / ( *dp2 );
    *x = *x - d;

    if ( webbur::r8_abs ( d ) <= eps * ( webbur::r8_abs ( *x ) + 1.0 ) )
    {
      return;
    }
  }

  return;
}
//****************************************************************************80

int i4_max ( int i1, int i2 )

//****************************************************************************80
//
//  Purpose:
//
//    I4_MAX returns the maximum of two I4's.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    13 October 1998
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int I1, I2, are two integers to be compared.
//
//    Output, int I4_MAX, the larger of I1 and I2.
//
{
  int value;

  if ( i2 < i1 )
  {
    value = i1;
  }
  else
  {
    value = i2;
  }
  return value;
}
//****************************************************************************80

int i4_min ( int i1, int i2 )

//****************************************************************************80
//
//  Purpose:
//
//    I4_MIN returns the minimum of two I4's.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    13 October 1998
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int I1, I2, two integers to be compared.
//
//    Output, int I4_MIN, the smaller of I1 and I2.
//
{
  int value;

  if ( i1 < i2 )
  {
    value = i1;
  }
  else
  {
    value = i2;
  }
  return value;
}
//****************************************************************************80

int i4_power ( int i, int j )

//****************************************************************************80
//
//  Purpose:
//
//    I4_POWER returns the value of I^J.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    01 April 2004
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int I, J, the base and the power.  J should be nonnegative.
//
//    Output, int I4_POWER, the value of I^J.
//
{
  int k;
  int value;

  if ( j < 0 )
  {
    if ( i == 1 )
    {
      value = 1;
    }
    else if ( i == 0 )
    {
      std::cerr << "\n";
      std::cerr << "I4_POWER - Fatal error!\n";
      std::cerr << "  I^J requested, with I = 0 and J negative.\n";
      std::exit ( 1 );
    }
    else
    {
      value = 0;
    }
  }
  else if ( j == 0 )
  {
    if ( i == 0 )
    {
      std::cerr << "\n";
      std::cerr << "I4_POWER - Fatal error!\n";
      std::cerr << "  I^J requested, with I = 0 and J = 0.\n";
      std::exit ( 1 );
    }
    else
    {
      value = 1;
    }
  }
  else if ( j == 1 )
  {
    value = i;
  }
  else
  {
    value = 1;
    for ( k = 1; k <= j; k++ )
    {
      value = value * i;
    }
  }
  return value;
}
//****************************************************************************80

int i4vec_product ( int n, int a[] )

//****************************************************************************80
//
//  Purpose:
//
//    I4VEC_PRODUCT multiplies the entries of an I4VEC.
//
//  Discussion:
//
//    An I4VEC is a vector of integer values.
//
//  Example:
//
//    Input:
//
//      A = ( 1, 2, 3, 4 )
//
//    Output:
//
//      I4VEC_PRODUCT = 24
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    17 May 2003
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the number of entries in the vector.
//
//    Input, int A[N], the vector
//
//    Output, int I4VEC_PRODUCT, the product of the entries of A.
//
{
  int i;
  int product;

  product = 1;
  for ( i = 0; i < n; i++ )
  {
    product = product * a[i];
  }

  return product;
}
//****************************************************************************80

void jacobi_compute ( int order, double alpha, double beta, double x[], 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    JACOBI_COMPUTE computes a Jacobi quadrature rule.
//
//  Discussion:
//
//    The integration interval is [ -1, 1 ].
//
//    The weight function is w(x) = (1-X)^ALPHA * (1+X)^BETA.
//
//    The integral to approximate:
//
//      Integral ( -1 <= X <= 1 ) (1-X)^ALPHA * (1+X)^BETA * F(X) dX
//
//    The quadrature rule:
//
//      Sum ( 1 <= I <= ORDER ) W(I) * F ( X(I) )
//
//    Thanks to Xu Xiang of Fudan University for pointing out that
//    an earlier implementation of this routine was incorrect!
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, the exponents of (1-X) and
//    (1+X) in the quadrature rule.  For simple Legendre quadrature,
//    set ALPHA = BETA = 0.0.  -1.0 < ALPHA and -1.0 < BETA are required.
//
//    Output, double X[ORDER], the abscissas.
//
//    Output, double W[ORDER], the weights.
//
{
  double an;
  double *b;
  double bn;
  double *c;
  double cc;
  double delta;
  double dp2;
  int i;
  double p1;
  double prod;
  double r1;
  double r2;
  double r3;
  double temp;
  double x0;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "JACOBI_COMPUTE - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  b = new double[order];
  c = new double[order];
//
//  Check ALPHA and BETA.
//
  if ( alpha <= -1.0 )
  {
    std::cerr << "\n";
    std::cerr << "JACOBI_COMPUTE - Fatal error!\n";
    std::cerr << "  -1.0 < ALPHA is required.\n";
    std::exit ( 1 );
  }

  if ( beta <= -1.0 )
  {
    std::cerr << "\n";
    std::cerr << "JACOBI_COMPUTE - Fatal error!\n";
    std::cerr << "  -1.0 < BETA is required.\n";
    std::exit ( 1 );
  }
//
//  Set the recursion coefficients.
//
  for ( i = 1; i <= order; i++ )
  {
    if ( alpha + beta == 0.0 || beta - alpha == 0.0 )
    {
      b[i-1] = 0.0;
    }
    else
    {
      b[i-1] = ( alpha + beta ) * ( beta - alpha ) / 
             ( ( alpha + beta + ( double ) ( 2 * i ) ) 
             * ( alpha + beta + ( double ) ( 2 * i - 2 ) ) );
    }

    if ( i == 1 )
    {
      c[i-1] = 0.0;
    }
    else
    {
      c[i-1] = 4.0 * ( double ) ( i - 1 ) 
         * ( alpha + ( double ) ( i - 1 ) ) 
          * ( beta + ( double ) ( i - 1 ) ) 
            * ( alpha + beta + ( double ) ( i - 1 ) ) / 
            ( ( alpha + beta + ( double ) ( 2 * i - 1 ) ) 
            * std::pow ( alpha + beta + ( double ) ( 2 * i - 2 ), 2 ) 
            * ( alpha + beta + ( double ) ( 2 * i - 3 ) ) );
    }
  }

  delta = webbur::r8_gamma ( alpha        + 1.0 ) 
        * webbur::r8_gamma (         beta + 1.0 ) 
        / webbur::r8_gamma ( alpha + beta + 2.0 );

  prod = 1.0;
  for ( i = 2; i <= order; i++ )
  {
    prod = prod * c[i-1];
  }
  cc = delta * std::pow ( 2.0, alpha + beta + 1.0 ) * prod;

  for ( i = 1; i <= order; i++ )
  {
    if ( i == 1 )
    {
      an = alpha / ( double ) ( order );
      bn = beta / ( double ) ( order );

      r1 = ( 1.0 + alpha ) 
        * ( 2.78 / ( 4.0 + ( double ) ( order * order ) ) 
        + 0.768 * an / ( double ) ( order ) );

      r2 = 1.0 + 1.48 * an + 0.96 * bn 
        + 0.452 * an * an + 0.83 * an * bn;

      x0 = ( r2 - r1 ) / r2;
    }
    else if ( i == 2 )
    {
      r1 = ( 4.1 + alpha ) / 
        ( ( 1.0 + alpha ) * ( 1.0 + 0.156 * alpha ) );

      r2 = 1.0 + 0.06 * ( ( double ) ( order ) - 8.0 ) * 
        ( 1.0 + 0.12 * alpha ) / ( double ) ( order );

      r3 = 1.0 + 0.012 * beta * 
        ( 1.0 + 0.25 * r8_abs ( alpha ) ) / ( double ) ( order );

      x0 = x0 - r1 * r2 * r3 * ( 1.0 - x0 );
    }
    else if ( i == 3 )
    {
      r1 = ( 1.67 + 0.28 * alpha ) / ( 1.0 + 0.37 * alpha );

      r2 = 1.0 + 0.22 * ( ( double ) ( order ) - 8.0 ) 
        / ( double ) ( order );

      r3 = 1.0 + 8.0 * beta / 
        ( ( 6.28 + beta ) * ( double ) ( order * order ) );

      x0 = x0 - r1 * r2 * r3 * ( x[0] - x0 );
    }
    else if ( i < order - 1 )
    {
      x0 = 3.0 * x[i-2] - 3.0 * x[i-3] + x[i-4];
    }
    else if ( i == order - 1 )
    {
      r1 = ( 1.0 + 0.235 * beta ) / ( 0.766 + 0.119 * beta );

      r2 = 1.0 / ( 1.0 + 0.639 
        * ( ( double ) ( order ) - 4.0 ) 
        / ( 1.0 + 0.71 * ( ( double ) ( order ) - 4.0 ) ) );

      r3 = 1.0 / ( 1.0 + 20.0 * alpha / ( ( 7.5 + alpha ) * 
        ( double ) ( order * order ) ) );

      x0 = x0 + r1 * r2 * r3 * ( x0 - x[i-3] );
    }
    else if ( i == order )
    {
      r1 = ( 1.0 + 0.37 * beta ) / ( 1.67 + 0.28 * beta );

      r2 = 1.0 / 
        ( 1.0 + 0.22 * ( ( double ) ( order ) - 8.0 ) 
        / ( double ) ( order ) );

      r3 = 1.0 / ( 1.0 + 8.0 * alpha / 
        ( ( 6.28 + alpha ) * ( double ) ( order * order ) ) );

      x0 = x0 + r1 * r2 * r3 * ( x0 - x[i-3] );
    }

    webbur::jacobi_root ( &x0, order, alpha, beta, &dp2, &p1, b, c );

    x[i-1] = x0;
    w[i-1] = cc / ( dp2 * p1 );
  }
//
//  Reverse the order of the values.
//
  for ( i = 1; i <= order/2; i++ )
  {
    temp       = x[i-1];
    x[i-1]     = x[order-i];
    x[order-i] = temp;
  }

  for ( i = 1; i <=order/2; i++ )
  {
    temp       = w[i-1];
    w[i-1]     = w[order-i];
    w[order-i] = temp;
  }

  delete [] b;
  delete [] c;

  return;
}
//****************************************************************************80

void jacobi_compute_points ( int order, double alpha, double beta, 
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    JACOBI_COMPUTE_POINTS computes Jacobi quadrature points.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    14 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, BETA, the exponents of the (1-X) and (1+X) factors.
//
//    Output, double X[ORDER], the abscissas.
//
{
  double *w;

  w = new double[order];

  webbur::jacobi_compute ( order, alpha, beta, x, w );

  delete [] w;

  return;
}
//****************************************************************************80

void jacobi_compute_weights ( int order, double alpha, double beta, 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    JACOBI_COMPUTE_WEIGHTS computes Jacobi quadrature weights.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    14 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, BETA, the exponents of the (1-X) and (1+X) factors.
//
//    Output, double W[ORDER], the weights.
//
{
  double *x;

  x = new double[order];

  webbur::jacobi_compute ( order, alpha, beta, x, w );

  delete [] x;

  return;
}
//****************************************************************************80

double jacobi_integral ( int expon, double alpha, double beta )

//****************************************************************************80
//
//  Purpose:
//
//    JACOBI_INTEGRAL integrates a monomial with Jacobi weight.
//
//  Discussion:
//
//    VALUE = Integral ( -1 <= X <= +1 ) x^EXPON (1-x)^ALPHA (1+x)^BETA dx
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    08 September 2007
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int EXPON, the exponent.
//
//    Input, double ALPHA, the exponent of (1-X) in the weight factor.
//
//    Input, double BETA, the exponent of (1+X) in the weight factor.
//
//    Output, double JACOBI_INTEGRAL, the value of the integral.
//
{
  double arg1;
  double arg2;
  double arg3;
  double arg4;
  double c;
  double s;
  double value;
  double value1;
  double value2;

  c = ( double ) ( expon );

  if ( ( expon % 2 ) == 0 )
  {
    s = +1.0;
  }
  else
  {
    s = -1.0;
  }

  arg1 = - alpha;
  arg2 =   1.0 + c;
  arg3 =   2.0 + beta + c;
  arg4 = - 1.0;

  value1 = webbur::r8_hyper_2f1 ( arg1, arg2, arg3, arg4 );

  arg1 = - beta;
  arg2 =   1.0 + c;
  arg3 =   2.0 + alpha + c;
  arg4 = - 1.0;

  value2 = webbur::r8_hyper_2f1 ( arg1, arg2, arg3, arg4 );

  value = webbur::r8_gamma ( 1.0 + c ) * ( 
      s * webbur::r8_gamma ( 1.0 + beta  ) * value1 
    / webbur::r8_gamma ( 2.0 + beta  + c ) 
    +     webbur::r8_gamma ( 1.0 + alpha ) * value2 
    / webbur::r8_gamma ( 2.0 + alpha + c ) );

  return value;
}
//****************************************************************************80

void jacobi_recur ( double *p2, double *dp2, double *p1, double x, int order, 
  double alpha, double beta, double b[], double c[] )

//****************************************************************************80
//
//  Purpose:
//
//    JACOBI_RECUR evaluates a Jacobi polynomial.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Output, double *P2, the value of J(ORDER)(X).
//
//    Output, double *DP2, the value of J'(ORDER)(X).
//
//    Output, double *P1, the value of J(ORDER-1)(X).
//
//    Input, double X, the point at which polynomials are evaluated.
//
//    Input, int ORDER, the order of the polynomial to be computed.
//
//    Input, double ALPHA, BETA, the exponents of (1+X) and
//    (1-X) in the quadrature rule.
//
//    Input, double B[ORDER], C[ORDER], the recursion coefficients.
//
{
  double dp0;
  double dp1;
  int i;
  double p0;

  *p1 = 1.0;
  dp1 = 0.0;

  *p2 = x + ( alpha - beta ) / ( alpha + beta + 2.0 );
  *dp2 = 1.0;

  for ( i = 2; i <= order; i++ )
  {
    p0 = *p1;
    dp0 = dp1;

    *p1 = *p2;
    dp1 = *dp2;

    *p2 = ( x - b[i-1] ) *  ( *p1 ) - c[i-1] * p0;
    *dp2 = ( x - b[i-1] ) * dp1 + ( *p1 ) - c[i-1] * dp0;
  }
  return;
}
//****************************************************************************80

void jacobi_root ( double *x, int order, double alpha, double beta, 
  double *dp2, double *p1, double b[], double c[] )

//****************************************************************************80
//
//  Purpose:
//
//    JACOBI_ROOT improves an approximate root of a Jacobi polynomial.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input/output, double *X, the approximate root, which
//    should be improved on output.
//
//    Input, int ORDER, the order of the polynomial to be computed.
//
//    Input, double ALPHA, BETA, the exponents of (1+X) and
//    (1-X) in the quadrature rule.
//
//    Output, double *DP2, the value of J'(ORDER)(X).
//
//    Output, double *P1, the value of J(ORDER-1)(X).
//
//    Input, double B[ORDER], C[ORDER], the recursion coefficients.
//
{
  double d;
  double eps;
  double p2;
  int step;
  int step_max = 10;

  eps = webbur::r8_epsilon ( );

  for ( step = 1; step <= step_max; step++ )
  {
    webbur::jacobi_recur ( &p2, dp2, p1, *x, order, alpha, beta, b, c );

    d = p2 / ( *dp2 );
    *x = *x - d;

    if ( webbur::r8_abs ( d ) <= eps * ( webbur::r8_abs ( *x ) + 1.0 ) )
    {
      return;
    }
  }
  return;
}
//****************************************************************************80

void laguerre_compute ( int order, double alpha, double beta, double x[], 
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    LAGUERRE_COMPUTE computes a Laguerre quadrature rule.
//
//  Discussion:
//
//    The integration interval is [ 0, +oo ).
//
//    The weight function is w(x) = exp ( -x );.
//
//    If the integral to approximate is:
//
//        Integral ( 0 <= X < +oo ) exp ( - X ) * F(X) dX
//
//    then the quadrature rule is:
//
//      Sum ( 1 <= I <= ORDER ) W(I) * F ( X(I) )
//
//    If the integral to approximate is:
//
//        Integral ( A <= X < +oo ) F(X) dX
//
//    then the quadrature rule is:
//
//      Sum ( 1 <= I <= ORDER ) W(I) * exp ( X(I) ) * F ( X(I) )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
//    Output, double W[ORDER], the weights.
//
{
  double *b;
  double *c;
  double cc;
  double dp2;
  int i;
  double p1;
  double prod;
  double r1;
  double r2;
  double ratio;
  double x0;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "LAGUERRE_COMPUTE - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  b = new double[order];
  c = new double[order];
//
//  Set the recursion coefficients.
//
  for ( i = 0; i < order; i++ )
  {
    b[i] = ( double ) ( 2 * i + 1 );
  }

  for ( i = 0; i < order; i++ )
  {
    c[i] = ( double ) ( i * i );
  }
  prod = 1.0;
  for ( i = 1; i < order; i++ )
  {
    prod = prod * c[i];
  }
  cc = prod;

  for ( i = 0; i < order; i++ )
  {
//
//  Compute an estimate for the root.
//
    if ( i == 0 )
    {
      x0 =  3.0 / ( 1.0 + 2.4 * ( double ) ( order ) );
    }
    else if ( i == 1 )
    {
      x0 = x0 + 15.0 / ( 1.0 + 2.5 * ( double ) ( order ) );
    }
    else
    {
      r1 = ( 1.0 + 2.55 * ( double ) ( i - 1 ) ) 
        / ( 1.9 * ( double ) ( i - 1 ) );

      x0 = x0 + r1 * ( x0 - x[i-2] );
    }
//
//  Use iteration to find the root.
//
    webbur::laguerre_root ( &x0, order, &dp2, &p1, b, c );
//
//  Set the abscissa and weight.
//
    x[i] = x0;
    w[i] = ( cc / dp2 ) / p1;
  }

  delete [] b;
  delete [] c;

  return;
}
//****************************************************************************80

void laguerre_compute_points ( int order, double alpha, double beta,
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    LAGUERRE_COMPUTE_POINTS computes Laguerre quadrature points.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  double *w;

  w = new double[order];

  webbur::laguerre_compute ( order, alpha, beta, x, w );

  delete [] w;

  return;
}
//****************************************************************************80

void laguerre_compute_weights ( int order, double alpha, double beta,
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    LAGUERRE_COMPUTE_WEIGHTS computes Laguerre quadrature weights.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  double *x;

  x = new double[order];

  webbur::laguerre_compute ( order, alpha, beta, x, w );

  delete [] x;

  return;
}
//****************************************************************************80

double laguerre_integral ( int expon )

//****************************************************************************80
//
//  Purpose:
//
//    LAGUERRE_INTEGRAL evaluates a monomial Laguerre integral.
//
//  Discussion:
//
//    The integral being computed is
//
//      integral ( 0 <= x < +oo ) x^n * exp ( -x ) dx
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int EXPON, the exponent.
//    0 <= EXPON.
//
//    Output, double EXACT, the value of the integral.
//
{
  double exact;

  exact = webbur::r8_factorial ( expon );

  return exact;
}
//****************************************************************************80

void laguerre_lookup_points ( int order, double alpha, double beta,
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    LAGUERRE_LOOKUP_POINTS looks up Laguerre quadrature points.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on [0,+oo).
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.  Legal values 
//    are 1, 3, 7, 15, 31, 63 and 127.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  static double x_001[1] =
  {
    1.0E+00
  };
  static double x_003[3] =
  {
    0.415774556783479083311533873128E+00, 
    0.229428036027904171982205036136E+01, 
    0.628994508293747919686641576551E+01
  };
  static double x_007[7] = 
  {
    0.193043676560362413838247885004E+00, 
    0.102666489533919195034519944317E+01, 
    0.256787674495074620690778622666E+01, 
    0.490035308452648456810171437810E+01, 
    0.818215344456286079108182755123E+01, 
    0.127341802917978137580126424582E+02, 
    0.193957278622625403117125820576E+02
  };
  static double x_015[15] =
  {
    0.933078120172818047629030383672E-01, 
    0.492691740301883908960101791412E+00, 
    0.121559541207094946372992716488E+01, 
    0.226994952620374320247421741375E+01, 
    0.366762272175143727724905959436E+01, 
    0.542533662741355316534358132596E+01, 
    0.756591622661306786049739555812E+01, 
    0.101202285680191127347927394568E+02, 
    0.131302824821757235640991204176E+02, 
    0.166544077083299578225202408430E+02, 
    0.207764788994487667729157175676E+02, 
    0.256238942267287801445868285977E+02, 
    0.314075191697539385152432196202E+02, 
    0.385306833064860094162515167595E+02, 
    0.480260855726857943465734308508E+02
  };
  static double x_031[31] =
  {
    0.45901947621108290743496080275224E-01, 
    0.24198016382477204890408974151714E+00, 
    0.59525389422235073707330165005414E+00, 
    1.1066894995329987162111308789792E+00, 
    1.7775956928747727211593727482675E+00, 
    2.6097034152566806503893375925315E+00, 
    3.6051968023400442698805817554243E+00, 
    4.7667470844717611313629127271123E+00, 
    6.0975545671817409269925429328463E+00, 
    7.6014009492331374229360106942867E+00, 
    9.2827143134708894182536695297710E+00, 
    11.146649755619291358993815629587E+00, 
    13.199189576244998522464925028637E+00, 
    15.447268315549310075809325891801E+00, 
    17.898929826644757646725793817752E+00, 
    20.563526336715822170743048968779E+00, 
    23.451973482011858591050255575933E+00, 
    26.577081352118260459975876986478E+00, 
    29.953990872346445506951917840024E+00, 
    33.600759532902202735410313885784E+00, 
    37.539164407330440882887902558001E+00, 
    41.795830870182219981347945853330E+00, 
    46.403866806411123136029227604386E+00, 
    51.405314476797755161861461088395E+00, 
    56.854992868715843620511922055660E+00, 
    62.826855908786321453677523304806E+00, 
    69.425277191080345623322251656443E+00, 
    76.807047763862732837609972285484E+00, 
    85.230358607545669169387065607043E+00, 
    95.188939891525629981308606853957E+00, 
    107.95224382757871475002440117666E+00
  };
  static double x_063[63] =
  {
    0.22768893732576153785994330248562E-01, 
    0.11998325242727824715771416426383E+00, 
    0.29494185444770149577427738517405E+00, 
    0.54779087896237725363865073775856E+00, 
    0.87869061179931901673895567052285E+00, 
    1.2878464335919706302309207788611E+00, 
    1.7755123815388553763979463268728E+00, 
    2.3419925567085989256055628337716E+00, 
    2.9876423223246473939976731053629E+00, 
    3.7128695992018000346299637413422E+00, 
    4.5181363349503584391105568561550E+00, 
    5.4039601781825946286902599782736E+00, 
    6.3709163787865330220392250891777E+00, 
    7.4196399339311711154888493199004E+00, 
    8.5508280008403328312589048722235E+00, 
    9.7652425999245366807004592977996E+00, 
    11.063713635140661736220550410604E+00, 
    12.447142262356492749798687569289E+00, 
    13.916504641057818562912967008183E+00, 
    15.472856110036296424777143607779E+00, 
    17.117335833863588753116900303886E+00, 
    18.851171974154856850873483787506E+00, 
    20.675687448056515660377265667433E+00, 
    22.592306346311528381292277759986E+00, 
    24.602561094972638883700642760037E+00, 
    26.708100458737343969779087998829E+00, 
    28.910698500451382640177718103234E+00, 
    31.212264631175912885477773820802E+00, 
    33.614854909101154836598842888345E+00, 
    36.120684774484823056306328740825E+00, 
    38.732143442933582145626041607663E+00, 
    41.451810222318741191114726181363E+00, 
    44.282473071479233839358857134636E+00, 
    47.227149784295686898935095231536E+00, 
    50.289112264240695761749021839419E+00, 
    53.471914456788652808348280619542E+00, 
    56.779424636342062213099781057119E+00, 
    60.215862909019862886417550114424E+00, 
    63.785845004235974631701139601836E+00, 
    67.494433702293885830374325695045E+00, 
    71.347199604295266286654803376075E+00, 
    75.350293425653234254290504744279E+00, 
    79.510532629986309149555391354778E+00, 
    83.835506080872257843339817658508E+00, 
    88.333701570354369086112766326498E+00, 
    93.014662728558547405303399037100E+00, 
    97.889184147578140043386727677112E+00, 
    102.96955690741381650783952746778E+00, 
    108.26988161961595392226350967206E+00, 
    113.80647350287462738934485955901E+00, 
    119.59839538830458666962452963285E+00, 
    125.66817255856119431291196303280E+00, 
    132.04277272091165746585590583045E+00, 
    138.75498418103789078167590567526E+00, 
    145.84541318313540358283994248439E+00, 
    153.36548459497863623710815962660E+00, 
    161.38215194813761243562172669592E+00, 
    169.98570600665839438795175301156E+00, 
    179.30366247401580910251827858515E+00, 
    189.52789596532475473668721332981E+00, 
    200.97521159924656741628671841018E+00, 
    214.25368536638788642698056296400E+00, 
    230.93465747089703971246562985079E+00
  };
  static double x_127[127] =
  {
    0.11339635298518611691893169631306E-01, 
    0.59749753435726620281348237057387E-01, 
    0.14685098690746167612388223687431E+00, 
    0.27267590735859553131378008278900E+00, 
    0.43724600644192665554577035869932E+00, 
    0.64058688222566929533576416399983E+00, 
    0.88272968639058364481487653650042E+00, 
    1.1637114160166537661560584700951E+00, 
    1.4835750152834613891313584861012E+00, 
    1.8423694351613565380686320809853E+00, 
    2.2401496839579024244513315656522E+00, 
    2.6769768780141303692167869961238E+00, 
    3.1529182957082825565771508308846E+00, 
    3.6680474360304752540226339926515E+00, 
    4.2224440823301888455977876667425E+00, 
    4.8161943715870502475665535087286E+00, 
    5.4493908694559416755862178908416E+00, 
    6.1221326512997254193944584763155E+00, 
    6.8345253894122668112237994973336E+00, 
    7.5866814466367472174205986836847E+00, 
    8.3787199765932725254842120659452E+00, 
    9.2107670307426558777922506102445E+00, 
    10.082955672528643809166439353647E+00, 
    10.995426098858125429803147358780E+00, 
    11.948325769197725997610605127857E+00, 
    12.941809542585531053723381098192E+00, 
    13.976039822878506520014405668679E+00, 
    15.051186712579523631574796365435E+00, 
    16.167428175612852922977395051768E+00, 
    17.324950209443673446561163712616E+00, 
    18.523947026965688560811711309349E+00, 
    19.764621248611504104071669386884E+00, 
    21.047184105173183606877044020054E+00, 
    22.371855651855542817648123918101E+00, 
    23.738864994122497183652313788712E+00, 
    25.148450525937368234077278385644E+00, 
    26.600860181041749607253384279755E+00, 
    28.096351697964619201753961292129E+00, 
    29.635192899504178910610227138642E+00, 
    31.217661987479759144214467152615E+00, 
    32.844047853610430460522951341338E+00, 
    34.514650407441149149105635947422E+00, 
    36.229780922306804019615388508885E+00, 
    37.989762400399956435968780140278E+00, 
    39.794929958089961778396437141707E+00, 
    41.645631232730180705153990897484E+00, 
    43.542226812286859549950892993822E+00, 
    45.485090689228791137996151336673E+00, 
    47.474610740231964719468766599146E+00, 
    49.511189233379087716728884584381E+00, 
    51.595243364671244443182771266934E+00, 
    53.727205825819316758288140069145E+00, 
    55.907525405447553305830605991732E+00, 
    58.136667626022439197077526025660E+00, 
    60.415115419018590295707192053805E+00, 
    62.743369841051809700207126742685E+00, 
    65.121950833949996311956025417139E+00, 
    67.551398031997886314411872443149E+00, 
    70.032271619884584511229871192030E+00, 
    72.565153245206849090888669416801E+00, 
    75.150646989739935299354362325096E+00, 
    77.789380404085816000647405462136E+00, 
    80.482005610750729205803962926758E+00, 
    83.229200481195914886796120019048E+00, 
    86.031669892953582966798238732643E+00, 
    88.890147073512051099652518544282E+00, 
    91.805395038358177994971250170499E+00, 
    94.778208131331583205387031034825E+00, 
    97.809413676305116411054110115424E+00, 
    100.89987375017285940371939762172E+00, 
    104.05048708821598934704076845022E+00, 
    107.26219113414600428423116401414E+00, 
    110.53596424851500530602771351277E+00, 
    113.87282809075839485348376187652E+00, 
    117.27385019192517774095477886379E+00, 
    120.74014673718880106173978002719E+00, 
    124.27288557955698354259506446928E+00, 
    127.87328950885942645093841745425E+00, 
    131.54263980314366921809377742137E+00, 
    135.28228009311836970132738106369E+00, 
    139.09362057432970013964422086977E+00, 
    142.97814260643601776808227753574E+00, 
    146.93740374437366549441080969072E+00, 
    150.97304325252187127492511437460E+00, 
    155.08678816034612572229641420609E+00, 
    159.28045992663288235401956989889E+00, 
    163.55598178957571104015967182053E+00, 
    167.91538689194360134245547184721E+00, 
    172.36082728473812536838156191681E+00, 
    176.89458392960192176311674993508E+00, 
    181.51907784036813069227528834025E+00, 
    186.23688252828112373861202530357E+00, 
    191.05073794450929196790836610789E+00, 
    195.96356614879879837839002542988E+00, 
    200.97848897600025153696475526130E+00, 
    206.09884802468871112127283042753E+00, 
    211.32822735671655260572377256981E+00, 
    216.67047937658230323477089465777E+00, 
    222.12975445929687246267304963754E+00, 
    227.71053502072232419089132431317E+00, 
    233.41767488282602453367775322563E+00, 
    239.25644498830308620018749667089E+00, 
    245.23258677871567172531254018984E+00, 
    251.35237488718128030005500991754E+00, 
    257.62269123792061413076191882313E+00, 
    264.05111322908240551754377241831E+00, 
    270.64601945722796749299111718606E+00, 
    277.41671750163651071798388218104E+00, 
    284.37359974220870326674402873120E+00, 
    291.52833521346495719581282021650E+00, 
    298.89410837028248600878895615414E+00, 
    306.48591978262611320418112423947E+00, 
    314.32096986471177487400007507615E+00, 
    322.41915589128679683349440361344E+00, 
    330.80372663802405651933847334878E+00, 
    339.50216127832433747735367595958E+00, 
    348.54737559472697355480761787441E+00, 
    357.97942028029845454049007443090E+00, 
    367.84794520076004578858341422871E+00, 
    378.21590623135532818332979188889E+00, 
    389.16539141251004101579475325153E+00, 
    400.80729331451702589996361286427E+00, 
    413.29853681779384418008260081859E+00, 
    426.87579153663675538288509017051E+00, 
    441.93085485310841412460309271842E+00, 
    459.21804639888429981971267313224E+00, 
    480.69378263388373859704269229304E+00
  };

  if ( order == 1 )
  {
    webbur::r8vec_copy ( order, x_001, x );
  }
  else if ( order == 3 )
  {
    webbur::r8vec_copy ( order, x_003, x );
  }
  else if ( order == 7 )
  {
    webbur::r8vec_copy ( order, x_007, x );
  }
  else if ( order == 15 )
  {
    webbur::r8vec_copy ( order, x_015, x );
  }
  else if ( order == 31 )
  {
    webbur::r8vec_copy ( order, x_031, x );
  }
  else if ( order == 63 )
  {
    webbur::r8vec_copy ( order, x_063, x );
  }
  else if ( order == 127 )
  {
    webbur::r8vec_copy ( order, x_127, x );
  }
  else
  {
    std::cerr << "\n";
    std::cerr << "LAGUERRE_LOOKUP_POINTS - Fatal error!\n";
    std::cerr << "  Unexpected value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }
  return;
}
//****************************************************************************80

void laguerre_lookup_weights ( int order, double alpha, double beta,
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    LAGUERRE_LOOKUP_WEIGHTS looks up Laguerre quadrature weights.
//
//  Discussion:
//
//    The allowed orders are 1, 3, 7, 15, 31, 63 and 127.
//
//    The weights are positive, symmetric and should sum to 2.
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Milton Abramowitz, Irene Stegun,
//    Handbook of Mathematical Functions,
//    National Bureau of Standards, 1964,
//    ISBN: 0-486-61272-4,
//    LC: QA47.A34.
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    ORDER must be 1, 3, 7, 15, 31, 63 or 127.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  static double w_001[1] =
  {
    1.0E+00
  };
  static double w_003[3] = 
  {
    0.711093009929173015449590191143E+00,
    0.278517733569240848801444888457E+00,
    0.103892565015861357489649204007E-01
  };
  static double w_007[7] =
  {
    0.409318951701273902130432880018E+00,
    0.421831277861719779929281005417E+00,
    0.147126348657505278395374184637E+00,
    0.206335144687169398657056149642E-01,
    0.107401014328074552213195962843E-02,
    0.158654643485642012687326223234E-04,
    0.317031547899558056227132215385E-07
  };
  static double w_015[15] =
  {
    0.218234885940086889856413236448E+00,
    0.342210177922883329638948956807E+00,
    0.263027577941680097414812275022E+00,
    0.126425818105930535843030549378E+00,
    0.402068649210009148415854789871E-01,
    0.856387780361183836391575987649E-02,
    0.121243614721425207621920522467E-02,
    0.111674392344251941992578595518E-03,
    0.645992676202290092465319025312E-05,
    0.222631690709627263033182809179E-06,
    0.422743038497936500735127949331E-08,
    0.392189726704108929038460981949E-10,
    0.145651526407312640633273963455E-12,
    0.148302705111330133546164737187E-15,
    0.160059490621113323104997812370E-19
  };
  static double w_031[31] =
  {
     0.11252789550372583820847728082801E+00,
     0.21552760818089123795222505285045E+00,
     0.23830825164569654731905788089234E+00,
     0.19538830929790229249915303390711E+00,
     0.12698283289306190143635272904602E+00,
     0.67186168923899300670929441993508E-01,
     0.29303224993879487404888669311974E-01,
     0.10597569915295736089529380314433E-01,
     0.31851272582386980320974842433019E-02,
     0.79549548307940382922092149012477E-03,
     0.16480052126636687317862967116412E-03,
     0.28229237864310816393860971468993E-04,
     0.39802902551008580387116174900106E-05,
     0.45931839841801061673729694510289E-06,
     0.43075545187731100930131457465897E-07,
     0.32551249938271570855175749257884E-08,
     0.19620246675410594996247151593142E-09,
     0.93190499086617587129534716431331E-11,
     0.34377541819411620520312597898311E-12,
     0.96795247130446716997405035776206E-14,
     0.20368066110115247398010624219291E-15,
     0.31212687280713526831765358632585E-17,
     0.33729581704161052453395678308350E-19,
     0.24672796386616696011038363242541E-21,
     0.11582201904525643634834564576593E-23,
     0.32472922591425422434798022809020E-26,
     0.49143017308057432740820076259666E-29,
     0.34500071104808394132223135953806E-32,
     0.87663710117162041472932760732881E-36,
     0.50363643921161490411297172316582E-40,
     0.19909984582531456482439549080330E-45
  };
  static double w_063[63] =
  {
     0.57118633213868979811587283390476E-01,
     0.12067476090640395283319932036351E+00,
     0.15925001096581873723870561096472E+00,
     0.16875178327560799234596192963585E+00,
     0.15366641977668956696193711310131E+00,
     0.12368770614716481641086652261948E+00,
     0.89275098854848671545279150057422E-01,
     0.58258485446105944957571825725160E-01,
     0.34546657545992580874717085812508E-01,
     0.18675685985714656798286552591203E-01,
     0.92233449044093536528490075241649E-02,
     0.41671250684839592762582663470209E-02,
     0.17238120299900582715386728541955E-02,
     0.65320845029716311169340559359043E-03,
     0.22677644670909586952405173207471E-03,
     0.72127674154810668410750270234861E-04,
     0.21011261180466484598811536851241E-04,
     0.56035500893357212749181536071292E-05,
     0.13673642785604888017836641282292E-05,
     0.30507263930195817240736097189550E-06,
     0.62180061839309763559981775409241E-07,
     0.11566529551931711260022448996296E-07,
     0.19614588267565478081534781863335E-08,
     0.30286171195709411244334756404054E-09,
     0.42521344539400686769012963452599E-10,
     0.54202220578073819334698791381873E-11,
     0.62627306838597672554166850420603E-12,
     0.65474443156573322992307089591924E-13,
     0.61815575808729181846302500000047E-14,
     0.52592721363507381404263991342633E-15,
     0.40230920092646484015391506025408E-16,
     0.27600740511819536505013824207729E-17,
     0.16936946756968296053322009855265E-18,
     0.92689146872177087314963772462726E-20,
     0.45093739060365632939780140603959E-21,
     0.19435162876132376573629962695374E-22,
     0.73926270895169207037999639194513E-24,
     0.24714364154434632615980126000066E-25,
     0.72288649446741597655145390616476E-27,
     0.18407617292614039362985209905608E-28,
     0.40583498566841960105759537058880E-30,
     0.77000496416438368114463925286343E-32,
     0.12488505764999334328843314866038E-33,
     0.17185000226767010697663950619912E-35,
     0.19896372636672396938013975755522E-37,
     0.19199671378804058267713164416870E-39,
     0.15278588285522166920459714708240E-41,
     0.99054752688842142955854138884590E-44,
     0.51597523673029211884228858692990E-46,
     0.21249846664084111245693912887783E-48,
     0.67903852766852910591172042494884E-51,
     0.16466654148296177467908300517887E-53,
     0.29509065402691055027053659375033E-56,
     0.37838420647571051984882241014675E-59,
     0.33358130068542431878174667995217E-62,
     0.19223461022273880981363303073329E-65,
     0.67812696961083016872779388922288E-69,
     0.13404752802440604607620468935693E-72,
     0.13109745101805029757648048223928E-76,
     0.52624863881401787388694579143866E-81,
     0.63780013856587414257760666006511E-86,
     0.12997078942372924566347473916943E-91,
     0.10008511496968754063443740168421E-98
  };
  static double w_127[127] =
  {
     0.28773246692000124355770010301506E-01,
     0.63817468175134649363480949265236E-01,
     0.91919669721570571389864194652717E-01,
     0.11054167914413766381245463002967E+00,
     0.11879771633375850188328329422643E+00,
     0.11737818530052695148804451630074E+00,
     0.10819305984180551488335145581193E+00,
     0.93827075290489628080377261401107E-01,
     0.76966450960588843995822485928431E-01,
     0.59934903912939714332570730063476E-01,
     0.44417742073889001371708316272923E-01,
     0.31385080966252320983009372215062E-01,
     0.21172316041924506411370709025015E-01,
     0.13650145364230541652171185564626E-01,
     0.84172852710599172279366657385445E-02,
     0.49674990059882760515912858620175E-02,
     0.28069903895001884631961957446400E-02,
     0.15192951003941952460445341057817E-02,
     0.78789028751796084086217287140548E-03,
     0.39156751064868450584507324648999E-03,
     0.18652434268825860550093566260060E-03,
     0.85173160415576621908809828160247E-04,
     0.37285639197853037712145321577724E-04,
     0.15648416791712993947447805296768E-04,
     0.62964340695224829035692735524979E-05,
     0.24288929711328724574541379938222E-05,
     0.89824607890051007201922871545035E-06,
     0.31844174740760353710742966328091E-06,
     0.10821272905566839211861807542741E-06,
     0.35245076750635536015902779085340E-07,
     0.11001224365719347407063839761738E-07,
     0.32904079616717932125329343003261E-08,
     0.94289145237889976419772700772988E-09,
     0.25882578904668318184050195309296E-09,
     0.68047437103370762630942259017560E-10,
     0.17131398805120837835399564475632E-10,
     0.41291744524052865469443922304935E-11,
     0.95264189718807273220707664873469E-12,
     0.21032604432442425932962942047474E-12,
     0.44427151938729352860940434285789E-13,
     0.89760500362833703323319846405449E-14,
     0.17341511407769287074627948346848E-14,
     0.32028099548988356631494379835210E-15,
     0.56531388950793682022660742095189E-16,
     0.95329672799026591234588044025896E-17,
     0.15353453477310142565288509437552E-17,
     0.23608962179467365686057842132176E-18,
     0.34648742794456611332193876653230E-19,
     0.48515241897086461320126957663545E-20,
     0.64786228633519813428137373790678E-21,
     0.82476020965403242936448553126316E-22,
     0.10005361880214719793491658282977E-22,
     0.11561395116207304954233181263632E-23,
     0.12719342731167922655612134264961E-24,
     0.13316584714165372967340004160814E-25,
     0.13261218454678944033646108509198E-26,
     0.12554995447643949807286074138324E-27,
     0.11294412178579462703240913107219E-28,
     0.96491020279562119228500608131696E-30,
     0.78241846768302099396733076955632E-31,
     0.60181503542219626658249939076636E-32,
     0.43882482704961741551510518054138E-33,
     0.30314137647517256304035802501863E-34,
     0.19826016543944539545224676057020E-35,
     0.12267623373665926559013654872402E-36,
     0.71763931692508888943812834967620E-38,
     0.39659378833836963584113716149270E-39,
     0.20688970553868040099581951696677E-40,
     0.10179587017979517245268418427523E-41,
     0.47200827745986374625714293679649E-43,
     0.20606828985553374825744353490744E-44,
     0.84627575907305987245899032156188E-46,
     0.32661123687088798658026998931647E-47,
     0.11833939207883162380564134612682E-48,
     0.40211209123895013807243250164050E-50,
     0.12799824394111125389430292847476E-51,
     0.38123877747548846504399051365162E-53,
     0.10612057542701156767898551949650E-54,
     0.27571446947200403594113572720812E-56,
     0.66772544240928492881306904862856E-58,
     0.15052438383868234954068178600268E-59,
     0.31538986800113758526689068500772E-61,
     0.61326614299483180785237418887960E-63,
     0.11048510030324810567549119229368E-64,
     0.18410563538091348076979665543900E-66,
     0.28323926570052832195543883237652E-68,
     0.40154409843763655508670978777418E-70,
     0.52351530215683708779772201956106E-72,
     0.62634476665005100555787696642851E-74,
     0.68612210535666530365348093803922E-76,
     0.68651298840956019297134099761855E-78,
     0.62581388433728084867318704240915E-80,
     0.51833271237514904046803469968027E-82,
     0.38893621571918443533108973497673E-84,
     0.26357711379476932781525533730623E-86,
     0.16078851293917979699005509638883E-88,
     0.87978042070968939637972577886624E-91,
     0.43013405077495109903408697802188E-93,
     0.18713435881342838527144321803729E-95,
     0.72125744708060471675805761366523E-98,
     0.24508746062177874383231742333023E-100,
     0.73042094619470875777647865078327E-103,
     0.18983290818383463537886818579820E-105,
     0.42757400244246684123093264825902E-108,
     0.82894681420515755691423485228897E-111,
     0.13729432219324400013067050156048E-113,
     0.19265464126404973222043166489406E-116,
     0.22693344503301354826140809941334E-119,
     0.22209290603717355061909071271535E-122,
     0.17851087685544512662856555121755E-125,
     0.11630931990387164467431190485525E-128,
     0.60524443584652392290952805077893E-132,
     0.24729569115063528647628375096400E-135,
     0.77789065006489410364997205809045E-139,
     0.18409738662712607039570678274636E-142,
     0.31900921131079114970179071968597E-146,
     0.39179487139174199737617666077555E-150,
     0.32782158394188697053774429820559E-154,
     0.17793590713138888062819640128739E-158,
     0.58882353408932623157467835381214E-163,
     0.10957236509071169877747203273886E-167,
     0.10281621114867000898285076975760E-172,
     0.41704725557697758145816510853967E-178,
     0.58002877720316101774638319601971E-184,
     0.18873507745825517106171619101120E-190,
     0.69106601826730911682786705950895E-198,
     0.43506813201105855628383313334402E-207
  };
  if ( order == 1 )
  {
    webbur::r8vec_copy ( order, w_001, w );
  }
  else if ( order == 3 )
  {
    webbur::r8vec_copy ( order, w_003, w );
  }
  else if ( order == 7 )
  {
    webbur::r8vec_copy ( order, w_007, w );
  }
  else if ( order == 15 )
  {
    webbur::r8vec_copy ( order, w_015, w );
  }
  else if ( order == 31 )
  {
    webbur::r8vec_copy ( order, w_031, w );
  }
  else if ( order == 63 )
  {
    webbur::r8vec_copy ( order, w_063, w );
  }
  else if ( order == 127 )
  {
    webbur::r8vec_copy ( order, w_127, w );
  }
  else
  {
    std::cerr << "\n";
    std::cerr << "LAGUERRE_LOOKUP_WEIGHTS - Fatal error!\n";
    std::cerr << "  Unexpected value of ORDER = " << order << ".\n";
    std::exit ( 1 );
  }
  return;
}
//****************************************************************************80

void laguerre_recur ( double *p2, double *dp2, double *p1, double x, 
  int order, double b[], double c[] )

//****************************************************************************80
//
//  Purpose:
//
//    LAGUERRE_RECUR evaluates a Laguerre polynomial.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Output, double *P2, the value of L(ORDER)(X).
//
//    Output, double *DP2, the value of L'(ORDER)(X).
//
//    Output, double *P1, the value of L(ORDER-1)(X).
//
//    Input, double X, the point at which polynomials are evaluated.
//
//    Input, int ORDER, the order of the polynomial to be computed.
//
//    Input, double B[ORDER], C[ORDER], the recursion coefficients.
//
{
  double dp0;
  double dp1;
  int i;
  double p0;

  *p1 = 1.0;
  dp1 = 0.0;

  *p2 = x - 1.0;
  *dp2 = 1.0;

  for ( i = 1; i < order; i++ )
  {
    p0 = *p1;
    dp0 = dp1;

    *p1 = *p2;
    dp1 = *dp2;

    *p2  = ( x - b[i] ) * ( *p1 ) - c[i] * p0;
    *dp2 = ( x - b[i] ) * dp1 + ( *p1 ) - c[i] * dp0;
  }

  return;
}
//****************************************************************************80

void laguerre_root ( double *x, int order, double *dp2, double *p1, 
  double b[], double c[] )

//****************************************************************************80
//
//  Purpose:
//
//    LAGUERRE_ROOT improves a root of a Laguerre polynomial.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by Arthur Stroud, Don Secrest.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input/output, double *X, the approximate root, which
//    should be improved on output.
//
//    Input, int ORDER, the order of the polynomial to be computed.
//
//    Output, double *DP2, the value of L'(ORDER)(X).
//
//    Output, double *P1, the value of L(ORDER-1)(X).
//
//    Input, double B[ORDER], C[ORDER], the recursion coefficients.
//
{
  double d;
  double eps;
  double p2;
  int step;
  int step_max = 10;

  eps = webbur::r8_epsilon ( );

  for ( step = 1; step <= step_max; step++ )
  {
    webbur::laguerre_recur ( &p2, dp2, p1, *x, order, b, c );

    d = p2 / ( *dp2 );
    *x = *x - d;

    if ( webbur::r8_abs ( d ) <= eps * ( webbur::r8_abs ( *x ) + 1.0 ) )
    {
      break;
    }
  }

  return;
}
//****************************************************************************80

void legendre_compute ( int order, double alpha, double beta, double x[],
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    LEGENDRE_COMPUTE computes a Legendre quadrature rule.
//
//  Discussion:
//
//    The integration interval is [ -1, 1 ].
//
//    The weight function is w(x) = 1.0.
//
//    The integral to approximate:
//
//      Integral ( -1 <= X <= 1 ) F(X) dX
//
//    The quadrature rule:
//
//      Sum ( 1 <= I <= ORDER ) W(I) * F ( X(I) )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    Original FORTRAN77 version by Philip Davis, Philip Rabinowitz.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Philip Davis, Philip Rabinowitz,
//    Methods of Numerical Integration,
//    Second Edition,
//    Dover, 2007,
//    ISBN: 0486453391,
//    LC: QA299.3.D28.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    1 <= ORDER.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas of the rule.
//
//    Output, double W[ORDER], the weights of the rule.
//    The weights are positive, symmetric, and should sum to 2.
//
{
  double d1;
  double d2pn;
  double d3pn;
  double d4pn;
  double dp;
  double dpn;
  double e1;
  double fx;
  double h;
  int i;
  int iback;
  int k;
  int m;
  int mp1mi;
  int ncopy;
  int nmove;
  double p;
  double pi = 3.141592653589793;
  double pk;
  double pkm1;
  double pkp1;
  double t;
  double u;
  double v;
  double x0;
  double xtemp;

  if ( order < 1 )
  {
    std::cerr << "\n";
    std::cerr << "LEGENDRE_COMPUTE - Fatal error!\n";
    std::cerr << "  Illegal value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }

  e1 = ( double ) ( order * ( order + 1 ) );

  m = ( order + 1 ) / 2;

  for ( i = 1; i <= m; i++ )
  {
    mp1mi = m + 1 - i;

    t = ( double ) ( 4 * i - 1 ) * pi / ( double ) ( 4 * order + 2 );

    x0 =  std::cos ( t ) * ( 1.0 - ( 1.0 - 1.0 / ( double ) ( order ) ) 
      / ( double ) ( 8 * order * order ) );

    pkm1 = 1.0;
    pk = x0;

    for ( k = 2; k <= order; k++ )
    {
      pkp1 = 2.0 * x0 * pk - pkm1 - ( x0 * pk - pkm1 ) / ( double ) ( k );
      pkm1 = pk;
      pk = pkp1;
    }

    d1 = ( double ) ( order ) * ( pkm1 - x0 * pk );

    dpn = d1 / ( 1.0 - x0 * x0 );

    d2pn = ( 2.0 * x0 * dpn - e1 * pk ) / ( 1.0 - x0 * x0 );

    d3pn = ( 4.0 * x0 * d2pn + ( 2.0 - e1 ) * dpn ) / ( 1.0 - x0 * x0 );

    d4pn = ( 6.0 * x0 * d3pn + ( 6.0 - e1 ) * d2pn ) / ( 1.0 - x0 * x0 );

    u = pk / dpn;
    v = d2pn / dpn;
//
//  Initial approximation H:
//
    h = -u * ( 1.0 + 0.5 * u * ( v + u * ( v * v - d3pn / ( 3.0 * dpn ) ) ) );
//
//  Refine H using one step of Newton's method:
//
    p = pk + h * ( dpn + 0.5 * h * ( d2pn + h / 3.0 
      * ( d3pn + 0.25 * h * d4pn ) ) );

    dp = dpn + h * ( d2pn + 0.5 * h * ( d3pn + h * d4pn / 3.0 ) );

    h = h - p / dp;

    xtemp = x0 + h;

    x[mp1mi-1] = xtemp;

    fx = d1 - h * e1 * ( pk + 0.5 * h * ( dpn + h / 3.0 
      * ( d2pn + 0.25 * h * ( d3pn + 0.2 * h * d4pn ) ) ) );

    w[mp1mi-1] = 2.0 * ( 1.0 - xtemp * xtemp ) / ( fx * fx );
  }

  if ( ( order % 2 ) == 1 )
  {
    x[0] = 0.0;
  }
//
//  Shift the data up.
//
  nmove = ( order + 1 ) / 2;
  ncopy = order - nmove;

  for ( i = 1; i <= nmove; i++ )
  {
    iback = order + 1 - i;
    x[iback-1] = x[iback-ncopy-1];
    w[iback-1] = w[iback-ncopy-1];
  }
//
//  Reflect values for the negative abscissas.
//
  for ( i = 1; i <= order - nmove; i++ )
  {
    x[i-1] = - x[order-i];
    w[i-1] = w[order-i];
  }

  return;
}
//****************************************************************************80

void legendre_compute_points ( int order, double alpha, double beta,
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    LEGENDRE_COMPUTE_POINTS computes Legendre quadrature points.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  double *w;

  w= new double[order];

  webbur::legendre_compute ( order, alpha, beta, x, w );

  delete [] w;

  return;
}
//****************************************************************************80

void legendre_compute_weights ( int order, double alpha, double beta,
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    LEGENDRE_COMPUTE_WEIGHTS computes Legendre quadrature weights.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license.
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  double *x;

  x = new double[order];

  webbur::legendre_compute ( order, alpha, beta, x, w );

  delete [] x;

  return;
}
//****************************************************************************80

double legendre_integral ( int expon )

//****************************************************************************80
//
//  Purpose:
//
//    LEGENDRE_INTEGRAL evaluates a monomial Legendre integral.
//
//  Discussion:
//
//    To test a Legendre quadrature rule, we use it to approximate the
//    integral of a monomial:
//
//      integral ( -1 <= x <= +1 ) x^n dx
///
//    This routine is given the value of the exponent, and returns the
//    exact value of the integral.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int EXPON, the exponent.
//
//    Output, double LEGENDRE_INTEGRAL, the value of the exact integral.
//
{
  double exact;
//
//  Get the exact value of the integral.
//
  if ( ( expon % 2 ) == 0 )
  {
    exact = 2.0 / ( double ) ( expon + 1 );
  }
  else
  {
    exact = 0.0;
  }

  return exact;
}
//****************************************************************************80

void legendre_lookup_points ( int order, double alpha, double beta,
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    LEGENDRE_LOOKUP_POINTS looks up Legendre quadrature points.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on [-1,1].
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    Legal values are 1, 3, 7, 15, 31, 63 and 127.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  static double x_001[1] =
  {
       0.0E+00
  };
  static double x_003[3] =
  {
     - 0.774596669241483377035853079956E+00, 
       0.0E+00, 
       0.774596669241483377035853079956E+00
  };
  static double x_007[7] =
  {
     - 0.949107912342758524526189684048E+00, 
     - 0.741531185599394439863864773281E+00, 
     - 0.405845151377397166906606412077E+00, 
       0.0E+00, 
       0.405845151377397166906606412077E+00, 
       0.741531185599394439863864773281E+00, 
       0.949107912342758524526189684048E+00
  };
  static double x_015[15] =
  {
      - 0.987992518020485428489565718587E+00, 
      - 0.937273392400705904307758947710E+00, 
      - 0.848206583410427216200648320774E+00, 
      - 0.724417731360170047416186054614E+00, 
      - 0.570972172608538847537226737254E+00, 
      - 0.394151347077563369897207370981E+00, 
      - 0.201194093997434522300628303395E+00, 
        0.0E+00, 
        0.201194093997434522300628303395E+00, 
       0.394151347077563369897207370981E+00, 
       0.570972172608538847537226737254E+00, 
       0.724417731360170047416186054614E+00, 
       0.848206583410427216200648320774E+00, 
       0.937273392400705904307758947710E+00, 
       0.987992518020485428489565718587E+00
  };
  static double x_031[31] =
  {
      -0.99708748181947707454263838179654,    
      -0.98468590966515248400211329970113,    
      -0.96250392509294966178905249675943,    
      -0.93075699789664816495694576311725,    
      -0.88976002994827104337419200908023,    
      -0.83992032014626734008690453594388,    
      -0.78173314841662494040636002019484,    
      -0.71577678458685328390597086536649,    
      -0.64270672292426034618441820323250,    
      -0.56324916140714926272094492359516,    
      -0.47819378204490248044059403935649,    
      -0.38838590160823294306135146128752,    
      -0.29471806998170161661790389767170,    
      -0.19812119933557062877241299603283,    
      -0.99555312152341520325174790118941E-01,
       0.00000000000000000000000000000000,
       0.99555312152341520325174790118941E-01,
       0.19812119933557062877241299603283,    
       0.29471806998170161661790389767170,    
       0.38838590160823294306135146128752,    
       0.47819378204490248044059403935649,    
       0.56324916140714926272094492359516,
       0.64270672292426034618441820323250,    
       0.71577678458685328390597086536649,    
       0.78173314841662494040636002019484,    
       0.83992032014626734008690453594388,    
       0.88976002994827104337419200908023,    
       0.93075699789664816495694576311725,    
       0.96250392509294966178905249675943,    
       0.98468590966515248400211329970113,   
       0.99708748181947707454263838179654
  };
  static double x_063[63] = 
  {
      -0.99928298402912378050701628988630E+00,     
      -0.99622401277797010860209018267357E+00,     
      -0.99072854689218946681089469460884E+00,     
      -0.98280881059372723486251140727639E+00,     
      -0.97248403469757002280196067864927E+00,     
      -0.95977944975894192707035416626398E+00,     
      -0.94472613404100980296637531962798E+00,     
      -0.92736092062184320544703138132518E+00,     
      -0.90772630277853155803695313291596E+00,     
      -0.88587032850785342629029845731337E+00,     
      -0.86184648236412371953961183943106E+00,     
      -0.83571355431950284347180776961571E+00,     
      -0.80753549577345676005146598636324E+00,     
      -0.77738126299037233556333018991104E+00,     
      -0.74532464831784741782932166103759E+00,     
      -0.71144409958484580785143153770401E+00,     
      -0.67582252811498609013110331596954E+00,     
      -0.63854710582136538500030695387338E+00,     
      -0.59970905187762523573900892686880E+00,     
      -0.55940340948628501326769780007005E+00,     
      -0.51772881329003324812447758452632E+00,     
      -0.47478724799480439992221230985149E+00,     
      -0.43068379879511160066208893391863E+00,     
      -0.38552639421224789247761502227440E+00,     
      -0.33942554197458440246883443159432E+00,     
      -0.29249405858625144003615715555067E+00,     
      -0.24484679324595336274840459392483E+00,     
      -0.19660034679150668455762745706572E+00,     
      -0.14787278635787196856983909655297E+00,     
      -0.98783356446945279529703669453922E-01, 
      -0.49452187116159627234233818051808E-01, 
       0.00000000000000000000000000000000E+00,     
       0.49452187116159627234233818051808E-01, 
       0.98783356446945279529703669453922E-01, 
       0.14787278635787196856983909655297E+00,     
       0.19660034679150668455762745706572E+00,     
       0.24484679324595336274840459392483E+00,     
       0.29249405858625144003615715555067E+00,     
       0.33942554197458440246883443159432E+00,     
       0.38552639421224789247761502227440E+00,     
       0.43068379879511160066208893391863E+00,     
       0.47478724799480439992221230985149E+00,     
       0.51772881329003324812447758452632E+00,     
       0.55940340948628501326769780007005E+00,     
       0.59970905187762523573900892686880E+00,     
       0.63854710582136538500030695387338E+00,     
       0.67582252811498609013110331596954E+00,     
       0.71144409958484580785143153770401E+00,     
       0.74532464831784741782932166103759E+00,     
       0.77738126299037233556333018991104E+00,     
       0.80753549577345676005146598636324E+00,     
       0.83571355431950284347180776961571E+00,     
       0.86184648236412371953961183943106E+00,     
       0.88587032850785342629029845731337E+00,     
       0.90772630277853155803695313291596E+00,     
       0.92736092062184320544703138132518E+00,     
       0.94472613404100980296637531962798E+00,     
       0.95977944975894192707035416626398E+00,     
       0.97248403469757002280196067864927E+00,     
       0.98280881059372723486251140727639E+00,     
       0.99072854689218946681089469460884E+00,     
       0.99622401277797010860209018267357E+00,  
       0.99928298402912378050701628988630E+00
  };
  static double x_127[127] = 
  {
       -0.99982213041530614629963254927125E+00,     
       -0.99906293435531189513828920479421E+00,     
       -0.99769756618980462107441703193392E+00,     
       -0.99572655135202722663543337085008E+00,     
       -0.99315104925451714736113079489080E+00,     
       -0.98997261459148415760778669967548E+00,     
       -0.98619317401693166671043833175407E+00,     
       -0.98181502080381411003346312451200E+00,     
       -0.97684081234307032681744391886221E+00,     
       -0.97127356816152919228894689830512E+00,     
       -0.96511666794529212109082507703391E+00,     
       -0.95837384942523877114910286998060E+00,     
       -0.95104920607788031054790764659636E+00,     
       -0.94314718462481482734544963026201E+00,     
       -0.93467258232473796857363487794906E+00,     
       -0.92563054405623384912746466814259E+00,     
       -0.91602655919146580931308861741716E+00,     
       -0.90586645826182138280246131760282E+00,     
       -0.89515640941708370896904382642451E+00,     
       -0.88390291468002656994525794802849E+00,     
       -0.87211280599856071141963753428864E+00,     
       -0.85979324109774080981203134414483E+00,     
       -0.84695169913409759845333931085437E+00,     
       -0.83359597615489951437955716480123E+00,     
       -0.81973418036507867415511910167470E+00,     
       -0.80537472720468021466656079404644E+00,     
       -0.79052633423981379994544995252740E+00,     
       -0.77519801587020238244496276354566E+00,     
       -0.75939907785653667155666366659810E+00,     
       -0.74313911167095451292056688997595E+00,    
       -0.72642798867407268553569290153270E+00,     
       -0.70927585412210456099944463906757E+00,     
       -0.69169312100770067015644143286666E+00,     
       -0.67369046373825048534668253831602E+00,     
       -0.65527881165548263027676505156852E+00,     
       -0.63646934240029724134760815684175E+00,     
       -0.61727347512685828385763916340822E+00,     
       -0.59770286357006522938441201887478E+00,     
       -0.57776938897061258000325165713764E+00,     
       -0.55748515286193223292186190687872E+00,     
       -0.53686246972339756745816636353452E+00,     
       -0.51591385950424935727727729906662E+00,     
       -0.49465204002278211739494017368636E+00,     
       -0.47308991924540524164509989939699E+00,     
       -0.45124058745026622733189858020729E+00,     
       -0.42911730928019337626254405355418E+00,     
       -0.40673351568978256340867288124339E+00,     
       -0.38410279579151693577907781452239E+00,     
       -0.36123888860586970607092484346723E+00,     
       -0.33815567472039850137600027657095E+00,     
       -0.31486716786289498148601475374890E+00,     
       -0.29138750639370562079451875284568E+00,     
       -0.26773094472238862088834352027938E+00,     
       -0.24391184465391785797071324453138E+00,     
       -0.21994466666968754245452337866940E+00,     
       -0.19584396114861085150428162519610E+00,     
       -0.17162435953364216500834492248954E+00,     
       -0.14730056544908566938932929319807E+00,     
       -0.12288734577408297172603365288567E+00,     
       -0.98399521677698970751091751509101E-01, 
       -0.73851959621048545273440409360569E-01, 
       -0.49259562331926630315379321821927E-01, 
       -0.24637259757420944614897071846088E-01, 
        0.00000000000000000000000000000000E+00,     
        0.24637259757420944614897071846088E-01, 
        0.49259562331926630315379321821927E-01, 
        0.73851959621048545273440409360569E-01, 
        0.98399521677698970751091751509101E-01, 
        0.12288734577408297172603365288567E+00,     
        0.14730056544908566938932929319807E+00,     
        0.17162435953364216500834492248954E+00,     
        0.19584396114861085150428162519610E+00,     
        0.21994466666968754245452337866940E+00,     
        0.24391184465391785797071324453138E+00,     
        0.26773094472238862088834352027938E+00,     
        0.29138750639370562079451875284568E+00,     
        0.31486716786289498148601475374890E+00,     
        0.33815567472039850137600027657095E+00,     
        0.36123888860586970607092484346723E+00,     
        0.38410279579151693577907781452239E+00,     
        0.40673351568978256340867288124339E+00,     
        0.42911730928019337626254405355418E+00,     
        0.45124058745026622733189858020729E+00,     
        0.47308991924540524164509989939699E+00,     
        0.49465204002278211739494017368636E+00,     
        0.51591385950424935727727729906662E+00,     
        0.53686246972339756745816636353452E+00,     
        0.55748515286193223292186190687872E+00,     
        0.57776938897061258000325165713764E+00,     
        0.59770286357006522938441201887478E+00,     
        0.61727347512685828385763916340822E+00,     
        0.63646934240029724134760815684175E+00,     
        0.65527881165548263027676505156852E+00,     
        0.67369046373825048534668253831602E+00,     
        0.69169312100770067015644143286666E+00,    
        0.70927585412210456099944463906757E+00,     
        0.72642798867407268553569290153270E+00,     
        0.74313911167095451292056688997595E+00,     
        0.75939907785653667155666366659810E+00,     
        0.77519801587020238244496276354566E+00,     
        0.79052633423981379994544995252740E+00,     
        0.80537472720468021466656079404644E+00,     
        0.81973418036507867415511910167470E+00,     
        0.83359597615489951437955716480123E+00,     
        0.84695169913409759845333931085437E+00,     
        0.85979324109774080981203134414483E+00,     
        0.87211280599856071141963753428864E+00,     
        0.88390291468002656994525794802849E+00,     
        0.89515640941708370896904382642451E+00,     
        0.90586645826182138280246131760282E+00,     
        0.91602655919146580931308861741716E+00,     
        0.92563054405623384912746466814259E+00,     
        0.93467258232473796857363487794906E+00,     
        0.94314718462481482734544963026201E+00,     
        0.95104920607788031054790764659636E+00,     
        0.95837384942523877114910286998060E+00,     
        0.96511666794529212109082507703391E+00,     
        0.97127356816152919228894689830512E+00,     
        0.97684081234307032681744391886221E+00,     
        0.98181502080381411003346312451200E+00,     
        0.98619317401693166671043833175407E+00,     
        0.98997261459148415760778669967548E+00,     
        0.99315104925451714736113079489080E+00,     
        0.99572655135202722663543337085008E+00,     
        0.99769756618980462107441703193392E+00,     
        0.99906293435531189513828920479421E+00,     
        0.99982213041530614629963254927125E+00 
  };

  if ( order == 1 )
  {
    webbur::r8vec_copy ( order, x_001, x );
  }
  else if ( order == 3 )
  {
    webbur::r8vec_copy ( order, x_003, x );
  }
  else if ( order == 7 )
  {
    webbur::r8vec_copy ( order, x_007, x);
  }
  else if ( order == 15 )
  {
    webbur::r8vec_copy ( order, x_015, x );
  }
  else if ( order == 31 )
  {
    webbur::r8vec_copy ( order, x_031, x );
  }
  else if ( order == 63 )
  {
    webbur::r8vec_copy ( order, x_063, x );
  }
  else if ( order == 127 )
  {
    webbur::r8vec_copy ( order, x_127, x );
  }
  else
  {
    std::cerr << "\n";
    std::cerr << "LEGENDRE_LOOKUP_POINTS - Fatal error!\n";
    std::cerr << "  Unexpected value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }
  return;
}
//****************************************************************************80

void legendre_lookup_weights ( int order, double alpha, double beta,
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    LEGENDRE_LOOKUP_WEIGHTS looks up Legendre quadrature weights.
//
//  Discussion:
//
//    The allowed orders are 1, 3, 7, 15, 31, 63 and 127.
//
//    The weights are positive, symmetric and should sum to 2.
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Milton Abramowitz, Irene Stegun,
//    Handbook of Mathematical Functions,
//    National Bureau of Standards, 1964,
//    ISBN: 0-486-61272-4,
//    LC: QA47.A34.
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    ORDER must be 1, 3, 7, 15, 31, 63 or 127.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  static double w_001[1] =
  {
    2.0E+00
  };
  static double w_003[3] = 
  {
    0.555555555555555555555555555555E+00,
    0.888888888888888888888888888888E+00,
    0.555555555555555555555555555555E+00
  };
  static double w_007[7] =
  {
    0.129484966168869693270611432679E+00,
    0.279705391489276667901467771424E+00,
    0.381830050505118944950369775489E+00,
    0.417959183673469387755102040816E+00,
    0.381830050505118944950369775489E+00,
    0.279705391489276667901467771424E+00,
    0.129484966168869693270611432679E+00
  };
  static double w_015[15] =
  {
    0.307532419961172683546283935772E-01,
    0.703660474881081247092674164507E-01,
    0.107159220467171935011869546686E+00,
    0.139570677926154314447804794511E+00,
    0.166269205816993933553200860481E+00,
    0.186161000015562211026800561866E+00,
    0.198431485327111576456118326444E+00,
    0.202578241925561272880620199968E+00,
    0.198431485327111576456118326444E+00,
    0.186161000015562211026800561866E+00,
    0.166269205816993933553200860481E+00,
    0.139570677926154314447804794511E+00,
    0.107159220467171935011869546686E+00,
    0.703660474881081247092674164507E-01,
    0.307532419961172683546283935772E-01
  };
  static double w_031[31] =
  {
    0.74708315792487746093913218970494E-02,
    0.17318620790310582463552990782414E-01,
    0.27009019184979421800608642617676E-01,
    0.36432273912385464024392008749009E-01,
    0.45493707527201102902315857856518E-01,
    0.54103082424916853711666259085477E-01,
    0.62174786561028426910343543686657E-01,
    0.69628583235410366167756126255124E-01,
    0.76390386598776616426357674901331E-01,
    0.82392991761589263903823367431962E-01,
    0.87576740608477876126198069695333E-01,
    0.91890113893641478215362871607150E-01,
    0.95290242912319512807204197487597E-01,
    0.97743335386328725093474010978997E-01,
    0.99225011226672307874875514428615E-01,
    0.99720544793426451427533833734349E-01,
    0.99225011226672307874875514428615E-01,
    0.97743335386328725093474010978997E-01,
    0.95290242912319512807204197487597E-01,
    0.91890113893641478215362871607150E-01,
    0.87576740608477876126198069695333E-01,
    0.82392991761589263903823367431962E-01,
    0.76390386598776616426357674901331E-01,
    0.69628583235410366167756126255124E-01,
    0.62174786561028426910343543686657E-01,
    0.54103082424916853711666259085477E-01,
    0.45493707527201102902315857856518E-01,
    0.36432273912385464024392008749009E-01,
    0.27009019184979421800608642617676E-01,
    0.17318620790310582463552990782414E-01,
    0.74708315792487746093913218970494E-02
  };
  static double w_063[63] =
  {
    0.18398745955770837880499331680577E-02,
    0.42785083468637618661951422543371E-02,
    0.67102917659601362519069109850892E-02,
    0.91259686763266563540586445877022E-02,
    0.11519376076880041750750606118707E-01,
    0.13884612616115610824866086365937E-01,
    0.16215878410338338882283672974995E-01,
    0.18507464460161270409260545805144E-01,
    0.20753761258039090775341953421471E-01,
    0.22949271004889933148942319561770E-01,
    0.25088620553344986618630138068443E-01,
    0.27166574359097933225189839439413E-01,
    0.29178047208280526945551502154029E-01,
    0.31118116622219817508215988557189E-01,
    0.32982034883779341765683179672459E-01,
    0.34765240645355877697180504642788E-01,
    0.36463370085457289630452409787542E-01,
    0.38072267584349556763638324927889E-01,
    0.39587995891544093984807928149202E-01,
    0.41006845759666398635110037009072E-01,
    0.42325345020815822982505485403028E-01,
    0.43540267083027590798964315704401E-01,
    0.44648638825941395370332669516813E-01,
    0.45647747876292608685885992608542E-01,
    0.46535149245383696510395418746953E-01,
    0.47308671312268919080604988338844E-01,
    0.47966421137995131411052756195132E-01,
    0.48506789097883847864090099145802E-01,
    0.48928452820511989944709361549215E-01,
    0.49230380423747560785043116988145E-01,
    0.49411833039918178967039646116705E-01,
    0.49472366623931020888669360420926E-01,
    0.49411833039918178967039646116705E-01,
    0.49230380423747560785043116988145E-01,
    0.48928452820511989944709361549215E-01,
    0.48506789097883847864090099145802E-01,
    0.47966421137995131411052756195132E-01,
    0.47308671312268919080604988338844E-01,
    0.46535149245383696510395418746953E-01,
    0.45647747876292608685885992608542E-01,
    0.44648638825941395370332669516813E-01,
    0.43540267083027590798964315704401E-01,
    0.42325345020815822982505485403028E-01,
    0.41006845759666398635110037009072E-01,
    0.39587995891544093984807928149202E-01,
    0.38072267584349556763638324927889E-01,
    0.36463370085457289630452409787542E-01,
    0.34765240645355877697180504642788E-01,
    0.32982034883779341765683179672459E-01,
    0.31118116622219817508215988557189E-01,
    0.29178047208280526945551502154029E-01,
    0.27166574359097933225189839439413E-01,
    0.25088620553344986618630138068443E-01,
    0.22949271004889933148942319561770E-01,
    0.20753761258039090775341953421471E-01,
    0.18507464460161270409260545805144E-01,
    0.16215878410338338882283672974995E-01,
    0.13884612616115610824866086365937E-01,
    0.11519376076880041750750606118707E-01,
    0.91259686763266563540586445877022E-02,
    0.67102917659601362519069109850892E-02,
    0.42785083468637618661951422543371E-02,
    0.18398745955770837880499331680577E-02
  };
  static double w_127[127] =
  {
    0.45645726109586654495731936146574E-03,
    0.10622766869538486959954760554099E-02,
    0.16683488125171936761028811985672E-02,
    0.22734860707492547802810838362671E-02,
    0.28772587656289004082883197417581E-02,
    0.34792893810051465908910894094105E-02,
    0.40792095178254605327114733456293E-02,
    0.46766539777779034772638165662478E-02,
    0.52712596565634400891303815906251E-02,
    0.58626653903523901033648343751367E-02,
    0.64505120486899171845442463868748E-02,
    0.70344427036681608755685893032552E-02,
    0.76141028256526859356393930849227E-02,
    0.81891404887415730817235884718726E-02,
    0.87592065795403145773316804234385E-02,
    0.93239550065309714787536985834029E-02,
    0.98830429087554914716648010899606E-02,
    0.10436130863141005225673171997668E-01,
    0.10982883090068975788799657376065E-01,
    0.11522967656921087154811609734510E-01,
    0.12056056679400848183529562144697E-01,
    0.12581826520465013101514365424172E-01,
    0.13099957986718627426172681912499E-01,
    0.13610136522139249906034237533759E-01,
    0.14112052399003395774044161633613E-01,
    0.14605400905893418351737288078952E-01,
    0.15089882532666922992635733981431E-01,
    0.15565203152273955098532590262975E-01,
    0.16031074199309941802254151842763E-01,
    0.16487212845194879399346060358146E-01,
    0.16933342169871654545878815295200E-01,
    0.17369191329918731922164721250350E-01,
    0.17794495722974774231027912900351E-01,
    0.18208997148375106468721469154479E-01,
    0.18612443963902310429440419898958E-01,
    0.19004591238555646611148901044533E-01,
    0.19385200901246454628112623489471E-01,
    0.19754041885329183081815217323169E-01,
    0.20110890268880247225644623956287E-01,
    0.20455529410639508279497065713301E-01,
    0.20787750081531811812652137291250E-01,
    0.21107350591688713643523847921658E-01,
    0.21414136912893259295449693233545E-01,
    0.21707922796373466052301324695331E-01,
    0.21988529885872983756478409758807E-01,
    0.22255787825930280235631416460158E-01,
    0.22509534365300608085694429903050E-01,
    0.22749615455457959852242553240982E-01,
    0.22975885344117206754377437838947E-01,
    0.23188206663719640249922582981729E-01,
    0.23386450514828194170722043496950E-01,
    0.23570496544381716050033676844306E-01,
    0.23740233018760777777714726703424E-01,
    0.23895556891620665983864481754172E-01,
    0.24036373866450369675132086026456E-01,
    0.24162598453819584716522917710986E-01,
    0.24274154023278979833195063936748E-01,
    0.24370972849882214952813561907241E-01,
    0.24452996155301467956140198471529E-01,
    0.24520174143511508275183033290175E-01,
    0.24572466031020653286354137335186E-01,
    0.24609840071630254092545634003360E-01,
    0.24632273575707679066033370218017E-01,
    0.24639752923961094419579417477503E-01,
    0.24632273575707679066033370218017E-01,
    0.24609840071630254092545634003360E-01,
    0.24572466031020653286354137335186E-01,
    0.24520174143511508275183033290175E-01,
    0.24452996155301467956140198471529E-01,
    0.24370972849882214952813561907241E-01,
    0.24274154023278979833195063936748E-01,
    0.24162598453819584716522917710986E-01,
    0.24036373866450369675132086026456E-01,
    0.23895556891620665983864481754172E-01,
    0.23740233018760777777714726703424E-01,
    0.23570496544381716050033676844306E-01,
    0.23386450514828194170722043496950E-01,
    0.23188206663719640249922582981729E-01,
    0.22975885344117206754377437838947E-01,
    0.22749615455457959852242553240982E-01,
    0.22509534365300608085694429903050E-01,
    0.22255787825930280235631416460158E-01,
    0.21988529885872983756478409758807E-01,
    0.21707922796373466052301324695331E-01,
    0.21414136912893259295449693233545E-01,
    0.21107350591688713643523847921658E-01,
    0.20787750081531811812652137291250E-01,
    0.20455529410639508279497065713301E-01,
    0.20110890268880247225644623956287E-01,
    0.19754041885329183081815217323169E-01,
    0.19385200901246454628112623489471E-01,
    0.19004591238555646611148901044533E-01,
    0.18612443963902310429440419898958E-01,
    0.18208997148375106468721469154479E-01,
    0.17794495722974774231027912900351E-01,
    0.17369191329918731922164721250350E-01,
    0.16933342169871654545878815295200E-01,
    0.16487212845194879399346060358146E-01,
    0.16031074199309941802254151842763E-01,
    0.15565203152273955098532590262975E-01,
    0.15089882532666922992635733981431E-01,
    0.14605400905893418351737288078952E-01,
    0.14112052399003395774044161633613E-01,
    0.13610136522139249906034237533759E-01,
    0.13099957986718627426172681912499E-01,
    0.12581826520465013101514365424172E-01,
    0.12056056679400848183529562144697E-01,
    0.11522967656921087154811609734510E-01,
    0.10982883090068975788799657376065E-01,
    0.10436130863141005225673171997668E-01,
    0.98830429087554914716648010899606E-02,
    0.93239550065309714787536985834029E-02,
    0.87592065795403145773316804234385E-02,
    0.81891404887415730817235884718726E-02,
    0.76141028256526859356393930849227E-02,
    0.70344427036681608755685893032552E-02,
    0.64505120486899171845442463868748E-02,
    0.58626653903523901033648343751367E-02,
    0.52712596565634400891303815906251E-02,
    0.46766539777779034772638165662478E-02,
    0.40792095178254605327114733456293E-02,
    0.34792893810051465908910894094105E-02,
    0.28772587656289004082883197417581E-02,
    0.22734860707492547802810838362671E-02,
    0.16683488125171936761028811985672E-02,
    0.10622766869538486959954760554099E-02,
    0.45645726109586654495731936146574E-03
  };
  if ( order == 1 )
  {
    webbur::r8vec_copy ( order, w_001, w );
  }
  else if ( order == 3 )
  {
    webbur::r8vec_copy ( order, w_003, w );
  }
  else if ( order == 7 )
  {
    webbur::r8vec_copy ( order, w_007, w );
  }
  else if ( order == 15 )
  {
    webbur::r8vec_copy ( order, w_015, w );
  }
  else if ( order == 31 )
  {
    webbur::r8vec_copy ( order, w_031, w );
  }
  else if ( order == 63 )
  {
    webbur::r8vec_copy ( order, w_063, w );
  }
  else if ( order == 127 )
  {
    webbur::r8vec_copy ( order, w_127, w );
  }
  else
  {
    std::cerr << "\n";
    std::cerr << "LEGENDRE_LOOKUP_WEIGHTS - Fatal error!\n";
    std::cerr << "  Unexpected value of ORDER = " << order << ".\n";
    std::exit ( 1 );
  }
  return;
}
//****************************************************************************80

void patterson_lookup_points ( int order, double alpha, double beta,
  double x[] )

//****************************************************************************80
//
//  Purpose:
//
//    PATTERSON_LOOKUP_POINTS looks up Patterson quadrature points.
//
//  Discussion:
//
//    Our convention is that the abscissas are numbered from left to right.
//
//    The rule is defined on [-1,1],
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Prem Kythe, Michael Schaeferkotter,
//    Handbook of Computational Methods for Integration,
//    Chapman and Hall, 2004,
//    ISBN: 1-58488-428-2,
//    LC: QA299.3.K98.
//
//    Thomas Patterson,
//    The Optimal Addition of Points to Quadrature Formulae,
//    Mathematics of Computation,
//    Volume 22, Number 104, October 1968, pages 847-856.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    Legal values are 1, 3, 7, 15, 31, 63 and 127.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double X[ORDER], the abscissas.
//
{
  static double x_001[1] = 
  {
     0.0 
  };
  static double x_003[3] =
  {
    -0.77459666924148337704, 
     0.0, 
     0.77459666924148337704
  };
  static double x_007[7] =
  {
    -0.96049126870802028342, 
    -0.77459666924148337704, 
    -0.43424374934680255800, 
     0.0, 
     0.43424374934680255800, 
     0.77459666924148337704, 
     0.96049126870802028342 
  };
  static double x_015[15] =
  {
    -0.99383196321275502221, 
    -0.96049126870802028342, 
    -0.88845923287225699889, 
    -0.77459666924148337704, 
    -0.62110294673722640294, 
    -0.43424374934680255800, 
    -0.22338668642896688163, 
     0.0, 
     0.22338668642896688163, 
     0.43424374934680255800, 
     0.62110294673722640294, 
     0.77459666924148337704, 
     0.88845923287225699889, 
     0.96049126870802028342, 
     0.99383196321275502221
  };
  static double x_031[31] =
  {
    -0.99909812496766759766, 
    -0.99383196321275502221, 
    -0.98153114955374010687, 
    -0.96049126870802028342, 
    -0.92965485742974005667, 
    -0.88845923287225699889, 
    -0.83672593816886873550, 
    -0.77459666924148337704, 
    -0.70249620649152707861, 
    -0.62110294673722640294, 
    -0.53131974364437562397, 
    -0.43424374934680255800, 
    -0.33113539325797683309, 
    -0.22338668642896688163, 
    -0.11248894313318662575, 
     0.0, 
     0.11248894313318662575, 
     0.22338668642896688163, 
     0.33113539325797683309, 
     0.43424374934680255800, 
     0.53131974364437562397, 
     0.62110294673722640294, 
     0.70249620649152707861, 
     0.77459666924148337704, 
     0.83672593816886873550, 
     0.88845923287225699889, 
     0.92965485742974005667, 
     0.96049126870802028342, 
     0.98153114955374010687, 
     0.99383196321275502221, 
     0.99909812496766759766
  };
  static double x_063[63] =
  {
    -0.99987288812035761194, 
    -0.99909812496766759766, 
    -0.99720625937222195908, 
    -0.99383196321275502221, 
    -0.98868475754742947994, 
    -0.98153114955374010687, 
    -0.97218287474858179658, 
    -0.96049126870802028342, 
    -0.94634285837340290515, 
    -0.92965485742974005667, 
    -0.91037115695700429250, 
    -0.88845923287225699889, 
    -0.86390793819369047715, 
    -0.83672593816886873550, 
    -0.80694053195021761186, 
    -0.77459666924148337704, 
    -0.73975604435269475868, 
    -0.70249620649152707861, 
    -0.66290966002478059546, 
    -0.62110294673722640294, 
    -0.57719571005204581484, 
    -0.53131974364437562397, 
    -0.48361802694584102756, 
    -0.43424374934680255800, 
    -0.38335932419873034692, 
    -0.33113539325797683309, 
    -0.27774982202182431507, 
    -0.22338668642896688163, 
    -0.16823525155220746498, 
    -0.11248894313318662575, 
    -0.056344313046592789972, 
     0.0, 
     0.056344313046592789972, 
     0.11248894313318662575, 
     0.16823525155220746498, 
     0.22338668642896688163, 
     0.27774982202182431507, 
     0.33113539325797683309, 
     0.38335932419873034692, 
     0.43424374934680255800, 
     0.48361802694584102756, 
     0.53131974364437562397, 
     0.57719571005204581484, 
     0.62110294673722640294, 
     0.66290966002478059546, 
     0.70249620649152707861, 
     0.73975604435269475868, 
     0.77459666924148337704, 
     0.80694053195021761186, 
     0.83672593816886873550, 
     0.86390793819369047715, 
     0.88845923287225699889, 
     0.91037115695700429250, 
     0.92965485742974005667, 
     0.94634285837340290515, 
     0.96049126870802028342, 
     0.97218287474858179658, 
     0.98153114955374010687, 
     0.98868475754742947994, 
     0.99383196321275502221, 
     0.99720625937222195908, 
     0.99909812496766759766, 
     0.99987288812035761194
  };
  static double x_127[127] =
  {
    -0.99998243035489159858, 
    -0.99987288812035761194, 
    -0.99959879967191068325, 
    -0.99909812496766759766, 
    -0.99831663531840739253, 
    -0.99720625937222195908, 
    -0.99572410469840718851, 
    -0.99383196321275502221, 
    -0.99149572117810613240, 
    -0.98868475754742947994, 
    -0.98537149959852037111, 
    -0.98153114955374010687, 
    -0.97714151463970571416, 
    -0.97218287474858179658, 
    -0.96663785155841656709, 
    -0.96049126870802028342, 
    -0.95373000642576113641, 
    -0.94634285837340290515, 
    -0.93832039777959288365, 
    -0.92965485742974005667, 
    -0.92034002547001242073, 
    -0.91037115695700429250, 
    -0.89974489977694003664, 
    -0.88845923287225699889, 
    -0.87651341448470526974, 
    -0.86390793819369047715, 
    -0.85064449476835027976, 
    -0.83672593816886873550, 
    -0.82215625436498040737, 
    -0.80694053195021761186, 
    -0.79108493379984836143, 
    -0.77459666924148337704, 
    -0.75748396638051363793, 
    -0.73975604435269475868, 
    -0.72142308537009891548, 
    -0.70249620649152707861, 
    -0.68298743109107922809, 
    -0.66290966002478059546, 
    -0.64227664250975951377, 
    -0.62110294673722640294, 
    -0.59940393024224289297, 
    -0.57719571005204581484, 
    -0.55449513263193254887, 
    -0.53131974364437562397, 
    -0.50768775753371660215, 
    -0.48361802694584102756, 
    -0.45913001198983233287, 
    -0.43424374934680255800, 
    -0.40897982122988867241, 
    -0.38335932419873034692, 
    -0.35740383783153215238, 
    -0.33113539325797683309, 
    -0.30457644155671404334, 
    -0.27774982202182431507, 
    -0.25067873030348317661, 
    -0.22338668642896688163, 
    -0.19589750271110015392, 
    -0.16823525155220746498, 
    -0.14042423315256017459, 
    -0.11248894313318662575, 
    -0.084454040083710883710, 
    -0.056344313046592789972, 
    -0.028184648949745694339, 
     0.0, 
     0.028184648949745694339, 
     0.056344313046592789972, 
     0.084454040083710883710, 
     0.11248894313318662575, 
     0.14042423315256017459, 
     0.16823525155220746498, 
     0.19589750271110015392, 
     0.22338668642896688163, 
     0.25067873030348317661, 
     0.27774982202182431507, 
     0.30457644155671404334, 
     0.33113539325797683309, 
     0.35740383783153215238, 
     0.38335932419873034692, 
     0.40897982122988867241, 
     0.43424374934680255800, 
     0.45913001198983233287, 
     0.48361802694584102756, 
     0.50768775753371660215, 
     0.53131974364437562397, 
     0.55449513263193254887, 
     0.57719571005204581484, 
     0.59940393024224289297, 
     0.62110294673722640294, 
     0.64227664250975951377, 
     0.66290966002478059546, 
     0.68298743109107922809, 
     0.70249620649152707861, 
     0.72142308537009891548, 
     0.73975604435269475868, 
     0.75748396638051363793, 
     0.77459666924148337704, 
     0.79108493379984836143, 
     0.80694053195021761186, 
     0.82215625436498040737, 
     0.83672593816886873550, 
     0.85064449476835027976, 
     0.86390793819369047715, 
     0.87651341448470526974, 
     0.88845923287225699889, 
     0.89974489977694003664, 
     0.91037115695700429250, 
     0.92034002547001242073, 
     0.92965485742974005667, 
     0.93832039777959288365, 
     0.94634285837340290515, 
     0.95373000642576113641, 
     0.96049126870802028342, 
     0.96663785155841656709, 
     0.97218287474858179658, 
     0.97714151463970571416, 
     0.98153114955374010687, 
     0.98537149959852037111, 
     0.98868475754742947994, 
     0.99149572117810613240, 
     0.99383196321275502221, 
     0.99572410469840718851, 
     0.99720625937222195908, 
     0.99831663531840739253, 
     0.99909812496766759766, 
     0.99959879967191068325, 
     0.99987288812035761194, 
     0.99998243035489159858
  };

  if ( order == 1 )
  {
    webbur::r8vec_copy ( order, x_001, x );
  }
  else if ( order == 3 )
  {
    webbur::r8vec_copy ( order, x_003, x );
  }
  else if ( order == 7 )
  {
    webbur::r8vec_copy ( order, x_007, x );
  }
  else if ( order == 15 )
  {
    webbur::r8vec_copy ( order, x_015, x );
  }
  else if ( order == 31 )
  {
    webbur::r8vec_copy ( order, x_031, x );
  }
  else if ( order == 63 )
  {
    webbur::r8vec_copy ( order, x_063, x );
  }
  else if ( order == 127 )
  {
    webbur::r8vec_copy ( order, x_127, x );
  }
  else
  {
    std::cerr << "\n";
    std::cerr << "PATTERSON_LOOKUP_POINTS - Fatal error!\n";
    std::cerr << "  Unexpected value of ORDER = " << order << "\n";
    std::exit ( 1 );
  }
  return;
}
//****************************************************************************80

void patterson_lookup_weights ( int order, double alpha, double beta,
  double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    PATTERSON_LOOKUP_WEIGHTS looks up Patterson quadrature weights.
//
//  Discussion:
//
//    The allowed orders are 1, 3, 7, 15, 31, 63 and 127.
//
//    The weights are positive, symmetric and should sum to 2.
//
//    The user must preallocate space for the output array W.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 March 2009
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    Milton Abramowitz, Irene Stegun,
//    Handbook of Mathematical Functions,
//    National Bureau of Standards, 1964,
//    ISBN: 0-486-61272-4,
//    LC: QA47.A34.
//
//    Arthur Stroud, Don Secrest,
//    Gaussian Quadrature Formulas,
//    Prentice Hall, 1966,
//    LC: QA299.4G3S7.
//
//  Parameters:
//
//    Input, int ORDER, the order of the rule.
//    ORDER must be 1, 3, 7, 15, 31, 63 or 127.
//
//    Input, double ALPHA, BETA, parameters which are not needed by this
//    function, but which are needed by certain quadrature rules.
//
//    Output, double W[ORDER], the weights.
//
{
  static double w_001[1] =
  {
    2.0
  };
  static double w_003[3] = 
  {
    0.555555555555555555556,
    0.888888888888888888889,
    0.555555555555555555556
  };
  static double w_007[7] =
  {
    0.104656226026467265194,
    0.268488089868333440729,
    0.401397414775962222905,
    0.450916538658474142345,
    0.401397414775962222905,
    0.268488089868333440729,
    0.104656226026467265194
  };
  static double w_015[15] =
  {
    0.0170017196299402603390,
    0.0516032829970797396969,
    0.0929271953151245376859,
    0.134415255243784220360,
    0.171511909136391380787,
    0.200628529376989021034,
    0.219156858401587496404,
    0.225510499798206687386,
    0.219156858401587496404,
    0.200628529376989021034,
    0.171511909136391380787,
    0.134415255243784220360,
    0.0929271953151245376859,
    0.0516032829970797396969,
    0.0170017196299402603390
  };
  static double w_031[31] =
  {
    0.00254478079156187441540,
    0.00843456573932110624631,
    0.0164460498543878109338,
    0.0258075980961766535646,
    0.0359571033071293220968,
    0.0464628932617579865414,
    0.0569795094941233574122,
    0.0672077542959907035404,
    0.0768796204990035310427,
    0.0857559200499903511542,
    0.0936271099812644736167,
    0.100314278611795578771,
    0.105669893580234809744,
    0.109578421055924638237,
    0.111956873020953456880,
    0.112755256720768691607,
    0.111956873020953456880,
    0.109578421055924638237,
    0.105669893580234809744,
    0.100314278611795578771,
    0.0936271099812644736167,
    0.0857559200499903511542,
    0.0768796204990035310427,
    0.0672077542959907035404,
    0.0569795094941233574122,
    0.0464628932617579865414,
    0.0359571033071293220968,
    0.0258075980961766535646,
    0.0164460498543878109338,
    0.00843456573932110624631,
    0.00254478079156187441540
  };
  static double w_063[63] =
  {
    0.000363221481845530659694,
    0.00126515655623006801137,
    0.00257904979468568827243,
    0.00421763044155885483908,
    0.00611550682211724633968,
    0.00822300795723592966926,
    0.0104982469096213218983,
    0.0129038001003512656260,
    0.0154067504665594978021,
    0.0179785515681282703329,
    0.0205942339159127111492,
    0.0232314466399102694433,
    0.0258696793272147469108,
    0.0284897547458335486125,
    0.0310735511116879648799,
    0.0336038771482077305417,
    0.0360644327807825726401,
    0.0384398102494555320386,
    0.0407155101169443189339,
    0.0428779600250077344929,
    0.0449145316536321974143,
    0.0468135549906280124026,
    0.0485643304066731987159,
    0.0501571393058995374137,
    0.0515832539520484587768,
    0.0528349467901165198621,
    0.0539054993352660639269,
    0.0547892105279628650322,
    0.0554814043565593639878,
    0.0559784365104763194076,
    0.0562776998312543012726,
    0.0563776283603847173877,
    0.0562776998312543012726,
    0.0559784365104763194076,
    0.0554814043565593639878,
    0.0547892105279628650322,
    0.0539054993352660639269,
    0.0528349467901165198621,
    0.0515832539520484587768,
    0.0501571393058995374137,
    0.0485643304066731987159,
    0.0468135549906280124026,
    0.0449145316536321974143,
    0.0428779600250077344929,
    0.0407155101169443189339,
    0.0384398102494555320386,
    0.0360644327807825726401,
    0.0336038771482077305417,
    0.0310735511116879648799,
    0.0284897547458335486125,
    0.0258696793272147469108,
    0.0232314466399102694433,
    0.0205942339159127111492,
    0.0179785515681282703329,
    0.0154067504665594978021,
    0.0129038001003512656260,
    0.0104982469096213218983,
    0.00822300795723592966926,
    0.00611550682211724633968,
    0.00421763044155885483908,
    0.00257904979468568827243,
    0.00126515655623006801137,
    0.000363221481845530659694
  };
  static double w_127[127] =
  {
    0.0000505360952078625176247,
    0.000180739564445388357820,
    0.000377746646326984660274,
    0.000632607319362633544219,
    0.000938369848542381500794,
    0.00128952408261041739210,
    0.00168114286542146990631,
    0.00210881524572663287933,
    0.00256876494379402037313,
    0.00305775341017553113613,
    0.00357289278351729964938,
    0.00411150397865469304717,
    0.00467105037211432174741,
    0.00524912345480885912513,
    0.00584344987583563950756,
    0.00645190005017573692280,
    0.00707248999543355546805,
    0.00770337523327974184817,
    0.00834283875396815770558,
    0.00898927578406413572328,
    0.00964117772970253669530,
    0.0102971169579563555237,
    0.0109557333878379016480,
    0.0116157233199551347270,
    0.0122758305600827700870,
    0.0129348396636073734547,
    0.0135915710097655467896,
    0.0142448773729167743063,
    0.0148936416648151820348,
    0.0155367755558439824399,
    0.0161732187295777199419,
    0.0168019385741038652709,
    0.0174219301594641737472,
    0.0180322163903912863201,
    0.0186318482561387901863,
    0.0192199051247277660193,
    0.0197954950480974994880,
    0.0203577550584721594669,
    0.0209058514458120238522,
    0.0214389800125038672465,
    0.0219563663053178249393,
    0.0224572658268160987071,
    0.0229409642293877487608,
    0.0234067774953140062013,
    0.0238540521060385400804,
    0.0242821652033365993580,
    0.0246905247444876769091,
    0.0250785696529497687068,
    0.0254457699654647658126,
    0.0257916269760242293884,
    0.0261156733767060976805,
    0.0264174733950582599310,
    0.0266966229274503599062,
    0.0269527496676330319634,
    0.0271855132296247918192,
    0.0273946052639814325161,
    0.0275797495664818730349,
    0.0277407021782796819939,
    0.0278772514766137016085,
    0.0279892182552381597038,
    0.0280764557938172466068,
    0.0281388499156271506363,
    0.0281763190330166021307,
    0.0281888141801923586938,
    0.0281763190330166021307,
    0.0281388499156271506363,
    0.0280764557938172466068,
    0.0279892182552381597038,
    0.0278772514766137016085,
    0.0277407021782796819939,
    0.0275797495664818730349,
    0.0273946052639814325161,
    0.0271855132296247918192,
    0.0269527496676330319634,
    0.0266966229274503599062,
    0.0264174733950582599310,
    0.0261156733767060976805,
    0.0257916269760242293884,
    0.0254457699654647658126,
    0.0250785696529497687068,
    0.0246905247444876769091,
    0.0242821652033365993580,
    0.0238540521060385400804,
    0.0234067774953140062013,
    0.0229409642293877487608,
    0.0224572658268160987071,
    0.0219563663053178249393,
    0.0214389800125038672465,
    0.0209058514458120238522,
    0.0203577550584721594669,
    0.0197954950480974994880,
    0.0192199051247277660193,
    0.0186318482561387901863,
    0.0180322163903912863201,
    0.0174219301594641737472,
    0.0168019385741038652709,
    0.0161732187295777199419,
    0.0155367755558439824399,
    0.0148936416648151820348,
    0.0142448773729167743063,
    0.0135915710097655467896,
    0.0129348396636073734547,
    0.0122758305600827700870,
    0.0116157233199551347270,
    0.0109557333878379016480,
    0.0102971169579563555237,
    0.00964117772970253669530,
    0.00898927578406413572328,
    0.00834283875396815770558,
    0.00770337523327974184817,
    0.00707248999543355546805,
    0.00645190005017573692280,
    0.00584344987583563950756,
    0.00524912345480885912513,
    0.00467105037211432174741,
    0.00411150397865469304717,
    0.00357289278351729964938,
    0.00305775341017553113613,
    0.00256876494379402037313,
    0.00210881524572663287933,
    0.00168114286542146990631,
    0.00128952408261041739210,
    0.000938369848542381500794,
    0.000632607319362633544219,
    0.000377746646326984660274,
    0.000180739564445388357820,
    0.0000505360952078625176247
  };
  if ( order == 1 )
  {
    webbur::r8vec_copy ( order, w_001, w );
  }
  else if ( order == 3 )
  {
    webbur::r8vec_copy ( order, w_003, w );
  }
  else if ( order == 7 )
  {
    webbur::r8vec_copy ( order, w_007, w );
  }
  else if ( order == 15 )
  {
    webbur::r8vec_copy ( order, w_015, w );
  }
  else if ( order == 31 )
  {
    webbur::r8vec_copy ( order, w_031, w );
  }
  else if ( order == 63 )
  {
    webbur::r8vec_copy ( order, w_063, w );
  }
  else if ( order == 127 )
  {
    webbur::r8vec_copy ( order, w_127, w );
  }
  else
  {
    std::cerr << "\n";
    std::cerr << "PATTERSON_LOOKUP_WEIGHTS - Fatal error!\n";
    std::cerr << "  Unexpected value of ORDER = " << order << ".\n";
    std::exit ( 1 );
  }
  return;
}
//****************************************************************************80

double r8_abs ( double x )

//****************************************************************************80
//
//  Purpose:
//
//    R8_ABS returns the absolute value of an R8.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, double X, the quantity whose absolute value is desired.
//
//    Output, double R8_ABS, the absolute value of X.
//
{
  double value;

  if ( 0.0 <= x )
  {
    value = x;
  } 
  else
  {
    value = -x;
  }
  return value;
}
//****************************************************************************80

double r8_choose ( int n, int k )

//****************************************************************************80
//
//  Purpose:
//
//    R8_CHOOSE computes the binomial coefficient C(N,K) as an R8.
//
//  Discussion:
//
//    The value is calculated in such a way as to avoid overflow and
//    roundoff.  The calculation is done in R8 arithmetic.
//
//    The formula used is:
//
//      C(N,K) = N! / ( K! * (N-K)! )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 March 2008
//
//  Author:
//
//    John Burkardt
//
//  Reference:
//
//    ML Wolfson, HV Wright,
//    Algorithm 160:
//    Combinatorial of M Things Taken N at a Time,
//    Communications of the ACM,
//    Volume 6, Number 4, April 1963, page 161.
//
//  Parameters:
//
//    Input, int N, K, the values of N and K.
//
//    Output, double R8_CHOOSE, the number of combinations of N
//    things taken K at a time.
//
{
  int i;
  int mn;
  int mx;
  int value;

  mn = webbur::i4_min ( k, n - k );

  if ( mn < 0 )
  {
    value = 0.0;
  }
  else if ( mn == 0 )
  {
    value = 1.0;
  }
  else
  {
    mx = webbur::i4_max ( k, n - k );
    value = ( double ) ( mx + 1 );

    for ( i = 2; i <= mn; i++ )
    {
      value = ( value * ( double ) ( mx + i ) ) / ( double ) i;
    }
  }
  return value;
}
//****************************************************************************80

double r8_epsilon ( )

//****************************************************************************80
//
//  Purpose:
//
//    R8_EPSILON returns the R8 roundoff unit.
//
//  Discussion:
//
//    The roundoff unit is a number R which is a power of 2 with the 
//    property that, to the precision of the computer's arithmetic,
//      1 < 1 + R
//    but 
//      1 = ( 1 + R / 2 )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 February 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Output, double R8_EPSILON, the R8 round-off unit.
//
{
  double value;

  value = 1.0;

  while ( 1.0 < ( double ) ( 1.0 + value )  )
  {
    value = value / 2.0;
  }

  value = 2.0 * value;

  return value;
}
//****************************************************************************80

double r8_factorial ( int n )

//****************************************************************************80
//
//  Purpose:
//
//    R8_FACTORIAL computes the factorial of N, also denoted "N!".
//
//  Discussion:
//
//    factorial ( N ) = N! = product ( 1 <= I <= N ) I
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    16 January 1999
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the argument of the factorial function.
//    If N is less than 1, the function value is returned as 1.
//
//    Output, double R8_FACTORIAL, the factorial of N.
//
{
  int i;
  double value;

  value = 1.0;

  for ( i = 1; i <= n; i++ )
  {
    value = value * ( double ) ( i );
  }

  return value;
}
//****************************************************************************80

double r8_factorial2 ( int n )

//****************************************************************************80
//
//  Purpose:
//
//    R8_FACTORIAL2 computes the double factorial function N!!
//
//  Discussion:
//
//    FACTORIAL2( N ) = Product ( N * (N-2) * (N-4) * ... * 2 )  (N even)
//                    = Product ( N * (N-2) * (N-4) * ... * 1 )  (N odd)
//
//  Example:
//
//     N    N!!
//
//     0     1
//     1     1
//     2     2
//     3     3
//     4     8
//     5    15
//     6    48
//     7   105
//     8   384
//     9   945
//    10  3840
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    22 January 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the argument of the double factorial 
//    function.  If N is less than 1, R8_FACTORIAL2 is returned as 1.0.
//
//    Output, double R8_FACTORIAL2, the value of N!!.
//
{
  int n_copy;
  double value;

  value = 1.0;

  if ( n < 1 )
  {
    return value;
  }

  n_copy = n;

  while ( 1 < n_copy )
  {
    value = value * ( double ) n_copy;
    n_copy = n_copy - 2;
  }

  return value;
}
//****************************************************************************80

double r8_gamma ( double x )

//****************************************************************************80
//
//  Purpose:
//
//    R8_GAMMA evaluates Gamma(X) for a real argument.
//
//  Discussion:
//
//    This routine calculates the gamma function for a real argument X.
//
//    Computation is based on an algorithm outlined in reference 1.
//    The program uses rational functions that approximate the gamma
//    function to at least 20 significant decimal digits.  Coefficients
//    for the approximation over the interval (1,2) are unpublished.
//    Those for the approximation for 12 <= X are from reference 2.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 January 2008
//
//  Author:
//
//    Original FORTRAN77 version by William Cody, Laura Stoltz.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    William Cody,
//    An Overview of Software Development for Special Functions,
//    in Numerical Analysis Dundee, 1975,
//    edited by GA Watson,
//    Lecture Notes in Mathematics 506,
//    Springer, 1976.
//
//    John Hart, Ward Cheney, Charles Lawson, Hans Maehly,
//    Charles Mesztenyi, John Rice, Henry Thatcher,
//    Christoph Witzgall,
//    Computer Approximations,
//    Wiley, 1968,
//    LC: QA297.C64.
//
//  Parameters:
//
//    Input, double X, the argument of the function.
//
//    Output, double R8_GAMMA, the value of the function.
//
{
//
//  Coefficients for minimax approximation over (12, INF).
//
  double c[7] = {
   -1.910444077728E-03, 
    8.4171387781295E-04, 
   -5.952379913043012E-04, 
    7.93650793500350248E-04, 
   -2.777777777777681622553E-03, 
    8.333333333333333331554247E-02, 
    5.7083835261E-03 };
  double eps = 2.22E-16;
  double fact;
  double half = 0.5;
  int i;
  int n;
  double one = 1.0;
  double p[8] = {
  -1.71618513886549492533811E+00,
   2.47656508055759199108314E+01, 
  -3.79804256470945635097577E+02,
   6.29331155312818442661052E+02, 
   8.66966202790413211295064E+02,
  -3.14512729688483675254357E+04, 
  -3.61444134186911729807069E+04,
   6.64561438202405440627855E+04 };
  bool parity;
  double pi = 3.1415926535897932384626434;
  double q[8] = {
  -3.08402300119738975254353E+01,
   3.15350626979604161529144E+02, 
  -1.01515636749021914166146E+03,
  -3.10777167157231109440444E+03, 
   2.25381184209801510330112E+04,
   4.75584627752788110767815E+03, 
  -1.34659959864969306392456E+05,
  -1.15132259675553483497211E+05 };
  double res;
  double sqrtpi = 0.9189385332046727417803297;
  double sum;
  double twelve = 12.0;
  double two = 2.0;
  double value;
  double xbig = 171.624;
  double xden;
  double xinf = 1.79E+308;
  double xminin = 2.23E-308;
  double xnum;
  double y;
  double y1;
  double ysq;
  double z;
  double zero = 0.0;;

  parity = false;
  fact = one;
  n = 0;
  y = x;
//
//  Argument is negative.
//
  if ( y <= zero )
  {
    y = - x;
    y1 = ( double ) ( int ) ( y );
    res = y - y1;

    if ( res != zero )
    {
      if ( y1 != ( double ) ( int ) ( y1 * half ) * two )
      {
        parity = true;
      }

      fact = - pi / std::sin ( pi * res );
      y = y + one;
    }
    else
    {
      res = xinf;
      value = res;
      return value;
    }
  }
//
//  Argument is positive.
//
  if ( y < eps )
  {
//
//  Argument < EPS.
//
    if ( xminin <= y )
    {
      res = one / y;
    }
    else
    {
      res = xinf;
      value = res;
      return value;
    }
  }
  else if ( y < twelve )
  {
    y1 = y;
//
//  0.0 < argument < 1.0.
//
    if ( y < one )
    {
      z = y;
      y = y + one;
    }
//
//  1.0 < argument < 12.0.
//  Reduce argument if necessary.
//
    else
    {
      n = ( int ) ( y ) - 1;
      y = y - ( double ) ( n );
      z = y - one;
    }
//
//  Evaluate approximation for 1.0 < argument < 2.0.
//
    xnum = zero;
    xden = one;
    for ( i = 0; i < 8; i++ )
    {
      xnum = ( xnum + p[i] ) * z;
      xden = xden * z + q[i];
    }
    res = xnum / xden + one;
//
//  Adjust result for case  0.0 < argument < 1.0.
//
    if ( y1 < y )
    {
      res = res / y1;
    }
//
//  Adjust result for case 2.0 < argument < 12.0.
//
    else if ( y < y1 )
    {
      for ( i = 1; i <= n; i++ )
      {
        res = res * y;
        y = y + one;
      }
    }
  }
  else
  {
//
//  Evaluate for 12.0 <= argument.
//
    if ( y <= xbig )
    {
      ysq = y * y;
      sum = c[6];
      for ( i = 0; i < 6; i++ )
      {
        sum = sum / ysq + c[i];
      }
      sum = sum / y - y + sqrtpi;
      sum = sum + ( y - half ) * std::log ( y );
      res = std::exp ( sum );
    }
    else
    {
      res = xinf;
      value = res;
      return value;
    }
  }
//
//  Final adjustments and return.
//
  if ( parity )
  {
    res = - res;
  }

  if ( fact != one )
  {
    res = fact / res;
  }

  value = res;

  return value;
}
//****************************************************************************80

double r8_huge ( )

//****************************************************************************80
//
//  Purpose:
//
//    R8_HUGE returns a "huge" R8.
//
//  Discussion:
//
//    The value returned by this function is NOT required to be the
//    maximum representable R8.  This value varies from machine to machine,
//    from compiler to compiler, and may cause problems when being printed.
//    We simply want a "very large" but non-infinite number.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    06 October 2007
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Output, double R8_HUGE, a "huge" R8 value.
//
{
  double value;

  value = 1.0E+30;

  return value;
}
//****************************************************************************80

double r8_hyper_2f1 ( double a, double b, double c, double x )

//****************************************************************************80
//
//  Purpose:
//
//    R8_HYPER_2F1 evaluates the hypergeometric function 2F1(A,B,C,X).
//
//  Discussion:
//
//    A minor bug was corrected.  The HW variable, used in several places as
//    the "old" value of a quantity being iteratively improved, was not
//    being initialized.  JVB, 11 February 2008.
//
//    The FORTRAN77 original version of this routine is copyrighted by
//    Shanjie Zhang and Jianming Jin.  However, they give permission to
//    incorporate this routine into a user program provided that the copyright
//    is acknowledged.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    11 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by Shanjie Zhang, Jianming Jin.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    Shanjie Zhang, Jianming Jin,
//    Computation of Special Functions,
//    Wiley, 1996,
//    ISBN: 0-471-11963-6,
//    LC: QA351.C45
//
//  Parameters:
//
//    Input, double A, B, C, X, the arguments of the function.
//    C must not be equal to a nonpositive integer.
//    X < 1.
//
//    Output, double R8_HYPER_2F1, the value of the function.
//
{
  double a0;
  double aa;
  double bb;
  double c0;
  double c1;
  double el = 0.5772156649015329;
  double eps;
  double f0;
  double f1;
  double g0;
  double g1;
  double g2;
  double g3;
  double ga;
  double gabc;
  double gam;
  double gb;
  double gbm;
  double gc;
  double gca;
  double gcab;
  double gcb;
  double gm;
  double hf;
  double hw;
  int j;
  int k;
  bool l0;
  bool l1;
  bool l2;
  bool l3;
  bool l4;
  bool l5;
  int m;
  int nm;
  double pa;
  double pb;
  double pi = 3.141592653589793;
  double r;
  double r0;
  double r1;
  double rm;
  double rp;
  double sm;
  double sp;
  double sp0;
  double x1;

  l0 = ( c == ( int ) ( c ) ) && ( c < 0.0 );
  l1 = ( 1.0 - x < 1.0E-15 ) && ( c - a - b <= 0.0 );
  l2 = ( a == ( int ) ( a ) ) && ( a < 0.0 );
  l3 = ( b == ( int ) ( b ) ) && ( b < 0.0 );
  l4 = ( c - a == ( int ) ( c - a ) ) && ( c - a <= 0.0 );
  l5 = ( c - b == ( int ) ( c - b ) ) && ( c - b <= 0.0 );

  if ( l0 || l1 )
  {
    std::cerr << "\n";
    std::cerr << "R8_HYPER_2F1 - Fatal error!\n";
    std::cerr << "  The hypergeometric series is divergent.\n";
    hf = 0.0;
    return hf;
  }

  if ( 0.95 < x )
  {
    eps = 1.0E-08;
  }
  else
  {
    eps = 1.0E-15;
  }

  if ( x == 0.0 || a == 0.0 || b == 0.0 )
  {
    hf = 1.0;
    return hf;
  }
  else if ( 1.0 - x == eps && 0.0 < c - a - b )
  {
    gc = webbur::r8_gamma ( c );
    gcab = webbur::r8_gamma ( c - a - b );
    gca = webbur::r8_gamma ( c - a );
    gcb = webbur::r8_gamma ( c - b );
    hf = gc * gcab / ( gca * gcb );
    return hf;
  }
  else if ( 1.0 + x <= eps && r8_abs ( c - a + b - 1.0 ) <= eps )
  {
    g0 = std::sqrt ( pi ) * std::pow ( 2.0, - a );
    g1 = webbur::r8_gamma ( c );
    g2 = webbur::r8_gamma ( 1.0 + a / 2.0 - b );
    g3 = webbur::r8_gamma ( 0.5 + 0.5 * a );
    hf = g0 * g1 / ( g2 * g3 );
    return hf;
  }
  else if ( l2 || l3 )
  {
    if ( l2 )
    {
      nm = ( int ) ( webbur::r8_abs ( a ) );
    }

    if ( l3 )
    {
      nm = ( int ) ( webbur::r8_abs ( b ) );
    }

    hf = 1.0;
    r = 1.0;

    for ( k = 1; k <= nm; k++ )
    {
      r = r * ( a + k - 1.0 ) * ( b + k - 1.0 ) 
        / ( k * ( c + k - 1.0 ) ) * x;
      hf = hf + r;
    }

    return hf;
  }
  else if ( l4 || l5 )
  {
    if ( l4 )
    {
      nm = ( int ) ( webbur::r8_abs ( c - a ) );
    }

    if ( l5 )
    {
      nm = ( int ) ( webbur::r8_abs ( c - b ) );
    }

    hf = 1.0;
    r  = 1.0;
    for ( k = 1; k <= nm; k++ )
    {
      r = r * ( c - a + k - 1.0 ) * ( c - b + k - 1.0 ) 
        / ( k * ( c + k - 1.0 ) ) * x;
      hf = hf + r;
    }
    hf = std::pow ( 1.0 - x, c - a - b ) * hf;
    return hf;
  }

  aa = a;
  bb = b;
  x1 = x;

  if ( x < 0.0 )
  {
    x = x / ( x - 1.0 );
    if ( a < c && b < a && 0.0 < b )
    {
      a = bb;
      b = aa;
    }
    b = c - b;
  }

  if ( 0.75 <= x )
  {
    gm = 0.0;

    if ( webbur::r8_abs ( c - a - b - ( int ) ( c - a - b ) ) < 1.0E-15 )
    {
      m = int ( c - a - b );
      ga = webbur::r8_gamma ( a );
      gb = webbur::r8_gamma ( b );
      gc = webbur::r8_gamma ( c );
      gam = webbur::r8_gamma ( a + m );
      gbm = webbur::r8_gamma ( b + m );

      pa = webbur::r8_psi ( a );
      pb = webbur::r8_psi ( b );

      if ( m != 0 )
      {
        gm = 1.0;
      }

      for ( j = 1; j <= std::abs ( m ) - 1; j++ )
      {
        gm = gm * j;
      }

      rm = 1.0;
      for ( j = 1; j <= std::abs ( m ); j++ )
      {
        rm = rm * j;
      }

      f0 = 1.0;
      r0 = 1.0;;
      r1 = 1.0;
      sp0 = 0.0;;
      sp = 0.0;

      if ( 0 <= m )
      {
        c0 = gm * gc / ( gam * gbm );
        c1 = - gc * std::pow ( x - 1.0, m ) / ( ga * gb * rm );

        for ( k = 1; k <= m - 1; k++ )
        {
          r0 = r0 * ( a + k - 1.0 ) * ( b + k - 1.0 ) 
            / ( k * ( k - m ) ) * ( 1.0 - x );
          f0 = f0 + r0;
        }

        for ( k = 1; k <= m; k++ )
        {
          sp0 = sp0 + 1.0 / ( a + k - 1.0 ) + 1.0 / ( b + k - 1.0 ) 
          - 1.0 / ( double ) ( k );
        }

        f1 = pa + pb + sp0 + 2.0 * el + std::log ( 1.0 - x );
        hw = f1;

        for ( k = 1; k <= 250; k++ )
        {
          sp = sp + ( 1.0 - a ) / ( k * ( a + k - 1.0 ) ) 
            + ( 1.0 - b ) / ( k * ( b + k - 1.0 ) );

          sm = 0.0;
          for ( j = 1; j <= m; j++ )
          {
            sm = sm + ( 1.0 - a ) 
              / ( ( j + k ) * ( a + j + k - 1.0 ) ) 
              + 1.0 / ( b + j + k - 1.0 );
          }

          rp = pa + pb + 2.0 * el + sp + sm + std::log ( 1.0 - x );

          r1 = r1 * ( a + m + k - 1.0 ) * ( b + m + k - 1.0 ) 
            / ( k * ( m + k ) ) * ( 1.0 - x );

          f1 = f1 + r1 * rp;

          if ( r8_abs ( f1 - hw ) < r8_abs ( f1 ) * eps )
          {
            break;
          }
          hw = f1;
        }
        hf = f0 * c0 + f1 * c1;
      }
      else if ( m < 0 )
      {
        m = - m;
        c0 = gm * gc / ( ga * gb * std::pow ( 1.0 - x, m ) );
        c1 = - ( - 1.0, m ) * gc / ( gam * gbm * rm );

        for ( k = 1; k <= m - 1; k++ )
        {
          r0 = r0 * ( a - m + k - 1.0 ) * ( b - m + k - 1.0 ) 
            / ( k * ( k - m ) ) * ( 1.0 - x );
          f0 = f0 + r0;
        }

        for ( k = 1; k <= m; k++ )
        {
          sp0 = sp0 + 1.0 / ( double ) ( k );
        }

        f1 = pa + pb - sp0 + 2.0 * el + std::log ( 1.0 - x );
        hw = f1;

        for ( k = 1; k <= 250; k++ )
        {
          sp = sp + ( 1.0 - a ) 
            / ( k * ( a + k - 1.0 ) ) 
            + ( 1.0 - b ) / ( k * ( b + k - 1.0 ) );

          sm = 0.0;
          for ( j = 1; j <= m; j++ )
          {
            sm = sm + 1.0 / ( double ) ( j + k );
          }

          rp = pa + pb + 2.0 * el + sp - sm + std::log ( 1.0 - x );

          r1 = r1 * ( a + k - 1.0 ) * ( b + k - 1.0 ) 
            / ( k * ( m + k ) ) * ( 1.0 - x );

          f1 = f1 + r1 * rp;

          if ( webbur::r8_abs ( f1 - hw ) < webbur::r8_abs ( f1 ) * eps )
          {
            break;
          }

          hw = f1;
        }

        hf = f0 * c0 + f1 * c1;
      }
    }
    else
    {
      ga = webbur::r8_gamma ( a );
      gb = webbur::r8_gamma ( b );
      gc = webbur::r8_gamma ( c );
      gca = webbur::r8_gamma ( c - a );
      gcb = webbur::r8_gamma ( c - b );
      gcab = webbur::r8_gamma ( c - a - b );
      gabc = webbur::r8_gamma ( a + b - c );
      c0 = gc * gcab / ( gca * gcb );
      c1 = gc * gabc / ( ga * gb ) * std::pow ( 1.0 - x, c - a - b );
      hf = 0.0;
      hw = hf;
      r0 = c0;
      r1 = c1;

      for ( k = 1; k <= 250; k++ )
      {
        r0 = r0 * ( a + k - 1.0 ) * ( b + k - 1.0 ) 
          / ( k * ( a + b - c + k ) ) * ( 1.0 - x );

        r1 = r1 * ( c - a + k - 1.0 ) * ( c - b + k - 1.0 ) 
          / ( k * ( c - a - b + k ) ) * ( 1.0 - x );

        hf = hf + r0 + r1;

        if ( webbur::r8_abs ( hf - hw ) < webbur::r8_abs ( hf ) * eps )
        {
          break;
        }
        hw = hf;
      }
      hf = hf + c0 + c1;
    }
  }
  else
  {
    a0 = 1.0;

    if ( a < c && c < 2.0 * a && b < c && c < 2.0 * b )
    {
      a0 = std::pow ( 1.0 - x, c - a - b );
      a = c - a;
      b = c - b;
    }

    hf = 1.0;
    hw = hf;
    r = 1.0;

    for ( k = 1; k <= 250; k++ )
    {
      r = r * ( a + k - 1.0 ) * ( b + k - 1.0 ) 
        / ( k * ( c + k - 1.0 ) ) * x;

      hf = hf + r;

      if ( webbur::r8_abs ( hf - hw ) <= webbur::r8_abs ( hf ) * eps )
      {
        break;
      }

      hw = hf;
    }
    hf = a0 * hf;
  }

  if ( x1 < 0.0 )
  {
    x = x1;
    c0 = 1.0 / std::pow ( 1.0 - x, aa );
    hf = c0 * hf;
  }

  a = aa;
  b = bb;

  if ( 120 < k )
  {
    std::cerr << "\n";
    std::cerr << "R8_HYPER_2F1 - Warning!\n";
    std::cerr << "  A large number of iterations were needed.\n";
    std::cerr << "  The accuracy of the results should be checked.\n";
  }

  return hf;
}
//****************************************************************************80

double r8_max ( double x, double y )

//****************************************************************************80
//
//  Purpose:
//
//    R8_MAX returns the maximum of two R8's.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 August 2004
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, double X, Y, the quantities to compare.
//
//    Output, double R8_MAX, the maximum of X and Y.
//
{
  double value;

  if ( y < x )
  {
    value = x;
  } 
  else
  {
    value = y;
  }
  return value;
}
//****************************************************************************80

double r8_mop ( int i )

//****************************************************************************80
//
//  Purpose:
//
//    R8_MOP returns the I-th power of -1 as an R8 value.
//
//  Discussion:
//
//    An R8 is an double value.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    16 November 2007
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int I, the power of -1.
//
//    Output, double R8_MOP, the I-th power of -1.
//
{
  double value;

  if ( ( i % 2 ) == 0 )
  {
    value = 1.0;
  }
  else
  {
    value = -1.0;
  }

  return value;
}
//****************************************************************************80

double r8_psi ( double xx )

//****************************************************************************80
//
//  Purpose:
//
//    R8_PSI evaluates the function Psi(X).
//
//  Discussion:
//
//    This routine evaluates the logarithmic derivative of the
//    Gamma function,
//
//      PSI(X) = d/dX ( GAMMA(X) ) / GAMMA(X)
//             = d/dX LN ( GAMMA(X) )
//
//    for real X, where either
//
//      - XMAX1 < X < - XMIN, and X is not a negative integer,
//
//    or
//
//      XMIN < X.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    09 February 2008
//
//  Author:
//
//    Original FORTRAN77 version by William Cody.
//    C++ version by John Burkardt.
//
//  Reference:
//
//    William Cody, Anthony Strecok, Henry Thacher,
//    Chebyshev Approximations for the Psi Function,
//    Mathematics of Computation,
//    Volume 27, Number 121, January 1973, pages 123-127.
//
//  Parameters:
//
//    Input, double XX, the argument of the function.
//
//    Output, double R8_PSI, the value of the function.
//
{
  double aug;
  double den;
  double four = 4.0;
  double fourth = 0.25;
  double half = 0.5;
  int i;
  int n;
  int nq;
  double one = 1.0;
  double p1[9] = { 
   4.5104681245762934160E-03, 
   5.4932855833000385356, 
   3.7646693175929276856E+02, 
   7.9525490849151998065E+03, 
   7.1451595818951933210E+04, 
   3.0655976301987365674E+05, 
   6.3606997788964458797E+05, 
   5.8041312783537569993E+05, 
   1.6585695029761022321E+05 };
  double p2[7] = { 
  -2.7103228277757834192, 
  -1.5166271776896121383E+01, 
  -1.9784554148719218667E+01, 
  -8.8100958828312219821, 
  -1.4479614616899842986, 
  -7.3689600332394549911E-02, 
  -6.5135387732718171306E-21 };
  double piov4 = 0.78539816339744830962;
  double q1[8] = { 
   9.6141654774222358525E+01, 
   2.6287715790581193330E+03, 
   2.9862497022250277920E+04, 
   1.6206566091533671639E+05, 
   4.3487880712768329037E+05, 
   5.4256384537269993733E+05, 
   2.4242185002017985252E+05, 
   6.4155223783576225996E-08 };
  double q2[6] = { 
   4.4992760373789365846E+01, 
   2.0240955312679931159E+02, 
   2.4736979003315290057E+02, 
   1.0742543875702278326E+02, 
   1.7463965060678569906E+01, 
   8.8427520398873480342E-01 };
  double sgn;
  double three = 3.0;
  double upper;
  double value;
  double w;
  double x;
  double x01 = 187.0;
  double x01d = 128.0;
  double x02 = 6.9464496836234126266E-04;
  double xinf = 1.70E+38;
  double xlarge = 2.04E+15;
  double xmax1 = 3.60E+16;
  double xmin1 = 5.89E-39;
  double xsmall = 2.05E-09;
  double z;
  double zero = 0.0;

  x = xx;
  w = webbur::r8_abs ( x );
  aug = zero;
//
//  Check for valid arguments, then branch to appropriate algorithm.
//
  if ( xmax1 <= - x || w < xmin1 )
  {
    if ( zero < x )
    {
      value = - xinf;
    }
    else
    {
      value = xinf;
    }
    return value;
  }

  if ( x < half )
  {
//
//  X < 0.5, use reflection formula: psi(1-x) = psi(x) + pi * cot(pi*x)
//  Use 1/X for PI*COTAN(PI*X)  when  XMIN1 < |X| <= XSMALL.
//
    if ( w <= xsmall )
    {
      aug = - one / x;
    }
//
//  Argument reduction for cotangent.
//
    else
    {
      if ( x < zero )
      {
        sgn = piov4;
      }
      else
      {
        sgn = - piov4;
      }

      w = w - ( double ) ( ( int ) ( w ) );
      nq = int ( w * four );
      w = four * ( w - ( double ) ( nq ) * fourth );
//
//  W is now related to the fractional part of 4.0 * X.
//  Adjust argument to correspond to values in the first
//  quadrant and determine the sign.
//
      n = nq / 2;

      if ( n + n != nq )
      {
        w = one - w;
      }

      z = piov4 * w;

      if ( ( n % 2 ) != 0 )
      {
        sgn = - sgn;
      }
//
//  Determine the final value for  -pi * cotan(pi*x).
//
      n = ( nq + 1 ) / 2;
      if ( ( n % 2 ) == 0 )
      {
//
//  Check for singularity.
//
        if ( z == zero )
        {
          if ( zero < x )
          {
            value = -xinf;
          }
          else
          {
            value = xinf;
          }
          return value;
        }
        aug = sgn * ( four / std::tan ( z ) );
      }
      else
      {
        aug = sgn * ( four * std::tan ( z ) );
      }
    }
    x = one - x;
  }
//
//  0.5 <= X <= 3.0.
//
  if ( x <= three )
  {
    den = x;
    upper = p1[0] * x;
    for ( i = 1; i <= 7; i++ )
    {
      den = ( den + q1[i-1] ) * x;
      upper = ( upper + p1[i]) * x;
    }
    den = ( upper + p1[8] ) / ( den + q1[7] );
    x = ( x - x01 / x01d ) - x02;
    value = den * x + aug;
    return value;
  }
//
//  3.0 < X.
//
  if ( x < xlarge )
  {
    w = one / ( x * x );
    den = w;
    upper = p2[0] * w;
    for ( i = 1; i <= 5; i++ )
    {
      den = ( den + q2[i-1] ) * w;
      upper = ( upper + p2[i] ) * w;
    }
    aug = ( upper + p2[6] ) / ( den + q2[5] ) - half / x + aug;
  }

  value = aug + std::log ( x );

  return value;
}
//****************************************************************************80

int r8col_compare ( int m, int n, double a[], int i, int j )

//****************************************************************************80
//
//  Purpose:
//
//    R8COL_COMPARE compares two columns in an R8COL.
//
//  Discussion:
//
//    An R8COL is an M by N array of R8's, regarded as an array of N columns,
//    each of length M.
//
//  Example:
//
//    Input:
//
//      M = 3, N = 4, I = 2, J = 4
//
//      A = (
//        1.  2.  3.  4.
//        5.  6.  7.  8.
//        9. 10. 11. 12. )
//
//    Output:
//
//      R8COL_COMPARE = -1
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    13 September 2005
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int M, N, the number of rows and columns.
//
//    Input, double A[M*N], the M by N array.
//
//    Input, int I, J, the columns to be compared.
//    I and J must be between 1 and N.
//
//    Output, int R8COL_COMPARE, the results of the comparison:
//    -1, column I < column J,
//     0, column I = column J,
//    +1, column J < column I.
//
{
  int k;
  int value;
//
//  Check.
//
  if ( i < 1 || n < i )
  {
    std::cerr << "\n";
    std::cerr << "R8COL_COMPARE - Fatal error!\n";
    std::cerr << "  Column index I is out of bounds.\n";
    std::cerr << "  I = " << i << "\n";
    std::exit ( 1 );
  }

  if ( j < 1 || n < j )
  {
    std::cerr << "\n";
    std::cerr << "R8COL_COMPARE - Fatal error!\n";
    std::cerr << "  Column index J is out of bounds.\n";
    std::cerr << "  J = " << j << "\n";
    std::exit ( 1 );
  }

  value = 0;

  if ( i == j )
  {
    return value;
  }

  k = 0;

  while ( k < m )
  {
    if ( a[k+(i-1)*m] < a[k+(j-1)*m] )
    {
      value = -1;
      return value;
    }
    else if ( a[k+(j-1)*m] < a[k+(i-1)*m] )
    {
      value = +1;
      return value;
    }
    k = k + 1;
  }

  return value;
}
//****************************************************************************80

void r8col_sort_heap_a ( int m, int n, double a[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8COL_SORT_HEAP_A ascending heapsorts an R8COL.
//
//  Discussion:
//
//    An R8COL is an M by N array of R8's, regarded as an array of N columns,
//    each of length M.
//
//    In lexicographic order, the statement "X < Y", applied to two real
//    vectors X and Y of length M, means that there is some index I, with
//    1 <= I <= M, with the property that
//
//      X(J) = Y(J) for J < I,
//    and
//      X(I) < Y(I).
//
//    In other words, the first time they differ, X is smaller.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    15 September 2005
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int M, N, the number of rows and columns.
//
//    Input/output, double A[M*N].
//    On input, the array of N columns of M-vectors.
//    On output, the columns of A have been sorted in lexicographic order.
//
{
  int i;
  int indx;
  int isgn;
  int j;

  if ( m <= 0 )
  {
    return;
  }

  if ( n <= 1 )
  {
    return;
  }
//
//  Initialize.
//
  i = 0;
  indx = 0;
  isgn = 0;
  j = 0;
//
//  Call the external heap sorter.
//
  for ( ; ; )
  {
    webbur::sort_heap_external ( n, &indx, &i, &j, isgn );
//
//  Interchange the I and J objects.
//
    if ( 0 < indx )
    {
      webbur::r8col_swap ( m, n, a, i, j );
    }
//
//  Compare the I and J objects.
//
    else if ( indx < 0 )
    {
      isgn = webbur::r8col_compare ( m, n, a, i, j );
    }
    else if ( indx == 0 )
    {
      break;
    }
  }

  return;
}
//****************************************************************************80

int *r8col_sort_heap_index_a ( int m, int n, double a[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8COL_SORT_HEAP_INDEX_A does an indexed heap ascending sort of an R8COL.
//
//  Discussion:
//
//    An R8COL is an M by N array of R8's, regarded as an array of N columns,
//    each of length M.
//
//    The sorting is not actually carried out.  Rather an index array is
//    created which defines the sorting.  This array may be used to sort
//    or index the array, or to sort or index related arrays keyed on the
//    original array.
//
//    A(*,J1) < A(*,J2) if the first nonzero entry of A(*,J1)-A(*,J2) 
//    is negative.
//
//    Once the index array is computed, the sorting can be carried out
//    "implicitly:
//
//      A(*,INDX(*)) is sorted,
//
//    Note that the index vector is 0-based.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    01 November 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int M, the number of rows in each column of A.
//
//    Input, int N, the number of columns in A.
//
//    Input, double A[M*N], the array.
//
//    Output, int R8COL_SORT_HEAP_INDEX_A[N], contains the sort index.  The
//    I-th column of the sorted array is A(*,INDX(I)).
//
{
  double *column;
  int i;
  int *indx;
  int indxt;
  int ir;
  int isgn;
  int j;
  int k;
  int l;

  if ( n < 1 )
  {
    return NULL;
  }

  indx = new int[n];

  for ( i = 0; i < n; i++ )
  {
    indx[i] = i;
  }

  if ( n == 1 )
  {
    return indx;
  }

  column = new double[m];

  l = n / 2 + 1;
  ir = n;

  for ( ; ; )
  {
    if ( 1 < l )
    {
      l = l - 1;
      indxt = indx[l-1];
      for ( k = 0; k < m; k++ )
      {
        column[k] = a[k+indxt*m];
      }
    }
    else
    {
      indxt = indx[ir-1];
      for ( k = 0; k < m; k++ )
      {
        column[k] = a[k+indxt*m];
      }
      indx[ir-1] = indx[0];
      ir = ir - 1;

      if ( ir == 1 )
      {
        indx[0] = indxt;
        break;
      }
    }

    i = l;
    j = l + l;

    while ( j <= ir )
    {
      if ( j < ir )
      {
        isgn = webbur::r8vec_compare ( m, a+indx[j-1]*m, a+indx[j]*m );

        if ( isgn < 0 )
        {
          j = j + 1;
        }
      }

      isgn = webbur::r8vec_compare ( m, column, a+indx[j-1]*m );

      if ( isgn < 0 )
      {
        indx[i-1] = indx[j-1];
        i = j;
        j = j + j;
      }
      else
      {
        j = ir + 1;
      }
    }
    indx[i-1] = indxt;
  }
  delete [] column;

  return indx;
}
//****************************************************************************80

int r8col_sorted_unique_count ( int m, int n, double a[], double tol )

//****************************************************************************80
//
//  Purpose:
//
//    R8COL_SORTED_UNIQUE_COUNT counts unique elements in a sorted R8COL.
//
//  Discussion:
//
//    An R8COL is an M by N array of R8's, regarded as an array of N columns,
//    each of length M.
//
//    The columns of the array may be ascending or descending sorted.
//
//    If the tolerance is large enough, then the concept of uniqueness
//    can become ambiguous.  If we have a tolerance of 1.5, then in the
//    list ( 1, 2, 3, 4, 5, 6, 7, 8, 9 ) is it fair to say we have only
//    one unique entry?  That would be because 1 may be regarded as unique,
//    and then 2 is too close to 1 to be unique, and 3 is too close to 2 to
//    be unique and so on.
//
//    This seems wrongheaded.  So I prefer the idea that an item is not
//    unique under a tolerance only if it is close to something that IS unique.
//    Thus, the unique items are guaranteed to cover the space if we include
//    a disk of radius TOL around each one.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    01 November 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int M, N, the number of rows and columns.
//
//    Input, double A[M*N], a sorted array, containing
//    N columns of data.
//
//    Input, double TOL, a tolerance for equality.
//
//    Output, int R8COL_SORTED_UNIQUE_COUNT, the number of unique columns.
//
{
  double diff;
  int i;
  int j1;
  int j2;
  int unique_num;

  unique_num = 0;

  if ( n <= 0 )
  {
    return unique_num;
  }

  unique_num = 1;
  j1 = 0;

  for ( j2 = 1; j2 < n; j2++ )
  {
    diff = 0.0;
    for ( i = 0; i < m; i++ )
    {
      diff = webbur::r8_max ( diff, webbur::r8_abs ( a[i+j1*m] - a[i+j2*m] ) );
    }
    if ( tol < diff )
    {
      unique_num = unique_num + 1;
      j1 = j2;
    }
  }

  return unique_num;
}
//****************************************************************************80

void r8col_swap ( int m, int n, double a[], int j1, int j2 )

//****************************************************************************80
//
//  Purpose:
//
//    R8COL_SWAP swaps columns J1 and J2 of an R8COL.
//
//  Discussion:
//
//    An R8COL is an M by N array of R8's, regarded as an array of N columns,
//    each of length M.
//
//  Example:
//
//    Input:
//
//      M = 3, N = 4, J1 = 2, J2 = 4
//
//      A = (
//        1.  2.  3.  4.
//        5.  6.  7.  8.
//        9. 10. 11. 12. )
//
//    Output:
//
//      A = (
//        1.  4.  3.  2.
//        5.  8.  7.  6.
//        9. 12. 11. 10. )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    23 October 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int M, N, the number of rows and columns.
//
//    Input/output, double A[M*N], the M by N array.
//
//    Input, int J1, J2, the columns to be swapped.
//    These columns are 1-based.
//
{
  int i;
  double temp;

  if ( j1 < 1 || n < j1 || j2 < 1 || n < j2 )
  {
    std::cerr << "\n";
    std::cerr << "R8COL_SWAP - Fatal error!\n";
    std::cerr << "  J1 or J2 is out of bounds.\n";
    std::cerr << "  J1 =   " << j1 << "\n";
    std::cerr << "  J2 =   " << j2 << "\n";
    std::cerr << "  NCOL = " << n << "\n";
    std::exit ( 1 );
  }

  if ( j1 == j2 )
  {
    return;
  }

  for ( i = 0; i < m; i++ )
  {
    temp          = a[i+(j1-1)*m];
    a[i+(j1-1)*m] = a[i+(j2-1)*m];
    a[i+(j2-1)*m] = temp;
  }

  return;
}
//****************************************************************************80

void r8col_undex ( int x_dim, int x_num, double x_val[], int x_unique_num, 
  double tol, int undx[], int xdnu[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8COL_UNDEX returns unique sorted indexes for an R8COL.
//
//  Discussion:
//
//    An R8COL is an M by N array of R8's, regarded as an array of N columns,
//    each of length M.
//
//    The goal of this routine is to determine a vector UNDX,
//    which points, to the unique elements of X, in sorted order,
//    and a vector XDNU, which identifies, for each entry of X, the index of
//    the unique sorted element of X.
//
//    This is all done with index vectors, so that the elements of
//    X are never moved.
//
//    The first step of the algorithm requires the indexed sorting
//    of X, which creates arrays INDX and XDNI.  (If all the entries
//    of X are unique, then these arrays are the same as UNDX and XDNU.)
//
//    We then use INDX to examine the entries of X in sorted order,
//    noting the unique entries, creating the entries of XDNU and
//    UNDX as we go.
//
//    Once this process has been completed, the vector X could be
//    replaced by a compressed vector XU, containing the unique entries
//    of X in sorted order, using the formula
//
//      XU(*) = X(UNDX(*)).
//
//    We could then, if we wished, reconstruct the entire vector X, or
//    any element of it, by index, as follows:
//
//      X(I) = XU(XDNU(I)).
//
//    We could then replace X by the combination of XU and XDNU.
//
//    Later, when we need the I-th entry of X, we can locate it as
//    the XDNU(I)-th entry of XU.
//
//    Here is an example of a vector X, the sort and inverse sort
//    index vectors, and the unique sort and inverse unique sort vectors
//    and the compressed unique sorted vector.
//
//      I     X  Indx  Xdni       XU  Undx  Xdnu
//    ----+-----+-----+-----+--------+-----+-----+
//      0 | 11.     0     0 |    11.     0     0
//      1 | 22.     2     4 |    22.     1     1
//      2 | 11.     5     1 |    33.     3     0
//      3 | 33.     8     7 |    55.     4     2
//      4 | 55.     1     8 |                  3
//      5 | 11.     6     2 |                  0
//      6 | 22.     7     5 |                  1
//      7 | 22.     3     6 |                  1
//      8 | 11.     4     3 |                  0
//
//    INDX(2) = 3 means that sorted item(2) is X(3).
//    XDNI(2) = 5 means that X(2) is sorted item(5).
//
//    UNDX(3) = 4 means that unique sorted item(3) is at X(4).
//    XDNU(8) = 2 means that X(8) is at unique sorted item(2).
//
//    XU(XDNU(I))) = X(I).
//    XU(I)        = X(UNDX(I)).
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    02 November 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int X_DIM, the dimension of the data values.
//    (the number of rows in the R8COL).
//
//    Input, int X_NUM, the number of data values,
//    (the number of columns in the R8COL).
//
//    Input, double X_VAL[X_DIM*X_NUM], the data values.
//
//    Input, int X_UNIQUE_NUM, the number of unique values in X_VAL.
//    This value is only required for languages in which the size of
//    UNDX must be known in advance.
//
//    Input, double TOL, a tolerance for equality.
//
//    Output, int UNDX[X_UNIQUE_NUM], the UNDX vector.
//
//    Output, int XDNU[X_NUM], the XDNU vector.
//
{
  double diff;
  int i;
  int *indx;
  int j;
  int k;
//
//  Implicitly sort the array.
//
  indx = webbur::r8col_sort_heap_index_a ( x_dim, x_num, x_val );
//
//  Walk through the implicitly sorted array X.
//
  i = 0;

  j = 0;
  undx[j] = indx[i];

  xdnu[indx[i]] = j;

  for ( i = 1; i < x_num; i++ )
  {
    diff = 0.0;
    for ( k = 0; k < x_dim; k++ )
    {
      diff = r8_max ( diff, 
        webbur::r8_abs ( x_val[k+indx[i]*x_dim] - x_val[k+undx[j]*x_dim] ) );
    }
    if ( tol < diff )
    {
      j = j + 1;
      undx[j] = indx[i];
    }
    xdnu[indx[i]] = j;
  }
  delete [] indx;

  return;
}
//****************************************************************************80

void r8col_unique_index ( int m, int n, double a[], double tol, 
  int unique_index[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8COL_UNIQUE_INDEX indexes the first occurrence of values in an R8COL.
//
//  Discussion:
//
//    An R8COL is an M by N array of R8 values.
//    It is regarded as an array of N columns of length M.
//
//    For element A(1:M,J) of the matrix, UNIQUE_INDEX(J) is the uniqueness
//   index of A(1:M,J).  That is, if A_UNIQUE contains the unique elements 
//    of A, gathered in order, then 
//
//      A_UNIQUE ( 1:M, UNIQUE_INDEX(J) ) = A(1:M,J)
//
//    The user must preallocate space for the output array UNIQUE_INDEX.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 November 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int M, N, the number of rows and columns of A.
//    The length of an "element" of A, and the number of "elements".
//
//    Input, double A[M*N], the array.
//
//    Input, double TOL, a tolerance for equality.
//
//    Output, int UNIQUE_INDEX[N], the unique index.
//
{
  double diff;
  int i;
  int j1;
  int j2;
  int unique_num;

  for ( j1 = 0; j1 < n; j1++ )
  {
    unique_index[j1] = -1;
  }
  unique_num = 0;

  for ( j1 = 0; j1 < n; j1++ )
  {
    if ( unique_index[j1] == -1 )
    {
      unique_index[j1] = unique_num;

      for ( j2 = j1 + 1; j2 < n; j2++ )
      {
        diff = 0.0;
        for ( i = 0; i < m; i++ )
        {
          diff = webbur::r8_max ( diff, 
            webbur::r8_abs ( a[i+j1*m] - a[i+j2*m] ) );
        }
        if ( diff <= tol )
        {
          unique_index[j2] = unique_num;
        }
      }
      unique_num = unique_num + 1;
    }
  }
  return;
}
//****************************************************************************80

int r8vec_compare ( int n, double a[], double b[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8VEC_COMPARE compares two R8VEC's.
//
//  Discussion:
//
//    An R8VEC is a vector of R8's.
//
//    The lexicographic ordering is used.
//
//  Example:
//
//    Input:
//
//      A1 = ( 2.0, 6.0, 2.0 )
//      A2 = ( 2.0, 8.0, 12.0 )
//
//    Output:
//
//      ISGN = -1
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    23 September 2005
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the number of entries in the vectors.
//
//    Input, double A[N], B[N], the vectors to be compared.
//
//    Output, int R8VEC_COMPARE, the results of the comparison:
//    -1, A is lexicographically less than B,
//     0, A is equal to B,
//    +1, A is lexicographically greater than B.
//
{
  int isgn;
  int k;

  isgn = 0;

  for ( k = 0; k < n; k++ )
  {
    if ( a[k] < b[k] )
    {
      isgn = -1;
      return isgn;
    }
    else if ( b[k] < a[k] )
    {
      isgn = +1;
      return isgn;
    }
  }
  return isgn;
}
//****************************************************************************80

void r8vec_copy ( int n, double a1[], double a2[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8VEC_COPY copies an R8VEC.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    03 July 2005
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the number of entries in the vectors.
//
//    Input, double A1[N], the vector to be copied.
//
//    Output, double A2[N], the copy of A1.
//
{
  int i;

  for ( i = 0; i < n; i++ )
  {
    a2[i] = a1[i];
  }
  return;
}
//****************************************************************************80

void r8vec_direct_product2 ( int factor_index, int factor_order, 
  double factor_value[], int factor_num, int point_num, double w[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8VEC_DIRECT_PRODUCT2 creates a direct product of R8VEC's.
//
//  Discussion:
//
//    An R8VEC is a vector of R8's.
//
//    To explain what is going on here, suppose we had to construct
//    a multidimensional quadrature rule as the product of K rules
//    for 1D quadrature.
//
//    The product rule will be represented as a list of points and weights.
//
//    The J-th item in the product rule will be associated with
//      item J1 of 1D rule 1,
//      item J2 of 1D rule 2, 
//      ..., 
//      item JK of 1D rule K.
//
//    In particular, 
//      X(J) = ( X(1,J1), X(2,J2), ..., X(K,JK))
//    and
//      W(J) = W(1,J1) * W(2,J2) * ... * W(K,JK)
//
//    So we can construct the quadrature rule if we can properly
//    distribute the information in the 1D quadrature rules.
//
//    This routine carries out that task for the weights W.
//
//    Another way to do this would be to compute, one by one, the
//    set of all possible indices (J1,J2,...,JK), and then index
//    the appropriate information.  An advantage of the method shown
//    here is that you can process the K-th set of information and
//    then discard it.
//
//  Example:
//
//    Rule 1: 
//      Order = 4
//      W(1:4) = ( 2, 3, 5, 7 )
//
//    Rule 2:
//      Order = 3
//      W(1:3) = ( 11, 13, 17 )
//
//    Rule 3:
//      Order = 2
//      W(1:2) = ( 19, 23 )
//
//    Product Rule:
//      Order = 24
//      W(1:24) =
//        ( 2 * 11 * 19 )
//        ( 3 * 11 * 19 )
//        ( 4 * 11 * 19 )
//        ( 7 * 11 * 19 )
//        ( 2 * 13 * 19 )
//        ( 3 * 13 * 19 )
//        ( 5 * 13 * 19 )
//        ( 7 * 13 * 19 )
//        ( 2 * 17 * 19 )
//        ( 3 * 17 * 19 )
//        ( 5 * 17 * 19 )
//        ( 7 * 17 * 19 )
//        ( 2 * 11 * 23 )
//        ( 3 * 11 * 23 )
//        ( 5 * 11 * 23 )
//        ( 7 * 11 * 23 )
//        ( 2 * 13 * 23 )
//        ( 3 * 13 * 23 )
//        ( 5 * 13 * 23 )
//        ( 7 * 13 * 23 )
//        ( 2 * 17 * 23 )
//        ( 3 * 17 * 23 )
//        ( 5 * 17 * 23 )
//        ( 7 * 17 * 23 )
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    18 April 2009
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int FACTOR_INDEX, the index of the factor being processed.
//    The first factor processed must be factor 0.
//
//    Input, int FACTOR_ORDER, the order of the factor.
//
//    Input, double FACTOR_VALUE[FACTOR_ORDER], the factor values for
//    factor FACTOR_INDEX.
//
//    Input, int FACTOR_NUM, the number of factors.
//
//    Input, int POINT_NUM, the number of elements in the direct product.
//
//    Input/output, double W[POINT_NUM], the elements of the
//    direct product, which are built up gradually.  
//
//  Local Parameters:
//
//    Local, integer START, the first location of a block of values to set.
//
//    Local, integer CONTIG, the number of consecutive values to set.
//
//    Local, integer SKIP, the distance from the current value of START
//    to the next location of a block of values to set.
//
//    Local, integer REP, the number of blocks of values to set.
//
{
  static int contig = 0;
  int i;
  int j;
  int k;
  static int rep = 0;
  static int skip = 0;
  int start;

  if ( factor_index == 0 )
  {
    contig = 1;
    skip = 1;
    rep = point_num;
    for ( i = 0; i < point_num; i++ )
    {
      w[i] = 1.0;
    }
  }

  rep = rep / factor_order;
  skip = skip * factor_order;

  for ( j = 0; j < factor_order; j++ )
  {
    start = 0 + j * contig;

    for ( k = 1; k <= rep; k++ )
    {
      for ( i = start; i < start + contig; i++ )
      {
        w[i] = w[i] * factor_value[j];
      }
      start = start + skip;
    }
  }

  contig = contig * factor_order;

  return;
}
//****************************************************************************80

double r8vec_sum ( int n, double a[] )

//****************************************************************************80
//
//  Purpose:
//
//    R8VEC_SUM returns the sum of an R8VEC.
//
//  Discussion:
//
//    An R8VEC is a double precision vector.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    15 October 2004
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int N, the number of entries in the vector.
//
//    Input, double A[N], the vector.
//
//    Output, double R8VEC_SUM, the sum of the vector.
//
{
  int i;
  double value;

  value = 0.0;
  for ( i = 0; i < n; i++ )
  {
    value = value + a[i];
  }

  return value;
}
//****************************************************************************80

void sort_heap_external ( int n, int *indx, int *i, int *j, int isgn )

//****************************************************************************80
//
//  Purpose:
//
//    SORT_HEAP_EXTERNAL externally sorts a list of items into ascending order.
//
//  Discussion:
//
//    The actual list is not passed to the routine.  Hence it may
//    consist of integers, reals, numbers, names, etc.  The user,
//    after each return from the routine, will be asked to compare or
//    interchange two items.
//
//    The current version of this code mimics the FORTRAN version,
//    so the values of I and J, in particular, are FORTRAN indices.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    05 February 2004
//
//  Author:
//
//    Original FORTRAN77 version by Albert Nijenhuis, Herbert Wilf.
//    C++ version by John Burkardt
//
//  Reference:
//
//    Albert Nijenhuis, Herbert Wilf,
//    Combinatorial Algorithms,
//    Academic Press, 1978, second edition,
//    ISBN 0-12-519260-6.
//
//  Parameters:
//
//    Input, int N, the length of the input list.
//
//    Input/output, int *INDX.
//    The user must set INDX to 0 before the first call.
//    On return,
//      if INDX is greater than 0, the user must interchange
//      items I and J and recall the routine.
//      If INDX is less than 0, the user is to compare items I
//      and J and return in ISGN a negative value if I is to
//      precede J, and a positive value otherwise.
//      If INDX is 0, the sorting is done.
//
//    Output, int *I, *J.  On return with INDX positive,
//    elements I and J of the user's list should be
//    interchanged.  On return with INDX negative, elements I
//    and J are to be compared by the user.
//
//    Input, int ISGN. On return with INDX negative, the
//    user should compare elements I and J of the list.  If
//    item I is to precede item J, set ISGN negative,
//    otherwise set ISGN positive.
//
{
  static int i_save = 0;
  static int j_save = 0;
  static int k = 0;
  static int k1 = 0;
  static int n1 = 0;
//
//  INDX = 0: This is the first call.
//
  if ( *indx == 0 )
  {

    i_save = 0;
    j_save = 0;
    k = n / 2;
    k1 = k;
    n1 = n;
  }
//
//  INDX < 0: The user is returning the results of a comparison.
//
  else if ( *indx < 0 )
  {
    if ( *indx == -2 ) 
    {
      if ( isgn < 0 ) 
      {
        i_save = i_save + 1;
      }
      j_save = k1;
      k1 = i_save;
      *indx = -1;
      *i = i_save;
      *j = j_save;
      return;
    }

    if ( 0 < isgn ) 
    {
      *indx = 2;
      *i = i_save;
      *j = j_save;
      return;
    }

    if ( k <= 1 ) 
    {
      if ( n1 == 1 ) 
      {
        i_save = 0;
        j_save = 0;
        *indx = 0;
      }
      else 
      {
        i_save = n1;
        j_save = 1;
        n1 = n1 - 1;
        *indx = 1;
      }
      *i = i_save;
      *j = j_save;
      return;
    }
    k = k - 1;
    k1 = k;
  }
//
//  0 < INDX: the user was asked to make an interchange.
//
  else if ( *indx == 1 ) 
  {
    k1 = k;
  }

  for ( ; ; )
  {

    i_save = 2 * k1;

    if ( i_save == n1 ) 
    {
      j_save = k1;
      k1 = i_save;
      *indx = -1;
      *i = i_save;
      *j = j_save;
      return;
    }
    else if ( i_save <= n1 ) 
    {
      j_save = i_save + 1;
      *indx = -2;
      *i = i_save;
      *j = j_save;
      return;
    }

    if ( k <= 1 ) 
    {
      break;
    }

    k = k - 1;
    k1 = k;
  }

  if ( n1 == 1 ) 
  {
    i_save = 0;
    j_save = 0;
    *indx = 0;
    *i = i_save;
    *j = j_save;
  }
  else 
  {
    i_save = n1;
    j_save = 1;
    n1 = n1 - 1;
    *indx = 1;
    *i = i_save;
    *j = j_save;
  }

  return;
}
//****************************************************************************80

void timestamp ( )

//****************************************************************************80
//
//  Purpose:
//
//    TIMESTAMP prints the current YMDHMS date as a time stamp.
//
//  Example:
//
//    31 May 2001 09:45:54 AM
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 September 2003
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    None
//
{
# define TIME_SIZE 40

  static char time_buffer[TIME_SIZE];
  const struct tm *tm;
  size_t len;
  time_t now;

  now = time ( NULL );
  tm = localtime ( &now );

  len = strftime ( time_buffer, TIME_SIZE, "%d %B %Y %I:%M:%S %p", tm );

  std::cout << time_buffer << "\n";

  return;
# undef TIME_SIZE
}
//****************************************************************************80

char *timestring ( )

//****************************************************************************80
//
//  Purpose:
//
//    TIMESTRING returns the current YMDHMS date as a string.
//
//  Example:
//
//    31 May 2001 09:45:54 AM
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    24 September 2003
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Output, char *TIMESTRING, a string containing the current YMDHMS date.
//
{
# define TIME_SIZE 40

  const struct tm *tm;
  size_t len;
  time_t now;
  char *s;

  now = time ( NULL );
  tm = localtime ( &now );

  s = new char[TIME_SIZE];

  len = strftime ( s, TIME_SIZE, "%d %B %Y %I:%M:%S %p", tm );

  return s;
# undef TIME_SIZE
}
//****************************************************************************80

void vec_colex_next3 ( int dim_num, int base[], int a[], bool *more )

//****************************************************************************80
//
//  Purpose:
//
//    VEC_COLEX_NEXT3 generates vectors in colex order.
//
//  Discussion:
//
//    The vectors are produced in colexical order, starting with
//
//    (1,        1,        ...,1),
//    (2,        1,        ...,1),
//     ...
//    (BASE(1),  1,        ...,1)
//
//    (1,        2,        ...,1)
//    (2,        2,        ...,1)
//    ...
//    (BASE(1),  2,        ...,1)
//
//    (1,        3,        ...,1)
//    (2,        3,        ...,1)
//    ...
//    (BASE(1),  BASE(2), ...,BASE(DIM_NUM)).
//
//  Example:
//
//    DIM_NUM = 2,
//    BASE = { 3, 3 }
//
//    1   1
//    2   1
//    3   1
//    1   2
//    2   2
//    3   2
//    1   3
//    2   3
//    3   3
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    19 August 2008
//
//  Author:
//
//    John Burkardt
//
//  Parameters:
//
//    Input, int DIM_NUM, the spatial dimension.
//
//    Input, int BASE[DIM_NUM], the bases to be used in each dimension.
//    In dimension I, entries will range from 1 to BASE[I].
//
//    Output, int A[DIM_NUM], the next vector.
//
//    Input/output, bool *MORE.  Set this variable false before
//    the first call.  On return, MORE is TRUE if another vector has
//    been computed.  If MORE is returned FALSE, ignore the output 
//    vector and stop calling the routine.
//
{
  int i;

  if ( !( *more ) )
  {
    for ( i = 0; i < dim_num; i++ )
    {
      a[i] = 1;
    }
    *more = true;
  }
  else
  {
    for ( i = 0; i < dim_num; i++ )
    {
      a[i] = a[i] + 1;

      if ( a[i] <= base[i] )
      {
        return;
      }
      a[i] = 1;
    }
    *more = false;
  }

  return;
}


}
