// $Id: APPSPACK_Mediator.cpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Mediator.cpp,v $

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Mediator.cpp
  \brief Implements APPSPACK::Mediator
*/

#include "APPSPACK_Mediator.hpp"
#include "APPSPACK_Print.hpp"
#include <numeric>

APPSPACK::Mediator::Mediator(APPSPACK::Parameter::List& params_in,
			     APPSPACK::Conveyor& conveyor_in) :
  conveyor(conveyor_in),
  params(params_in),
  debug(params_in.getParameter("Debug", 0)),
  maxEvaluations(params.getParameter("Maximum Evaluations", -1))
{
}

APPSPACK::Mediator::~Mediator()
{
}

void APPSPACK::Mediator::push(Citizen::Interface& citizen)
{
  const string& name = citizen.getName();
  if (ownermap.find(name) != ownermap.end())
  {
    cout << "APPSPACK::Mediator::push -- Citizen" << name << "already exists!" << endl;
    throw "APPSPACK Error";
  }

  vector<int> empty;
  ownermap[name] = empty;
  town.push_back(&citizen);
  record[name] = 0;
}

void APPSPACK::Mediator::mediate()
{
  preprocess();

  ConveyorList W;
  ConveyorList R;
  int loop = 0;
  while (townActive())
  {
    if (debug > 0)
      cout << "*** MEDIATOR ITERATION " << loop++ << ": Noncached evals: " 
           << conveyor.getCounter().getNumEvaluations() << endl;
    // Citizens can add to W and look at R.
    citizenExchange(W,R);
    // R has now been processed by all citizens--delete.
    pruneExchangeList(R);
    // Now vote on W order.
    citizenSelectOrder(W);
    // Communicate with Conveyor.
    conveyorExchange(W,R);

    if (maxEvaluations != -1)
    {
      // Only break on max evaluations if parameter has been set.
      if (conveyor.getCounter().getNumEvaluations() > maxEvaluations)
      {
        cerr << "Expended max function evaluations." << endl;
        cerr << "Max Evaluations = " << maxEvaluations << endl;
        break;
      }
    }
  }

  if (debug > 0)
    cout << "*** FINAL MEDIATOR ITERATION " << loop++ << ": Noncached evals: " 
         << conveyor.getCounter().getNumEvaluations() << endl;
  // We are done, no more submitting points.  
  W.prune();
  // Let Citizens see any evalautions that returned from
  citizenExchange(W,R);
  // Prune are and record evaluations."
  pruneExchangeList(R);

  postprocess();

  // Now collect pending points.
  cout << "Citizen Evaluations breakdown:" << endl;
  map<string,int>::iterator it;
  for (it = record.begin(); it!=record.end(); it++)
    cout << "Citizen " << it->first << ": " << it->second << " points evaluated. \n";
  cout << "Pending evaluations: " << conveyor.getNumPending() << endl;

  if ((maxEvaluations != -1) && (conveyor.getCounter().getNumEvaluations() > maxEvaluations))
    cout << "Exit Status: Expended max function evaluations." << endl;
  else
    cout << "Normal exit." << endl;
}

void APPSPACK::Mediator::deleteProcessedTags(const vector<int>& t)
{
  // vit used to point to a tag list in ownermap.
  vector<int>::iterator vit;
  // mit used to iterate through ownermap.
  map<string, vector<int> >::iterator mit;
  for (mit = ownermap.begin(); mit != ownermap.end(); mit++)
  {
    // v denotes a taglist in ownermap.
    vector<int>& v = mit->second;
    // Loop over entries of t.
    for (int i=0; i<t.size(); i++)
    {
      vit = find(v.begin(), v.end(), t[i]);
      // If vit != v.end(), then t[i] is contained in v.
      if (vit != v.end())
      {
	v.erase(vit);
        record[mit->first]++;
      }
    }
  }
}

//PRIVATE
bool APPSPACK::Mediator::townActive()
{
  bool isActive=false;
  APPSPACK::Citizen::State state;
  for (int i=0; i<town.size(); i++)
  {
    state = town[i]->getState();
    // Citizen wishes everyone to stop.
    if (state == APPSPACK::Citizen::MustStop)
      return false;
    
    // Check if at least one guy is active.
    if (state == APPSPACK::Citizen::PassiveContinue)
      isActive=true;
    if (state == APPSPACK::Citizen::MustContinue)
      isActive=true;
  }

  return isActive;
}

//PRIVATE
void APPSPACK::Mediator::citizenExchange(APPSPACK::ConveyorList& W,
					 const APPSPACK::ConveyorList& R)
{
  string name;
  APPSPACK::Citizen::State state;

  if (debug > 3)
  {
    cout << "==================================================" << endl;
    cout << "<Exchanging points with Citizens>" << endl;
    cout << "--------------------------------------------------" << endl;
  }

  for (int c=0; c<town.size(); c++)
  {
    state = town[c]->getState();
    name = town[c]->getName();
    
    if (debug > 3)
    {
      cout << "Name: " << name << " state = " << state << endl;
      cout << "--------------------------------------------------" << endl;
    }

    // Call exchange for active citizens.
    if (state != Citizen::Retired)
    {
      ConveyorList Wnew;
      town[c]->exchange(R, Wnew, ownermap);
      if (debug > 4)
      {
        cout << "Submitting trial points" << endl;
        Wnew.print("Wnew");
      }
      updateOwnerMap(name, Wnew);
      W.appendList(Wnew);
    }   

    if (debug > 3)
      cout << "--------------------------------------------------" << endl;
  }

  if (debug > 3)
  {
    cout << "<Exchange complete.>" << endl;
    cout << "==================================================" << endl;
  }
}

//PRIVATE
void APPSPACK::Mediator::citizenSelectOrder(APPSPACK::ConveyorList& W)
{
  // Map from priority level to vector of Citizens at same priority.
  map<int, vector<int> > lmap;   
  for (int i=0; i<priority.size(); i++)
    lmap[priority[i]].push_back(i);  
  
  vector<int> newq;
  int maxp=*max_element(priority.begin(), priority.end());
  for (int p=1; p<=maxp; p++)
  {
    // Get list of tags from each citizen at level p.  Place in cmap.

    // Processing level p.
    if (lmap.find(p) != lmap.end())
    {
      vector<int> clist = lmap[p];
      map<int, vector<int> > cmap;
      // Get votes from citizen that are at priority level p.
      int numpoint=getTagOrderFromCitizenList(W, clist, cmap); 
      spliceTags(clist, cmap, newq);
    }
  } // Now queue is formed.
  
  if (debug > 0)
  {
    map<string, int>::iterator it;
    cout << "Eval Breakdown: ";
    for (it=record.begin(); it!=record.end(); it++)
    {
      cout << it->first << " = " << it->second << " | ";
    }
    cout << endl;
  }

  ConveyorList oldW;
  oldW.prependList(W);

  while (newq.size() > 0)
  {
    W.push(oldW.pop(newq[0])); 
    newq.erase(newq.begin());
  }

}

//PRIVATE
void APPSPACK::Mediator::conveyorExchange(APPSPACK::ConveyorList& W,
					  APPSPACK::ConveyorList& R)
{
  conveyor.exchange(W,R);
}

int APPSPACK::Mediator::getTagOrderFromCitizenList(const APPSPACK::ConveyorList& W,
                                                   const vector<int>& clist,
                                                   map<int, vector<int> >& cmap)
{
  vector<int> wtags;
  W.getTagList(wtags);
   
  int numpoint=0; // Counts number of points at level p.
  vector<int>::const_iterator it;
  for (it=clist.begin(); it!=clist.end(); it++)
  {
    // Get new votes for citizen clist[i].
    int c=*it;
    vector<int> cvotes;
    if (town[c]->getState() != Citizen::Retired)
      town[c]->vote(W, ownermap, cvotes); // not voting, but rather ordering
    
    // Error check. Make sure Citizen only votes on active queue.
    for (int k = 0; k < cvotes.size(); k ++)
    {
      int tag = cvotes[k];
      // Verify that the Citizen is not voting on points outside the queue W.
      if (find(wtags.begin(),wtags.end(), tag) == wtags.end())
      {
        cerr << "APPSPACK::Mediator::citizenSelectOrder: " << town[c]->getName()
             << " voting on point " << tag << " not in queue W." << endl;
        throw "APPSPACK Error";
      }
    } // Finished check.
    
    // Now add to cmap.  Tags ok.
    cmap[c] = cvotes;
    numpoint += cvotes.size();
  }

  return numpoint;
}

void  APPSPACK::Mediator::spliceTags(const vector<int>& clist, 
                                     map<int, vector<int> >& cmap,
                                     vector<int>& newq)
{
  int numpoint=0;
  for (int i=0; i<cmap.size(); i++)
    numpoint += cmap[i].size();

  // Let's order clist so that citizens at same priority level
  // with fewer points evaluated are submitted first.
  vector<int> orderedClist;
  {
    // First create reverse map to order Citizens by eval count.
    multimap<int,int> reverseRecord;  // maps evals to citizens.
    for (int i=0; i<clist.size(); i++)
      reverseRecord.insert(make_pair(record[town[clist[i]]->getName()], clist[i]));
    
    multimap<int,int>::const_iterator it;
    for(it = reverseRecord.begin(); it != reverseRecord.end(); it++)
      orderedClist.push_back(it->second);
  }

  // Note: Important to order this way in case only 1 processors is returning
  // at at time.

  int add_count=0;
  int del_count=0;

  for (int i=0; i<numpoint; i++)
  {
    // Add a point from top of each list
    for (int j=0; j<orderedClist.size(); j++)
    {
      int cj=orderedClist[j];
      if (cmap[cj].size() > 0)
      {
        int tag = cmap[cj][0];
        cmap[cj].erase(cmap[cj].begin());
        if (find(newq.begin(), newq.end(), tag) == newq.end())
        {
          newq.push_back(tag);
          add_count++;
        }
        else
          del_count++;
      }
    }
  }

  if ((add_count+del_count) != numpoint)
  {
    cerr << "APPSPACK::Mediator: Problem with tag formation." << endl;
    throw "APPSPACK Error";
  }

}

//PRIVATE
void APPSPACK::Mediator::pruneExchangeList(APPSPACK::ConveyorList& R)
{
  vector<int> tagr;
  R.getTagList(tagr);
  deleteProcessedTags(tagr);
  R.prune();
}

void APPSPACK::Mediator::preprocess()
{
  /*
  cout << "\n"
       << "-----------------------------------------------------\n"
       << "APPSPACK: Asynchronous Parallel Pattern Search\n"
       << "Written by T. G. Kolda et al., Sandia National Labs\n"
       << "For more information visit \n"
       << "http://software.sandia.gov/appspack\n"
       << "-----------------------------------------------------\n"
       << "\n";
  */

  cout << "\n" << "*** Mediator ***" << "\n";
  conveyor.print();
 
  if (debug > 3)
  {
    cout << "==================================================" << endl;
    cout << "<Preprocessing>" << endl;
    cout << "--------------------------------------------------" << endl;
  }

  for (int i = 0; i < town.size(); i++)
  {
    if (debug > 3)
    {
      cout << "Running preprocess commands on " << town[i]->getName() << endl;
      cout << "---------------------------------------------------" << endl;
    }
    
    town[i]->preprocess();

    if (debug > 3)
      cout << "---------------------------------------------------" << endl;
  }

  if (debug > 3)
  {
    cout << "<Preprocessing Complete>" << endl;
    cout << "==================================================" << endl;
  }

  // Set default values.
  priority.assign(town.size(), 1);

  cout << "Number of citizens: " << town.size() << endl;

  if (params.isParameterVector("Priority Level"))
  {
    Vector pv = params.getVectorParameter("Priority Level"); 

    if (pv.size() != town.size())
    {
      cerr << "APPSPACK::Mediator::preprocess(): \"Priority Level\" vector has incorrect size" << endl;
      throw "APPSPACK Error";
    }

    for (int i = 0; i < pv.size(); i++)
      priority[i] = int(pv[i]);
  } 
}

void APPSPACK::Mediator::postprocess()
{
  if (debug > 3)
  {
    cout << "==================================================" << endl;
    cout << "<Postprocessing>" << endl;
    cout << "--------------------------------------------------" << endl;
  }

  for (int i=0; i<town.size(); i++)
  {
    if (debug > 3)
    {
      cout << "Running postprocess commands on " << town[i]->getName() << endl;
      cout << "---------------------------------------------------" << endl;
    }

    town[i]->postprocess();
    
    if (debug > 3)
      cout << "---------------------------------------------------" << endl;
  }
 
  if (debug > 3)
  {
    cout << "<Postprocessing Complete>" << endl;
    cout << "==================================================" << endl;
  }
 
  cout << "\n" << "*** Mediator exiting: all points processed ***" << endl << endl;
  cout << "Evaluation breakdown:";
  conveyor.getCounter().print();

  // Print the final solution
  if (Print::doPrint(Print::FinalSolution))
  {
    /*
    cout << "\n"
	 << "-----------------------------------------------------\n"
	 << "Thank you!\n"
	 << "APPSPACK: Asynchronous Parallel Pattern Search\n"
	 << "Written by T. G. Kolda et al., Sandia National Labs\n"
	 << "For more information visit \n"
	 << "http://software.sandia.gov/appspack\n"
	 << "-----------------------------------------------------\n"
	 << "\n";
    */
  }

}

void APPSPACK::Mediator::updateOwnerMap(const string& name, const APPSPACK::ConveyorList& P)
{
  vector<int> tags;
  P.getTagList(tags);
  ownermap[name].insert(ownermap[name].end(),tags.begin(),tags.end());
}

ostream& operator<<(ostream& stream, const map<string, vector<int> >& m)
{
  map<string, vector<int> >::const_iterator it;
  for( it = m.begin(); it != m.end(); it++)
    stream << "Citizen  " << it->first << " tags: " << it->second << endl;

  return stream;
}

//! Prints the contents of a vector of ints.
ostream& operator<<(ostream& stream, const vector<int>& v)
{
  if (v.size() == 0)
  {
    stream << "<empty>";
    return stream;
  }
  stream << v[0];
  for (int i=1; i<v.size(); i++)
    stream << " " << v[i];

  return stream;
}

