// $Id: APPSPACK_Citizen_SystemCall.cpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_SystemCall.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_SystemCall.cpp
  \brief Implements APPSPACK::Citizen_SystemCall

  \todo Josh Comment parameters.
*/

#include "APPSPACK_Citizen_SystemCall.hpp"
#include "APPSPACK_Combiner_Generic.hpp"

int APPSPACK::Citizen::SystemCall::staticCount = 0;

APPSPACK::Citizen::SystemCall::SystemCall(Parameter::List& params_in,
					  Combiner::Generic& combiner_in,
					  const string name_in) :
  params(params_in),
  name(name_in),
  executableName(params.getParameter("Executable Name", "a.out")),
  inputPrefix(params.getParameter("Input Prefix", "oracle_input")),
  outputPrefix(params.getParameter("Output Prefix", "oracle_output")),
  defaultStep(params.getParameter("Default Step", 1.0)),
  defaultVotes(params.getParameter("Default Votes", 1)),
  isInputFile(false),
  isOutputFile(false),
  precision(params.getParameter("Output File Precision", 14)),
  combiner(combiner_in)
{    
  
}

APPSPACK::Citizen::SystemCall::~SystemCall()
{
}

void APPSPACK::Citizen::SystemCall::preprocess()
{
}

void APPSPACK::Citizen::SystemCall::postprocess()
{
}

void APPSPACK::Citizen::SystemCall::exchange(const APPSPACK::ConveyorList& R,
					     APPSPACK::ConveyorList& Wnew, 
                                             const map<string, vector<int> >& ownermap)
{
  // Plan 1.
  // If a worker is free, then we call the executable, by calling the
  // evaluator worker.  Reset deep copies Wk and Rk to empty.
  // Otherwise, append points to a records of W and R since last system call.
  // Quueue is always up to date.  Just need to keep a record of R.

  // Processing new batch of points.  Clear votes from last batch.
  const vector<int>& citizen_tags = ownermap.find(name)->second;

  tagVote.clear();
  createStrings();
  writeInputFile(R, citizen_tags);
  runProgram();
  readOutputFile(Wnew);
  deleteFiles();
}

void APPSPACK::Citizen::SystemCall::vote(const ConveyorList& W, 
					 const map<string, vector<int> >& ownermap,
					 map<int,int>& tagVote_out)
{
  tagVote_out = tagVote;
}

APPSPACK::Citizen::State APPSPACK::Citizen::SystemCall::getState()
{
  return APPSPACK::Citizen::StopRequest;
}

const string& APPSPACK::Citizen::SystemCall::getName() const
{
  return name;
}

void APPSPACK::Citizen::SystemCall::createStrings()
{
  char tmp[128];
  sprintf(tmp, "%d", staticCount++);
  fileTag = tmp;
  
  inputFileName = inputPrefix + "." + fileTag;
  outputFileName = outputPrefix + "." + fileTag;
  execString = executableName + " " + inputFileName + " " 
    + outputFileName + " " + fileTag;
}

void APPSPACK::Citizen::SystemCall::writeInputFile(const APPSPACK::ConveyorList& R,
						   const vector<int>& citizen_tags)
{
  ofstream stream;
  stream.open(inputFileName.c_str());

  if (!stream) 
  {
    cerr << "APPSPACK::Citizen::SystemCall::writeInputFile - Unable to open the following file: " 
	 << inputFileName << endl;
    return;
  }

  isInputFile = true;
  list<Point*>::const_iterator it;

  stream << "New Evaluations: " << R.size() << endl;
  for (it = R.begin(); it != R.end(); it++)
  {
    stream << "f=[ ";
    (*it)->getVecF().leftshift(stream,precision);
    stream << " ]";
    
    stream << " x=[ ";
    (*it)->getX().leftshift(stream,precision);
    stream << " ] ";

    int tag = (*it)->getTag();
    if (find(citizen_tags.begin(), citizen_tags.end(), tag) != citizen_tags.end())
      stream << "own=[ true ] " << endl;
    else
      stream << "own=[ false ] " << endl;
  }
  
  stream.close();
}

void  APPSPACK::Citizen::SystemCall::runProgram()
{
  //  system(execString.c_str());
}

void APPSPACK::Citizen::SystemCall::readOutputFile(APPSPACK::ConveyorList& Wnew)
{
  ifstream fin;
  fin.open(outputFileName.c_str());
  
  if (!fin) 
  {
    cerr << "\nWarning - no output file for the point corresponding to tag=" << fileTag << endl;
    fin.close();
    return;
  }
  
  isOutputFile = true;
  
  if (fin.eof())
  {
    cerr << "\nWarning - empty output file for the point corresponding to tag=" << fileTag << endl;
    fin.close();
    return;
  }

  string line;
  while(getline(fin,line))
  {
    if (line.find("x") != string::npos)
    {
      // Line contains a new point to add to the W.
      Wnew.push(processNewX(line));
    }
  }
}

void APPSPACK::Citizen::SystemCall::processTagVote(const string& line)
{
  // Get tag.
  int tag = str2int(getContent(line, "tag"));
  
  // Get votes.
  int votes = str2int(getContent(line, "votes"));

  // Update votes.  
  tagVote[tag] = votes;
}

APPSPACK::Point* APPSPACK::Citizen::SystemCall::processNewX(const string& line)
{
  // Get x.
  Vector x;
  str2vec(getContent(line, "x"), x);

  // Get votes.
  int votes=defaultVotes;
  string vs = getContent(line, "votes");
  if (vs.size() > 0)
    votes = str2int(vs);
  
  // Get step.
  double step = defaultStep;
  string ss = getContent(line, "step");
  if (ss.size() > 0)
    step = str2dbl(ss);

  // Create empty f.
  Vector f;
  string msg="(" + name + ")";

  Point* p = new Point(x, f, step, 0.0, combiner, msg);

  // Add votes to tagVote list.
  tagVote[p->getTag()] = votes;

  return p;
}

string APPSPACK::Citizen::SystemCall::getContent(const string& line, 
						 const string& name) const
{
  // Get start and stop of data.  i.e. find insides of [ ].
  string s = name + "=[";
  string::size_type start = line.find(s) + s.size();
  string::size_type stop  = line.find("]", start);
  return line.substr(start, stop-start);
}

int APPSPACK::Citizen::SystemCall::str2int(const string& item) const
{
  int k;
  istringstream ss(item);
  ss >> k;  
  return k;
}

double APPSPACK::Citizen::SystemCall::str2dbl(const string& item) const
{
  double d;
  istringstream ss(item);
  ss >> d;
  return d;
}

void APPSPACK::Citizen::SystemCall::str2vec(const string& item, 
					    APPSPACK::Vector& x) const
{
  x.resize(0);
  istringstream ss(item);
  
  double xi;
  while (ss >> xi)
    x.push_back(xi);
}

void APPSPACK::Citizen::SystemCall::deleteFiles()
{
  if (1)
  {
#ifndef SNL_TFLOPS_ENV
  if (isInputFile)
    unlink(inputFileName.c_str());
  if (isOutputFile)
    unlink(outputFileName.c_str());
#endif
  }

  isInputFile = false;
  isOutputFile = false;
}
