// $Id: APPSPACK_Citizen_NAPPSPACK.cpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_NAPPSPACK.cpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_NAPPSPACK.cpp
  \brief Implements APPSPACK::Citizen_NAPPSPACK
*/

#include "APPSPACK_Citizen_NAPPSPACK.hpp"
#include "NAPPSPACK_Method_Smooth.hpp"
#include <cassert>

APPSPACK::Citizen::NAPPSPACK::NAPPSPACK(Parameter::List& params_in,
			    const Constraints::Linear& constraints_in,
			    Combiner::Generic& combiner_in,
			    const string name_in) :
  params(params_in),
  name(name_in),
  constraints(constraints_in),
  debug(0)
{
}

APPSPACK::Citizen::NAPPSPACK::NAPPSPACK(Parameter::List& params_in,
			    const Constraints::Linear& constraints_in,
			    const string name_in) :
  params(params_in),
  name(name_in),
  constraints(constraints_in),
  solver(NULL),
  method(NULL),
  debug(0)
{
}

APPSPACK::Citizen::NAPPSPACK::~NAPPSPACK()
{
}

void APPSPACK::Citizen::NAPPSPACK::copyDoubleParameter(string paramName)
{
  // Now set some individual parameters.
  if (params.sublist(name).isParameter(paramName))
  {
    double d=params.sublist(name).getParameter(paramName, 1.0);
    params.sublist("Nonlinear").setParameter(paramName, d);
  }
}

void APPSPACK::Citizen::NAPPSPACK::preprocess()
{
  /*
  cout << "\n"
       << "-----------------------------------------------------\n"
       << "NAPPSPACK: Nonlinear Asynchronous Parallel Pattern Search\n"
       << "Sandia National Labs\n"
       << "For more information visit \n"
       << "http://software.sandia.gov/appspack\n"
       << "-----------------------------------------------------\n"
       << "\n";
  */

  debug=params.sublist("Nonlinear").getParameter("Debug", 0);
  params.sublist("Solver").setParameter("Mediator Debug", debug);

  filterLevel=params.sublist("Nonlinear").getParameter("Filter Level", 0);
  canForceQuit=params.sublist(name).getParameter("Can Force Quit", false);

  //  maxTotalEvaluations=params.sublist("Nonlinear").getParameter("Maximum Total Evaluations", 100000);
  maxMajorEvaluations=params.sublist("Nonlinear").getParameter("Maximum Major Evaluations", 1000);
  maxMajorIterations=params.sublist("Nonlinear").getParameter("Maximum Major Iterations", 1000);
 
  copyDoubleParameter("Initial Penalty Value");
  copyDoubleParameter("Initial Smoothing Value");
  copyDoubleParameter("Constraint Tolerance");
  copyDoubleParameter("Maximum Penalty Value");
  copyDoubleParameter("Minimum Smoothing Value");
  copyDoubleParameter("Penalty Value Increase");
  copyDoubleParameter("Smoothing Value Decrease");
  
  // Get number of nonlinear constraints.
  nlec = params.sublist("Nonlinear").getParameter("Number Equality", 0);
  nlic = params.sublist("Nonlinear").getParameter("Number Inequality", 0);

  cout << "=====================================================\n";
  cout << "Nonlinear/linear constraint summary.\n";
  cout << "=====================================================";
  constraints.print();
  cout << "Number of nonlinear equality: " << nlec << endl;
  cout << "Number of nonlinear inequality: " << nlic << endl;

  // Now add in method type specified by Citizen.
  string methodType=params.sublist(name).getParameter("Method", "undefined");
  params.sublist("Nonlinear").setParameter("Method", methodType);

  int oldEval=0; 
  int oldCach=0; 

  majorState=::NAPPSPACK::NLSolver::MaximumMajors;
  iterationCount=0;
  evalCount=0;
  // Create method.
  method = (::NAPPSPACK::Method::Generic*)(new ::NAPPSPACK::Method::Smooth(params));
  cout << "=====================================================\n" 
       << "Solving with " << method->getName() << ":\n" 
       << "=====================================================\n";

  // Create Citizen solver.
  solver = new Citizen::GSS(params, constraints, 
			    method->getCombiner(), "Solver");
  success="f";
  solver->preprocess();
  
  // Called last as this may be set by solver.
  etaStar = params.sublist("Nonlinear").getParameter("Constraint Tolerance", 1e-5);
}

void APPSPACK::Citizen::NAPPSPACK::postprocess()
{
  // Get best found so far.
  solver->getBestX(xBest);
  solver->getBestVecF(fcBest);

  cout.precision(12);
  cout << "-------------------------------------\n";
  cout << "Exit message from " << name << endl;
  cout << "-------------------------------------\n";
  cout << "(f[0], cinf, minorState, nvar, nlec, nlic) = " 
       << fcBest[0] << ", " << method->infnorm(fcBest) << ", " << success
       << ", " << xBest.size() << ", " << nlec << ", " << nlic << endl;
  cout << "Best Found = " 
       << fcBestTol[0] << ", " << method->infnorm(fcBestTol) << ", " << success
       << ", " << xBest.size() << ", " << nlec << ", " << nlic << endl;
  cout << "Major State = " << majorState << endl;
  cout << "(Use method " << method->getName() << ")" << endl;
  cout << "=============================================\n" << endl;

  // Clean up.
  delete method;
  delete solver;
}

void APPSPACK::Citizen::NAPPSPACK::updateSubproblem()
{
  minorState=APPSPACK::Solver::Continue;
  solver->getBestX(xBest);
  solver->getBestVecF(fcBest);
  params.sublist("Solver").setParameter("Initial X", xBest);
  params.sublist("Solver").setParameter("Initial F", fcBest); 
  method->update(xBest, fcBest, minorState);
  solver->postprocess();
  delete solver;
  solver = new Citizen::GSS(params, constraints, 
			    method->getCombiner(), "Solver");
  solver->preprocess();
}

void APPSPACK::Citizen::NAPPSPACK::exchange(const APPSPACK::ConveyorList& R,
                                            APPSPACK::ConveyorList& Wnew, 
                                            const map<string, vector<int> >& ownermap)
{
  const vector<int>& citizen_tags = ownermap.find(name)->second;

  // Count number of evaluations used.
  vector<int> mytags;
  R.getTagList(mytags);
  for (int i=0; i<citizen_tags.size(); i++)
    if (find(mytags.begin(), mytags.end(), citizen_tags[i]) != mytags.end())
      evalCount++;

  if ((solver->getState() == APPSPACK::Citizen::StopRequest) ||
      (evalCount > maxMajorEvaluations))
  {
    // Update subproblem.
    if (evalCount > maxMajorEvaluations)
      cout << "NLP expended maximum evaluations" << endl;

    updateSubproblem();
    iterationCount++;
    cout << name << " Major: " << iterationCount << ") ";
    method->printMajor(xBest,fcBest);
    cout << ", evals=" << evalCount << endl;



    if (method->isOptimal(fcBest))
    {
      cout << "NLP Converged!" << endl;  
      success="s";
      majorState=::NAPPSPACK::NLSolver::SuccessfulExit;
    }

    evalCount = 0;
  }

  if (debug > 3)
    R.print("\n****\n"+name+" received from mediator");

  // Make sure all points have the correct combiner.
  APPSPACK::ConveyorList Rthis;
  list<APPSPACK::Point*>::const_iterator itr;
  for (itr=R.begin(); itr!=R.end(); itr++)
  {
    int tag = (*itr)->getTag();
    double step = (*itr)->getStep();
    Vector fc = (*itr)->getVecF();
    Vector x = (*itr)->getX();
    bool isGoodFeasible=updateBest(x,fc);

    if (find(citizen_tags.begin(), citizen_tags.end(), tag) != citizen_tags.end())
    {
      // This is our point.  Let's copy it.
      Rthis.push(new Point(**itr));
    }
    else
    {
      // Not our point.  
      if ((filterLevel==0) || ((filterLevel==1) && isGoodFeasible))
      {        
        Point* p = new Point(x, fc, step, 0.0, 
                             method->getCombiner(), "External");
        Rthis.push(p);
      }
    }
  }
  
  // Must copy map to pass "Solver" as name.
  map<string, vector<int> > tmpmap = ownermap;
  tmpmap["Solver"] = ownermap.find(name)->second;
  solver->exchange(Rthis, Wnew, tmpmap);

  if (debug > 3)
  {
    Vector tmpX;
    Vector tmpF;
    solver->getBestX(tmpX);
    solver->getBestVecF(tmpF);
    cout << name << " Best x=" << tmpX << endl;
    cout << name << " Best fc=" << tmpF << ", combined(fc)="
         << method->getCombiner()(tmpF) << endl;
    Wnew.print(name+": Submitting following points to queue");
    cout << "*****\n" << endl;
  }

  Rthis.prune();

}

void APPSPACK::Citizen::NAPPSPACK::vote(const APPSPACK::ConveyorList& W,
				  const map<string, vector<int> >& ownermap,
				  vector<int>& tagOrder)
{
  // Must copy map to pass "Solver" as name.
  map<string, vector<int> > tmpmap = ownermap;
  tmpmap["Solver"] = ownermap.find(name)->second;
  solver->vote(W, tmpmap, tagOrder);
}

APPSPACK::Citizen::State APPSPACK::Citizen::NAPPSPACK::getState()
{
  if ((majorState != ::NAPPSPACK::NLSolver::MaximumMajors) ||
      iterationCount > maxMajorIterations)
  {
    // We are finished!!!
    if (canForceQuit==true)
      return APPSPACK::Citizen::MustStop;
    else
      return APPSPACK::Citizen::StopRequest;
  }

  return APPSPACK::Citizen::PassiveContinue;
}

const string& APPSPACK::Citizen::NAPPSPACK::getName() const
{
  return name;
}

double APPSPACK::Citizen::NAPPSPACK::infnorm(APPSPACK::Vector fc)
{
  assert( (fc.size() == (nlic+nlec+1)) && "fc has incorrect size!");

  // Equality constraints (recall fc[0] = f(x)):
  // fc[1], ... , fc[nlec] 
  // Inequality constraints. (note that we want fc[i] <= 0):
  // fc[nlec+1], ... , fc[fc.size() - 1]

  double norm=0;
  for (int i=1; i<fc.size(); i++)
  {
    if (i <= nlec)
      norm = max(fabs(fc[i]), norm);
    else
      norm = max(fc[i], norm);
  }
  
  return norm;
}

bool APPSPACK::Citizen::NAPPSPACK::updateBest(const APPSPACK::Vector& x, 
                                              const APPSPACK::Vector& fc)
{
  bool goodFeasible=false;

  // Always take first evaluations.
  if (fcBestTol.size() == 0)
  {
    xBestTol=x;
    fcBestTol=fc;
  }
  
  double fcnorm=infnorm(fc);
  double fcnormTol=infnorm(fcBestTol);
  
  // Case I.  Both fc and fcBestTol satisfy tolerance.    
  if ((fcnorm < etaStar) && (fcnormTol < etaStar))
  {
    goodFeasible=true;
    if (fc[0] < fcBestTol[0])
    {
      xBestTol=x;
      fcBestTol=fc;
    }
  }
  else if (fcnorm < fcnormTol)
  {
    goodFeasible=true;
    // Neither satisfies tolerance, but x is closer to feasible
    xBestTol=x;
    fcBestTol=fc;
  }

  return goodFeasible;
}
