// $Id: APPSPACK_Citizen_GSS.hpp,v 1.2 2008/05/02 00:57:23 tgkolda Exp $ 
// $Source: /usr/local/cvsroot/hopspack/src/APPSPACK_Citizen_GSS.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_Citizen_GSS.hpp
  \brief Class definition for APPSPACK::Citizen_GSS
*/

#ifndef APPSPACK_CITIZEN_GSS_HPP
#define APPSPACK_CITIZEN_GSS_HPP

#include "APPSPACK_Citizen_Interface.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_Iterator.hpp"
#include "APPSPACK_List.hpp"

namespace APPSPACK
{
namespace Citizen
{

//! Wrapper for an instance of class Iterator. 
/*! Todo add comments. */
class GSS : public Interface
{
public:
  //! Constructor 
  GSS(Parameter::List& params_in, 
      const Constraints::Linear& constraints_in,
      Combiner::Generic& combiner_in,
      const string name);
  
  //! Destructor
  ~GSS();

  //! Prints initial information corresponding to solver parameter "Debug".
  virtual void preprocess();

  //! Print final information when solve has completed.
  virtual void postprocess();
  
  //! Performs a GSS.
  /*!
    This method performs the following tasks:
    <ul>

    <li> R is first copied to #exchangeList using a deep copy which
    converts external points to appear as though they have no parent.
    This ensures that direction steps sizes are only reduced
    corresponding to points that this object generates.

    <li> Points that have a better objective value but are not
    feasible are automatically removed from the #exchangeList.

    <li> Tags in #queueTags corresponding to evaluated points are
    erased from #queueTags.

    <li> The #exchangeList is then sent to the #gss_iterator which
    will process the newly evaluated points and replace these with
    new points to add to the queue Wnew.

    <li> If a new best point is found, #queueTags is pruned according
    to parameter #maxQueue.  Tags in Wnew are then added to #queueTags.

    <li> The #exchangeList (which is a deep copy of master list of
    evaluated points R, is then pruned and intermediate solve stage
    diagnostics printed.

    </ul>
    
  \see APPSPACK::Citizen::Interface::exchange().
  */
  virtual void exchange(const ConveyorList& R, ConveyorList& Wnew, 
                        const map<string, vector<int> >& ownermap);
  
  //! Derived. Votes according to stand APPS order.
  virtual void vote(const ConveyorList& W, 
                    const map<string, vector<int> >& ownermap,
                    vector<int>& tagOrder);
  
  //! Return citizen's solve state.
  /*!
    While the state of #gss_iterator is APPSPACK::Iterator::Continue, this method
    returns Citizen::PassiveContinue.  Otherwise Citizen::StopRequest is returned.
  */
  virtual Citizen::State getState();
 
  //! Return citizen's name.
  virtual const string& getName() const;

  //! Return best value of x found thus far.
  void getBestX(Vector& x);

  //! Return the best value of f found thus far.
  void getBestVecF(Vector& f);

private:

  //! Print diagnostic information prior to point generation.
  void printPreDiagnostics() const;

  //! Print diagnostic information after point generation.
  void printPostDiagnostics(bool newBestFound) const;

  //! Pop off infeasible points externally generated.
  void popInfeasibleBest();

private:

  //! Parameters
  Parameter::List& params;

  //! Constraints
  const Constraints::Linear& constraints;

  //! Performs a single APPSPACK iteration using reverse communication.
  Iterator gss_iterator;
  
  //! Trial points that have been evaluated evaluated
  List exchangeList;

  //! Optional citizen name.
  string name;

  //! Records tags of points that this currently wishes evaluated.
  /*!  This list is used to vote the order of the queue.  Only points
    in #queueTags will receive a nonzero vote.  

    \note If a point does not receive a vote from any Citizen, it is
    pruned.
  */
  vector<int> queueTags;

  //! How big the queue is allowed to be after a prune (set according to "Max Queue Size" parameter)
  int maxQueue;

  //! Tag of initial point.
  int initialTag;

  //! Debug level.
  int debug;

};

}
}
#endif
