//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file NAPPSPACK_NLSolver.hpp
  \brief Class description for NAPPSPACK::NLSolver
*/

#ifndef NAPPSPACK_NLSOLVER
#define NAPPSPACK_NLSOLVER

#include "APPSPACK_Common.hpp"
#include "APPSPACK_Solver.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_Executor_Interface.hpp"
#include "APPSPACK_Conveyor.hpp"
#include "NAPPSPACK_Method_Generic.hpp"

namespace NAPPSPACK
{

/*! \brief This class is designed to solve nonlinear minimization problems.

NLSovler uses the class APPSPACK::Solver to solve a series of linearly
constrained optimization problems of the form
\f[
\begin{array}{ll}
{\rm minimize} & M(x,\rho_k,\alpha_k) \\
{\rm subject to} & \ell \le A x \le u.
\end{array}
\f]
where \f$ M(x,\rho_k)\f$ denotes a merit function with the property that
\f[
\lim_{\rho_k \to \infty} M(x,\rho_k,\alpha_k) = \infty 
\f]
if a nonlinear constraint is violated.  The merit function \f$M(x,\rho_k,\alpha_k)\f$
is defined a subclass of APPSPACK::Combiner::Generic and the corresponding
method by NAPPSPACK::Method::Generic. \ref mainpage_intro

These parameters are are stored in the Parameter::List that is
passed to the NAPPSPACK Solver constructor. 

<ul>
<li> "Number Equality" (int) number of nonlinear equality constraints.
<li> "Number Inequality" (int) number of nonlinear inequality constraints.
<li> "Constraint Tolerance" (double) maximum constraint violation tolerance.
<li> "Final Step Tolerance" (double) final stopping tolerance for linearly constrainted subproblem.
<li> "Initial Step Tolerance" (double) initial step size for subproblem.
<li> "Minimum Step Tolerance" (double) smallest the step tolerance for a given subproblem.
<li> "Initial Penalty Value" (double) maximum value of the penalty term.
<li> "Maximum Penalty Value" (double) maximum value of the penalty term.
<li> "Maximum Total Evaluations" (int) maximum number of evaluations for all subproblems.
<li> "Maximum Major Evaluations" (int) maximum number of evaluations for a single subproblem.
<li> "Maximum Major Iterations" (int) maximum number of subproblems.
<li> "Initial Smoothing Value" (double) initial value for the smoothing parameter.
<li> "Minimum Smoothing Value" (double) minimum value for the smoothing parameter.
<li> "Method" (string) Can be one of the following: 
<ul>
<li> "Merit Max" Merit function based on \f$ \ell_\infty\f$ norm
<li> "Merit Max Smooth"  Merit function based on smoothed \f$ \ell_\infty\f$ norm
<li> "Merit One" Merit function based on \f$ \ell_1\f$ norm
<li> "Merit One Smooth"  Merit function based on smoothed \f$ \ell_1\f$ norm
<li> "Merit Two" Merit function based on \f$ \ell_2\f$ norm
<li> "Merit Two Smooth"  Merit function based on smoothed \f$ \ell_2\f$ norm
<li> "Merit Two Squared" Merit function based on \f$ \ell_2^2\f$ norm
</ul>
<li> "Smoothing Value Decrease" (double) scale term used to decrease smoothing parameter.
<li> "Step Tolerance Decrease" (double) scale term used to decrease step tolerance.
<li> "Penalty Value Increase" (double) scale term used to increase penalty parameter.
</ul>
*/
class NLSolver
{
public:

  //! State of the nonlinear solver
  enum State {
    SuccessfulExit,		//!< Solver step length converged with accpetable constraint violation.
    MaximumMajors,		//!< Solver exited with maximum major (outer) iterations.
    MaximumEvals                //!< Solver exited with maximum function evaluations.
  };
  
  //! Constructor 
  NLSolver(APPSPACK::Parameter::List& params_in, 
  APPSPACK::Executor::Interface& executor_in, 
	   const APPSPACK::Constraints::Linear& constraints_in);

  //! Destructor
  ~NLSolver();

  //! Return the x-vector corresponding to the best point
  const vector<double>& getBestX() const;
  
  //! Returns instance of class Value associated with current best point.
  const vector<double>& getBestFC() const;
  
  //! Returns total number of points evaluated.
  int getNumEvaluations() const;

  //! Returns total number of cached evaluations.
  int getNumCached() const;

  //! Use APPSPACK to solve linearly constrained subproblem
  void solveSubproblem();

  //! Solve nonlinearly constrained subproblem
  State solve();

private:
  //! Method which defines \f$M(x,\rho_k)\f$ and corresponding update strategies.
  Method::Generic* method;
  //! Stores parameter list to be passed to APPSPACK for each subproblem solve
  APPSPACK::Parameter::List& params;
  //! Stores executor passed to APPSPACK for each subproblem solve
  APPSPACK::Executor::Interface& executor; 
  //! Stores reference to constraints passed to APPSPACK for each subproblem solve
  const APPSPACK::Constraints::Linear& constraints;
  //! Current best point
  APPSPACK::Vector xBest;
  //! Function values corresponding to current best point
  APPSPACK::Vector fcBest;
  //! Stores cache of function values to be 
  APPSPACK::Cache::Manager cache;
  //! Sate of linear solver.
  APPSPACK::Solver::State minorState;
  //! State of nonlinear solver.
  NAPPSPACK::NLSolver::State majorState;

  //! Total number of cached evaluations for all solves.
  int nCached;
  //! Total number of function evaluations for all solves.
  int nEvaluated;
  //! Number of nonlinear equality constraints;
  int nlec; 
  //! Number nonlinear inequality constraints;
  int nlic;
  //! Maximum number of evaluations.
  int maxTotalEvaluations;
  //! Maximum number of evaluations per major iteration.
  int maxMajorEvaluations;
  //! Maximum number of major (or outer) iterations.
  int maxMajorIterations;
};



}
#endif
