//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*! 
  \file NAPPSPACK_Method_Lagrange.cpp
  \brief Implementation of NAPPSPACK::Method::Lagrange
*/

#include "NAPPSPACK_Method_Lagrange.hpp"
#include "APPSPACK_Common.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Cache_Point.hpp"

NAPPSPACK::Method::Lagrange::Lagrange(APPSPACK::Parameter::List& params_in):
  params(params_in),
  methodName("Lagrange")
 {
  // Set initial values.
  double mu0=1;
  maxit=30;

  // Constants specified in Conn, Gould, Sartenaer, Toint paper.
  alpha_eta=.1;
  tau=.5;
  gamma_s=.1;

  w_s=1;
  alpha_w=1;
  beta_w=.6;

  beta_eta=.5;

  eta_s=0.12589;
  etak=.01;

  alphak=min(gamma_s,mu0);
  wk=w_s*pow(alphak,alpha_w);

  // Convergence tolerance.
  etastar=1e-5;
  Deltastar=1e-5;

  //Update combiner.
  combiner.setMu(mu0);
  if (!params.sublist("Nonlinear").isParameter("Constraint Count"))
  {
    cerr << "Must set number of nonlinear constraints\n";
    throw "APPSPACK Error";
  }

  if (params.sublist("Nonlinear").isParameter("Number Inequality"))
  {
    int nlic = 0;;
    nlic = params.sublist("Nonlinear").getParameter("Number Inequality", nlic);
    if (nlic > 0)
    {
      cerr << "Only equality constraints supported: " << nlic 
           << " inequality constraints present." << endl;
      throw "APPSPACK Error";
    }
  }
  
  int nnlc = params.sublist("Nonlinear").getParameter("Constraint Count", 0);
  APPSPACK::Vector initialLambda(nnlc, 0.0);
  if (params.sublist("Nonlinear").isParameter("Initial Lambda"))
    initialLambda = params.sublist("Nonlinear").getVectorParameter("Initial Lambda");
  combiner.setLambda(initialLambda);

  if (nnlc != initialLambda.size())
  {
    cerr << "Lambda must have same size as number of nonlinear constraints\n";
    throw "APPSPACK Error";
  }

  theta_tol=100;
  Deltak=wk/theta();

  
  updateAPPSPACKParams(Deltak);
}

bool NAPPSPACK::Method::Lagrange::isOptimal(const APPSPACK::Vector& fc) const
{
  if (infnorm(fc) < etastar && Deltak < Deltastar)
    return true;
  else
    return false;
}

void NAPPSPACK::Method::Lagrange::printMajor(const APPSPACK::Vector& x, const APPSPACK::Vector& fc) const
{
  cout << "fc = " <<  fc << ", y = " << combiner.getLambda();
  cout << ", x = [";
  for (int i = 0; i < x.size(); i++)
    cout << " " << setprecision(14) << x[i];
  cout << "]';\n";

  cout << "| f = " << combiner(fc) 
       << " | etak = " << etak << " | wk = " << wk << " | alphak = " << alphak
       << " | Deltak = " << Deltak << " | n(ceq) =  " << infnorm(fc)
       << " | mu = " << combiner.getMu() << endl;
}

APPSPACK::Combiner::Lagrange& NAPPSPACK::Method::Lagrange::getCombiner()
{
  return combiner;
}

const string& NAPPSPACK::Method::Lagrange::getName()
{
  return methodName;
}

double NAPPSPACK::Method::Lagrange::theta() const
{
  double t = (1+combiner.getLambda().norm() + 1/combiner.getMu())/theta_tol;
  return  1.0 > t ? 1.0 : t;
}

void NAPPSPACK::Method::Lagrange::updateAPPSPACKParams(double tol)
{
  APPSPACK::Cache::Point::setStaticTolerance(tol/10);
  params.sublist("Solver").setParameter("Bounds Tolerance", tol/2);
  params.sublist("Solver").setParameter("Step Tolerance", tol);
  params.sublist("Solver").setParameter("Minimum Step", 2*tol);
  params.sublist("Solver").setParameter("Cache Comparison Tolerance", 1e-14);
  params.sublist("Solver").setParameter("Maximum Evaluations", 500);
  params.sublist("Solver").setParameter("Debug", 0);  
}

void NAPPSPACK::Method::Lagrange::updateLambda(const APPSPACK::Vector& fc)
{
  APPSPACK::Vector lambda = combiner.getLambda();
  double mu = combiner.getMu();
  for (int k = 0; k < lambda.size(); k++)
  {
      lambda[k] = lambda[k] + fc[k+1]/mu;
  }
  combiner.setLambda(lambda);
}

double NAPPSPACK::Method::Lagrange::infnorm(const APPSPACK::Vector& fc) const
{
  double norm = 0;
  for (int i = 1; i < fc.size(); i++)
    norm += fabs(fc[i]);
  return norm;
}

void NAPPSPACK::Method::Lagrange::update(APPSPACK::Vector& x, APPSPACK::Vector& fc,
					 const APPSPACK::Solver::State& state)
{
  if (infnorm(fc) > etak || state != APPSPACK::Solver::StepConverged) // Increase penalty
  {
    combiner.setMu(tau*combiner.getMu());
    alphak = min(gamma_s, combiner.getMu());
    //    wk     = w_s*pow(alphak, alpha_w);
    wk = .5*wk;
    etak   = eta_s*pow(alphak, alpha_eta);
  }
  else // Update lambda.
  {
    cout << "Lambda update successful" << endl;
    updateLambda(fc);
    alphak = min(gamma_s, combiner.getMu());
    //    wk     = wk*pow(alphak, beta_w);
    etak   = etak*pow(alphak, beta_eta);
  }
  
  // Update step tolerance for next solve. 
  Deltak = wk/theta();
  updateAPPSPACKParams(Deltak);
}
