//@HEADER
// ************************************************************************
// 
//         HOPSPACK: Hybrid Opitmization Parallel Search Package
//               Copyright (2008) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file NAPPSPACK_Method_Generic.hpp
  \brief Class description for NAPPSPACK::Method::Generic
*/

#ifndef NAPPSPACK_METHOD_GENERIC
#define NAPPSPACK_METHOD_GENERIC

#include "APPSPACK_Common.hpp"
#include "APPSPACK_Combiner_Generic.hpp"
#include "APPSPACK_Solver.hpp"

namespace NAPPSPACK
{

/*! \brief Namespace for classes that define an algorithm for solving nonlinearly
  constrained problems.
*/
namespace Method
{
  
/*! \brief Base class for a subclass that defines a strategy for solving nonlinearly
constrained subproblems.

The subproblems have the form
<ul>
<li> Setting up a sequence of linearly constrained subproblems of the form
\f[
\begin{array}{ll}
{\rm minimize} & M(x,\rho_k) \\
{\rm subject to} & \ell \le A x \le u.
\end{array}
\f]
where \f$ M(x,\rho_k)\f$ denotes a merit function with the property that
\f[
\lim_{\rho_k \to \infty} M(x,\rho_k) = \infty 
\f]
if a nonlinear constraint is violated.  The merit function \f$M(x,\rho_k)\f$
is defined a subclass of APPSPACK::Combiner::Generic and the corresponding
method by NAPPSPACK::Method::Generic.
<li> Defines an update strategy for increase \f$\rho_k\f$, modifying any
  other parameters that define the \f$M(x,\rho_k)\f$.
<li> Chooses an  appropriate stopping tolerance \f$\Delta_{\rm tol}\f$ for the next
  subproblem solve.
</ul>
*/
class Generic
{
public:
  //! Sets up merit function for next solve with APPSPACK::Solver.
  virtual void update(APPSPACK::Vector& x, APPSPACK::Vector& fc,
		      const APPSPACK::Solver::State& state) = 0;
  
  //! Returns true if method considers point optimal for user specified tolerances.
  virtual bool isOptimal(const APPSPACK::Vector& fc) const = 0;

  //! Prints summary of last linearly constrained subproblem solve.
  virtual void printMajor(const APPSPACK::Vector& x, 
			  const APPSPACK::Vector& fc) const = 0;

  //! Returns reference to current combiner to define merit function.
  virtual APPSPACK::Combiner::Generic& getCombiner() = 0;

  //! Returns string name of method.
  virtual const string& getName() = 0;

  //! Return inf norm of constraint violation.
  virtual double infnorm(const APPSPACK::Vector& fc) const = 0;

private:
};


}
}
#endif
